document.addEventListener('DOMContentLoaded', function () {

    // --- ELEMENTOS DEL DOM ---
    const projectsTableBody = document.getElementById('projects-table-body');
    const saveProjectBtn = document.getElementById('save-project-btn');
    const addProjectForm = document.getElementById('add-project-form');
    const projectModalElement = document.getElementById('addProjectModal');
    const addProjectModal = new bootstrap.Modal(projectModalElement);
    const projectModalTitle = document.getElementById('addProjectModalLabel');
    const clientIdDropdown = document.getElementById('cliente_id');
    const professionalsCheckboxList = document.getElementById('professionals-checkbox-list');
    const searchInput = document.getElementById('search-input');
    const generateStrategyBtn = document.getElementById('generate-strategy-btn');

    // --- URLs DE LA API ---
    const PROJECTS_API_URL = 'api/projects.php';
    const CLIENTS_API_URL = 'api/clients.php';
    const PROFESSIONALS_API_URL = 'api/professionals.php';

    // --- ESTADO CENTRALIZADO ---
    const appState = {
        clients: [],
        professionals: [],
        projects: []
    };
    let editMode = false;
    let currentProjectId = null;

    // --- FUNCIONES PRINCIPALES ---

    async function initialLoad() {
        try {
            // Cargar todo en paralelo para mayor eficiencia
            const [clientsRes, professionalsRes, projectsRes] = await Promise.all([
                fetch(CLIENTS_API_URL),
                fetch(PROFESSIONALS_API_URL),
                fetch(PROJECTS_API_URL)
            ]);

            if (!clientsRes.ok) throw new Error('Error al cargar clientes.');
            if (!professionalsRes.ok) throw new Error('Error al cargar profesionales.');
            if (!projectsRes.ok) throw new Error('Error al cargar proyectos.');

            appState.clients = await clientsRes.json();
            appState.professionals = await professionalsRes.json();
            appState.projects = await projectsRes.json();

            // Renderizar todo
            renderProjectsTable(appState.projects, appState.clients, appState.professionals);
            populateClientsDropdown(appState.clients);
            populateProfessionalsCheckbox(appState.professionals);

            // Revisar si hay un parámetro de búsqueda en la URL
            const urlParams = new URLSearchParams(window.location.search);
            const searchTerm = urlParams.get('search');
            if (searchTerm) {
                searchInput.value = searchTerm;
                filterProjects();
            }

        } catch (error) {
            console.error('Error en la carga inicial:', error);
            projectsTableBody.innerHTML = `<tr><td colspan="7" class="text-center">${error.message}</td></tr>`;
        }
    }

    async function fetchProjects() {
        try {
            const response = await fetch(PROJECTS_API_URL);
            if (!response.ok) throw new Error('Error al obtener los proyectos.');
            appState.projects = await response.json();
            renderProjectsTable(appState.projects, appState.clients, appState.professionals);
        } catch (error) {
            console.error('Error:', error);
            projectsTableBody.innerHTML = `<tr><td colspan="7" class="text-center">${error.message}</td></tr>`;
        }
    }

    async function saveProject() {
        // Recolectar IDs de los profesionales seleccionados
        const selectedProfessionals = Array.from(professionalsCheckboxList.querySelectorAll('input[type="checkbox"]:checked'))
            .map(cb => parseInt(cb.value));

        const projectData = {
            nombre_proyecto: document.getElementById('nombre_proyecto').value,
            cliente_id: document.getElementById('cliente_id').value,
            descripcion: document.getElementById('descripcion').value,
            fecha_inicio: document.getElementById('fecha_inicio').value,
            fecha_fin_estimada: document.getElementById('fecha_fin_estimada').value,
            tipo_marketing: document.getElementById('tipo_marketing').value,
            estado: document.getElementById('estado_proyecto').value,
            profesionales_asignados: selectedProfessionals,
            estrategia_estrategico: document.getElementById('estrategia_estrategico').value,
            estrategia_tactico: document.getElementById('estrategia_tactico').value,
            estrategia_operativo: document.getElementById('estrategia_operativo').value
        };

        const url = editMode ? `${PROJECTS_API_URL}?id=${currentProjectId}` : PROJECTS_API_URL;
        const method = editMode ? 'PUT' : 'POST';

        try {
            const response = await fetch(url, { method, headers: { 'Content-Type': 'application/json' }, body: JSON.stringify(projectData) });
            if (!response.ok) {
                const errorData = await response.json();
                throw new Error(errorData.error || `Error al ${editMode ? 'actualizar' : 'crear'} el proyecto.`);
            }
            addProjectModal.hide();
            fetchProjects(); // Recargar solo los proyectos
        } catch (error) {
            console.error('Error:', error); alert(error.message);
        }
    }

    async function deleteProject(id) {
        try {
            const response = await fetch(`${PROJECTS_API_URL}?id=${id}`, { method: 'DELETE' });
            if (!response.ok) throw new Error('Error al eliminar el proyecto.');
            fetchProjects();
        } catch (error) {
            console.error('Error:', error); alert(error.message);
        }
    }

    // --- FUNCIONES DE RENDERIZADO Y UI ---

    function renderProjectsTable(projects, clients, professionals) {
        projectsTableBody.innerHTML = '';
        if (!projects || projects.length === 0) {
            projectsTableBody.innerHTML = '<tr><td colspan="7" class="text-center">No hay proyectos registrados.</td></tr>';
            return;
        }
        const clientMap = new Map(clients.map(c => [c.id, c.nombre_empresa]));
        const professionalMap = new Map(professionals.map(p => [p.id, p.nombre_completo]));

        const formatDate = (dateString) => {
            if (!dateString) return 'N/A';
            const [year, month, day] = dateString.split('-');
            if (!year || !month || !day) return dateString;
            return `${day}/${month}/${year}`;
        };

        projects.forEach(project => {
            const row = document.createElement('tr');
            const statusBadge = `<span class="badge bg-info text-dark">${project.estado}</span>`;
            
            // Obtener nombres de profesionales asignados
            const assignedNames = project.profesionales_asignados
                .map(id => professionalMap.get(parseInt(id)))
                .filter(Boolean) // Filtrar por si algún profesional fue eliminado
                .join(', ');

            row.innerHTML = `
                <td>${project.id}</td>
                <td><a href="project-detail.html?id=${project.id}">${project.nombre_proyecto}</a></td>
                <td>${clientMap.get(parseInt(project.cliente_id)) || 'N/A'}</td>
                <td>${formatDate(project.fecha_inicio)}</td>
                <td>${assignedNames || '<span class="text-muted">Sin asignar</span>'}</td>
                <td>${statusBadge}</td>
                <td>
                    <button class="btn btn-sm btn-outline-secondary edit-project-btn" data-id="${project.id}" title="Editar"><i class="bi bi-pencil-fill"></i></button>
                    <button class="btn btn-sm btn-outline-danger delete-project-btn" data-id="${project.id}" title="Eliminar"><i class="bi bi-trash-fill"></i></button>
                </td>
            `;
            projectsTableBody.appendChild(row);
        });
    }

    function populateClientsDropdown(clients) {
        clientIdDropdown.innerHTML = '<option value="">Seleccione un cliente</option>';
        clients.forEach(client => {
            if (client.estado === 'Activo') {
                const option = document.createElement('option');
                option.value = client.id;
                option.textContent = client.nombre_empresa;
                clientIdDropdown.appendChild(option);
            }
        });
    }

    function populateProfessionalsCheckbox(professionals) {
        professionalsCheckboxList.innerHTML = '';
        professionals.forEach(prof => {
            if (prof.estado === 'Activo') {
                const div = document.createElement('div');
                div.classList.add('form-check', 'form-check-inline');
                div.innerHTML = `
                    <input class="form-check-input" type="checkbox" value="${prof.id}" id="prof-${prof.id}">
                    <label class="form-check-label" for="prof-${prof.id}">${prof.nombre_completo}</label>
                `;
                professionalsCheckboxList.appendChild(div);
            }
        });
    }

    async function handleEditClick(id) {
        const project = appState.projects.find(p => p.id === id);
        if (!project) return;

        editMode = true;
        currentProjectId = id;
        projectModalTitle.textContent = 'Editar Proyecto';

        // Llenar el formulario
        document.getElementById('nombre_proyecto').value = project.nombre_proyecto;
        document.getElementById('cliente_id').value = project.cliente_id;
        document.getElementById('descripcion').value = project.descripcion;
        document.getElementById('fecha_inicio').value = project.fecha_inicio;
        document.getElementById('fecha_fin_estimada').value = project.fecha_fin_estimada;
        document.getElementById('tipo_marketing').value = project.tipo_marketing;
        document.getElementById('estado_proyecto').value = project.estado;

        // Llenar la pestaña de estrategia
        document.getElementById('estrategia_estrategico').value = project.estrategia_estrategico || '';
        document.getElementById('estrategia_tactico').value = project.estrategia_tactico || '';
        document.getElementById('estrategia_operativo').value = project.estrategia_operativo || '';

        // Marcar los checkboxes de profesionales
        professionalsCheckboxList.querySelectorAll('input[type="checkbox"]').forEach(cb => {
            cb.checked = project.profesionales_asignados.includes(parseInt(cb.value));
        });

        addProjectModal.show();
    }

    function prepareAddModal() {
        editMode = false;
        currentProjectId = null;
        projectModalTitle.textContent = 'Abrir Nuevo Proyecto';
        addProjectForm.reset();
        // Desmarcar todos los checkboxes
        professionalsCheckboxList.querySelectorAll('input[type="checkbox"]').forEach(cb => cb.checked = false);
    }

    function filterProjects() {
        const searchTerm = searchInput.value.toLowerCase();
        const clientMap = new Map(appState.clients.map(c => [c.id, c.nombre_empresa.toLowerCase()]));

        const filteredProjects = appState.projects.filter(project => {
            const projectName = project.nombre_proyecto.toLowerCase();
            const clientName = clientMap.get(parseInt(project.cliente_id)) || '';
            
            return projectName.includes(searchTerm) || clientName.includes(searchTerm);
        });

        renderProjectsTable(filteredProjects, appState.clients, appState.professionals);
    }

    async function handleGenerateWithAIClick() {
        // 1. Recopilar datos del formulario
        const clientId = parseInt(document.getElementById('cliente_id').value);
        const client = appState.clients.find(c => c.id === clientId);

        const context = {
            clientName: client ? client.nombre_empresa : 'un cliente',
            projectName: document.getElementById('nombre_proyecto').value,
            projectDescription: document.getElementById('descripcion').value,
            marketingType: document.getElementById('tipo_marketing').value
        };

        if (!context.projectDescription) {
            alert("Por favor, ingrese la descripción del proyecto (necesidades y objetivos) antes de generar la estrategia.");
            return;
        }

        // 2. Mostrar un indicador de carga
        generateStrategyBtn.disabled = true;
        generateStrategyBtn.innerHTML = '<span class="spinner-border spinner-border-sm" role="status" aria-hidden="true"></span> Generando...';

        try {
            // 3. Llamar a nuestro nuevo endpoint de IA
            const response = await fetch('api/ai_handler.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    task_type: 'generate_strategy',
                    context: context
                })
            });
            if (!response.ok) {
                const errorData = await response.json();
                const errorMessage = errorData.error || 'La IA no pudo generar una respuesta.';
                throw new Error(errorMessage);
            }

            const strategy = await response.json();

            // 4. Poblar los textareas con la respuesta
            document.getElementById('estrategia_estrategico').value = strategy.estrategico || '';
            document.getElementById('estrategia_tactico').value = strategy.tactico || '';
            document.getElementById('estrategia_operativo').value = strategy.operativo || '';

        } catch (error) {
            console.error("Error con la IA:", error);
            alert(error.message);
        } finally {
            // 5. Restaurar el botón
            generateStrategyBtn.disabled = false;
            generateStrategyBtn.innerHTML = '<i class="bi bi-robot"></i> Generar con IA';
        }
    }

    // --- EVENT LISTENERS ---

    projectsTableBody.addEventListener('click', function(event) {
        const button = event.target.closest('.edit-project-btn, .delete-project-btn');
        if (!button) return;

        const id = parseInt(button.dataset.id);

        if (button.classList.contains('delete-project-btn')) {
            if (confirm(`¿Estás seguro de que quieres eliminar el proyecto con ID ${id}?`)) deleteProject(id);
        }
        if (button.classList.contains('edit-project-btn')) {
            handleEditClick(id);
        }
    });

    searchInput.addEventListener('input', filterProjects);

    generateStrategyBtn.addEventListener('click', handleGenerateWithAIClick);

    saveProjectBtn.addEventListener('click', saveProject);

    document.querySelector('[data-bs-target="#addProjectModal"]').addEventListener('click', prepareAddModal);

    projectModalElement.addEventListener('hidden.bs.modal', prepareAddModal);

    // --- CARGA INICIAL ---
    initialLoad();
});