document.addEventListener('DOMContentLoaded', function () {

    // --- ELEMENTOS DEL DOM ---
    const projectNameTitle = document.getElementById('project-name-title');
    const projectDetailsContainer = document.getElementById('project-details-container');
    const tasksTableBody = document.getElementById('tasks-table-body');
    const saveTaskBtn = document.getElementById('save-task-btn');
    const kanbanBoard = document.getElementById('kanban-board');
    const taskDetailModalEl = document.getElementById('taskDetailModal');
    const taskDetailModal = new bootstrap.Modal(taskDetailModalEl);
    const taskDetailModalLabel = document.getElementById('taskDetailModalLabel');
    const taskDetailContent = document.getElementById('task-detail-content');
    const commentList = document.getElementById('comment-list');
    const postCommentBtn = document.getElementById('post-comment-btn');
    const completeTaskBtn = document.getElementById('complete-task-btn');
    const tagList = document.getElementById('tag-list');
    const addTagBtn = document.getElementById('add-tag-btn');
    const fileList = document.getElementById('file-list');
    const uploadFileBtn = document.getElementById('upload-file-btn');
    const linkList = document.getElementById('link-list');
    const addLinkBtn = document.getElementById('add-link-btn');
    const tagFilterDropdown = document.getElementById('tag-filter');
    const priorityFilterDropdown = document.getElementById('priority-filter');
    const professionalFilterDropdown = document.getElementById('professional-filter');
    const budgetSummary = document.getElementById('budget-summary');
    const expenseTableBody = document.getElementById('expense-table-body');
    const saveBudgetItemBtn = document.getElementById('save-budget-item-btn');
    const ganttChartContainer = document.getElementById('gantt-chart-container');

    // Nuevos elementos para Grillas y Planes de Medios
    const gridsContainer = document.getElementById('grids-container');
    const mediaPlansContainer = document.getElementById('media-plans-container');
    const addGridModalEl = document.getElementById('addGridModal');
    const addGridModal = new bootstrap.Modal(addGridModalEl);
    const saveGridBtn = document.getElementById('save-grid-btn');
    const addMediaPlanModalEl = document.getElementById('addMediaPlanModal');
    const addMediaPlanModal = new bootstrap.Modal(addMediaPlanModalEl);
    const saveMediaPlanBtn = document.getElementById('save-media-plan-btn');

    const addTaskModalEl = document.getElementById('addTaskModal');
    const assignedToDropdown = document.getElementById('asignado_a');
    let addTaskModal; // La declaramos aquí para que sea accesible en todo el script

    // --- URLs Y ESTADO ---
    const urlParams = new URLSearchParams(window.location.search);
    const projectId = urlParams.get('id');

    const PROJECTS_API_URL = 'api/projects.php';
    const PROFESSIONALS_API_URL = 'api/professionals.php';
    const TASKS_API_URL = 'api/tasks.php';
    const GRIDS_API_URL = 'api/grids.php';
    const MEDIA_PLANS_API_URL = 'api/media_plans.php';


    let appState = {
        project: null,
        professionals: [],
        tasks: [],
        grids: [],
        mediaPlans: [],
        // Simulación de usuario logueado. En una app real, esto vendría del servidor.
        currentUser: { id: 1, nombre_completo: 'Director de Cuentas' },
        currentOpenTaskId: null,
        currentProfessionalFilter: 'all',
        currentPriorityFilter: 'all',
        currentTagFilter: 'all',
        taskEditMode: false,
        currentTaskId: null,
        ganttChart: null
    };

    /**
     * Formatea una fecha en formato YYYY-MM-DD a DD/MM/YYYY.
     * @param {string} dateString La fecha en formato YYYY-MM-DD.
     * @returns {string} La fecha formateada o 'N/A' si la entrada es inválida.
     */
    const formatDate = (dateString) => {
        if (!dateString) return 'N/A';
        const [year, month, day] = dateString.split('-');
        if (!year || !month || !day) return dateString;
        return `${day}/${month}/${year}`;
    };

    // --- FUNCIONES PRINCIPALES ---

    async function initialLoad() {
        if (!projectId) {
            projectNameTitle.textContent = "Proyecto no encontrado";
            return;
        }

        // Inicializamos todos los modales y elementos relacionados aquí, dentro de initialLoad
        if (document.getElementById('addBudgetItemModal')) {
            new bootstrap.Modal(document.getElementById('addBudgetItemModal'));
        }
        addTaskModal = new bootstrap.Modal(addTaskModalEl);

        try {
            const [projectRes, professionalsRes, tasksRes, gridsRes, mediaPlansRes] = await Promise.all([
                fetch(`${PROJECTS_API_URL}?id=${projectId}`),
                fetch(PROFESSIONALS_API_URL),
                fetch(`${TASKS_API_URL}?project_id=${projectId}`),
                fetch(`${GRIDS_API_URL}?project_id=${projectId}`),
                fetch(`${MEDIA_PLANS_API_URL}?project_id=${projectId}`),
            ]);

            if (!projectRes.ok) throw new Error('No se pudo cargar el proyecto.');
            if (!professionalsRes.ok) throw new Error('No se pudieron cargar los profesionales.');
            if (!tasksRes.ok) throw new Error('No se pudieron cargar las tareas.');
            if (!gridsRes.ok) throw new Error('No se pudieron cargar las grillas de contenido.');
            if (!mediaPlansRes.ok) throw new Error('No se pudieron cargar los planes de medios.');

            appState.project = await projectRes.json();
            appState.professionals = await professionalsRes.json();
            appState.tasks = await tasksRes.json();
            appState.grids = await gridsRes.json();
            appState.mediaPlans = await mediaPlansRes.json();

            renderProjectDetails();
            renderBudgetSection();
            populateProfessionalsDropdown();
            populateProfessionalsFilterDropdown();
            renderTasksTable();
            populateTagFilterDropdown();
            renderGridsSection();
            renderMediaPlansSection();
            // El gantt se renderizará cuando se haga clic en su pestaña
            populateTagsDatalist();

        } catch (error) {
            console.error("Error en la carga inicial:", error);
            if (projectDetailsContainer) projectDetailsContainer.innerHTML = `<div class="alert alert-danger">${error.message}</div>`;
        }
    }

    async function saveTask() {
        const taskData = {
            project_id: projectId,
            nombre_tarea: document.getElementById('nombre_tarea').value,
            asignado_a: parseInt(document.getElementById('asignado_a').value) || null,
            prioridad: document.getElementById('prioridad').value,
            fecha_inicio: document.getElementById('fecha_inicio_tarea').value,
            fecha_fin: document.getElementById('fecha_fin_tarea').value,
            fecha_fin_real: document.getElementById('fecha_fin_real').value || null,
            porcentaje: document.getElementById('porcentaje').value
        };

        if (!taskData.nombre_tarea) {
            alert('El nombre de la tarea es obligatorio.');
            return;
        }

        const url = appState.taskEditMode ? `${TASKS_API_URL}?id=${appState.currentTaskId}` : TASKS_API_URL;
        const method = appState.taskEditMode ? 'PUT' : 'POST';

        try {
            const response = await fetch(url, {
                method: method,
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(taskData)
            });
            if (!response.ok) throw new Error('Error al guardar la tarea.');

            addTaskModal.hide();
            document.getElementById('add-task-form').reset();
            fetchTasks();
        } catch (error) {
            console.error("Error al guardar tarea:", error);
            alert(error.message);
        }
    }

    async function saveGrid() {
        const gridData = {
            project_id: projectId,
            name: document.getElementById('grid-name').value,
            start_date: document.getElementById('grid-start-date').value,
            end_date: document.getElementById('grid-end-date').value,
            networks: Array.from(document.querySelectorAll('#grid-networks input:checked')).map(el => el.value),
            responsables: Array.from(document.querySelectorAll('#grid-responsables option:checked')).map(el => parseInt(el.value))
        };

        if (!gridData.name || !gridData.start_date || !gridData.end_date) {
            alert('Nombre y fechas son obligatorios para la grilla.');
            return;
        }

        try {
            const response = await fetch(GRIDS_API_URL, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(gridData)
            });
            if (!response.ok) throw new Error('Error al guardar la grilla.');

            addGridModal.hide();
            document.getElementById('add-grid-form').reset();
            await fetchGrids();
        } catch (error) {
            console.error("Error al guardar grilla:", error);
            alert(error.message);
        }
    }

    async function saveMediaPlan() {
        const planData = {
            project_id: projectId,
            name: document.getElementById('plan-name').value,
            start_date: document.getElementById('plan-start-date').value,
            end_date: document.getElementById('plan-end-date').value,
            budget: parseFloat(document.getElementById('plan-budget').value) || 0,
            objective: document.getElementById('plan-objective').value,
            audience: document.getElementById('plan-audience').value,
            responsables: Array.from(document.querySelectorAll('#plan-responsables option:checked')).map(el => parseInt(el.value))
        };

        if (!planData.name || !planData.start_date || !planData.end_date) {
            alert('Nombre y fechas son obligatorios para el plan.');
            return;
        }

        try {
            const response = await fetch(MEDIA_PLANS_API_URL, { method: 'POST', headers: { 'Content-Type': 'application/json' }, body: JSON.stringify(planData) });
            if (!response.ok) throw new Error('Error al guardar el plan de medios.');
            addMediaPlanModal.hide();
            document.getElementById('add-media-plan-form').reset(); // Corregido el ID del formulario
            await fetchMediaPlans();
        } catch (error) {
            console.error("Error al guardar plan de medios:", error);
            alert(error.message);
        }
    }

    async function addLink() {
        const linkNameInput = document.getElementById('new-link-name');
        const linkUrlInput = document.getElementById('new-link-url');
        const name = linkNameInput.value.trim();
        const url = linkUrlInput.value.trim();

        if (!name || !url) {
            alert('Por favor, ingrese un nombre y una URL para el enlace.');
            return;
        }

        try {
            const response = await fetch(`${TASKS_API_URL}?id=${appState.currentOpenTaskId}`, {
                method: 'PUT',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ new_link: { name, url } })
            });
            if (!response.ok) throw new Error('No se pudo añadir el enlace.');

            linkNameInput.value = '';
            linkUrlInput.value = '';
            await fetchTasks();
            const updatedTask = appState.tasks.find(t => t.id === appState.currentOpenTaskId);
            renderLinks(updatedTask);
        } catch (error) {
            console.error("Error al añadir enlace:", error);
            alert(error.message);
        }
    }

    async function saveBudgetItem() {
        const description = document.getElementById('budget-item-description').value;
        const amount = parseFloat(document.getElementById('budget-item-amount').value);
        const date = document.getElementById('budget-item-date').value;

        if (!description || isNaN(amount) || !date) {
            alert('Todos los campos son requeridos.');
            return;
        }

        const budgetItem = {
            id: `mov_${Date.now()}`,
            description,
            amount,
            date
        };

        const updateData = { new_egreso: budgetItem };
        // Esta lógica está desactualizada. Los egresos ahora se manejan en la tabla `expenses`.
        // La dejaremos así por ahora, pero debería migrarse a su propio endpoint.
        
        try {
            const response = await fetch(`${PROJECTS_API_URL}?id=${projectId}`, {
                method: 'PUT',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(updateData)
            });
            if (!response.ok) throw new Error('No se pudo guardar el movimiento.');

            const budgetModalInstance = bootstrap.Modal.getInstance(document.getElementById('addBudgetItemModal'));
            if (budgetModalInstance) budgetModalInstance.hide();

            document.getElementById('add-budget-item-form').reset();
            await initialLoad(); // Recargar todo para actualizar el presupuesto y el proyecto
        } catch (error) {
            console.error("Error al guardar movimiento:", error);
            alert(error.message);
        }
    }

    async function addComment() {
        const text = document.getElementById('new-comment-text').value;
        if (!text.trim()) return;

        const commentData = {
            text: text,
            author_id: appState.currentUser.id,
            author_name: appState.currentUser.nombre_completo
        };

        try {
            const response = await fetch(`${TASKS_API_URL}?action=add_comment&id=${appState.currentOpenTaskId}`, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(commentData)
            });

            if (!response.ok) throw new Error('No se pudo publicar el comentario.');

            document.getElementById('new-comment-text').value = '';
            await fetchTasks(); // Recargar tareas para actualizar el estado
            
            const updatedTask = appState.tasks.find(t => t.id === appState.currentOpenTaskId);
            if (updatedTask) {
                renderComments(updatedTask);
            }

        } catch (error) {
            console.error("Error al añadir comentario:", error);
            alert(error.message);
        }
    }

    async function updateTaskProgress(taskId, newStatus) {
        let newPercentage;
        if (newStatus === 'Por Hacer') {
            newPercentage = 0;
        } else if (newStatus === 'En Progreso') {
            newPercentage = 50; // Un valor intermedio por defecto
        } else if (newStatus === 'Hecho') {
            newPercentage = 100;
        } else {
            return; // No hacer nada si el estado no es válido
        }

        try {
            const response = await fetch(`${TASKS_API_URL}?id=${taskId}`, {
                method: 'PUT',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ porcentaje: newPercentage })
            });
            if (!response.ok) throw new Error('No se pudo actualizar la tarea.');
            await fetchTasks(); // Recargar tareas para actualizar el estado
        } catch (error) {
            console.error("Error al actualizar la tarea:", error);
            alert(error.message);
        }
    }

    async function addTag() {
        const newTagInput = document.getElementById('new-tag-input');
        const tagText = newTagInput.value.trim();
        if (!tagText) return;

        try {
            const response = await fetch(`${TASKS_API_URL}?id=${appState.currentOpenTaskId}`, {
                method: 'PUT',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ new_tag: tagText })
            });
            if (!response.ok) throw new Error('No se pudo añadir la etiqueta.');

            newTagInput.value = '';
            await fetchTasks();
            const updatedTask = appState.tasks.find(t => t.id === appState.currentOpenTaskId);
            renderTags(updatedTask);
        } catch (error) {
            console.error("Error al añadir etiqueta:", error);
            alert(error.message);
        }
    }

    async function uploadFile() {
        const fileInput = document.getElementById('file-input');
        if (fileInput.files.length === 0) {
            alert('Por favor, seleccione un archivo para subir.');
            return;
        }

        const file = fileInput.files[0];
        const formData = new FormData();
        formData.append('file', file);

        try {
            // 1. Subir el archivo al servidor
            const uploadResponse = await fetch('api/upload.php', {
                method: 'POST',
                body: formData
            });
            if (!uploadResponse.ok) throw new Error('Error en la subida del archivo.');
            const uploadResult = await uploadResponse.json();

            // 2. Asociar el archivo a la tarea
            const updateResponse = await fetch(`${TASKS_API_URL}?id=${appState.currentOpenTaskId}`, {
                method: 'PUT',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ new_file: { path: uploadResult.filePath, name: uploadResult.fileName } })
            });
            if (!updateResponse.ok) throw new Error('No se pudo asociar el archivo a la tarea.');

            fileInput.value = ''; // Limpiar el input
            await fetchTasks();
            const updatedTask = appState.tasks.find(t => t.id === appState.currentOpenTaskId);
            renderFiles(updatedTask);

        } catch (error) {
            console.error("Error al subir archivo:", error);
            alert(error.message);
        }
    }

    async function completeTask() {
        if (!appState.currentOpenTaskId) return;

        if (!confirm('¿Estás seguro de que quieres marcar esta tarea como completada? Se establecerá el progreso al 100% y la fecha de finalización a hoy.')) return;

        const today = new Date().toISOString().split('T')[0]; // Formato YYYY-MM-DD

        try {
            const response = await fetch(`${TASKS_API_URL}?id=${appState.currentOpenTaskId}`, {
                method: 'PUT',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ porcentaje: 100, fecha_fin_real: today })
            });
            if (!response.ok) throw new Error('No se pudo completar la tarea.');
            
            await fetchTasks();
            taskDetailModal.hide();
        } catch (error) {
            console.error("Error al completar la tarea:", error);
            alert(error.message);
        }
    }

    function openTaskDetailModal(taskId) {
        appState.currentOpenTaskId = taskId;
        const task = appState.tasks.find(t => t.id === taskId);
        if (!task) return;

        const professionalMap = new Map(appState.professionals.map(p => [parseInt(p.id), p.nombre_completo]));
        const assignedName = professionalMap.get(task.asignado_a) || 'N/A';

        taskDetailModalLabel.textContent = task.nombre_tarea;
        taskDetailContent.innerHTML = `<p><strong>Asignado a:</strong> ${assignedName}</p><p><strong>Progreso:</strong> ${task.porcentaje}%</p><p><strong>Fecha Fin Real:</strong> ${task.fecha_fin_real || 'Pendiente'}</p>`;

        // Mostrar u ocultar el botón de completar
        if (task.fecha_fin_real) {
            completeTaskBtn.style.display = 'none';
        } else {
            completeTaskBtn.style.display = 'block';
        }

        renderComments(task);
        renderTags(task);
        renderFiles(task);
        renderLinks(task);
        taskDetailModal.show();
    }

    function prepareAddTaskModal() {
        appState.taskEditMode = false;
        appState.currentTaskId = null;
        document.getElementById('addTaskModalLabel').textContent = 'Añadir Nueva Tarea';
        document.getElementById('add-task-form').reset();
    }

    async function handleTaskEditClick(taskId) {
        const task = appState.tasks.find(t => t.id === taskId);
        if (!task) return;

        appState.taskEditMode = true;
        appState.currentTaskId = taskId;

        document.getElementById('addTaskModalLabel').textContent = 'Editar Tarea';
        document.getElementById('nombre_tarea').value = task.nombre_tarea;
        document.getElementById('asignado_a').value = task.asignado_a;
        document.getElementById('prioridad').value = task.prioridad || 'Media';
        document.getElementById('fecha_inicio_tarea').value = task.fecha_inicio;
        document.getElementById('fecha_fin_tarea').value = task.fecha_fin;
        document.getElementById('fecha_fin_real').value = task.fecha_fin_real;
        document.getElementById('porcentaje').value = task.porcentaje;

        addTaskModal.show();
    }

    function getFilteredTasks() {
        const { tasks, currentTagFilter, currentPriorityFilter, currentProfessionalFilter } = appState;
        if (!tasks) return [];

        return tasks.filter(task => {
            const tagMatch = currentTagFilter === 'all' || (task.tags && task.tags.includes(currentTagFilter));
            const priorityMatch = currentPriorityFilter === 'all' || (task.prioridad || 'Media') === currentPriorityFilter;
            const professionalMatch = currentProfessionalFilter === 'all' || task.asignado_a === parseInt(currentProfessionalFilter);

            return tagMatch && priorityMatch && professionalMatch;
        });
    }

    async function fetchTasks() {
        const tasksRes = await fetch(`${TASKS_API_URL}?project_id=${projectId}`);
        appState.tasks = await tasksRes.json();
        // No es necesario repoblar todos los filtros, solo los de etiquetas que son dinámicos y pueden cambiar
        populateTagFilterDropdown();
        populateTagsDatalist();

        // Renderizar todas las vistas
        renderTasksTable();
        renderKanbanBoard();
        renderGanttChart(); // Actualizar el gantt si está visible
    }

    async function fetchGrids() {
        const gridsRes = await fetch(`${GRIDS_API_URL}?project_id=${projectId}`);
        appState.grids = await gridsRes.json();
        renderGridsSection();
    }

    async function fetchMediaPlans() {
        const mediaPlansRes = await fetch(`${MEDIA_PLANS_API_URL}?project_id=${projectId}`);
        appState.mediaPlans = await mediaPlansRes.json();
        renderMediaPlansSection();
    }


    async function deleteTask(taskId) {
        if (!confirm('¿Estás seguro de que quieres eliminar esta tarea? Esta acción no se puede deshacer.')) {
            return;
        }

        try {
            const response = await fetch(`${TASKS_API_URL}?id=${taskId}`, {
                method: 'DELETE'
            });

            if (!response.ok && response.status !== 204) { // 204 No Content es una respuesta exitosa para DELETE
                const errorData = await response.json();
                throw new Error(errorData.error || 'Error al eliminar la tarea.');
            } // Si es 204, no hay cuerpo de respuesta, no intentar json()
            await fetchTasks(); // Recargar las tareas para reflejar la eliminación
        } catch (error) {
            console.error('Error al eliminar la tarea:', error);
            alert(error.message);
        }
    }
    // --- FUNCIONES DE RENDERIZADO ---

    function renderProjectDetails() {
        const project = appState.project;
        if (projectNameTitle) projectNameTitle.textContent = project.nombre_proyecto;
        if (projectDetailsContainer) projectDetailsContainer.innerHTML = `
            <p><strong>Descripción:</strong> ${project.descripcion || 'Sin descripción'}</p>
            <p><strong>Fechas:</strong> ${formatDate(project.fecha_inicio)} a ${formatDate(project.fecha_fin_estimada)}</p>
            <p><strong>Estado:</strong> <span class="badge bg-info text-dark">${project.estado}</span></p>
        `;
    }

    function renderGridsSection() {
        if (!gridsContainer) return;
        gridsContainer.innerHTML = '';

        if (appState.grids.length === 0) {
            gridsContainer.innerHTML = '<p class="text-muted">No hay grillas de contenido para este proyecto.</p>';
            return;
        }

        const professionalMap = new Map(appState.professionals.map(p => [p.id, p.nombre_completo]));

        appState.grids.forEach(grid => {
            const responsablesNames = (grid.responsables || [])
                .map(id => professionalMap.get(parseInt(id)) || 'Desconocido')
                .join(', ');

            const card = `
                <div class="col-md-6 col-lg-4 mb-4">
                    <div class="card h-100">
                        <div class="card-body d-flex flex-column">
                            <h5 class="card-title">${grid.name}</h5>
                            <h6 class="card-subtitle mb-2 text-muted">${formatDate(grid.start_date)} - ${formatDate(grid.end_date)}</h6>
                            <p class="card-text">
                                <strong>Redes:</strong> ${(grid.networks || []).join(', ') || 'N/A'}<br>
                                <strong>Responsables:</strong> ${responsablesNames || 'N/A'}
                            </p>
                        </div>
                        <div class="card-footer text-end">
                            <a href="grid-editor.html?grid_id=${grid.id}" class="btn btn-primary btn-sm">Abrir Editor</a>
                        </div>
                    </div>
                </div>
            `;
            gridsContainer.innerHTML += card;
        });
    }

    function renderMediaPlansSection() {
        if (!mediaPlansContainer) return;
        mediaPlansContainer.innerHTML = '';

        if (appState.mediaPlans.length === 0) {
            mediaPlansContainer.innerHTML = '<p class="text-muted">No hay planes de medios para este proyecto.</p>';
            return;
        }

        const professionalMap = new Map(appState.professionals.map(p => [p.id, p.nombre_completo]));

        appState.mediaPlans.forEach(plan => {
            const responsablesNames = (plan.responsables || [])
                .map(id => professionalMap.get(parseInt(id)) || 'Desconocido')
                .join(', ');

            const card = `
                <div class="col-md-6 col-lg-4 mb-4">
                    <div class="card h-100 d-flex flex-column">
                        <div class="card-body">
                            <h5 class="card-title">${plan.name}</h5>
                            <h6 class="card-subtitle mb-2 text-muted">${formatDate(plan.start_date)} - ${formatDate(plan.end_date)}</h6>
                            <p class="card-text">
                                <strong>Objetivo:</strong> ${plan.objective || 'N/A'}<br>
                                <strong>Presupuesto:</strong> ${new Intl.NumberFormat('es-PY', { style: 'currency', currency: 'PYG' }).format(plan.budget || 0)}<br>
                                <strong>Responsables:</strong> ${responsablesNames || 'N/A'}
                            </p>
                        </div>
                        <div class="card-footer text-end">
                            <a href="media-plan-editor.html?plan_id=${plan.id}" class="btn btn-primary btn-sm">Abrir Editor</a>
                        </div>
                    </div>
                </div>
            `;
            mediaPlansContainer.innerHTML += card;
        });
    }


    function renderBudgetSection() {
        const { project } = appState;
        if (!project) return;

        const formatCurrency = (value) => new Intl.NumberFormat('es-PY', { style: 'currency', currency: 'PYG' }).format(value);

        const totalIngresos = (project.ingresos || []).reduce((sum, item) => sum + parseFloat(item.amount), 0);
        const totalEgresos = (project.egresos || []).reduce((sum, item) => sum + parseFloat(item.amount), 0);

        if (budgetSummary) {
            budgetSummary.innerHTML = `
            <div class="d-flex justify-content-end">
                 <div class="card card-body text-center">
                    <h6 class="card-title mb-0">Total Egresos</h6>
                    <p class="card-text fs-4 text-danger mb-0">${formatCurrency(totalEgresos)}</p>
                </div>
            </div>
            <div class="d-grid">
                <button class="btn btn-sm btn-outline-success mt-3" data-bs-toggle="modal" data-bs-target="#addBudgetItemModal">
                    <i class="bi bi-plus-circle"></i> Registrar Egreso
                </button>
            </div>
        `;
        }

        if (expenseTableBody) {
            expenseTableBody.innerHTML = (project.egresos || []).map(item => `
            <tr>
                <td>${item.description}</td>
                <td>${formatDate(item.date)}</td>
                <td class="text-end text-danger">${formatCurrency(item.amount)}</td>
            </tr>
        `).join('');
        }
    }

    function renderTasksTable() {
        tasksTableBody.innerHTML = '';
        const tasksToRender = getFilteredTasks();
        if (!tasksToRender) return;

        const professionalMap = new Map(appState.professionals.map(p => [parseInt(p.id), p.nombre_completo]));
        
        if (tasksTableBody && tasksToRender.length === 0) {
            tasksTableBody.innerHTML = `<tr><td colspan="8" class="text-center">No se encontraron tareas con los filtros seleccionados.</td></tr>`;
            return;
        }

        tasksToRender.forEach(task => {
            const row = document.createElement('tr');
            const assignedName = professionalMap.get(parseInt(task.asignado_a)) || '<span class="text-muted">N/A</span>';
            
            let priorityBadge;
            switch (task.prioridad) {
                case 'Alta':
                    priorityBadge = '<span class="badge bg-danger">Alta</span>';
                    break;
                case 'Baja':
                    priorityBadge = '<span class="badge bg-secondary">Baja</span>';
                    break;
                default:
                    priorityBadge = '<span class="badge bg-primary">Media</span>';
            }

            row.innerHTML = `
                <td><a href="#" class="task-detail-link" data-id="${task.id}">${task.nombre_tarea}</a></td>
                <td>${assignedName}</td>
                <td>${priorityBadge}</td>
                <td>${formatDate(task.fecha_inicio)}</td>
                <td>${formatDate(task.fecha_fin)}</td>
                <td>${formatDate(task.fecha_fin_real)}</td>
                <td>
                    <div class="progress">
                        <div class="progress-bar" role="progressbar" style="width: ${task.porcentaje}%;" aria-valuenow="${task.porcentaje}" aria-valuemin="0" aria-valuemax="100">${task.porcentaje}%</div>
                    </div>
                </td>
                <td>
                    <button class="btn btn-sm btn-outline-secondary edit-task-btn" data-id="${task.id}" title="Editar"><i class="bi bi-pencil-fill"></i></button>
                    <button class="btn btn-sm btn-outline-danger delete-task-btn" data-id="${task.id}" title="Eliminar"><i class="bi bi-trash-fill"></i></button>
                </td>
            `;
            tasksTableBody.appendChild(row);
        });
    }

    function renderKanbanBoard() {
        if (!kanbanBoard) return;
        kanbanBoard.innerHTML = '';
        const tasksToRender = getFilteredTasks();
        if (!tasksToRender) return;

        const professionalMap = new Map(appState.professionals.map(p => [parseInt(p.id), p.nombre_completo]));

        const columns = {
            'Por Hacer': { tasks: [] },
            'En Progreso': { tasks: [] },
            'Hecho': { tasks: [] }
        };

        tasksToRender.forEach(task => {
            if (task.porcentaje == 100) {
                columns['Hecho'].tasks.push(task);
            } else if (task.porcentaje > 0) {
                columns['En Progreso'].tasks.push(task);
            } else {
                columns['Por Hacer'].tasks.push(task);
            }
        });

        for (const [title, data] of Object.entries(columns)) {
            const columnEl = document.createElement('div');
            columnEl.classList.add('col-md-4', 'kanban-column');
            
            let tasksHtml = '';

            // Ordenar las tareas por prioridad
            const priorityOrder = { 'Alta': 1, 'Media': 2, 'Baja': 3 };
            data.tasks.sort((a, b) => {
                const priorityA = priorityOrder[a.prioridad || 'Media'];
                const priorityB = priorityOrder[b.prioridad || 'Media'];
                return priorityA - priorityB;
            });

            data.tasks.forEach(task => {
                const assignedName = professionalMap.get(parseInt(task.asignado_a)) || 'Sin asignar';
                let priorityIcon;
                switch (task.prioridad) {
                    case 'Alta':
                        priorityIcon = '<i class="bi bi-chevron-double-up text-danger" title="Prioridad Alta"></i> ';
                        break;
                    case 'Baja':
                        priorityIcon = '<i class="bi bi-chevron-double-down text-secondary" title="Prioridad Baja"></i> ';
                        break;
                    default:
                        priorityIcon = ''; // Media no lleva icono para no sobrecargar
                }

                tasksHtml += `
                    <div class="card mb-3 task-card" data-id="${task.id}" style="cursor: pointer;">
                        <div class="card-body">
                            <h6 class="card-title">${priorityIcon}${task.nombre_tarea}</h6>
                            <p class="card-text small text-muted">
                                Asignado a: <strong>${assignedName}</strong><br>
                                Fin: ${formatDate(task.fecha_fin)}
                            </p>
                            <div class="progress" style="height: 10px;">
                                <div class="progress-bar" role="progressbar" style="width: ${task.porcentaje}%;" aria-valuenow="${task.porcentaje}" aria-valuemin="0" aria-valuemax="100"></div>
                            </div>
                            <div class="d-flex justify-content-end mt-2">
                                <button class="btn btn-sm btn-outline-secondary edit-task-btn me-1" data-id="${task.id}" title="Editar"><i class="bi bi-pencil-fill"></i></button>
                                <button class="btn btn-sm btn-outline-danger delete-task-btn" data-id="${task.id}" title="Eliminar"><i class="bi bi-trash-fill"></i></button>
                            </div>
                        </div>
                    </div>
                `;
            });

            columnEl.innerHTML = `
                <div class="card bg-light">
                    <div class="card-header" data-status="${title}">${title} <span class="badge bg-secondary rounded-pill">${data.tasks.length}</span></div>
                    <div class="card-body kanban-tasks">
                        ${tasksHtml || '<p class="text-muted small">No hay tareas en esta columna.</p>'}
                    </div>
                </div>
            `;
            kanbanBoard.appendChild(columnEl);
        }

        // Inicializar SortableJS en las columnas
        const taskColumns = kanbanBoard.querySelectorAll('.kanban-tasks');
        taskColumns.forEach(column => {
            new Sortable(column, {
                group: 'kanban-tasks',
                animation: 150,
                onEnd: function (evt) {
                    const taskId = parseInt(evt.item.dataset.id);
                    const newStatus = evt.to.previousElementSibling.dataset.status;
                    updateTaskProgress(taskId, newStatus);
                }
            });
        });
    }

    function renderGanttChart() {
        // No renderizar si la pestaña no está activa para ahorrar recursos
        if (!document.getElementById('pills-gantt-tab')?.classList.contains('active')) {
            appState.ganttChart = null; // Limpiar la instancia si no está visible
            if (ganttChartContainer) ganttChartContainer.innerHTML = ''; // Limpiar el SVG
            return;
        }

        const tasksToRender = getFilteredTasks();
        if (tasksToRender.length === 0) {
            if (ganttChartContainer) {
                ganttChartContainer.innerHTML = ''; // Limpiar por si había algo
            }
            return;
        }

        const ganttTasks = tasksToRender.map(task => {
            // Frappe Gantt requiere fechas de inicio y fin. Si no existen, no se puede mostrar.
            if (!task.fecha_inicio || !task.fecha_fin) {
                return null;
            }
            return {
                id: String(task.id),
                name: task.nombre_tarea,
                start: task.fecha_inicio,
                end: task.fecha_fin,
                progress: task.porcentaje,
                // custom_class se usa para colorear las barras
                custom_class: task.porcentaje == 100 ? 'bar-complete' : (new Date(task.fecha_fin) < new Date() && task.porcentaje < 100 ? 'bar-overdue' : '')
            };
        }).filter(Boolean); // Eliminar tareas nulas (sin fechas)

        if (ganttTasks.length === 0) {
            if (ganttChartContainer) {
                ganttChartContainer.innerHTML = ''; // Limpiar
            }
            return;
        }

        // Limpiar el contenedor del gráfico antes de renderizar
        if (!ganttChartContainer) return;
        ganttChartContainer.innerHTML = '';

        // Crear una nueva instancia del gráfico
        appState.ganttChart = new Gantt("#gantt-chart-container", ganttTasks, {
            header_height: 50,
            column_width: 30,
            step: 24,
            view_modes: ['Quarter Day', 'Half Day', 'Day', 'Week', 'Month'],
            bar_height: 20, // (20px de alto + 18px de padding = 38px total)
            bar_corner_radius: 3,
            padding: 18,
            view_mode: 'Week', // Vista por defecto
            language: 'es', // Si tienes una traducción para 'es'
            on_click: (task) => {
                openTaskDetailModal(parseInt(task.id));
            }
        });
    }

    function renderComments(task) {
        if (commentList) commentList.innerHTML = '';
        if (task && task.comentarios && task.comentarios.length > 0) {
            task.comentarios.sort((a, b) => new Date(a.timestamp) - new Date(b.timestamp)).forEach(comment => { // Ordenar comentarios
                commentList.innerHTML += `<div class="card bg-light mb-2"><div class="card-body p-2"><p class="mb-0">${comment.text}</p><small class="text-muted"><strong>${comment.author_name}</strong> - ${new Date(comment.timestamp).toLocaleString()}</small></div></div>`;
            });
        } else {
            commentList.innerHTML = '<p class="text-muted small">No hay comentarios aún. ¡Sé el primero!</p>';
        }
    }

    function renderTags(task) {
        if (tagList) tagList.innerHTML = '';
        if (task && task.tags && task.tags.length > 0) {
            task.tags.forEach(tag => {
                tagList.innerHTML += `<span class="badge bg-secondary me-1">${tag}</span>`;
            });
        } else {
            tagList.innerHTML = '<span class="small text-muted">Sin etiquetas.</span>';
        }
    }

    function renderFiles(task) {
        if (fileList) fileList.innerHTML = '';
        if (task && task.archivos && task.archivos.length > 0) {
            task.archivos.forEach(file => {
                fileList.innerHTML += `<li class="list-group-item"><a href="${file.path}" target="_blank" download>${file.name}</a></li>`;
            });
        } else {
            fileList.innerHTML = '<li class="list-group-item text-muted small">No hay archivos adjuntos.</li>';
        }
    }

    function renderLinks(task) {
        if (linkList) linkList.innerHTML = '';
        if (task && task.links && task.links.length > 0) {
            task.links.forEach(link => {
                // Añadir https:// si no está presente para enlaces externos
                const safeUrl = link.url.startsWith('http') ? link.url : `https://${link.url}`;
                linkList.innerHTML += `<li class="list-group-item"><a href="${safeUrl}" target="_blank">${link.name}</a></li>`;
            });
        } else {
            linkList.innerHTML = '<li class="list-group-item text-muted small">No hay enlaces externos.</li>';
        }
    }

    function populateTagsDatalist() {
        const datalist = document.getElementById('tags-datalist');
        if (!datalist) return;
        datalist.innerHTML = '';
        
        const allTags = new Set();
        appState.tasks.forEach(task => {
            if (task.tags && Array.isArray(task.tags)) {
                task.tags.forEach(tag => allTags.add(tag));
            }
        });

        allTags.forEach(tag => {
            datalist.innerHTML += `<option value="${tag}">`;
        });
    }

    function populateTagFilterDropdown() {
        if (tagFilterDropdown) tagFilterDropdown.innerHTML = '<option value="all">Todas las etiquetas</option>';
        const allTags = new Set(); // Usar Set para evitar duplicados
        appState.tasks.forEach(task => {
            if (task.tags && Array.isArray(task.tags)) {
                task.tags.forEach(tag => allTags.add(tag));
            }
        });

        allTags.forEach(tag => {
            const option = document.createElement('option');
            option.value = tag;
            option.textContent = tag;
            if (tagFilterDropdown) tagFilterDropdown.appendChild(option);
        });
    }

    function populateProfessionalsFilterDropdown() {
        if (professionalFilterDropdown) professionalFilterDropdown.innerHTML = '<option value="all">Todos los profesionales</option>';
        if (!appState.project || !appState.project.profesionales_asignados) return; // Asegurarse de que el proyecto y los profesionales asignados existan

        const assignedProfessionalIds = new Set(appState.project.profesionales_asignados.map(id => parseInt(id)));
        
        appState.professionals.forEach(prof => {
            if (assignedProfessionalIds.has(parseInt(prof.id))) {
                const option = document.createElement('option');
                option.value = prof.id;
                option.textContent = prof.nombre_completo;
                if (professionalFilterDropdown) professionalFilterDropdown.appendChild(option);
            }
        });
    }

    function populateMultiSelect(selectElementId, options) {
        const select = document.getElementById(selectElementId);
        if (!select) return; // Verificar si el elemento existe
        select.innerHTML = '';
        options.forEach(opt => {
            const option = document.createElement('option');
            option.value = opt.id;
            option.textContent = opt.nombre_completo;
            select.appendChild(option);
        });
    }

    function prepareAddGridModal() {
        document.getElementById('add-grid-form').reset();
        const assignedProfessionals = appState.project.profesionales_asignados.map(id => parseInt(id)); // Asegurarse de que sean números
        const professionalsForGrid = appState.professionals.filter(p => assignedProfessionals.includes(p.id));
        populateMultiSelect('grid-responsables', professionalsForGrid);
    }

    function prepareAddMediaPlanModal() {
        document.getElementById('add-media-plan-form').reset();
        const assignedProfessionals = appState.project.profesionales_asignados.map(id => parseInt(id)); // Asegurarse de que sean números
        const professionalsForPlan = appState.professionals.filter(p => assignedProfessionals.includes(p.id));
        populateMultiSelect('plan-responsables', professionalsForPlan);
    }

    function populateProfessionalsDropdown() {
        if (assignedToDropdown) assignedToDropdown.innerHTML = '<option value="">Seleccione un profesional</option>';
        appState.professionals.forEach(prof => {
            if (prof.estado === 'Activo') {
                const option = document.createElement('option');
                option.value = prof.id;
                option.textContent = prof.nombre_completo;
                assignedToDropdown.appendChild(option);
            }
        });
    }

    // --- EVENT LISTENERS ---
    saveTaskBtn?.addEventListener('click', saveTask);
    saveGridBtn?.addEventListener('click', saveGrid);
    saveMediaPlanBtn?.addEventListener('click', saveMediaPlan);
    document.querySelector('[data-bs-target="#addGridModal"]')?.addEventListener('click', prepareAddGridModal);
    document.querySelector('[data-bs-target="#addMediaPlanModal"]')?.addEventListener('click', prepareAddMediaPlanModal);

    // Al abrir el modal para añadir (no para editar)
    document.querySelector('[data-bs-target="#addTaskModal"]')?.addEventListener('click', prepareAddTaskModal);

    // Al cerrar el modal de añadir/editar, resetearlo
    addTaskModalEl?.addEventListener('hidden.bs.modal', prepareAddTaskModal);

    document.body.addEventListener('click', function(event) {
        const target = event.target;

        const deleteBtn = target.closest('.delete-task-btn');
        if (deleteBtn) {
            event.stopPropagation();
            deleteTask(parseInt(deleteBtn.dataset.id));
            return;
        }

        const detailLink = target.closest('.task-detail-link');
        if (detailLink) {
            event.preventDefault();
            openTaskDetailModal(parseInt(detailLink.dataset.id));
            return;
        }

        const taskCard = target.closest('.task-card');
        if (taskCard) {
            openTaskDetailModal(parseInt(taskCard.dataset.id));
            return;
        }

        // Botón de editar tarea (en tabla o kanban)
        const editBtn = target.closest('.edit-task-btn');
        if (editBtn) {
            event.stopPropagation();
            handleTaskEditClick(parseInt(editBtn.dataset.id));
        }
    });

    const kanbanTab = document.getElementById('pills-kanban-tab');
    kanbanTab?.addEventListener('shown.bs.tab', function () {
        renderKanbanBoard();
    });

    const ganttTab = document.getElementById('pills-gantt-tab');
    ganttTab?.addEventListener('shown.bs.tab', function () {
        renderGanttChart();
    });

    const gridsTab = document.getElementById('pills-grids-tab');
    gridsTab?.addEventListener('shown.bs.tab', function () {
        renderGridsSection();
    });

    const mediaPlansTab = document.getElementById('pills-media-plans-tab');
    mediaPlansTab?.addEventListener('shown.bs.tab', function () {
        renderMediaPlansSection();
    });

    postCommentBtn?.addEventListener('click', addComment);
    addTagBtn?.addEventListener('click', addTag);
    uploadFileBtn?.addEventListener('click', uploadFile);
    addLinkBtn?.addEventListener('click', addLink);
    saveBudgetItemBtn?.addEventListener('click', saveBudgetItem);
    completeTaskBtn?.addEventListener('click', completeTask);

    tagFilterDropdown?.addEventListener('change', function() {
        appState.currentTagFilter = this.value;
        renderTasksTable();
        renderKanbanBoard();
        renderGanttChart();
    });

    priorityFilterDropdown?.addEventListener('change', function() {
        appState.currentPriorityFilter = this.value;
        renderTasksTable();
        renderKanbanBoard();
        renderGanttChart();
    });

    professionalFilterDropdown?.addEventListener('change', function() {
        appState.currentProfessionalFilter = this.value;
        renderTasksTable();
        renderKanbanBoard();
        renderGanttChart();
    });

    // --- CARGA INICIAL ---
    initialLoad();
});