document.addEventListener('DOMContentLoaded', function () {

    // --- URLs DE API ---
    const PROFESSIONALS_API_URL = 'api/professionals.php';

    // --- ELEMENTOS DEL DOM ---
    const tableBody = document.getElementById('professionals-table-body');
    const modalEl = document.getElementById('professionalModal');
    const modal = new bootstrap.Modal(modalEl);
    const modalLabel = document.getElementById('professionalModalLabel');
    const form = document.getElementById('professional-form');
    const saveBtn = document.getElementById('save-professional-btn');
    const addBtn = document.querySelector('[data-bs-target="#professionalModal"]'); // Selector corregido
    const fotoPreview = document.getElementById('foto-preview');
    const fotoInput = document.getElementById('foto');
    const puestoSelect = document.getElementById('puesto');

    // --- ESTADO ---
    let editMode = false;
    let currentProfessionalId = null;

    // --- FUNCIONES ---

    /**
     * Carga y renderiza los profesionales en la tabla.
     */
    async function loadProfessionals() {
        try {
            const response = await fetch(PROFESSIONALS_API_URL);
            if (!response.ok) throw new Error('No se pudieron cargar los profesionales.');
            
            const professionals = await response.json();
            renderTable(professionals);

        } catch (error) {
            console.error('Error al cargar profesionales:', error);
            tableBody.innerHTML = `<tr><td colspan="6" class="text-center text-danger">${error.message}</td></tr>`;
        }
    }

    /**
     * Renderiza la tabla de profesionales.
     * @param {Array} professionals - El array de objetos de profesionales.
     */
    function renderTable(professionals) {
        tableBody.innerHTML = '';
        if (professionals.length === 0) {
            tableBody.innerHTML = '<tr><td colspan="6" class="text-center">No hay profesionales registrados.</td></tr>';
            return;
        }

        professionals.forEach(prof => {
            const row = `
                <tr>
                    <td><img src="${prof.foto_url || 'https://via.placeholder.com/40'}" alt="${prof.nombre_completo}" class="rounded-circle" width="40" height="40" style="object-fit: cover;"></td>
                    <td>${prof.nombre_completo}</td>
                    <td>${prof.puesto || 'N/A'}</td>
                    <td>${prof.email}</td>
                    <td><span class="badge bg-${prof.estado === 'Activo' ? 'success' : 'secondary'}">${prof.estado}</span></td>
                    <td>
                        <button class="btn btn-sm btn-outline-secondary edit-btn" data-id="${prof.id}" title="Editar"><i class="bi bi-pencil-fill"></i></button>
                        <button class="btn btn-sm btn-outline-danger delete-btn" data-id="${prof.id}" title="Eliminar"><i class="bi bi-trash-fill"></i></button>
                    </td>
                </tr>
            `;
            tableBody.innerHTML += row;
        });
    }

    /**
     * Prepara y muestra el modal para editar un profesional.
     * @param {number} id - El ID del profesional a editar.
     */
    async function handleEdit(id) {
        try {
            const response = await fetch(`${PROFESSIONALS_API_URL}?id=${id}`);
            if (!response.ok) throw new Error('No se pudo obtener la información del profesional.');
            
            const professional = await response.json();
            
            editMode = true;
            currentProfessionalId = id;

            modalLabel.textContent = 'Editar Profesional';
            form.reset();

            // Llenar el formulario
            for (const key in professional) {
                const element = document.getElementById(key);
                if (element && key !== 'foto_url' && key !== 'password') {
                    element.value = professional[key];
                }
            }

            // Asegurarse de que el puesto se seleccione correctamente
            puestoSelect.value = professional.puesto;

            // Manejar la vista previa de la foto
            if (professional.foto_url) {
                fotoPreview.src = professional.foto_url;
                fotoPreview.style.display = 'block';
            } else {
                fotoPreview.style.display = 'none';
            }

            modal.show();

        } catch (error) {
            console.error('Error al preparar edición:', error);
            alert(error.message);
        }
    }

    /**
     * Guarda los datos del profesional (crear o actualizar).
     */
    async function saveProfessional() {
        const formData = new FormData(form);

        saveBtn.disabled = true;
        saveBtn.innerHTML = '<span class="spinner-border spinner-border-sm" role="status" aria-hidden="true"></span> Guardando...';

        try {
            const response = await fetch(PROFESSIONALS_API_URL, { // Siempre hacemos POST a la misma URL
                method: 'POST', 
                body: formData
            });

            if (!response.ok) throw new Error(`Error al guardar. Código: ${response.status} ${response.statusText}`);
            
            const result = await response.json();
            
            if (result.success) {
                modal.hide();
                loadProfessionals();
            } else {
                throw new Error(result.error || 'Ocurrió un error desconocido.');
            }

        } catch (error) {
            console.error('Error al guardar:', error);
            alert('Error: ' + error.message);
        } finally {
            saveBtn.disabled = false;
            saveBtn.innerHTML = 'Guardar Profesional';
        }
    }

    /**
     * Elimina un profesional.
     * @param {number} id - El ID del profesional a eliminar.
     */
    async function deleteProfessional(id) {
        if (!confirm('¿Estás seguro de que quieres eliminar este profesional?')) return;

        try {
            const response = await fetch(`${PROFESSIONALS_API_URL}?id=${id}`, { method: 'DELETE' });
            if (!response.ok) throw new Error('Error al eliminar.');
            loadProfessionals();
        } catch (error) {
            console.error('Error al eliminar:', error);
            alert(error.message);
        }
    }

    // --- EVENT LISTENERS ---

    addBtn.addEventListener('click', () => {
        editMode = false;
        currentProfessionalId = null;
        modalLabel.textContent = 'Añadir Profesional';
        form.reset();
        fotoPreview.style.display = 'none';
    });

    saveBtn.addEventListener('click', saveProfessional);

    tableBody.addEventListener('click', (e) => {
        if (e.target.closest('.edit-btn')) {
            const id = e.target.closest('.edit-btn').dataset.id;
            handleEdit(id);
        }
        if (e.target.closest('.delete-btn')) {
            const id = e.target.closest('.delete-btn').dataset.id;
            deleteProfessional(id);
        }
    });

    fotoInput.addEventListener('change', () => {
        const file = fotoInput.files[0];
        if (file) {
            fotoPreview.src = URL.createObjectURL(file);
            fotoPreview.style.display = 'block';
        }
    });

    // --- INICIO ---
    loadProfessionals();
});