document.addEventListener('DOMContentLoaded', function() {
    const loader = document.getElementById('loader');
    const errorContainer = document.getElementById('error-container');
    const dashboardContent = document.getElementById('dashboard-content');
    const sinceDateInput = document.getElementById('since-date');
    const untilDateInput = document.getElementById('until-date');
    const filterBtn = document.getElementById('filter-btn');

    const kpiFans = document.getElementById('total-fans-kpi');
    const kpiEngagements = document.getElementById('total-engagements-kpi');
    const kpiImpressions = document.getElementById('total-impressions-kpi');

    const comparisonFans = document.getElementById('total-fans-comparison');
    const comparisonEngagements = document.getElementById('total-engagements-comparison');
    const comparisonImpressions = document.getElementById('total-impressions-comparison');

    const performanceChartControls = document.getElementById('performance-chart-controls');

    let performanceChart = null;
    let countryChart = null;
    let topPostsChart = null;

    let currentData = null; // Almacenar los datos actuales para no tener que volver a pedirlos

    async function fetchDashboardData(since, until) {
        loader.style.display = 'block';
        dashboardContent.style.display = 'none';
        errorContainer.style.display = 'none';
        filterBtn.disabled = true;
        filterBtn.innerHTML = '<span class="spinner-border spinner-border-sm"></span> Cargando...';

        let url = 'api/meta_insights.php';
        const params = new URLSearchParams();
        if (since) params.append('since', since);
        if (until) params.append('until', until);

        if (params.toString()) {
            url += `?${params.toString()}`;
        }

        try {
            const response = await fetch(url);
            if (!response.ok) {
                const errorData = await response.json();
                throw new Error(errorData.error || `Error ${response.status}`);
            }
            const data = await response.json();
            currentData = data; // Guardar los datos completos para usar en los gráficos

            renderKpiCards(data);
            renderPerformanceChart();
            renderCountryChart();
            renderTopPostsChart();

            loader.style.display = 'none';
            dashboardContent.style.display = 'block';

        } catch (error) {
            console.error('Error al cargar las métricas de Meta:', error);
            errorContainer.textContent = `Error al cargar las métricas: ${error.message}. Asegúrate de que la cuenta de Meta esté conectada correctamente en Configuración y que tengas permisos para acceder a los datos de la página.`;
            errorContainer.style.display = 'block';
            loader.style.display = 'none';
        } finally {
            filterBtn.disabled = false;
            filterBtn.innerHTML = 'Aplicar Filtro';
        }
    }

    function renderKpiCards(data) {
        const numberFormatter = new Intl.NumberFormat('es-ES'); // Formateador de números

        // Función auxiliar para calcular y renderizar la comparación
        const renderComparison = (element, currentValue, previousValue) => {
            if (previousValue === 0 && currentValue === 0) {
                element.innerHTML = `<span class="text-white-50">Sin datos previos</span>`;
                return;
            }
            const percentageChange = ((currentValue - previousValue) / previousValue) * 100;
            const isPositive = percentageChange >= 0;
            const icon = isPositive ? 'bi-arrow-up' : 'bi-arrow-down';
            const colorClass = isPositive ? 'text-white' : 'text-white'; // Mantenemos blanco, el icono indica la dirección

            element.innerHTML = `
                <span class="${colorClass}">
                    <i class="bi ${icon}"></i> ${percentageChange.toFixed(1)}%
                </span>
                <small class="text-white-50">vs. período anterior</small>
            `;
        };

        // --- Total de Fans ---
        // page_fans es una métrica de "vida", no de período. Solo tiene un valor actual.
        const totalFans = data.page_fans?.current_values.slice(-1)[0]?.value || 0;
        document.getElementById('total-fans-kpi').textContent = numberFormatter.format(totalFans);
        document.getElementById('total-fans-comparison').innerHTML = '<span class="text-white-50">Total histórico</span>';

        // --- Interacciones ---
        const totalEngagements = data.page_post_engagements?.current_values.reduce((sum, item) => sum + item.value, 0) || 0;
        const previousEngagements = data.page_post_engagements?.previous_values.reduce((sum, item) => sum + item.value, 0) || 0;
        document.getElementById('total-engagements-kpi').textContent = numberFormatter.format(totalEngagements);
        renderComparison(document.getElementById('total-engagements-comparison'), totalEngagements, previousEngagements);

        // --- Impresiones ---
        const totalImpressions = data.page_impressions?.current_values.reduce((sum, item) => sum + item.value, 0) || 0;
        const previousImpressions = data.page_impressions?.previous_values.reduce((sum, item) => sum + item.value, 0) || 0;
        document.getElementById('total-impressions-kpi').textContent = numberFormatter.format(totalImpressions);
        renderComparison(document.getElementById('total-impressions-comparison'), totalImpressions, previousImpressions);
    }

    function renderPerformanceChart() {
        const ctx = document.getElementById('performance-chart').getContext('2d');
        if (chartInstances.performance) {
            chartInstances.performance.destroy();
        }

        // Asegurarse de que currentData esté disponible
        if (!currentData) return;

        const { since, until } = { since: sinceDateInput.value, until: untilDateInput.value };
        
        const labels = (currentData.page_impressions?.current_values || []).map(item => {
            const date = new Date(item.end_time);
            return date.toLocaleDateString('es-ES', { day: '2-digit', month: 'short' });
        });

        performanceChart = new Chart(ctx, {
            type: 'line',
            data: {
                labels: labels,
                datasets: [] // Se llenará dinámicamente
            },
            options: {
                responsive: true,
                interaction: {
                    mode: 'index',
                    intersect: false,
                },
                plugins: {
                    legend: {
                        position: 'top',
                    },
                    title: { // Título dinámico para el gráfico
                        display: true,
                        text: `Rendimiento de la Página ${since && until ? `(del ${since} al ${until})` : '(Últimos 30 días)'}`
                    }
                },
                scales: {
                    y: { type: 'linear', display: true, position: 'left', title: { display: true, text: 'Impresiones' } },
                    y1: { type: 'linear', display: true, position: 'right', title: { display: true, text: 'Interacciones' }, grid: { drawOnChartArea: false } },
                }
            }
        });
        updatePerformanceChartDatasets(); // Llamar para poblar los datasets iniciales
    }

    function updatePerformanceChartDatasets() {
        if (!performanceChart || !currentData) return;

        const metricConfig = {
            page_impressions: {
                label: 'Impresiones',
                borderColor: 'rgba(54, 162, 235, 1)',
                backgroundColor: 'rgba(54, 162, 235, 0.2)',
                yAxisID: 'y',
            },
            page_reach_unique: {
                label: 'Alcance',
                borderColor: 'rgba(255, 206, 86, 1)',
                backgroundColor: 'rgba(255, 206, 86, 0.2)',
                yAxisID: 'y',
            },
            page_post_engagements: {
                label: 'Interacciones',
                borderColor: 'rgba(75, 192, 192, 1)',
                backgroundColor: 'rgba(75, 192, 192, 0.2)',
                yAxisID: 'y1',
            }
        };

        const selectedMetrics = Array.from(performanceChartControls.querySelectorAll('input[type="checkbox"]:checked')).map(cb => cb.value);
        
        performanceChart.data.datasets = selectedMetrics.map(metricKey => {
            const config = metricConfig[metricKey];
            const values = currentData[metricKey]?.current_values.map(item => item.value) || [];
            return {
                label: config.label,
                data: values,
                borderColor: config.borderColor,
                backgroundColor: config.backgroundColor,
                yAxisID: config.yAxisID,
                fill: config.yAxisID === 'y', // Rellenar solo el eje principal
            };
        });

        performanceChart.update();
    }

    function renderCountryChart() {
        const ctx = document.getElementById('country-chart').getContext('2d');
        if (chartInstances.country) {
            chartInstances.country.destroy();
        }

        // Asegurarse de que currentData esté disponible
        if (!currentData) return;

        const countryData = currentData.page_fans_country?.current_values[0]?.value || {};
        if (Object.keys(countryData).length === 0) {
            // Mostrar un mensaje si no hay datos de país
            ctx.canvas.parentNode.innerHTML = '<p class="text-muted text-center">No hay datos de fans por país para mostrar.</p>';
            return;
        }

        const sortedCountries = Object.entries(countryData).sort(([,a],[,b]) => b-a);
        
        const topCountries = sortedCountries.slice(0, 7); // Top 7 países
        const otherCount = sortedCountries.slice(7).reduce((sum, [,count]) => sum + count, 0); // Suma del resto

        const labels = topCountries.map(([countryCode]) => countryCode);
        const values = topCountries.map(([,count]) => count);

        if (otherCount > 0) {
            labels.push('Otros');
            values.push(otherCount);
        }

        chartInstances.country = new Chart(ctx, {
            type: 'doughnut',
            data: {
                labels: labels,
                datasets: [{
                    label: 'Fans por País',
                    data: values,
                    backgroundColor: [
                        '#0d6efd', '#198754', '#ffc107', '#dc3545', '#0dcaf0', '#6f42c1', '#fd7e14', '#6c757d' // Colores de Bootstrap
                    ],
                }]
            },
            options: {
                responsive: true,
                plugins: {
                    legend: {
                        position: 'top',
                    },
                    title: {
                        display: true,
                        text: 'Distribución de Fans por País'
                    }
                }
            }
        });
    }

    function renderTopPostsChart() {
        const ctx = document.getElementById('top-posts-chart').getContext('2d');
        if (chartInstances.topPosts) {
            chartInstances.topPosts.destroy();
        }

        // Asegurarse de que currentData esté disponible
        if (!currentData) return;

        const topPostsData = currentData.top_posts || [];
        if (topPostsData.length === 0) {
            // Mostrar un mensaje si no hay datos de posts
            ctx.canvas.parentNode.innerHTML = '<p class="text-muted text-center">No hay datos de publicaciones recientes para mostrar.</p>';
            return;
        }

        // Ordenar por interacciones de mayor a menor
        topPostsData.sort((a, b) => b.engagements - a.engagements);

        const labels = topPostsData.map(post => post.message);
        const values = topPostsData.map(post => post.engagements);

        chartInstances.topPosts = new Chart(ctx, {
            type: 'bar',
            data: {
                labels: labels,
                datasets: [{
                    label: 'Interacciones Totales',
                    data: values,
                    backgroundColor: 'rgba(255, 159, 64, 0.6)', // Naranja de Bootstrap
                    borderColor: 'rgba(255, 159, 64, 1)',
                    borderWidth: 1
                }]
            },
            options: {
                indexAxis: 'y', // Gráfico de barras horizontales
                responsive: true,
                plugins: {
                    legend: {
                        display: false
                    },
                    title: {
                        display: true,
                        text: 'Top 5 Publicaciones Recientes por Interacciones'
                    }
                }
            }
        });
    }

    function initialize() {        
        // Establecer fechas por defecto en los inputs
        const today = new Date();
        const thirtyDaysAgo = new Date();
        thirtyDaysAgo.setDate(today.getDate() - 30);

        untilDateInput.value = today.toISOString().split('T')[0];
        sinceDateInput.value = thirtyDaysAgo.toISOString().split('T')[0];

        // Cargar datos iniciales
        fetchDashboardData(sinceDateInput.value, untilDateInput.value);

        // Añadir listener al botón de filtro
        filterBtn.addEventListener('click', () => {
            fetchDashboardData(sinceDateInput.value, untilDateInput.value);
        });

        // Añadir listener a los controles del gráfico de rendimiento
        performanceChartControls.addEventListener('change', () => {
            updatePerformanceChartDatasets();
        });
    }

    initialize();
});
        });
    }

    function updatePerformanceChartDatasets() {
        if (!performanceChart || !currentData) return;

        const metricConfig = {
            page_impressions: {
                label: 'Impresiones',
                borderColor: 'rgba(54, 162, 235, 1)',
                backgroundColor: 'rgba(54, 162, 235, 0.2)',
                yAxisID: 'y',
            },
            page_reach_unique: {
                label: 'Alcance',
                borderColor: 'rgba(255, 206, 86, 1)',
                backgroundColor: 'rgba(255, 206, 86, 0.2)',
                yAxisID: 'y',
            },
            page_post_engagements: {
                label: 'Interacciones',
                borderColor: 'rgba(75, 192, 192, 1)',
                backgroundColor: 'rgba(75, 192, 192, 0.2)',
                yAxisID: 'y1',
            }
        };

        const selectedMetrics = Array.from(performanceChartControls.querySelectorAll('input:checked')).map(cb => cb.value);
        
        performanceChart.data.datasets = selectedMetrics.map(metricKey => {
            const config = metricConfig[metricKey];
            const values = currentData[metricKey]?.current_values.map(item => item.value) || [];
            return {
                label: config.label,
                data: values,
                borderColor: config.borderColor,
                backgroundColor: config.backgroundColor,
                yAxisID: config.yAxisID,
                fill: config.yAxisID === 'y', // Rellenar solo el eje principal
            };
        });

        performanceChart.update();
    }

    function renderCountryChart() {
        if (!currentData) return;
        const countryData = currentData.page_fans_country?.current_values[0]?.value || {};
        const sortedCountries = Object.entries(countryData).sort(([,a],[,b]) => b-a);
        
        const topCountries = sortedCountries.slice(0, 7);
        const otherCount = sortedCountries.slice(7).reduce((sum, [,count]) => sum + count, 0);

        const labels = topCountries.map(([countryCode]) => countryCode);
        const values = topCountries.map(([,count]) => count);

        if (otherCount > 0) {
            labels.push('Otros');
            values.push(otherCount);
        }

        const ctx = document.getElementById('country-chart').getContext('2d');
        if (countryChart) {
            countryChart.destroy();
        }
        countryChart = new Chart(ctx, {
            type: 'doughnut',
            data: {
                labels: labels,
                datasets: [{
                    label: 'Fans por País',
                    data: values,
                    backgroundColor: [
                        '#0d6efd', '#198754', '#ffc107', '#dc3545', '#0dcaf0', '#6f42c1', '#fd7e14', '#6c757d'
                    ],
                }]
            },
            options: {
                responsive: true,
                plugins: {
                    legend: {
                        position: 'top',
                    },
                    title: {
                        display: true,
                        text: 'Distribución de Fans por País'
                    }
                }
            }
        });
    }

    function renderTopPostsChart() {
        if (!currentData) return;
        const topPostsData = currentData.top_posts || [];
        // Ordenar por interacciones de mayor a menor
        topPostsData.sort((a, b) => b.engagements - a.engagements);

        const labels = topPostsData.map(post => post.message);
        const values = topPostsData.map(post => post.engagements);

        const ctx = document.getElementById('top-posts-chart').getContext('2d');
        if (topPostsChart) {
            topPostsChart.destroy();
        }
        topPostsChart = new Chart(ctx, {
            type: 'bar',
            data: {
                labels: labels,
                datasets: [{
                    label: 'Interacciones Totales',
                    data: values,
                    backgroundColor: 'rgba(255, 159, 64, 0.6)',
                    borderColor: 'rgba(255, 159, 64, 1)',
                    borderWidth: 1
                }]
            },
            options: {
                indexAxis: 'y', // Gráfico de barras horizontales
                responsive: true,
                plugins: {
                    legend: {
                        display: false
                    },
                    title: {
                        display: true,
                        text: 'Top 5 Posts Recientes por Interacciones'
                    }
                }
            }
        });
    }

    function initialize() {        
        // Establecer fechas por defecto en los inputs
        const today = new Date();
        const thirtyDaysAgo = new Date();
        thirtyDaysAgo.setDate(today.getDate() - 30);

        untilDateInput.value = today.toISOString().split('T')[0];
        sinceDateInput.value = thirtyDaysAgo.toISOString().split('T')[0];

        // Cargar datos iniciales
        loadInsights(sinceDateInput.value, untilDateInput.value);

        // Añadir listener al botón de filtro
        filterBtn.addEventListener('click', () => {
            loadInsights(sinceDateInput.value, untilDateInput.value);
        });

        // Añadir listener a los controles del gráfico de rendimiento
        performanceChartControls.addEventListener('change', () => {
            updatePerformanceChartDatasets();
        });
    }

    initialize();
});