document.addEventListener('DOMContentLoaded', function () {

    // --- ELEMENTOS DEL DOM ---
    const planNameTitle = document.getElementById('plan-name-title');
    const projectNameSubtitle = document.getElementById('project-name-subtitle');
    const backToMediaPlansBtn = document.getElementById('back-to-media-plans-btn');
    const mediaItemsBody = document.getElementById('media-items-body');
    const budgetSummaryContainer = document.getElementById('budget-summary-container');
    
    const mediaItemModalEl = document.getElementById('mediaItemModal');
    const mediaItemModal = new bootstrap.Modal(mediaItemModalEl);
    const mediaItemModalLabel = document.getElementById('mediaItemModalLabel');
    const saveMediaItemBtn = document.getElementById('save-media-item-btn');

    // --- URLs Y ESTADO ---
    const urlParams = new URLSearchParams(window.location.search);
    const planId = urlParams.get('plan_id');

    const PROJECTS_API_URL = 'api/projects.php';
    const MEDIA_PLANS_API_URL = 'api/media_plans.php';

    const appState = {
        project: null,
        mediaPlan: null,
        isEditMode: false,
        currentItemId: null
    };

    // --- FUNCIONES DE CARGA ---

    async function initialLoad() {
        if (!planId) {
            planNameTitle.textContent = "Plan de medios no especificado";
            return;
        }

        try {
            const planRes = await fetch(`${MEDIA_PLANS_API_URL}?id=${planId}`);
            if (!planRes.ok) throw new Error('No se pudo cargar el plan de medios.');
            appState.mediaPlan = await planRes.json();

            const projectRes = await fetch(`${PROJECTS_API_URL}?id=${appState.mediaPlan.project_id}`);
            if (!projectRes.ok) throw new Error('No se pudo cargar el proyecto asociado.');
            appState.project = await projectRes.json();

            planNameTitle.textContent = appState.mediaPlan.name;
            projectNameSubtitle.textContent = `Proyecto: ${appState.project.nombre_proyecto}`;
            document.title = `Editor: ${appState.mediaPlan.name} - Platflow`;
            backToMediaPlansBtn.href = `project-detail.html?id=${appState.project.id}`;

            renderBudgetSummary();
            renderMediaItems();

        } catch (error) {
            console.error("Error en la carga inicial:", error);
            planNameTitle.textContent = "Error";
            mediaItemsBody.innerHTML = `<tr><td colspan="8" class="text-center text-danger">${error.message}</td></tr>`;
        }
    }

    // --- FUNCIONES DE DATOS Y UI ---

    function renderBudgetSummary() {
        if (!budgetSummaryContainer) return;

        const totalBudget = parseFloat(appState.mediaPlan.budget) || 0;
        const allocatedBudget = (appState.mediaPlan.items || []).reduce((sum, item) => sum + (parseFloat(item.budget) || 0), 0);
        const remainingBudget = totalBudget - allocatedBudget;

        const formatCurrency = (value) => new Intl.NumberFormat('es-PY', { style: 'currency', currency: 'PYG', maximumFractionDigits: 0 }).format(value);

        budgetSummaryContainer.innerHTML = `
            <div class="col-md-4">
                <div class="card text-white bg-primary"><div class="card-body"><h5 class="card-title">Presupuesto Total</h5><p class="card-text fs-4">${formatCurrency(totalBudget)}</p></div></div>
            </div>
            <div class="col-md-4">
                <div class="card text-white bg-warning"><div class="card-body"><h5 class="card-title">Presupuesto Asignado</h5><p class="card-text fs-4">${formatCurrency(allocatedBudget)}</p></div></div>
            </div>
            <div class="col-md-4">
                <div class="card text-white ${remainingBudget >= 0 ? 'bg-success' : 'bg-danger'}"><div class="card-body"><h5 class="card-title">Presupuesto Restante</h5><p class="card-text fs-4">${formatCurrency(remainingBudget)}</p></div></div>
            </div>
        `;
    }

    function renderMediaItems() {
        mediaItemsBody.innerHTML = '';
        const items = appState.mediaPlan.items || [];

        if (items.length === 0) {
            mediaItemsBody.innerHTML = '<tr><td colspan="8" class="text-center text-muted">No hay líneas de medios en este plan. ¡Añade la primera!</td></tr>';
            return;
        }

        const formatDate = (dateString) => {
            if (!dateString) return 'N/A';
            const [year, month, day] = dateString.split('-');
            if (!year || !month || !day) return dateString;
            return `${day}/${month}/${year}`;
        };

        const formatCurrency = (value) => new Intl.NumberFormat('es-PY', { style: 'currency', currency: 'PYG', maximumFractionDigits: 0 }).format(value);

        items.forEach(item => {
            const row = document.createElement('tr');
            row.innerHTML = `
                <td>${item.channel}</td>
                <td>${item.media}</td>
                <td>${item.format}</td>
                <td>${formatDate(item.startDate)} al ${formatDate(item.endDate)}</td>
                <td class="text-end">${formatCurrency(item.budget)}</td>
                <td>${item.kpis}</td>
                <td>${item.results || ''}</td>
                <td>
                    <button class="btn btn-sm btn-outline-secondary edit-item-btn" data-id="${item.id}" title="Editar"><i class="bi bi-pencil-fill"></i></button>
                    <button class="btn btn-sm btn-outline-danger delete-item-btn" data-id="${item.id}" title="Eliminar"><i class="bi bi-trash-fill"></i></button>
                </td>
            `;
            mediaItemsBody.appendChild(row);
        });
    }

    function openMediaItemModal(item = null) {
        document.getElementById('media-item-form').reset();
        if (item) { // Modo Edición
            appState.isEditMode = true;
            appState.currentItemId = item.id;
            mediaItemModalLabel.textContent = 'Editar Línea de Medio';
            document.getElementById('item-channel').value = item.channel;
            document.getElementById('item-media').value = item.media;
            document.getElementById('item-format').value = item.format;
            document.getElementById('item-start-date').value = item.startDate;
            document.getElementById('item-end-date').value = item.endDate;
            document.getElementById('item-budget').value = item.budget;
            document.getElementById('item-kpis').value = item.kpis;
            document.getElementById('item-results').value = item.results || '';
        } else { // Modo Creación
            appState.isEditMode = false;
            appState.currentItemId = null;
            mediaItemModalLabel.textContent = 'Añadir Línea de Medio';
        }
        mediaItemModal.show();
    }

    async function saveMediaItem() {
        const itemData = {
            id: appState.isEditMode ? appState.currentItemId : `item_${Date.now()}`,
            channel: document.getElementById('item-channel').value,
            media: document.getElementById('item-media').value,
            format: document.getElementById('item-format').value,
            startDate: document.getElementById('item-start-date').value,
            endDate: document.getElementById('item-end-date').value,
            budget: parseFloat(document.getElementById('item-budget').value) || 0,
            kpis: document.getElementById('item-kpis').value,
            results: document.getElementById('item-results').value
        };

        if (!itemData.media || !itemData.format || !itemData.budget) {
            alert('Medio, formato y presupuesto son obligatorios.');
            return;
        }

        let updatedItems = [...(appState.mediaPlan.items || [])];
        if (appState.isEditMode) {
            const index = updatedItems.findIndex(i => i.id === appState.currentItemId);
            if (index > -1) updatedItems[index] = itemData;
        } else {
            updatedItems.push(itemData);
        }

        await updatePlanOnServer({ items: updatedItems });
    }

    async function deleteMediaItem(itemId) {
        if (!confirm('¿Estás seguro de que quieres eliminar esta línea del plan?')) return;

        const updatedItems = (appState.mediaPlan.items || []).filter(i => i.id !== itemId);
        await updatePlanOnServer({ items: updatedItems });
    }

    async function updatePlanOnServer(updateData) {
        const fullUpdateData = { ...appState.mediaPlan, ...updateData };

        try {
            const response = await fetch(`${MEDIA_PLANS_API_URL}?id=${planId}`, {
                method: 'PUT',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(fullUpdateData)
            });
            if (!response.ok) throw new Error('Error al actualizar el plan de medios.');
            
            appState.mediaPlan = await response.json();
            renderBudgetSummary();
            renderMediaItems();
            mediaItemModal.hide();
        } catch (error) {
            console.error(error);
            alert(error.message);
        }
    }

    // --- EVENT LISTENERS ---
    document.getElementById('add-media-item-btn').addEventListener('click', () => openMediaItemModal());
    saveMediaItemBtn.addEventListener('click', saveMediaItem);

    mediaItemsBody.addEventListener('click', (e) => {
        const editBtn = e.target.closest('.edit-item-btn');
        if (editBtn) {
            const item = appState.mediaPlan.items.find(i => i.id === editBtn.dataset.id);
            if (item) openMediaItemModal(item);
        }

        const deleteBtn = e.target.closest('.delete-item-btn');
        if (deleteBtn) {
            deleteMediaItem(deleteBtn.dataset.id);
        }
    });

    // --- CARGA INICIAL ---
    initialLoad();
});