document.addEventListener('DOMContentLoaded', () => {
    const searchInput = document.getElementById('global-search-input');
    const searchResultsContainer = document.getElementById('global-search-results');

    if (!searchInput || !searchResultsContainer) return;

    let searchData = {
        projects: [],
        clients: [],
        tasks: []
    };

    let dataLoaded = false;

    // Carga los datos de búsqueda en segundo plano
    async function loadSearchData() {
        try {
            const [projectsRes, clientsRes, tasksRes] = await Promise.all([
                fetch('api/projects.php'),
                fetch('api/clients.php'),
                fetch('api/tasks.php')
            ]); // Corregido: fetch('api/tasks.php')

            // Verificar si las respuestas son OK antes de intentar parsear JSON
            if (!projectsRes.ok || !clientsRes.ok || !tasksRes.ok) {
                throw new Error('Una o más APIs de búsqueda fallaron al cargar.');
            }
            searchData.projects = await projectsRes.json();
            searchData.clients = await clientsRes.json();
            searchData.tasks = await tasksRes.json();
            dataLoaded = true;
        } catch (error) {
            console.error("Error loading global search data:", error);
        }
    }

    function performSearch(query) {
        if (!dataLoaded || query.length < 2) {
            searchResultsContainer.innerHTML = '';
            searchResultsContainer.classList.remove('show');
            return;
        }

        const lowerCaseQuery = query.toLowerCase();
        let resultsHtml = '';
        let resultCount = 0;

        // Buscar en Proyectos
        const projectResults = searchData.projects.filter(p => p.nombre_proyecto.toLowerCase().includes(lowerCaseQuery)).slice(0, 3);
        if (projectResults.length > 0) {
            resultsHtml += '<li><h6 class="dropdown-header">Proyectos</h6></li>';
            projectResults.forEach(p => {
                resultsHtml += `<li><a class="dropdown-item" href="project-detail.html?id=${p.id}">${p.nombre_proyecto}</a></li>`;
                resultCount++;
            });
        }

        // Buscar en Clientes
        const clientResults = searchData.clients.filter(c => c.nombre_empresa.toLowerCase().includes(lowerCaseQuery)).slice(0, 3);
        if (clientResults.length > 0) {
            resultsHtml += '<li><hr class="dropdown-divider"></li><li><h6 class="dropdown-header">Clientes</h6></li>';
            clientResults.forEach(c => {
                resultsHtml += `<li><a class="dropdown-item" href="clients.html?view_client_id=${c.id}">${c.nombre_empresa}</a></li>`;
                resultCount++;
            });
        }

        // Buscar en Tareas
        const taskResults = searchData.tasks.filter(t => t.nombre_tarea.toLowerCase().includes(lowerCaseQuery)).slice(0, 3);
        if (taskResults.length > 0) {
            resultsHtml += '<li><hr class="dropdown-divider"></li><li><h6 class="dropdown-header">Tareas</h6></li>';
            taskResults.forEach(t => {
                resultsHtml += `<li><a class="dropdown-item" href="project-detail.html?id=${t.project_id}">${t.nombre_tarea}</a></li>`;
                resultCount++;
            });
        }

        if (resultCount > 0) {
            searchResultsContainer.innerHTML = resultsHtml;
            searchResultsContainer.classList.add('show');
        } else {
            searchResultsContainer.innerHTML = '<li><span class="dropdown-item-text text-muted">No se encontraron resultados.</span></li>';
            searchResultsContainer.classList.add('show');
        }
    }

    // --- Event Listeners ---

    let debounceTimer;
    searchInput.addEventListener('input', () => {
        clearTimeout(debounceTimer);
        debounceTimer = setTimeout(() => {
            performSearch(searchInput.value);
        }, 250); // Espera 250ms después de que el usuario deja de escribir
    });

    searchInput.addEventListener('focus', () => {
        if (!dataLoaded) {
            loadSearchData();
        }
        if (searchInput.value.length >= 2) {
            searchResultsContainer.classList.add('show');
        }
    });

    // Ocultar resultados al hacer clic fuera
    document.addEventListener('click', (e) => {
        if (!e.target.closest('#global-search-container')) {
            searchResultsContainer.classList.remove('show');
        }
    });

    // Cargar datos al inicio para una búsqueda más rápida la primera vez
    loadSearchData();
});