document.addEventListener('DOMContentLoaded', function () { // Se mantiene para el código que necesita que el DOM esté cargado

    // --- URLs DE API ---
    const INVOICES_API_URL = 'api/invoices.php';
    const EXPENSES_API_URL = 'api/expenses.php';
    const CLIENTS_API_URL = 'api/clients.php';
    const PROJECTS_API_URL = 'api/projects.php';
    const SETTINGS_API_URL = 'api/settings.php';

    // --- ELEMENTOS DEL DOM ---
    // Ingresos
    const invoicesTableBody = document.getElementById('invoices-table-body');
    const saveInvoiceBtn = document.getElementById('save-invoice-btn');
    const invoiceModalEl = document.getElementById('addInvoiceModal');
    const invoiceModal = new bootstrap.Modal(invoiceModalEl);
    const invoiceModalLabel = document.getElementById('addInvoiceModalLabel');
    const invoiceForm = document.getElementById('invoice-form');
    const invoiceClientDropdown = document.getElementById('invoice-client-id');

    // Egresos
    const expensesTableBody = document.getElementById('expenses-table-body');
    const saveExpenseBtn = document.getElementById('save-expense-btn');
    const expenseModalEl = document.getElementById('addExpenseModal');
    const expenseModalLabel = document.getElementById('addExpenseModalLabel');
    const expenseModal = new bootstrap.Modal(expenseModalEl);
    const expenseForm = document.getElementById('expense-form');

    // Reporte
    const reportContentEl = document.getElementById('financial-report-content');
    const printReportBtn = document.getElementById('print-report-btn');
    const reportStartDateInput = document.getElementById('report-start-date');
    const reportEndDateInput = document.getElementById('report-end-date');
    const applyReportFilterBtn = document.getElementById('apply-report-filter-btn');
    const clearReportFilterBtn = document.getElementById('clear-report-filter-btn');

    // --- ESTADO DE LA APP (ahora es global para que las funciones CRUD puedan accederlo) ---
    window.appState = {
        invoices: [],
        expenses: [],
        projects: [],
        clients: [],
        invoiceEditMode: false,
        expenseEditMode: false,
        currentInvoiceId: null,
        currentExpenseId: null,
        systemSettings: {}, // Para guardar los ajustes del sistema
        evolutionChart: null // Para la instancia del gráfico
    };

    // --- FUNCIONES DE FORMATO ---
    /**
     * Formatea un número como moneda usando los ajustes del sistema.
     * @param {number} value El número a formatear.
     * @returns {string} El valor formateado como moneda.
     */
    const formatCurrency = (value) => {
        const settings = window.appState.systemSettings;
        const number = Number(value) || 0;

        // Usamos Intl.NumberFormat porque maneja correctamente los separadores y decimales.
        // Forzamos un locale que no use agrupamiento por defecto (como 'en-US') y luego reemplazamos los separadores.
        // Esto es más robusto que construir la cadena manualmente.
        const formatted = new Intl.NumberFormat('en-US', {
            minimumFractionDigits: settings.decimal_places || 0,
            maximumFractionDigits: settings.decimal_places || 0,
        }).format(number);

        // Reemplazar separadores por los definidos por el usuario
        const parts = formatted.split('.');
        const integerPart = parts[0].replace(/,/g, settings.thousands_separator || '.');
        const decimalPart = parts[1];

        const finalNumber = decimalPart !== undefined ? `${integerPart}${settings.decimal_separator || ','}${decimalPart}` : integerPart;

        return `${settings.currency_symbol || 'Gs.'} ${finalNumber}`;
    };

    /**
     * Formatea una fecha en formato YYYY-MM-DD a DD/MM/YYYY.
     * @param {string} dateString La fecha en formato YYYY-MM-DD.
     * @returns {string} La fecha formateada o 'N/A' si la entrada es inválida.
     */
    const formatDate = (dateString) => {
        if (!dateString) return 'N/A';
        const [year, month, day] = dateString.split('-');
        if (!year || !month || !day) return dateString;
        return `${day}/${month}/${year}`;
    };

    const getStatusBadge = (status) => {
        switch (status) {
            case 'Pagada': return `<span class="badge bg-success">${status}</span>`;
            case 'Pendiente': return `<span class="badge bg-warning text-dark">${status}</span>`;
            case 'Vencida': return `<span class="badge bg-danger">${status}</span>`;
            case 'Anulada': return `<span class="badge bg-secondary">${status}</span>`;
            default: return `<span class="badge bg-light text-dark">${status}</span>`;
        }
    };

    // --- FUNCIONES DE CARGA DE DATOS (FETCH) ---
    async function initialLoad() {
        try {
            const [settingsRes, invoicesRes, expensesRes, clientsRes, projectsRes] = await Promise.all([
                fetch(SETTINGS_API_URL),
                fetch(INVOICES_API_URL),
                fetch(EXPENSES_API_URL),
                fetch(CLIENTS_API_URL),
                fetch(PROJECTS_API_URL)
            ]);

            const settingsData = await settingsRes.json();
            appState.systemSettings = settingsData.system_settings || {};
            appState.invoices = await invoicesRes.json();
            appState.expenses = await expensesRes.json();
            appState.clients = await clientsRes.json();
            appState.projects = await projectsRes.json();

            renderFinancialReport();
            renderEvolutionChart();
            renderInvoicesTable();
            renderExpensesTable();
            populateClientsDropdown();

        } catch (error) {
            console.error("Error en la carga inicial:", error);
        }
    }

    // --- FUNCIONES DE RENDERIZADO ---
    function renderInvoicesTable() {
        invoicesTableBody.innerHTML = '';
        if (appState.invoices.length === 0) {
            invoicesTableBody.innerHTML = '<tr><td colspan="7" class="text-center">No hay facturas de venta registradas.</td></tr>';
            return;
        }
        appState.invoices.forEach(inv => {
            const row = `
                <tr>
                    <td>${inv.invoice_number}</td>
                    <td>${inv.client_name || 'N/A'}</td>
                    <td>${formatDate(inv.invoice_date)}</td>
                    <td>${formatDate(inv.due_date)}</td>
                    <td class="text-end">${formatCurrency(inv.totals.total)}</td>
                    <td>${getStatusBadge(inv.status)}</td>
                    <td>
                        <button class="btn btn-sm btn-outline-secondary" title="Editar" onclick="handleEditInvoice(${inv.id})"><i class="bi bi-pencil-fill"></i></button>
                        <button class="btn btn-sm btn-outline-danger" title="Eliminar" onclick="deleteInvoice(${inv.id})"><i class="bi bi-trash-fill"></i></button>
                    </td>
                </tr>
            `;
            invoicesTableBody.innerHTML += row;
        });
    }

    function renderExpensesTable() {
        expensesTableBody.innerHTML = '';
        if (appState.expenses.length === 0) {
            expensesTableBody.innerHTML = '<tr><td colspan="7" class="text-center">No hay facturas de gasto registradas.</td></tr>';
            return;
        }
        appState.expenses.forEach(exp => {
            const row = `
                <tr>
                    <td>${exp.expense_number}</td>
                    <td>${exp.provider}</td>
                    <td>${formatDate(exp.expense_date)}</td>
                    <td>${formatDate(exp.due_date)}</td>
                    <td class="text-end">${formatCurrency(exp.totals.total)}</td>
                    <td>${getStatusBadge(exp.status)}</td>
                    <td>
                        <button class="btn btn-sm btn-outline-secondary" title="Editar" onclick="handleEditExpense(${exp.id})"><i class="bi bi-pencil-fill"></i></button>
                        <button class="btn btn-sm btn-outline-danger" title="Eliminar" onclick="deleteExpense(${exp.id})"><i class="bi bi-trash-fill"></i></button>
                    </td>
                </tr>
            `;
            expensesTableBody.innerHTML += row;
        });
    }

    function renderFinancialReport(startDate = null, endDate = null) {
        // 1. Calcular Ingresos (Facturas de Venta Pagadas)
        const totalIncome = appState.invoices
            .filter(inv => {
                const isPaid = inv.status === 'Pagada';
                const isWithinDateRange = (!startDate || inv.invoice_date >= startDate) && (!endDate || inv.invoice_date <= endDate);
                return isPaid && isWithinDateRange;
            })
            .reduce((sum, inv) => sum + (parseFloat(inv.totals.total) || 0), 0);

        // 2. Calcular Costos de Proyectos (Egresos dentro de cada proyecto)
        const totalProjectCosts = appState.projects
            .flatMap(p => p.egresos || [])
            .filter(expense => {
                return (!startDate || expense.date >= startDate) && (!endDate || expense.date <= endDate);
            })
            .reduce((sum, expense) => sum + (parseFloat(expense.amount) || 0), 0);

        // 3. Calcular Gastos Generales (Facturas de Gasto Pagadas)
        const totalGeneralExpenses = appState.expenses
            .filter(exp => {
                const isPaid = exp.status === 'Pagada';
                const isWithinDateRange = (!startDate || exp.expense_date >= startDate) && (!endDate || exp.expense_date <= endDate);
                return isPaid && isWithinDateRange;
            })
            .reduce((sum, exp) => sum + (parseFloat(exp.totals.total) || 0), 0);

        // 4. Calcular Resultados
        const grossProfit = totalIncome - totalProjectCosts - totalGeneralExpenses;
        const taxableBase = grossProfit / 1.1; // Base imponible antes de IVA
        const iva = grossProfit - taxableBase; // IVA 10%
        const netProfit = taxableBase; // Utilidad Neta

        // Título dinámico para el informe
        const reportTitle = (startDate && endDate) 
            ? `Estado de Resultados (${formatDate(startDate)} - ${formatDate(endDate)})`
            : 'Estado de Resultados (General)';

        // 5. Renderizar
        reportContentEl.innerHTML = `
            <ul class="list-group list-group-flush">
                <li class="list-group-item d-flex justify-content-between align-items-center">
                    <strong>(+) Ingresos Totales (Pagados)</strong>
                    <span class="badge bg-success rounded-pill fs-6">${formatCurrency(totalIncome)}</span>
                </li>
                <li class="list-group-item d-flex justify-content-between align-items-center">
                    <strong>(-) Costos de Proyectos</strong>
                    <span class="badge bg-warning text-dark rounded-pill fs-6">${formatCurrency(totalProjectCosts)}</span>
                </li>
                <li class="list-group-item d-flex justify-content-between align-items-center">
                    <strong>(-) Gastos Generales (Pagados)</strong>
                    <span class="badge bg-danger rounded-pill fs-6">${formatCurrency(totalGeneralExpenses)}</span>
                </li>
                <li class="list-group-item d-flex justify-content-between align-items-center list-group-item-secondary">
                    <strong>(=) Utilidad Bruta</strong>
                    <strong class="fs-5">${formatCurrency(grossProfit)}</strong>
                </li>
                <li class="list-group-item d-flex justify-content-between align-items-center">
                    <span>(-) IVA (10%)</span>
                    <span>${formatCurrency(iva)}</span>
                </li>
                <li class="list-group-item d-flex justify-content-between align-items-center list-group-item-primary">
                    <strong class="fs-5">(=) Utilidad Neta (Resultado Final)</strong>
                    <strong class="fs-4">${formatCurrency(netProfit)}</strong>
                </li>
            </ul>
        `;

        // Actualizar el título en la tarjeta del informe
        const cardHeader = reportContentEl.closest('.card').querySelector('.card-header h4');
        if (cardHeader) cardHeader.textContent = reportTitle;

        renderEvolutionChart(startDate, endDate);
    }

    function printReport() {
        const reportHtml = reportContentEl.innerHTML;
        const printWindow = window.open('', '_blank', 'height=600,width=800');
        printWindow.document.write('<html><head><title>Informe de Resultados</title>');
        // Incluir Bootstrap para que se vea bien
        printWindow.document.write('<link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">');
        printWindow.document.write('<style>body { padding: 20px; } .badge { font-size: 1rem !important; }</style>');
        printWindow.document.write('</head><body>');
        printWindow.document.write('<h1>Informe de Resultados</h1>');
        printWindow.document.write(reportHtml);
        printWindow.document.write('</body></html>');
        printWindow.document.close();
        printWindow.print();
    }

    function renderEvolutionChart(startDate = null, endDate = null) {
        const monthlyData = {};

        // 1. Procesar Ingresos
        appState.invoices.forEach(inv => {
            const isPaid = inv.status === 'Pagada';
            const isWithinDateRange = (!startDate || inv.invoice_date >= startDate) && (!endDate || inv.invoice_date <= endDate);
            if (isPaid && isWithinDateRange) {
                const month = inv.invoice_date.substring(0, 7); // YYYY-MM
                if (!monthlyData[month]) monthlyData[month] = { income: 0, expenses: 0 };
                monthlyData[month].income += parseFloat(inv.totals.total) || 0;
            }
        });

        // 2. Procesar Costos de Proyectos
        appState.projects.flatMap(p => p.egresos || []).forEach(expense => {
            const isWithinDateRange = (!startDate || expense.date >= startDate) && (!endDate || expense.date <= endDate);
            if (isWithinDateRange) {
                const month = expense.date.substring(0, 7);
                if (!monthlyData[month]) monthlyData[month] = { income: 0, expenses: 0 };
                monthlyData[month].expenses += parseFloat(expense.amount) || 0;
            }
        });

        // 3. Procesar Gastos Generales
        appState.expenses.forEach(exp => {
            const isPaid = exp.status === 'Pagada';
            const isWithinDateRange = (!startDate || exp.expense_date >= startDate) && (!endDate || exp.expense_date <= endDate);
            if (isPaid && isWithinDateRange) {
                const month = exp.expense_date.substring(0, 7);
                if (!monthlyData[month]) monthlyData[month] = { income: 0, expenses: 0 };
                monthlyData[month].expenses += parseFloat(exp.totals.total) || 0;
            }
        });

        // 4. Preparar datos para el gráfico
        const sortedMonths = Object.keys(monthlyData).sort();
        const labels = sortedMonths.map(month => new Date(month + '-02').toLocaleString('es-ES', { month: 'short', year: 'numeric' }));
        const incomeData = sortedMonths.map(month => monthlyData[month].income);
        const expenseData = sortedMonths.map(month => monthlyData[month].expenses);

        // 5. Renderizar el gráfico
        const ctx = document.getElementById('evolution-chart').getContext('2d');
        if (appState.evolutionChart) {
            appState.evolutionChart.destroy(); // Destruir gráfico anterior para evitar solapamientos
        }
        appState.evolutionChart = new Chart(ctx, {
            type: 'line',
            data: {
                labels: labels,
                datasets: [{
                    label: 'Ingresos',
                    data: incomeData,
                    borderColor: 'rgba(25, 135, 84, 1)',
                    backgroundColor: 'rgba(25, 135, 84, 0.2)',
                    fill: true,
                    tension: 0.1
                }, {
                    label: 'Gastos Totales',
                    data: expenseData,
                    borderColor: 'rgba(220, 53, 69, 1)',
                    backgroundColor: 'rgba(220, 53, 69, 0.2)',
                    fill: true,
                    tension: 0.1
                }]
            },
            options: {
                responsive: true,
                scales: { y: { beginAtZero: true } }
            }
        });
    }

    function populateClientsDropdown() {
        invoiceClientDropdown.innerHTML = '<option value="">Seleccione un cliente...</option>';
        appState.clients.forEach(client => {
            if (client.estado === 'Activo') {
                invoiceClientDropdown.innerHTML += `<option value="${client.id}">${client.nombre_empresa}</option>`;
            }
        });
    }

    // --- FUNCIONES DE MANEJO DE EVENTOS Y FORMULARIOS ---

    // Guardar Factura de Venta
    saveInvoiceBtn.addEventListener('click', async () => {
        const total = parseFloat(document.getElementById('invoice-total').value);
        const iva10 = total / 11;

        const invoiceData = {
            client_id: document.getElementById('invoice-client-id').value,
            invoice_number: document.getElementById('invoice-number').value,
            invoice_date: document.getElementById('invoice-date').value,
            due_date: document.getElementById('invoice-due-date').value,
            status: document.getElementById('invoice-status').value,
            items: [], // Simplificado por ahora
            totals: {
                subtotal: total,
                iva10: iva10,
                iva5: 0,
                total: total
            }
        };

        if (!invoiceData.client_id || !invoiceData.invoice_number || !invoiceData.invoice_date) {
            alert('Cliente, Número de Factura y Fecha son obligatorios.');
            return;
        }

        const url = appState.invoiceEditMode ? `${INVOICES_API_URL}?id=${appState.currentInvoiceId}` : INVOICES_API_URL;
        const method = appState.invoiceEditMode ? 'PUT' : 'POST';

        try {
            const response = await fetch(url, {
                method: method,
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(invoiceData)
            });
            if (!response.ok) throw new Error(`Error al ${appState.invoiceEditMode ? 'actualizar' : 'guardar'} la factura.`);
            
            invoiceModal.hide();
            initialLoad();
        } catch (error) {
            alert(error.message);
        }
    });

    // Guardar Factura de Gasto
    saveExpenseBtn.addEventListener('click', async () => {
        const total = parseFloat(document.getElementById('expense-total').value);
        const iva10 = total / 11;

        const expenseData = {
            provider: document.getElementById('expense-provider').value,
            expense_number: document.getElementById('expense-number').value,
            expense_date: document.getElementById('expense-date').value,
            due_date: document.getElementById('expense-due-date').value,
            status: document.getElementById('expense-status').value,
            items: [], // Simplificado
            totals: {
                subtotal: total,
                iva10: iva10,
                iva5: 0,
                total: total
            }
        };

        if (!expenseData.provider || !expenseData.expense_number || !expenseData.expense_date) {
            alert('Proveedor, Número de Factura y Fecha son obligatorios.');
            return;
        }

        const url = appState.expenseEditMode ? `${EXPENSES_API_URL}?id=${appState.currentExpenseId}` : EXPENSES_API_URL;
        const method = appState.expenseEditMode ? 'PUT' : 'POST';

        try {
            const response = await fetch(url, {
                method: method,
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(expenseData)
            });
            if (!response.ok) throw new Error(`Error al ${appState.expenseEditMode ? 'actualizar' : 'guardar'} el gasto.`);
            
            expenseModal.hide();
            initialLoad();
        } catch (error) {
            alert(error.message);
        }
    });

    // Limpiar modales al cerrar
    invoiceModalEl.addEventListener('hidden.bs.modal', () => {
        invoiceModalEl.querySelector('form').reset();
        appState.invoiceEditMode = false;
        appState.currentInvoiceId = null;
        invoiceModalLabel.textContent = 'Añadir Factura de Venta';
    });

    expenseModalEl.addEventListener('hidden.bs.modal', () => {
        expenseModalEl.querySelector('form').reset();
        appState.expenseEditMode = false;
        appState.currentExpenseId = null;
        expenseModalLabel.textContent = 'Añadir Factura de Gasto';
    });

    // --- INICIO ---
    initialLoad();
});

// --- FUNCIONES CRUD GLOBALES (para onclick) ---
// Se mueven fuera del DOMContentLoaded para que sean accesibles globalmente

function handleEditInvoice(id) {
    const invoiceModalEl = document.getElementById('addInvoiceModal');
    const invoiceModal = bootstrap.Modal.getInstance(invoiceModalEl);
    const invoiceModalLabel = document.getElementById('addInvoiceModalLabel');

    const invoice = window.appState.invoices.find(inv => inv.id === id);
    if (!invoice) return;

    window.appState.invoiceEditMode = true;
    window.appState.currentInvoiceId = id;

    invoiceModalLabel.textContent = 'Editar Factura de Venta';
    document.getElementById('invoice-client-id').value = invoice.client_id;
    document.getElementById('invoice-number').value = invoice.invoice_number;
    document.getElementById('invoice-date').value = invoice.invoice_date;
    document.getElementById('invoice-due-date').value = invoice.due_date;
    document.getElementById('invoice-total').value = invoice.totals.total;
    document.getElementById('invoice-status').value = invoice.status;

    invoiceModal.show();
};

async function deleteInvoice(id) {
    if (!confirm('¿Estás seguro de que quieres eliminar esta factura de venta?')) return;

    try {
        const response = await fetch(`api/invoices.php?id=${id}`, { method: 'DELETE' });
        // DELETE puede devolver 204 No Content, que es una respuesta de éxito
        if (response.status !== 204 && !response.ok) {
             const errorData = await response.json().catch(() => ({}));
             throw new Error(errorData.error || 'Error al eliminar la factura.');
        }
        // Disparar un evento personalizado para que el otro script recargue los datos
        document.dispatchEvent(new CustomEvent('dataChanged'));
    } catch (error) {
        alert(error.message);
    }
};

// Placeholder para egresos
function handleEditExpense(id) {
    const expenseModalEl = document.getElementById('addExpenseModal');
    const expenseModal = bootstrap.Modal.getInstance(expenseModalEl);
    const expenseModalLabel = document.getElementById('addExpenseModalLabel');

    const expense = window.appState.expenses.find(exp => exp.id === id);
    if (!expense) return;

    window.appState.expenseEditMode = true;
    window.appState.currentExpenseId = id;

    expenseModalLabel.textContent = 'Editar Factura de Gasto';
    document.getElementById('expense-provider').value = expense.provider;
    document.getElementById('expense-number').value = expense.expense_number;
    document.getElementById('expense-date').value = expense.expense_date;
    document.getElementById('expense-due-date').value = expense.due_date;
    document.getElementById('expense-total').value = expense.totals.total;
    document.getElementById('expense-status').value = expense.status;

    expenseModal.show();
};

async function deleteExpense(id) {
    if (!confirm('¿Estás seguro de que quieres eliminar esta factura de gasto?')) return;

    try {
        const response = await fetch(`api/expenses.php?id=${id}`, { method: 'DELETE' });
        if (response.status !== 204 && !response.ok) {
             const errorData = await response.json().catch(() => ({}));
             throw new Error(errorData.error || 'Error al eliminar la factura de gasto.');
        }
        document.dispatchEvent(new CustomEvent('dataChanged'));
    } catch (error) {
        alert(error.message);
    }
};

// Listener para el evento personalizado de recarga
document.addEventListener('dataChanged', function() {
    // Buscamos la función initialLoad dentro del otro script y la llamamos si existe
    // Esto es un poco "hacky", pero funciona en este contexto.
    // La forma ideal sería tener un solo punto de entrada que maneje todo.
    location.reload(); // La forma más simple y robusta de asegurar que todo se actualice.
});