document.addEventListener('DOMContentLoaded', function() {

    // --- ELEMENTOS DEL DOM ---
    const activeProjectsCountEl = document.getElementById('active-projects-count');
    const pendingTasksCountEl = document.getElementById('pending-tasks-count');
    const overdueTasksCountEl = document.getElementById('overdue-tasks-count');
    const recentProjectsListEl = document.getElementById('recent-projects-list');
    const upcomingTasksListEl = document.getElementById('upcoming-tasks-list');
    const calendarEl = document.getElementById('calendar');

    // --- URLs DE LA API ---
    const PROJECTS_API_URL = 'api/projects.php'; // Corregido el path
    const TASKS_API_URL = 'api/tasks.php';
    const MEDIA_PLANS_API_URL = 'api/media_plans.php';
    const GRID_POSTS_API_URL = 'api/grid_posts.php'; // Nueva API para los posteos

    // --- FUNCIONES PRINCIPALES ---

    async function initialLoad() {
        try {
            const [projectsRes, tasksRes, mediaPlansRes, gridPostsRes] = await Promise.all([
                fetch(PROJECTS_API_URL),
                fetch(TASKS_API_URL),
                fetch(MEDIA_PLANS_API_URL),
                fetch(GRID_POSTS_API_URL) // Añadimos la llamada
            ]);

            if (!projectsRes.ok) throw new Error('No se pudieron cargar los proyectos.');
            if (!tasksRes.ok) throw new Error('No se pudieron cargar las tareas.');
            if (!mediaPlansRes.ok) throw new Error('No se pudieron cargar los planes de medios.');
            if (!gridPostsRes.ok) throw new Error('No se pudieron cargar los posteos de la grilla.');

            const projects = await projectsRes.json();
            const tasks = await tasksRes.json();
            const mediaPlans = await mediaPlansRes.json();
            const gridPosts = await gridPostsRes.json();

            renderStatCards(projects, tasks);
            renderRecentProjects(projects);
            renderUpcomingTasks(tasks, projects);
            renderCalendar(tasks, projects, mediaPlans, gridPosts);

        } catch (error) {
            console.error("Error en la carga inicial del dashboard:", error);
            // Podríamos mostrar un error general en el dashboard
        }
    }

    /**
     * Formatea una fecha en formato YYYY-MM-DD a DD/MM/YYYY.
     * @param {string} dateString La fecha en formato YYYY-MM-DD.
     * @returns {string} La fecha formateada o 'N/A' si la entrada es inválida.
     */
    function formatDate(dateString) {
        if (!dateString) return 'N/A';
        const [year, month, day] = dateString.split('-');
        if (!year || !month || !day) return dateString; // Devuelve original si no tiene el formato esperado
        return `${day}/${month}/${year}`;
    }

    // --- FUNCIONES DE RENDERIZADO ---

    function renderStatCards(projects, tasks) {
        const activeProjects = projects.filter(p => ['Aprobado', 'En Progreso'].includes(p.estado));
        activeProjectsCountEl.textContent = activeProjects.length;
        
        const today = new Date();
        today.setHours(0, 0, 0, 0); // Normalizar a medianoche

        // Tareas Vencidas: No están al 100% y su fecha de fin ya pasó.
        const overdueTasks = tasks.filter(t => {
            // Solo considerar tareas que tienen fecha de fin
            if (!t.fecha_fin || t.porcentaje >= 100) return false;
            const endDate = new Date(t.fecha_fin); // JS interpreta YYYY-MM-DD como UTC, lo cual es bueno para consistencia.
            return endDate < today;
        });
        overdueTasksCountEl.textContent = overdueTasks.length;

        // Tareas Pendientes: No están al 100% y su fecha de fin no ha pasado (o no tienen fecha).
        const pendingTasks = tasks.filter(t => t.porcentaje < 100);
        // Restamos las vencidas para que los contadores no se solapen.
        pendingTasksCountEl.textContent = pendingTasks.length - overdueTasks.length;
    }

    function renderRecentProjects(projects) {
        recentProjectsListEl.innerHTML = '';
        // Ordenar por ID descendente para obtener los más nuevos
        const recentProjects = projects.sort((a, b) => b.id - a.id).slice(0, 5);

        if (recentProjects.length === 0) {
            recentProjectsListEl.innerHTML = '<li class="list-group-item text-muted">No hay proyectos recientes.</li>';
            return;
        }

        recentProjects.forEach(project => {
            const li = document.createElement('li');
            li.className = 'list-group-item d-flex justify-content-between align-items-center';
            li.innerHTML = `
                <a href="project-detail.html?id=${project.id}">${project.nombre_proyecto}</a>
                <span class="badge bg-info text-dark">${project.estado}</span>
            `;
            recentProjectsListEl.appendChild(li);
        });
    }

    function renderUpcomingTasks(tasks, projects) {
        upcomingTasksListEl.innerHTML = '';
        const projectMap = new Map(projects.map(p => [p.id, p.nombre_proyecto]));

        const today = new Date();
        const nextWeek = new Date();
        nextWeek.setDate(today.getDate() + 7);

        const upcomingTasks = tasks.filter(t => {
            if (t.porcentaje >= 100 || !t.fecha_fin) return false;
            const endDate = new Date(t.fecha_fin + 'T00:00:00');
            return endDate >= today && endDate <= nextWeek;
        }).sort((a, b) => new Date(a.fecha_fin) - new Date(b.fecha_fin));

        if (upcomingTasks.length === 0) {
            upcomingTasksListEl.innerHTML = '<li class="list-group-item text-muted">No hay tareas venciendo en los próximos 7 días.</li>';
            return;
        }

        upcomingTasks.slice(0, 5).forEach(task => {
            const li = document.createElement('li');
            li.className = 'list-group-item';
            const projectName = projectMap.get(parseInt(task.project_id)) || 'Proyecto desconocido';
            li.innerHTML = `
                <a href="project-detail.html?id=${task.project_id}">${task.nombre_tarea}</a>
                <small class="d-block text-muted">
                    Proyecto: ${projectName} | Vence: ${formatDate(task.fecha_fin)}
                </small>
            `;
            upcomingTasksListEl.appendChild(li);
        });
    }

    function renderCalendar(tasks, projects, mediaPlans, gridPosts) {
        // 1. Eventos de Tareas
        const taskEvents = tasks.filter(task => task.fecha_fin).map(task => {
            let color = '#0d6efd'; // Azul por defecto
            if (task.porcentaje == 100) {
                color = '#198754'; // Verde (completada)
            } else if (new Date(task.fecha_fin) < new Date() && task.porcentaje < 100) {
                color = '#dc3545'; // Rojo (vencida)
            }

            return {
                title: `Vence Tarea: ${task.nombre_tarea}`,
                start: task.fecha_fin, // Evento de un solo día en la fecha de fin
                allDay: true,
                url: `project-detail.html?id=${task.project_id}`,
                backgroundColor: color,
                borderColor: color
            };
        });

        // 2. Eventos de Vencimiento de Proyectos
        const projectEvents = projects.filter(p => p.fecha_fin_estimada).map(project => ({
            title: `FIN PROYECTO: ${project.nombre_proyecto}`,
            start: project.fecha_fin_estimada,
            allDay: true,
            url: `project-detail.html?id=${project.id}`,
            backgroundColor: '#fd7e14', // Naranja
            borderColor: '#fd7e14'
        }));

        // 3. Eventos de Vencimiento de Planes de Medios
        const mediaPlanEvents = [];
        mediaPlans.forEach(plan => {
            // Evento para el fin del plan general
            if (plan.end_date) {
                mediaPlanEvents.push({
                    title: `Fin Plan: ${plan.name}`,
                    start: plan.end_date,
                    allDay: true,
                    url: `media-plan-detail.html?project_id=${plan.project_id}`,
                    backgroundColor: '#0dcaf0', // Cyan
                    borderColor: '#20c997'
                });
            }
            // Eventos para cada línea de pauta (item) dentro del plan
            (plan.items || []).forEach(item => {
                if (item.endDate) {
                    mediaPlanEvents.push({
                        title: `Fin Pauta: ${item.media}`,
                        start: item.endDate,
                        allDay: true,
                        url: `media-plan-editor.html?plan_id=${plan.id}`,
                        backgroundColor: '#6610f2', // Indigo para pautas
                        borderColor: '#6610f2'
                    });
                }
            });
        });

        // 4. Eventos de Posteos de Redes Sociales
        const postEvents = gridPosts.filter(post => post.fecha_posteo).map(post => ({
            title: `Posteo: ${post.tipo_contenido}`,
            start: `${post.fecha_posteo}T${post.hora_posteo || '12:00:00'}`, // Usar la hora si existe
            allDay: false, // No es todo el día si tiene hora
            url: `grid-editor.html?grid_id=${post.grid_id}`, // Enlace al editor de la grilla
            backgroundColor: '#fd7e14', // Naranja
            borderColor: '#fd7e14'
        }));

        // 5. Combinar todos los eventos
        const allEvents = [...taskEvents, ...projectEvents, ...mediaPlanEvents, ...postEvents];


        const calendar = new FullCalendar.Calendar(calendarEl, {
            initialView: 'dayGridMonth',
            locale: 'es',
            headerToolbar: {
                left: 'prev,next today',
                center: 'title',
                right: 'dayGridMonth,timeGridWeek,listWeek'
            },
            events: allEvents,
            eventTimeFormat: { // Ocultar la hora en la vista de mes
                hour: 'numeric',
                minute: '2-digit',
                meridiem: false
            }
        });
        calendar.render();
    }

    // --- CARGA INICIAL ---
    initialLoad();
});