document.addEventListener('DOMContentLoaded', function () {

    // --- ELEMENTOS DEL DOM ---
    const agencyForm = document.getElementById('agency-settings-form'); // Formulario de datos de la agencia
    const saveAgencyBtn = document.getElementById('save-settings-btn'); // Botón de guardar datos de la agencia
    const logoInput = document.getElementById('logo'); // Input de archivo para el logo
    const logoPreview = document.getElementById('logo-preview'); // Vista previa del logo

    const systemForm = document.getElementById('system-settings-form'); // Formulario de ajustes del sistema
    const saveSystemBtn = document.getElementById('save-system-settings-btn'); // Botón de guardar ajustes del sistema

    const aiForm = document.getElementById('ai-settings-form'); // Formulario de ajustes de IA
    // --- URLs DE LA API ---
    const SETTINGS_API_URL = 'api/settings.php';
    const COMPANY_INFO_KEY = 'company_info';

    // --- FUNCIONES ---

    /**
     * Obtiene un valor de un objeto de forma segura.
     * @param {object} obj El objeto.
     * @param {string} key La clave a buscar.
     * @param {*} defaultValue El valor por defecto si la clave no existe.
     * @returns {*} El valor de la clave o el valor por defecto.
     */
    const getOrDefault = (obj, key, defaultValue = '') => {
        return obj && obj[key] !== undefined ? obj[key] : defaultValue;
    };

    /**
     * Carga la configuración de la empresa desde la API y la muestra en el formulario.
     */
    async function loadCompanyInfo() {
        try {
            const response = await fetch(`${SETTINGS_API_URL}?key=${COMPANY_INFO_KEY}`);
            if (response.status === 404) {
                console.log('No se encontró configuración de la empresa. Se puede crear una nueva.');
                return; // No hay datos, es normal en la primera ejecución
            }
            if (!response.ok) {
                throw new Error('No se pudo cargar la configuración de la empresa.');
            }
            
            const settings = await response.json();
            const agencyData = settings.agency_data || {};
            const systemData = settings.system_settings || {};
            const aiData = settings.ai_settings || {};
            const metaData = settings.meta_settings || {};

            // Rellenar formulario de Agencia
            document.getElementById('agency_name').value = getOrDefault(agencyData, 'agency_name');
            document.getElementById('business_name').value = getOrDefault(agencyData, 'business_name');
            document.getElementById('ruc').value = getOrDefault(agencyData, 'ruc');
            document.getElementById('legal_rep').value = getOrDefault(agencyData, 'legal_rep');
            document.getElementById('commercial_email').value = getOrDefault(agencyData, 'commercial_email');
            document.getElementById('commercial_phone').value = getOrDefault(agencyData, 'commercial_phone');
            document.getElementById('finance_email').value = getOrDefault(agencyData, 'finance_email');
            document.getElementById('finance_phone').value = getOrDefault(agencyData, 'finance_phone');
            document.getElementById('website').value = getOrDefault(agencyData, 'website');
            document.getElementById('address').value = getOrDefault(agencyData, 'address');
            document.getElementById('city').value = getOrDefault(agencyData, 'city');
            document.getElementById('country').value = getOrDefault(agencyData, 'country');
            if (agencyData.logo_url) {
                logoPreview.src = agencyData.logo_url;
                logoPreview.style.display = 'block';
            } else {
                logoPreview.style.display = 'none';
            }

            // Rellenar formulario de Ajustes del Sistema
            document.getElementById('thousands_separator').value = getOrDefault(systemData, 'thousands_separator', '.');
            document.getElementById('decimal_separator').value = getOrDefault(systemData, 'decimal_separator', ',');
            document.getElementById('decimal_places').value = getOrDefault(systemData, 'decimal_places', 0);
            document.getElementById('percentage_decimals').value = getOrDefault(systemData, 'percentage_decimals', 1);
            document.getElementById('currency_code').value = getOrDefault(systemData, 'currency_code', 'PYG');
            document.getElementById('currency_symbol').value = getOrDefault(systemData, 'currency_symbol', 'Gs.');
            
            // Rellenar formulario de IA
            document.getElementById('ai_provider').value = getOrDefault(aiData, 'ai_provider', 'gemini');
            document.getElementById('ai_model').value = getOrDefault(aiData, 'ai_model');
            const apiKeyField = document.getElementById('api_key');
            if (apiKeyField && aiData.api_key) {
                apiKeyField.placeholder = "•••••••••••••••••••• (Clave guardada)"; // No mostrar la clave real
            }
        } catch (error) {
            console.error('Error al cargar la configuración:', error);
            alert('Error al cargar la configuración: ' + error.message);
        }
    }

    /**
     * Guarda la configuración de la empresa en la base de datos.
     * @param {Event} event El evento de submit.
     */
    async function saveSettings(event) {
        event.preventDefault();

        const form = event.target;
        const submitButton = form.querySelector('button[type="submit"]');
        const formType = form.querySelector('input[name="form_type"]').value;

        const formData = new FormData(form);

        // Manejo especial para la API Key de IA y Meta App Secret
        if (formType === 'ai_settings' && !document.getElementById('api_key').value) {
            formData.delete('api_key'); // No enviar si está vacío, para mantener la existente
        }
        if (formType === 'meta_settings' && !document.getElementById('meta_app_secret').value) {
            formData.delete('meta_app_secret'); // No enviar si está vacío, para mantener la existente
        }

        submitButton.disabled = true;
        submitButton.innerHTML = '<span class="spinner-border spinner-border-sm" role="status" aria-hidden="true"></span> Guardando...';

        try {
            const response = await fetch(SETTINGS_API_URL, {
                method: 'POST',
                body: formData
            });

            if (!response.ok) {
                const errorData = await response.json();
                throw new Error(errorData.error || 'Falló la solicitud para guardar la configuración.');
            }

            const result = await response.json();
            alert(result.message || 'Configuración guardada con éxito.');

            // Actualizar la vista previa del logo si se subió uno nuevo
            if (formType === 'agency_data' && result.data && result.data.agency_data && result.data.agency_data.logo_url) {
                logoPreview.src = result.data.agency_data.logo_url + '?t=' + new Date().getTime(); // Cache busting
                logoPreview.style.display = 'block';
            }
            
            loadSettings(); // Recargar para actualizar el estado de los botones de Meta y placeholders

        } catch (error) {
            console.error('Error al guardar la configuración:', error);
            alert('Error: ' + error.message);
        } finally {
            submitButton.disabled = false;
            submitButton.innerHTML = 'Guardar Cambios'; // O el texto original del botón
        }
    }

    async function disconnectFromMeta() {
        if (!confirm('¿Estás seguro de que quieres desconectar tu cuenta de Meta? Se eliminarán los tokens de acceso y no podrás programar posteos ni ver métricas.')) {
            return;
        }

        disconnectMetaBtn.disabled = true;
        disconnectMetaBtn.innerHTML = '<span class="spinner-border spinner-border-sm"></span> Desconectando...';

        try {
            const response = await fetch(SETTINGS_API_URL, {
                method: 'POST',
                headers: { 'Content-Type': 'application/x-www-form-urlencoded' }, // FormData no es necesario aquí
                body: 'form_type=disconnect_meta'
            });

            if (!response.ok) throw new Error('Falló la solicitud para desconectar la cuenta de Meta.');

            const result = await response.json();
            alert(result.message || 'Cuenta de Meta desconectada.');
            loadSettings(); // Recargar para actualizar la UI
        } catch (error) {
            console.error('Error al desconectar la cuenta de Meta:', error);
            alert('Error: ' + error.message);
        } finally {
            disconnectMetaBtn.disabled = false;
            disconnectMetaBtn.innerHTML = '<i class="bi bi-x-circle-fill"></i> Desconectar Cuenta';
        }
    }

    function connectToMeta() {
        // Añadimos una comprobación para no redirigir si el botón está deshabilitado
        if (connectMetaBtn.disabled) {
            alert('Por favor, primero guarda el "Meta App ID" y el "Meta App Secret" antes de conectar.');
            return;
        }
        window.location.href = 'api/meta_auth.php';
    }

    // --- EVENT LISTENERS ---
    agencyForm.addEventListener('submit', saveSettings);
    systemForm.addEventListener('submit', saveSettings);
    aiForm.addEventListener('submit', saveSettings);
    metaForm.addEventListener('submit', saveSettings);
    connectMetaBtn.addEventListener('click', connectToMeta);
    disconnectMetaBtn.addEventListener('click', disconnectFromMeta);

    logoInput.addEventListener('change', () => {
        const file = logoInput.files[0];
        if (file) {
            logoPreview.src = URL.createObjectURL(file);
        }
    }

    // --- EVENT LISTENERS ---
    // --- CARGA INICIAL ---
    loadCompanyInfo();
});