document.addEventListener('DOMContentLoaded', function () {

    // --- URLs DE API ---
    const ANALISIS_API_URL = 'api/analizar_datos.php';

    // --- ELEMENTOS DEL DOM ---
    const analisisForm = document.getElementById('analisis-form');
    const generateBtn = document.getElementById('generate-report-btn');
    const loadingSpinner = document.getElementById('loading-spinner');
    const reportContainer = document.getElementById('report-container');
    const reportContent = document.getElementById('report-content');
    const downloadBtn = document.getElementById('download-report-btn');

    /**
     * Maneja el envío del formulario de análisis.
     * Envía los archivos y el prompt al backend y muestra el resultado.
     */
    async function handleAnalysisSubmit(event) {
        event.preventDefault();

        const formData = new FormData(analisisForm);
        const prompt = formData.get('prompt');
        const files = document.getElementById('data-files').files;

        // Validación básica en el frontend
        if (!prompt && files.length === 0) {
            alert('Por favor, escriba una solicitud o adjunte al menos un archivo.');
            return;
        }

        // Mostrar estado de carga
        generateBtn.disabled = true;
        generateBtn.innerHTML = '<i class="bi bi-hourglass-split me-2"></i>Generando...';
        loadingSpinner.style.display = 'block';
        reportContainer.style.display = 'none'; // Ocultar resultados anteriores

        try {
            const response = await fetch(ANALISIS_API_URL, {
                method: 'POST',
                body: formData
            });

            if (!response.ok) {
                const errorData = await response.json().catch(() => null); // Intenta parsear el error
                throw new Error(errorData?.message || `Error del servidor: ${response.statusText}`);
            }

            const result = await response.json();

            if (result.success) {
                // Mostrar el reporte
                reportContent.innerHTML = result.data.report_html;
                reportContainer.style.display = 'block';
            } else {
                throw new Error(result.message || 'Ocurrió un error desconocido en el servidor.');
            }

        } catch (error) {
            console.error('Error al generar el reporte:', error);
            alert(`Error: ${error.message}`);
            // Opcional: mostrar el error en la UI
            reportContent.innerHTML = `<div class="alert alert-danger"><strong>Error:</strong> ${error.message}</div>`;
            reportContainer.style.display = 'block';
        } finally {
            // Restaurar estado del botón y ocultar spinner
            generateBtn.disabled = false;
            generateBtn.innerHTML = '<i class="bi bi-magic me-2"></i>Generar Reporte';
            loadingSpinner.style.display = 'none';
        }
    }

    /**
     * Descarga el contenido del reporte como un archivo HTML.
     */
    function downloadReport() {
        const reportHtml = `<!DOCTYPE html><html><head><title>Reporte IA</title><link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet"></head><body class="p-4">${reportContent.innerHTML}</body></html>`;
        const blob = new Blob([reportHtml], { type: 'text/html' });
        const url = URL.createObjectURL(blob);
        const a = document.createElement('a');
        a.href = url;
        a.download = 'reporte_ia.html';
        a.click();
        URL.revokeObjectURL(url);
    }

    // --- EVENT LISTENERS ---
    analisisForm.addEventListener('submit', handleAnalysisSubmit);
    downloadBtn.addEventListener('click', downloadReport);
});