document.addEventListener('DOMContentLoaded', function() {
    // Elementos del DOM
    const calendarEl = document.getElementById('task-calendar');
    const professionalFilterEl = document.getElementById('professional-filter');
    const priorityFilterEl = document.getElementById('priority-filter');
    const tagFilterEl = document.getElementById('tag-filter');

    // --- URLs DE LA API ---
    const TASKS_API_URL = 'api/tasks.php';
    const PROFESSIONALS_API_URL = 'api/professionals.php';

    // --- ESTADO ---
    let appState = {
        tasks: [],
        professionals: [],
        calendar: null
    };

    // --- INICIALIZACIÓN ---
    async function initialLoad() {
        try {
            const [tasksRes, professionalsRes] = await Promise.all([ // Cargar tareas y profesionales en paralelo
                fetch(TASKS_API_URL),
                fetch(PROFESSIONALS_API_URL)
            ]);
            
            if (!tasksRes.ok) throw new Error('No se pudieron cargar las tareas.');
            if (!professionalsRes.ok) throw new Error('No se pudieron cargar los profesionales.');
            
            appState.tasks = await tasksRes.json();
            appState.professionals = await professionalsRes.json();

            populateFilters();
            initializeCalendar();
            
        } catch (error) {
            console.error("Error en la carga inicial de la agenda:", error);
            if (calendarEl) {
                calendarEl.innerHTML = `<div class="alert alert-danger">No se pudo cargar la agenda. ${error.message}</div>`;
            }
        }
    }

    function initializeCalendar() {
        if (!calendarEl) return;

        appState.calendar = new FullCalendar.Calendar(calendarEl, {
            initialView: 'dayGridMonth',
            locale: 'es',
            headerToolbar: {
                left: 'prev,next today',
                center: 'title',
                right: 'dayGridMonth,timeGridWeek,listWeek'
            },
            buttonText: {
                today: 'Hoy',
                month: 'Mes',
                week: 'Semana',
                list: 'Lista'
            },
            events: getFilteredEvents(),
            eventClick: function(info) { // Manejar clic en un evento
                info.jsEvent.preventDefault(); // Prevenir la acción por defecto del navegador
                // Redirigir al detalle del proyecto de la tarea
                if (info.event.extendedProps.projectId) {
                    window.location.href = `project-detail.html?id=${info.event.extendedProps.projectId}`;
                }
            },
            eventDidMount: function(info) {
                // Añadir popover con más detalles al pasar el ratón
                new bootstrap.Popover(info.el, {
                    title: info.event.title,
                    content: info.event.extendedProps.description || 'Sin descripción.',
                    trigger: 'hover',
                    placement: 'top',
                    container: 'body',
                    html: true
                });
            }
        });
        appState.calendar.render();
    }

    // --- FILTROS Y EVENTOS ---

    function populateFilters() {
        // Profesionales
        professionalFilterEl.innerHTML = '<option value="all">Todos los Responsables</option>'; // Opción por defecto
        appState.professionals.forEach(p => {
            professionalFilterEl.innerHTML += `<option value="${p.id}">${p.nombre_completo}</option>`; // Añadir cada profesional
        });

        // Prioridades
        priorityFilterEl.innerHTML = '<option value="all">Todas las Prioridades</option>'; // Opción por defecto
        ['Alta', 'Media', 'Baja'].forEach(p => {
            priorityFilterEl.innerHTML += `<option value="${p}">${p}</option>`; // Añadir cada prioridad
        });

        // Etiquetas (Tags)
        const allTags = new Set(appState.tasks.flatMap(t => t.tags || [])); // Recopilar todas las etiquetas únicas
        tagFilterEl.innerHTML = '<option value="all">Todas las Etiquetas</option>'; // Opción por defecto
        allTags.forEach(tag => {
            tagFilterEl.innerHTML += `<option value="${tag}">${tag}</option>`;
        });
    }

    function getFilteredEvents() {
        const selectedProfessional = professionalFilterEl.value;
        const selectedPriority = priorityFilterEl.value;
        const selectedTag = tagFilterEl.value;

        const filteredTasks = appState.tasks.filter(task => {
            const professionalMatch = selectedProfessional === 'all' || task.asignado_a == selectedProfessional; // Filtrar por profesional
            const priorityMatch = selectedPriority === 'all' || (task.prioridad || 'Media') === selectedPriority;
            const tagMatch = selectedTag === 'all' || (task.tags && task.tags.includes(selectedTag));
            return professionalMatch && priorityMatch && tagMatch;
        });

        // Mapear tareas a eventos del calendario
        return filteredTasks.filter(task => task.fecha_fin).map(task => {
            let color = '#0d6efd'; // Color por defecto (azul)
            let title = `Vence: ${task.nombre_tarea}`; // Título del evento
            const today = new Date();
            today.setHours(0, 0, 0, 0); // Normalizar la fecha de hoy a medianoche

            if (task.porcentaje >= 100) { // Tarea completada
                color = '#198754'; // Verde (Completada)
                title = `Completada: ${task.nombre_tarea}`; // Cambiar título
            } else if (new Date(task.fecha_fin + 'T00:00:00') < today) {
                color = '#dc3545'; // Rojo (Vencida)
            }

            return {
                title: title,
                start: task.fecha_fin,
                allDay: true, // Evento de día completo
                backgroundColor: color,
                borderColor: color,
                extendedProps: { 
                    projectId: task.project_id, 
                    description: `Prioridad: ${task.prioridad || 'Media'}` 
                }
            };
        });
    }

    function handleFilterChange() {
        if (!appState.calendar) return;
        appState.calendar.removeAllEvents();
        appState.calendar.addEventSource(getFilteredEvents());
    }

    // --- LISTENERS ---
    [professionalFilterEl, priorityFilterEl, tagFilterEl].forEach(el => el.addEventListener('change', handleFilterChange));

    // --- INICIO ---
    initialLoad();
});