document.addEventListener('DOMContentLoaded', function() {
    const calendarEl = document.getElementById('media-plan-calendar');

    // --- URLs DE LA API ---
    const MEDIA_PLANS_API_URL = 'api/media_plans.php';
    const PROJECTS_API_URL = 'api/projects.php';

    const fetchData = async (url) => {
        const response = await fetch(url);
        if (!response.ok) throw new Error(`Error al cargar ${url}`);
        return response.json();
    };

    async function initialLoad() {
        try {
            const [mediaPlans, projects] = await Promise.all([
                fetchData(MEDIA_PLANS_API_URL),
                fetchData(PROJECTS_API_URL)
            ]);
            
            const projectsMap = new Map(projects.map(p => [p.id, p.nombre_proyecto]));
            renderCalendar(mediaPlans, projectsMap);

        } catch (error) {
            console.error("Error en la carga de la agenda de plan de medios:", error);
            calendarEl.innerHTML = `<div class="alert alert-danger">Error al cargar el calendario: ${error.message}</div>`;
        }
    }

    function renderCalendar(mediaPlans, projectsMap) {
        const events = [];
        mediaPlans.forEach(plan => {
            if (plan.end_date) {
                events.push({
                    title: `Fin Plan: ${plan.name}`,
                    start: plan.end_date,
                    allDay: true,
                    url: `media-plan-detail.html?project_id=${plan.project_id}&plan_id=${plan.id}`,
                    backgroundColor: '#0dcaf0', // Cyan
                    borderColor: '#0dcaf0',
                    extendedProps: {
                        projectName: projectsMap.get(parseInt(plan.project_id)) || 'N/A'
                    }
                });
            }
        });

        const calendar = new FullCalendar.Calendar(calendarEl, {
            initialView: 'dayGridMonth',
            locale: 'es',
            headerToolbar: {
                left: 'prev,next today',
                center: 'title',
                right: 'dayGridMonth,timeGridWeek,listWeek'
            },
            events: events
        });
        calendar.render();
    }

    initialLoad();
});