document.addEventListener('DOMContentLoaded', function() {
    const calendarEl = document.getElementById('editorial-calendar'); // Elemento del calendario
    // Función para obtener datos de una URL
    const fetchData = async (url) => {
        try {
            const response = await fetch(url);
            if (!response.ok) {
                throw new Error(`Error al cargar ${url}: ${response.statusText}`);
            }
            return await response.json();
        } catch (error) {
            console.error(error);
            return [];
        }
    };

    // Cargar todos los datos necesarios en paralelo
    const loadAllData = async () => {
        const [posts, grids, projects, clients, professionals] = await Promise.all([ // Cargar todas las APIs necesarias
            fetchData('api/grid_posts.php'),
            fetchData('api/grids.php'),
            fetchData('api/projects.php'),
            fetchData('api/clients.php'),
            fetchData('api/professionals.php')
        ]);
        // Crear mapas para búsqueda rápida de IDs a nombres
        // Crear mapas para búsqueda rápida
        const gridsMap = new Map(grids.map(grid => [grid.id, grid]));
        const projectsMap = new Map(projects.map(project => [project.id, project]));
        const clientsMap = new Map(clients.map(client => [client.id, client]));
        const professionalsMap = new Map(professionals.map(prof => [prof.id, prof]));

        // Procesar los posteos para crear eventos de calendario
        return posts.filter(post => post.fecha_posteo).map(post => { // Filtrar posteos sin fecha
            const grid = gridsMap.get(post.grid_id);
            if (!grid) return null; // Si la grilla no existe, omitir
            
            const project = projectsMap.get(parseInt(grid.project_id));
            if (!project) return null; // Si el proyecto no existe, omitir
            
            const client = clientsMap.get(parseInt(project.cliente_id));
            const clientName = client ? client.nombre_empresa : 'Cliente no encontrado';
            
            const responsables = (grid.responsables || []) // Obtener nombres de responsables
                .map(id => professionalsMap.get(parseInt(id))?.nombre_completo)
                .filter(Boolean);

            return {
                id: post.id,
                title: `${clientName}: ${post.tipo_contenido || 'Posteo'}`,
                start: `${post.fecha_posteo}T${post.hora_posteo || '09:00:00'}`,
                extendedProps: {
                    client: clientName,
                    objective: post.objetivo,
                    responsables: responsables.join(', ') || 'No asignado',
                    copy: post.copy, // Contenido del post para el tooltip
                    networks: (post.redes || []).join(', ')
                },
                // Asigna un color diferente si el posteo ya fue realizado
                backgroundColor: post.realizado ? '#198754' : '#0d6efd',
                borderColor: post.realizado ? '#146c43' : '#0a58ca'
            };
        }).filter(Boolean); // Eliminar nulos si no se encontró grid o proyecto
    };

    // Inicializar el calendario
    loadAllData().then(events => { // Cargar datos y luego inicializar el calendario
        const calendar = new FullCalendar.Calendar(calendarEl, {
            initialView: 'dayGridMonth',
            locale: 'es',
            headerToolbar: {
                left: 'prev,next today',
                center: 'title',
                right: 'dayGridMonth,timeGridWeek,timeGridDay,listYear' // Vistas de mes, semana, día y lista anual
            }, // Configuración de la barra de herramientas
            events: events,
            eventDidMount: function(info) {
                // Crear un tooltip con Bootstrap
                const props = info.event.extendedProps;
                const tooltipContent = `
                    <h5>${info.event.title}</h5>
                    <p><strong>Cliente:</strong> ${props.client}</p>
                    <p><strong>Objetivo:</strong> ${props.objective}</p>
                    <p><strong>Responsables:</strong> ${props.responsables}</p>
                    <p><strong>Redes:</strong> ${props.networks}</p>
                `;
                new bootstrap.Tooltip(info.el, { // Inicializar el tooltip de Bootstrap
                    title: tooltipContent,
                    html: true,
                    placement: 'top',
                    trigger: 'hover',
                    container: 'body'
                });
            }
        });

        calendar.render();
    }).catch(error => {
        console.error("Error al inicializar el calendario:", error);
        calendarEl.innerHTML = '<div class="alert alert-danger">No se pudo cargar la agenda editorial.</div>';
    });
});