document.addEventListener('DOMContentLoaded', function() {
    // No ejecutar si no estamos en una página principal de la app
    if (!document.querySelector('body')) return;

    let currentUserId = null;
    let currentConversations = [];
    let activeConversationId = null;
    let professionals = [];
    let selectedGroupMembers = new Map();
    let pollingInterval = null;
    let notificationPollingInterval = null;

    const appState = {
        isEditMode: false,
        isChatOpen: false
    };

    // --- Creación de la Interfaz del Chat ---

    function createChatInterface() {
        // 1. Referenciar el botón y el modal existentes en el HTML
        const chatButton = document.getElementById('global-chat-button');
        const chatModalEl = document.getElementById('globalChatModal');
        const chatModal = new bootstrap.Modal(chatModalEl);

        // 2. Añadir listeners
        chatButton.addEventListener('click', () => {
            chatModal.show();
        });
        const notificationBadge = document.getElementById('chat-notification-badge');

        // Cuando el modal se muestra, cargar conversaciones y empezar a refrescar
        chatModalEl.addEventListener('shown.bs.modal', () => {
            appState.isChatOpen = true;
            if (currentUserId) {
                loadConversations();
                startPolling();
            }
            // Ocultar la notificación al abrir el chat
            notificationBadge.style.display = 'none';
        });

        // Cuando el modal se oculta, dejar de refrescar para ahorrar recursos
        chatModalEl.addEventListener('hidden.bs.modal', () => {
            appState.isChatOpen = false;
            stopPolling();
        });

        // Inicializar el resto de la lógica del chat
        initializeChatLogic();
    }

    // --- Lógica del Chat (adaptada de chat.js) ---

    function initializeChatLogic() {
        // Referenciar elementos del DOM recién creados
        const conversationsList = document.getElementById('conv-list-items');
        const chatHeader = document.getElementById('chat-header');
        const chatMessages = document.getElementById('chat-messages');
        const messageInput = document.getElementById('message-input');
        const sendButton = document.getElementById('send-button');
        const newChatButton = document.getElementById('new-chat-button');
        const newChatModalEl = document.getElementById('new-chat-modal');
        const newChatModal = new bootstrap.Modal(newChatModalEl);
        const closeModalButtons = newChatModalEl.querySelectorAll('.close-button, [data-bs-dismiss="modal"]');
        const searchProfessionalInput = document.getElementById('search-professional');
        const groupNameInput = document.getElementById('group-name');
        const groupPhotoInput = document.getElementById('group-photo');
        const groupPhotoPreview = document.getElementById('group-photo-preview');
        const searchGroupMembersInput = document.getElementById('search-group-members');
        const groupMembersList = document.getElementById('group-members-list');
        const selectedMembersContainer = document.getElementById('selected-members-container');
        const createGroupButton = document.getElementById('create-group-button');
        const chatOptionsDropdown = document.getElementById('chat-options-dropdown');
        const editGroupBtn = document.getElementById('edit-group-btn');
        const deleteGroupBtn = document.getElementById('delete-group-btn');

        const chatInput = document.getElementById('chat-input');

        // --- Funciones Principales ---

        async function loadConversations() {
            try {
                const response = await fetch('api/chat.php?action=get_conversations');
                currentConversations = await response.json();
                const conversations = currentConversations;
                
                conversationsList.innerHTML = '';

                if (conversations.length === 0) {
                    conversationsList.innerHTML = '<p style="padding: 15px; text-align: center;">No tienes conversaciones.</p>';
                    return;
                }

                conversations.forEach(conv => {
                    const convElement = document.createElement('div');
                    convElement.href = '#';
                    convElement.className = 'list-group-item list-group-item-action conversation-item';
                    convElement.dataset.convId = conv.id;
                    convElement.innerHTML = `
                        <div class="d-flex w-100 align-items-center">
                            <img src="${conv.photo || 'https://via.placeholder.com/50'}" alt="${conv.name}" class="rounded-circle me-3" width="40" height="40">
                            <div class="flex-grow-1 conversation-details"><strong class="mb-1 name">${conv.name}</strong><div class="text-muted small last-message">${conv.last_message}</div></div>
                        </div>
                    `;
                    convElement.addEventListener('click', () => loadMessages(conv.id));
                    conversationsList.appendChild(convElement);
                });

                // Marcar la activa si existe
                if (activeConversationId) {
                    const activeEl = conversationsList.querySelector(`.conversation-item[data-conv-id='${activeConversationId}']`);
                    if (activeEl) activeEl.classList.add('active');
                }

            } catch (error) {
                console.error('Error cargando conversaciones:', error);
            }
        }

        async function loadMessages(conversationId) {
            if (!conversationId) return;
            activeConversationId = conversationId;
            document.getElementById('chat-input').style.display = 'block';

            const nearBottom = chatMessages.scrollTop + chatMessages.clientHeight >= chatMessages.scrollHeight - 50;

            try {
                const response = await fetch(`api/chat.php?action=get_messages&conversation_id=${conversationId}`);
                const data = await response.json();

                const newMessagesContent = JSON.stringify(data.messages);
                if (chatMessages.dataset.lastContent === newMessagesContent) {
                    return;
                }
                chatMessages.dataset.lastContent = newMessagesContent;

                // Inyectar el input si no existe
                if (!document.getElementById('message-input')) {
                    document.getElementById('chat-input').innerHTML = `<div class="input-group"><input type="text" id="message-input" class="form-control" placeholder="Escribe un mensaje..."><button class="btn btn-primary" type="button" id="send-button"><i class="bi bi-send-fill"></i></button></div>`;
                    addInputEventListeners();
                }

                document.getElementById('chat-header').innerHTML = `
                    <div class="me-auto d-flex align-items-center">
                        <img src="${data.conversation.photo || 'https://via.placeholder.com/40'}" alt="${data.conversation.name}" class="rounded-circle me-3" width="40" height="40">
                        <strong class="h5 mb-0">${data.conversation.name}</strong>
                    </div>
                    <div class="dropdown" id="chat-options-dropdown" style="display: ${data.conversation.is_group_chat ? 'block' : 'none'};">
                        <button class="btn btn-light" type="button" data-bs-toggle="dropdown" aria-expanded="false"><i class="bi bi-three-dots-vertical"></i></button>
                        <ul class="dropdown-menu dropdown-menu-end">
                            <li><a class="dropdown-item" href="#" id="edit-group-btn">Editar Grupo</a></li>
                            <li><a class="dropdown-item text-danger" href="#" id="delete-group-btn">Eliminar Grupo</a></li>
                        </ul>
                    </div>
                `;
                document.getElementById('chat-header').style.display = 'flex';
                addHeaderEventListeners(); // Re-asignar listeners a los nuevos botones

                chatMessages.innerHTML = '';
                data.messages.forEach(msg => {
                    appendMessageToUI(msg, false); // false = no hacer scroll
                });

                if (nearBottom) {
                    chatMessages.scrollTop = chatMessages.scrollHeight;
                }

                document.querySelectorAll('.conversation-item').forEach(el => {
                    el.classList.remove('active');
                    if (parseInt(el.dataset.convId) === conversationId) {
                        el.classList.add('active');
                    }
                });

            } catch (error) {
                console.error(`Error cargando mensajes para la conversación ${conversationId}:`, error);
            }
        }

        async function sendMessage() {
            // La referencia al input se debe tomar aquí, ya que se crea dinámicamente
            const messageInput = document.getElementById('message-input');
            const content = messageInput.value.trim();
            if (content === '' || !activeConversationId) return;

            const tempMessage = {
                message_content: content,
                sender_id: currentUserId
            };
            appendMessageToUI(tempMessage, true); // true = hacer scroll
            messageInput.value = '';

            try {
                const response = await fetch('api/chat.php?action=send_message', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({
                        conversation_id: activeConversationId,
                        message_content: content
                    })
                });

                const savedMessage = await response.json();
                if (savedMessage.error) {
                    console.error('Error al enviar el mensaje:', savedMessage.error);
                } else {
                    updateConversationList(activeConversationId, content);
                }
            } catch (error) {
                console.error('Error de red al enviar mensaje:', error);
            }
        }

        function addHeaderEventListeners() {
            const editBtn = document.getElementById('edit-group-btn');
            const deleteBtn = document.getElementById('delete-group-btn');
            if (editBtn) editBtn.addEventListener('click', handleEditGroup);
            if (deleteBtn) deleteBtn.addEventListener('click', handleDeleteGroup);
        }

        function addInputEventListeners() {
            const msgInput = document.getElementById('message-input');
            const sendBtn = document.getElementById('send-button');
            if (sendBtn) sendBtn.addEventListener('click', sendMessage);
            if (msgInput) msgInput.addEventListener('keypress', e => { if (e.key === 'Enter') sendMessage(); });
        }

        function appendMessageToUI(message, doScroll) {
            const messageElement = document.createElement('div');
            const messageClass = parseInt(message.sender_id) === currentUserId ? 'sent' : 'received';
            messageElement.className = `message ${messageClass}`;
            messageElement.dataset.messageId = message.id;

            let actions = '';
            if (message.id && parseInt(message.sender_id) === currentUserId) {
                actions = `
                    <div class="message-actions">
                        <button class="btn btn-link btn-sm p-1 edit-message-btn">Editar</button>
                        <button class="btn btn-link btn-sm p-1 text-danger delete-message-btn">Eliminar</button>
                    </div>
                `;
            }
            messageElement.innerHTML = `<div class="bubble">${message.message_content}</div> ${actions}`;
            chatMessages.appendChild(messageElement);
            if (doScroll) {
                chatMessages.scrollTop = chatMessages.scrollHeight;
            }
        }

        function updateConversationList(conversationId, lastMessage) {
            const convElement = document.querySelector(`.conversation-item[data-conv-id='${conversationId}']`);
            if (convElement) {
                convElement.querySelector('.last-message').textContent = lastMessage;
                convElement.parentElement.prepend(convElement);
            }
        }

        function renderProfessionalsList(profsToRender) {
            const professionalsList = document.getElementById('professionals-list');
            professionalsList.innerHTML = '';

            if (profsToRender.length === 0) {
                professionalsList.innerHTML = '<li class="list-group-item text-muted">No se encontraron profesionales.</li>';
                return;
            }

            profsToRender.forEach(prof => {
                if (prof.id === currentUserId) return;

                const li = document.createElement('li');
                li.className = 'list-group-item list-group-item-action d-flex align-items-center conversation-item';
                li.dataset.userId = prof.id;
                li.style.cursor = 'pointer';
                li.innerHTML = `
                    <img src="${prof.photo || 'https://via.placeholder.com/50'}" alt="${prof.name}" class="rounded-circle me-3" width="50" height="50">
                    <div class="conversation-details">
                        <p class="name mb-0 fw-bold">${prof.name}</p>
                    </div>
                `;
                li.addEventListener('click', () => startIndividualChat(prof.id));
                professionalsList.appendChild(li);
            });
        }

        async function loadProfessionals() {
            if (professionals.length > 0) {
                renderProfessionalsList(professionals);
                return;
            }

            try {
                const response = await fetch('api/chat.php?action=get_professionals');
                professionals = await response.json();
                renderProfessionalsList(professionals);
            } catch (error) {
                console.error('Error cargando profesionales:', error);
            }
        }

        async function startIndividualChat(otherUserId) {
            try {
                const response = await fetch(`api/chat.php?action=get_or_create_individual_conversation&user_id=${otherUserId}`);
                const data = await response.json();
                if (data.conversation_id) {
                    closeNewChatModal();
                    await loadConversations();
                    loadMessages(data.conversation_id);
                } else {
                    console.error('No se pudo obtener el ID de la conversación.', data.error);
                }
            } catch (error) {
                console.error('Error al iniciar el chat individual:', error);
            }
        }

        function handleProfessionalSearch() {
            const searchTerm = searchProfessionalInput.value.toLowerCase();
            const filteredProfessionals = professionals.filter(prof =>
                prof.name.toLowerCase().includes(searchTerm)
            );
            renderProfessionalsList(filteredProfessionals);
        }

        function handleGroupMemberSearch() {
            const searchTerm = searchGroupMembersInput.value.toLowerCase();
            groupMembersList.innerHTML = '';
            if (searchTerm.length === 0) return;

            const filtered = professionals.filter(prof =>
                prof.name.toLowerCase().includes(searchTerm) &&
                prof.id !== currentUserId &&
                !selectedGroupMembers.has(prof.id)
            );

            filtered.forEach(prof => {
                const li = document.createElement('li');
                li.className = 'list-group-item list-group-item-action';
                li.style.cursor = 'pointer';
                li.textContent = prof.name;
                li.addEventListener('click', () => selectMemberForGroup(prof));
                groupMembersList.appendChild(li);
            });
        }

        // --- Lógica del Modal ---

        function setupModalTabs() {
            const tabs = document.querySelectorAll('#modal-tabs .tab-link');
            const tabContents = document.querySelectorAll('#new-chat-modal .tab-content');

            tabs.forEach(tab => {
                tab.addEventListener('click', () => {
                    tabs.forEach(t => t.classList.remove('active'));
                    tabContents.forEach(c => c.classList.remove('active'));

                    tab.classList.add('active');
                    const targetContentId = tab.getAttribute('data-tab');
                    document.getElementById(targetContentId).classList.add('active');

                    if (targetContentId === 'group') {
                        createGroupButton.style.display = 'block';
                        if (!appState.isEditMode) {
                            createGroupButton.textContent = 'Crear Grupo';
                            newChatModalEl.querySelector('.modal-title').textContent = 'Nueva Conversación';
                        }
                    } else {
                        createGroupButton.style.display = 'none';
                    }
                });
            });
        }

        function selectMemberForGroup(prof) {
            if (selectedGroupMembers.has(prof.id) || prof.id === currentUserId) return;
            selectedGroupMembers.set(prof.id, prof);
            searchGroupMembersInput.value = '';
            groupMembersList.innerHTML = '';
            renderSelectedMembers();
        }

        function removeMemberFromGroup(userId) {
            selectedGroupMembers.delete(userId);
            renderSelectedMembers();
        }

        function renderSelectedMembers() {
            selectedMembersContainer.innerHTML = '';
            selectedGroupMembers.forEach(prof => {
                const pill = document.createElement('span');
                pill.className = 'selected-member-pill';
                pill.innerHTML = `${prof.name} <span class="remove-member-btn" data-user-id="${prof.id}">&times;</span>`;
                selectedMembersContainer.appendChild(pill);
            });
        }

        async function createGroup() {
            const groupName = groupNameInput.value.trim();
            const memberIds = Array.from(selectedGroupMembers.keys());
            const groupPhotoFile = groupPhotoInput.files[0];
            const isEditing = appState.isEditMode;

            if (isEditing && !activeConversationId) {
                alert('Error: No se ha identificado el grupo a editar.');
                return;
            }
            if (!groupName) {
                alert('Por favor, dale un nombre al grupo.');
                return;
            }
            if (memberIds.length === 0 && !isEditing) {
                alert('Debes añadir al menos un miembro al grupo.');
                return;
            }

            const formData = new FormData();
            formData.append('name', groupName);
            formData.append('members', JSON.stringify(memberIds));
            if (groupPhotoFile) {
                formData.append('group_photo', groupPhotoFile);
            }

            const url = isEditing 
                ? `api/chat.php?action=edit_group&id=${activeConversationId}`
                : 'api/chat.php?action=create_group';
            
            try {
                const response = await fetch(url, { method: 'POST', body: formData });
                const data = await response.json();
                if (data.conversation_id) {
                    closeNewChatModal();
                    await loadConversations();
                    loadMessages(data.conversation_id);
                } else {
                    throw new Error(data.error || `Error desconocido al ${isEditing ? 'editar' : 'crear'} el grupo.`);
                }
            } catch (error) {
                console.error(`Error al ${isEditing ? 'editar' : 'crear'} el grupo:`, error);
                alert(`No se pudo ${isEditing ? 'editar' : 'crear'} el grupo. Inténtalo de nuevo.`);
            }
        }

        function openNewChatModal() {
            newChatModal.show();
            searchProfessionalInput.value = '';
            loadProfessionals();
        }

        function closeNewChatModal() {
            groupNameInput.value = '';
            groupPhotoInput.value = '';
            groupPhotoPreview.style.display = 'none';
            searchGroupMembersInput.value = '';
            groupMembersList.innerHTML = '';
            selectedGroupMembers.clear();
            renderSelectedMembers();
            appState.isEditMode = false;
            newChatModal.hide();
        }

        // --- Event Listeners ---

        newChatButton.addEventListener('click', openNewChatModal);
        closeModalButtons.forEach(btn => btn.addEventListener('click', closeNewChatModal));

        addInputEventListeners();
        searchGroupMembersInput.addEventListener('input', handleGroupMemberSearch);
        groupPhotoInput.addEventListener('change', () => {
            const file = groupPhotoInput.files[0];
            if (file) {
                groupPhotoPreview.src = URL.createObjectURL(file);
                groupPhotoPreview.style.display = 'block';
            }
        });
        createGroupButton.addEventListener('click', createGroup);

        selectedMembersContainer.addEventListener('click', e => {
            if (e.target.classList.contains('remove-member-btn')) {
                removeMemberFromGroup(parseInt(e.target.dataset.userId));
            }
        });

        chatMessages.addEventListener('click', e => {
            const messageElement = e.target.closest('.message');
            if (!messageElement || !messageElement.dataset.messageId) return;
            const messageId = parseInt(messageElement.dataset.messageId);

            if (e.target.classList.contains('delete-message-btn')) deleteMessage(messageId, messageElement);
            if (e.target.classList.contains('edit-message-btn')) editMessage(messageId, messageElement);
        });

        const handleEditGroup = () => {
            const conversation = currentConversations.find(c => c.id === activeConversationId);
            if (!conversation || !conversation.is_group_chat) return;

            appState.isEditMode = true;
            newChatModalEl.querySelector('.modal-title').textContent = 'Editar Grupo';
            document.querySelector('.tab-link[data-tab="group"]').click();
            createGroupButton.textContent = 'Guardar Cambios';
            groupNameInput.value = conversation.name;
            selectedGroupMembers.clear();
            // TODO: Cargar miembros actuales del grupo desde la API para editarlos.
            newChatModal.show();
        };

        const handleDeleteGroup = async () => {
            if (!activeConversationId || !confirm('¿Estás seguro de que quieres eliminar este grupo?')) return;

            try {
                const response = await fetch(`api/chat.php?action=delete_group&id=${activeConversationId}`, { method: 'POST' });
                const result = await response.json();
                if (result.success) {
                    activeConversationId = null;
                    chatHeader.style.display = 'none !important';
                    chatMessages.innerHTML = '<div class="text-center p-5 text-muted">Selecciona una conversación para empezar.</div>';
                    chatInput.style.display = 'none';
                    await loadConversations();
                } else {
                    throw new Error(result.error || 'No se pudo eliminar el grupo.');
                }
            } catch (error) {
                alert(error.message);
            }
        };

        async function deleteMessage(messageId, element) {
            if (!confirm('¿Eliminar este mensaje?')) return;
            try {
                const response = await fetch(`api/chat.php?action=delete_message&id=${messageId}`, { method: 'POST' });
                const result = await response.json();
                if (result.success) {
                    element.remove();
                } else {
                    throw new Error(result.error || 'No se pudo eliminar el mensaje.');
                }
            } catch (error) {
                alert(error.message);
            }
        }

        function editMessage(messageId, element) {
            const bubble = element.querySelector('.bubble');
            const originalContent = bubble.innerHTML;
            bubble.setAttribute('contenteditable', 'true');
            bubble.focus();

            const saveChanges = async () => {
                bubble.removeAttribute('contenteditable');
                const newContent = bubble.innerText.trim();
                if (newContent && newContent !== originalContent) {
                    try {
                        const response = await fetch(`api/chat.php?action=edit_message`, {
                            method: 'POST',
                            headers: { 'Content-Type': 'application/json' },
                            body: JSON.stringify({ message_id: messageId, content: newContent })
                        });
                        const result = await response.json();
                        if (!result.success) throw new Error();
                        bubble.innerHTML = newContent; // Actualizar con el contenido sanitizado si la API lo devolviera
                    } catch (e) { bubble.innerHTML = originalContent; alert('No se pudo editar.'); }
                } else {
                    bubble.innerHTML = originalContent;
                }
            };

            bubble.addEventListener('blur', saveChanges, { once: true });
            bubble.addEventListener('keydown', e => {
                if (e.key === 'Enter' && !e.shiftKey) {
                    e.preventDefault();
                    bubble.blur();
                }
                if (e.key === 'Escape') {
                    bubble.innerHTML = originalContent;
                    bubble.blur();
                }
            });
        }

        // --- Inicialización y Polling ---

        async function initializeApp() {
            try {
                const response = await fetch('api/chat.php?action=get_session_user');
                const data = await response.json();
                if (data.user_id) {
                    currentUserId = data.user_id;
                } else {
                    console.error('No se pudo obtener el ID del usuario.');
                    document.getElementById('global-chat-button').style.display = 'none';
                }
            } catch (error) {
                console.error('Error al inicializar el chat:', error);
                document.getElementById('global-chat-button').style.display = 'none';
            }
        }

        function startPolling() {
            if (pollingInterval) return;
            pollingInterval = setInterval(() => {
                if (activeConversationId) {
                    loadMessages(activeConversationId);
                }
            }, 5000);
        }

        function stopPolling() {
            clearInterval(pollingInterval);
            pollingInterval = null;
        }

        function startNotificationPolling() {
            if (notificationPollingInterval) return;

            let lastCheckTimestamp = new Date().toISOString();

            const checkNotifications = async () => {
                // Solo comprobar si el chat está cerrado
                if (appState.isChatOpen) return;

                try {
                    const response = await fetch(`api/chat.php?action=check_new_messages&since=${lastCheckTimestamp}`);
                    const data = await response.json();
                    if (data.has_new_messages) {
                        document.getElementById('chat-notification-badge').style.display = 'block';
                    }
                    lastCheckTimestamp = new Date().toISOString(); // Actualizar para la próxima comprobación
                } catch (error) {
                    console.error("Error al comprobar notificaciones:", error);
                }
            };
            notificationPollingInterval = setInterval(checkNotifications, 15000); // Comprobar cada 15 segundos
        }

        setupModalTabs();
        initializeApp();
    }

    // Iniciar todo el proceso
    createChatInterface();
});