document.addEventListener('DOMContentLoaded', function() {
    // --- ELEMENTOS DEL DOM ---
    const activeProjectsCountEl = document.getElementById('active-projects-count');
    const pendingTasksCountEl = document.getElementById('pending-tasks-count');
    const overdueTasksCountEl = document.getElementById('overdue-tasks-count');
    const recentProjectsListEl = document.getElementById('recent-projects-list');
    const upcomingTasksListEl = document.getElementById('upcoming-tasks-list');
    const calendarEl = document.getElementById('calendar');

    // --- URLs DE LA API ---
    const PROJECTS_API_URL = 'api/projects.php';
    const TASKS_API_URL = 'api/tasks.php';

    // --- FUNCIONES DE UTILIDAD ---
    const formatDate = (dateString) => {
        if (!dateString) return 'N/A';
        const [year, month, day] = dateString.split('-');
        if (!year || !month || !day) return dateString;
        return `${day}/${month}/${year}`;
    };

    // --- FUNCIONES DE CARGA Y RENDERIZADO ---

    async function loadDashboardData() {
        try {
            // Cargar proyectos y tareas en paralelo
            const [projectsRes, tasksRes] = await Promise.all([
                fetch(PROJECTS_API_URL),
                fetch(TASKS_API_URL)
            ]);

            if (!projectsRes.ok) throw new Error('No se pudieron cargar los proyectos.');
            if (!tasksRes.ok) throw new Error('No se pudieron cargar las tareas.');

            const projects = await projectsRes.json();
            const tasks = await tasksRes.json();

            // Renderizar todos los componentes del dashboard
            renderKpiCards(projects, tasks);
            renderRecentProjects(projects);
            renderUpcomingTasks(tasks, projects);
            renderCalendar(projects, tasks);

        } catch (error) {
            console.error('Error al cargar el dashboard:', error);
            if (recentProjectsListEl) recentProjectsListEl.innerHTML = `<li class="list-group-item text-danger">${error.message}</li>`;
        }
    }

    function renderKpiCards(projects, tasks) {
        const today = new Date();
        today.setHours(0, 0, 0, 0); // Para comparar solo fechas

        // Proyectos Activos
        const activeProjects = projects.filter(p => ['En Progreso', 'Aprobado', 'Planificación'].includes(p.estado));
        if (activeProjectsCountEl) {
            activeProjectsCountEl.textContent = activeProjects.length;
        }

        // Tareas Pendientes
        const pendingTasks = tasks.filter(t => t.porcentaje < 100);
        if (pendingTasksCountEl) {
            pendingTasksCountEl.textContent = pendingTasks.length;
        }

        // Tareas Vencidas
        const overdueTasks = pendingTasks.filter(t => {
            if (!t.fecha_fin) return false;
            const dueDate = new Date(t.fecha_fin + 'T00:00:00'); // Asegurar que se interprete como fecha local
            return dueDate < today;
        });
        if (overdueTasksCountEl) {
            overdueTasksCountEl.textContent = overdueTasks.length;
        }
    }

    function renderRecentProjects(projects) {
        if (!recentProjectsListEl) return;
        recentProjectsListEl.innerHTML = '';

        // Ordenar por fecha de inicio descendente y tomar los 5 más recientes
        const recentProjects = projects.sort((a, b) => new Date(b.fecha_inicio) - new Date(a.fecha_inicio)).slice(0, 5);

        if (recentProjects.length === 0) {
            recentProjectsListEl.innerHTML = '<li class="list-group-item text-muted">No hay proyectos recientes.</li>';
            return;
        }

        recentProjects.forEach(project => {
            const li = document.createElement('li');
            li.className = 'list-group-item';
            li.innerHTML = `
                <a href="project-detail.html?id=${project.id}" class="text-decoration-none">
                    <strong>${project.nombre_proyecto}</strong>
                </a>
                <small class="d-block text-muted">Cliente: ${project.cliente_nombre}</small>
            `;
            recentProjectsListEl.appendChild(li);
        });
    }

    function renderUpcomingTasks(tasks, projects) {
        if (!upcomingTasksListEl) return;
        upcomingTasksListEl.innerHTML = '';

        const projectMap = new Map(projects.map(p => [p.id, p.nombre_proyecto]));

        // Filtrar tareas pendientes y ordenarlas por fecha de fin
        const upcomingTasks = tasks
            .filter(t => t.porcentaje < 100 && t.fecha_fin)
            .sort((a, b) => new Date(a.fecha_fin) - new Date(b.fecha_fin))
            .slice(0, 5);

        if (upcomingTasks.length === 0) {
            upcomingTasksListEl.innerHTML = '<li class="list-group-item text-muted">No hay tareas próximas a vencer.</li>';
            return;
        }

        upcomingTasks.forEach(task => {
            const li = document.createElement('li');
            li.className = 'list-group-item';
            li.innerHTML = `
                <a href="project-detail.html?id=${task.project_id}" class="text-decoration-none">
                    <strong>${task.nombre_tarea}</strong>
                </a>
                <small class="d-block text-muted">
                    Proyecto: ${projectMap.get(task.project_id) || 'N/A'} | Vence: ${formatDate(task.fecha_fin)}
                </small>
            `;
            upcomingTasksListEl.appendChild(li);
        });
    }

    function renderCalendar(projects, tasks) {
        if (!calendarEl) return;

        const projectEvents = projects.map(p => ({
            title: `PROY: ${p.nombre_proyecto}`,
            start: p.fecha_inicio,
            end: p.fecha_fin_estimada ? new Date(new Date(p.fecha_fin_estimada).getTime() + 86400000).toISOString().split('T')[0] : p.fecha_inicio, // +1 día para que sea inclusivo
            url: `project-detail.html?id=${p.id}`,
            color: '#0d6efd', // Azul para proyectos
            allDay: true
        }));

        const taskEvents = tasks.map(t => ({
            title: `Tarea: ${t.nombre_tarea}`,
            start: t.fecha_fin,
            url: `project-detail.html?id=${t.project_id}`,
            color: t.porcentaje === 100 ? '#198754' : '#ffc107', // Verde si está completa, amarillo si no
            allDay: true
        }));

        const calendar = new FullCalendar.Calendar(calendarEl, {
            initialView: 'dayGridMonth',
            headerToolbar: {
                left: 'prev,next today',
                center: 'title',
                right: 'dayGridMonth,timeGridWeek,listWeek'
            },
            events: [...projectEvents, ...taskEvents],
            locale: 'es',
            buttonText: {
                today: 'Hoy',
                month: 'Mes',
                week: 'Semana',
                list: 'Lista'
            },
            eventTimeFormat: { // Ocultar la hora para eventos de todo el día
                hour: 'numeric',
                minute: '2-digit',
                meridiem: false
            }
        });

        calendar.render();
    }

    // --- CARGA INICIAL ---
    loadDashboardData();
});