<?php
require_once 'api/check_session.php';

// --- 1. Verificación de Permisos ---
// Solo los administradores pueden ejecutar este script.
if (($_SESSION['user_puesto'] ?? '') !== 'Administrador') {
    http_response_code(403);
    die("Acceso denegado. Solo los administradores pueden realizar copias de seguridad.");
}

require_once 'config/db.php';

// --- 2. Configuración del Backup ---
$backup_dir = __DIR__ . '/backups/';
$db_name = DB_NAME;
$db_user = DB_USER;
$db_pass = DB_PASS;
$db_host = DB_HOST;

$file_name = 'backup-' . $db_name . '-' . date("Y-m-d_H-i-s") . '.sql';
$file_path = $backup_dir . $file_name;
$compressed_file_path = $file_path . '.gz';

// --- 3. Crear el directorio de backups si no existe ---
if (!is_dir($backup_dir)) {
    if (!mkdir($backup_dir, 0755, true)) {
        die("Error: No se pudo crear el directorio de backups. Verifique los permisos.");
    }
}

// --- 4. Construir y ejecutar el comando mysqldump ---
// Usar putenv es ligeramente más seguro que poner la contraseña en la línea de comandos.
putenv('MYSQL_PWD=' . $db_pass);
$command = sprintf(
    'mysqldump --host=%s --user=%s %s > %s',
    escapeshellarg($db_host),
    escapeshellarg($db_user),
    escapeshellarg($db_name),
    escapeshellarg($file_path)
);

// Ejecutar el comando
@exec($command, $output, $return_var);
putenv('MYSQL_PWD='); // Limpiar la variable de entorno

// Verificar si mysqldump falló
if ($return_var !== 0) {
    die("Error al ejecutar mysqldump. Asegúrese de que esté en el PATH del sistema o proporcione la ruta completa. Código de error: " . $return_var);
}

// --- 5. Comprimir el archivo SQL ---
$gz_command = sprintf(
    'gzip -9 %s',
    escapeshellarg($file_path)
);
@exec($gz_command, $output_gz, $return_var_gz);

if ($return_var_gz !== 0) {
    if (file_exists($file_path)) unlink($file_path); // Limpiar el archivo .sql si la compresión falla
    die("Error al comprimir el archivo de backup. Código de error: " . $return_var_gz);
}

// --- 6. Forzar la descarga del archivo comprimido ---
if (file_exists($compressed_file_path)) {
    header('Content-Description: File Transfer');
    header('Content-Type: application/octet-stream');
    header('Content-Disposition: attachment; filename="' . basename($compressed_file_path) . '"');
    header('Expires: 0');
    header('Cache-Control: must-revalidate');
    header('Pragma: public');
    header('Content-Length: ' . filesize($compressed_file_path));
    
    // Limpiar el buffer de salida antes de leer el archivo
    ob_clean();
    flush();
    
    readfile($compressed_file_path);

    // --- 7. Limpiar el archivo del servidor después de la descarga ---
    unlink($compressed_file_path);
    
    exit;
} else {
    die("Error: El archivo de backup comprimido no se pudo encontrar.");
}

?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <title>Backup de Base de Datos</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
</head>
<body class="bg-light">
    <div class="container mt-5">
        <div class="card">
            <div class="card-body text-center">
                <h1 class="card-title">Backup de Base de Datos</h1>
                <p class="lead">Si la descarga no comenzó automáticamente, haz clic en el siguiente enlace.</p>
                <?php if (isset($compressed_file_path) && file_exists($compressed_file_path)): ?>
                    <a href="backups/<?php echo basename($compressed_file_path); ?>" class="btn btn-primary btn-lg">
                        Descargar Backup (<?php echo basename($compressed_file_path); ?>)
                    </a>
                <?php else: ?>
                    <div class="alert alert-danger">
                        Ocurrió un error al generar el archivo de backup.
                    </div>
                <?php endif; ?>
                <p class="mt-3 text-muted">
                    <a href="index.php">Volver al Dashboard</a>
                </p>
            </div>
        </div>
    </div>
</body>
</html>