<?php
header("Content-Type: application/json");
require_once __DIR__ . '/../config/db.php';

$method = $_SERVER['REQUEST_METHOD'];
$project_id = isset($_GET['project_id']) ? (int)$_GET['project_id'] : null;
$task_id = isset($_GET['id']) ? (int)$_GET['id'] : null;
$pdo = get_db_connection();

/**
 * Decodifica los campos JSON de una tarea.
 * @param array $task La tarea de la BD.
 * @return array La tarea con los campos JSON decodificados.
 */
function decode_task_json_fields($task) {
    if ($task) {
        $json_fields = ['comentarios', 'archivos', 'links', 'tags'];
        foreach ($json_fields as $field) {
            $task[$field] = $task[$field] ? json_decode($task[$field], true) : [];
        }
    }
    return $task;
}

if ($method === 'GET') {
    if ($project_id !== null) {
        $stmt = $pdo->prepare("SELECT * FROM tasks WHERE project_id = ? ORDER BY fecha_fin ASC");
        $stmt->execute([$project_id]);
        $project_tasks = $stmt->fetchAll();
        $project_tasks = array_map('decode_task_json_fields', $project_tasks);
        echo json_encode($project_tasks);
    } else {
        $stmt = $pdo->query("SELECT * FROM tasks ORDER BY fecha_fin ASC");
        $tasks = $stmt->fetchAll();
        $tasks = array_map('decode_task_json_fields', $tasks);
        echo json_encode($tasks);
    }
} elseif ($method === 'POST' && isset($_GET['action']) && $_GET['action'] === 'add_comment' && $task_id !== null) {
    // Endpoint para añadir un comentario a una tarea
    $input = json_decode(file_get_contents('php://input'), true);

    if (empty($input['text']) || !isset($input['author_id']) || !isset($input['author_name'])) {
        http_response_code(400);
        echo json_encode(['error' => 'El texto y el autor del comentario son requeridos.']);
        exit;
    }

    // 1. Obtener comentarios actuales
    $stmt = $pdo->prepare("SELECT comentarios FROM tasks WHERE id = ?");
    $stmt->execute([$task_id]);
    $current_comments_json = $stmt->fetchColumn();
    $comments = $current_comments_json ? json_decode($current_comments_json, true) : [];

    // 2. Añadir nuevo comentario
    $new_comment = [
        'id' => uniqid('c_'),
        'author_id' => $input['author_id'],
        'author_name' => $input['author_name'],
        'text' => htmlspecialchars($input['text']),
        'timestamp' => date('Y-m-d H:i:s')
    ];
    $comments[] = $new_comment;

    // 3. Actualizar la base de datos
    $stmt = $pdo->prepare("UPDATE tasks SET comentarios = ? WHERE id = ?");
    if ($stmt->execute([json_encode($comments), $task_id])) {
        echo json_encode($new_comment);
    } else {
        http_response_code(500);
        echo json_encode(['error' => 'No se pudo añadir el comentario.']);
    }

} elseif ($method === 'POST') {
    $input = json_decode(file_get_contents('php://input'), true);

    if (empty($input['nombre_tarea']) || empty($input['project_id'])) {
        http_response_code(400);
        echo json_encode(['error' => 'El nombre de la tarea y el ID del proyecto son obligatorios.']);
        exit;
    }

    try {
        $sql = "INSERT INTO tasks (project_id, nombre_tarea, asignado_a, fecha_inicio, fecha_fin, porcentaje, prioridad, comentarios, archivos, links, tags) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            (int)$input['project_id'],
            $input['nombre_tarea'],
            isset($input['asignado_a']) && $input['asignado_a'] !== '' ? (int)$input['asignado_a'] : null,
            empty($input['fecha_inicio']) ? null : $input['fecha_inicio'],
            empty($input['fecha_fin']) ? null : $input['fecha_fin'],
            isset($input['porcentaje']) ? (int)$input['porcentaje'] : 0,
            $input['prioridad'] ?? 'Media',
            json_encode([]), // comentarios
            json_encode([]), // archivos
            json_encode([]), // links
            json_encode([])  // tags
        ]);

        $new_id = $pdo->lastInsertId();
        $stmt = $pdo->prepare("SELECT * FROM tasks WHERE id = ?");
        $stmt->execute([$new_id]);
        $new_task = $stmt->fetch();

        http_response_code(201);
        echo json_encode(decode_task_json_fields($new_task));
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Error al crear tarea: ' . $e->getMessage()]);
    }

} elseif ($method === 'PUT' && $task_id !== null) {
    $input = json_decode(file_get_contents('php://input'), true);

    // Obtener la tarea actual para manejar campos JSON
    $stmt = $pdo->prepare("SELECT * FROM tasks WHERE id = ?");
    $stmt->execute([$task_id]);
    $current_task = decode_task_json_fields($stmt->fetch());

    if (!$current_task) {
        http_response_code(404);
        echo json_encode(['error' => 'Tarea no encontrada para actualizar']);
        exit;
    }

    // Lógica para añadir elementos a campos JSON sin sobreescribir
    if (isset($input['new_file'])) {
        $current_task['archivos'][] = $input['new_file'];
    }
    if (isset($input['new_tag']) && !in_array($input['new_tag'], $current_task['tags'])) {
        $current_task['tags'][] = $input['new_tag'];
    }
    if (isset($input['new_link'])) {
        $current_task['links'][] = $input['new_link'];
    }

    // Lógica para fecha de fin real
    $fecha_fin_real = $current_task['fecha_fin_real'];
    if (isset($input['porcentaje']) && (int)$input['porcentaje'] >= 100 && (int)$current_task['porcentaje'] < 100) {
        $fecha_fin_real = date('Y-m-d');
    } elseif (isset($input['porcentaje']) && (int)$input['porcentaje'] < 100) {
        $fecha_fin_real = null;
    }

    try {
        $sql = "UPDATE tasks SET 
                    nombre_tarea = ?, 
                    asignado_a = ?, 
                    fecha_inicio = ?, 
                    fecha_fin = ?, 
                    fecha_fin_real = ?,
                    porcentaje = ?, 
                    prioridad = ?, 
                    comentarios = ?, 
                    archivos = ?, 
                    links = ?, 
                    tags = ?
                WHERE id = ?";
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            $input['nombre_tarea'] ?? $current_task['nombre_tarea'],
            (isset($input['asignado_a']) && $input['asignado_a'] !== '') ? (int)$input['asignado_a'] : $current_task['asignado_a'],
            empty($input['fecha_inicio']) ? $current_task['fecha_inicio'] : $input['fecha_inicio'],
            empty($input['fecha_fin']) ? $current_task['fecha_fin'] : $input['fecha_fin'],
            $fecha_fin_real,
            $input['porcentaje'] ?? $current_task['porcentaje'],
            $input['prioridad'] ?? $current_task['prioridad'],
            json_encode($input['comentarios'] ?? $current_task['comentarios']),
            json_encode($current_task['archivos']), // Usamos la versión actualizada
            json_encode($current_task['links']),     // Usamos la versión actualizada
            json_encode($current_task['tags']),       // Usamos la versión actualizada
            $task_id
        ]);

        if ($stmt->rowCount() || count($current_task['archivos']) !== count($input['archivos'] ?? []) || count($current_task['links']) !== count($input['links'] ?? []) || count($current_task['tags']) !== count($input['tags'] ?? [])) {
            $stmt = $pdo->prepare("SELECT * FROM tasks WHERE id = ?");
            $stmt->execute([$task_id]);
            $updated_task = decode_task_json_fields($stmt->fetch());
            echo json_encode($updated_task);
        } else {
            // Si no se afectaron filas, puede ser porque los datos eran idénticos.
            // Devolvemos la tarea actual para mantener consistencia en el frontend.
            echo json_encode($current_task);
        }
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Error al actualizar tarea: ' . $e->getMessage()]);
    }

} elseif ($method === 'DELETE' && $task_id !== null) {
    try {
        $stmt = $pdo->prepare("DELETE FROM tasks WHERE id = ?");
        $stmt->execute([$task_id]);

        if ($stmt->rowCount()) {
            http_response_code(204);
        } else {
            http_response_code(404);
            echo json_encode(['error' => 'Tarea no encontrada']);
        }
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Error al eliminar tarea: ' . $e->getMessage()]);
    }
} else {
    http_response_code(405);
    echo json_encode(['error' => 'Método no permitido']);
}
?>