<?php
header('Content-Type: application/json');

// Prevenir la ejecución si ya está instalado
if (file_exists(__DIR__ . '/../config/db.php')) {
    echo json_encode(['success' => false, 'message' => 'La aplicación ya está instalada. Elimine el archivo config/db.php para reinstalar.']);
    exit;
}

$db_host = $_POST['db_host'] ?? '';
$db_name = $_POST['db_name'] ?? '';
$db_user = $_POST['db_user'] ?? '';
$db_pass = $_POST['db_pass'] ?? '';
$admin_email = $_POST['admin_email'] ?? '';
$admin_name = $_POST['admin_name'] ?? '';
$admin_password = $_POST['admin_password'] ?? '';
$admin_puesto = $_POST['admin_puesto'] ?? 'Administrador'; // Rol por defecto si no se envía

if (empty($db_host) || empty($db_name) || empty($db_user) || empty($admin_name) || empty($admin_email) || empty($admin_password) || empty($admin_puesto)) {
    echo json_encode(['success' => false, 'message' => 'Todos los campos son obligatorios.']);
    exit;
}

// --- 1. Probar la conexión a la base de datos ---
try {
    $dsn = "mysql:host=$db_host;dbname=$db_name;charset=utf8mb4";
    $pdo = new PDO($dsn, $db_user, $db_pass);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} catch (PDOException $e) {
    echo json_encode(['success' => false, 'message' => 'No se pudo conectar a la base de datos: ' . $e->getMessage()]);
    exit;
}

// --- 2. Crear el archivo de configuración `config/db.php` ---
$config_content = "<?php
/*
 * Archivo de configuración de la base de datos.
 * Generado automáticamente por el instalador de Platflow.
 */

require_once __DIR__ . '/../config/config.php';

define('DB_HOST', '{$db_host}');
define('DB_NAME', '{$db_name}');
define('DB_USER', '{$db_user}');
define('DB_PASS', '{$db_pass}');

/**
 * Establece y retorna una conexión PDO a la base de datos.
 * @return PDO|null
 */
function get_db_connection() {
    static \$pdo = null;
    if (\$pdo === null) {
        try {
            \$dsn = 'mysql:host=' . DB_HOST . ';dbname=' . DB_NAME . ';charset=utf8mb4';
            \$pdo = new PDO(\$dsn, DB_USER, DB_PASS);
            \$pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
            \$pdo->setAttribute(PDO::ATTR_DEFAULT_FETCH_MODE, PDO::FETCH_ASSOC);
        } catch (PDOException \$e) {
            // En un entorno de producción, esto debería registrarse en un log y no mostrarse al usuario.
            die('Error de conexión a la base de datos: ' . \$e->getMessage());
        }
    }
    return \$pdo;
}
?>";

$config_path = __DIR__ . '/../config/db.php';
if (!file_put_contents($config_path, $config_content)) {
    echo json_encode(['success' => false, 'message' => 'Error: No se pudo escribir el archivo de configuración en config/db.php. Verifique los permisos.']);
    exit;
}

// --- 3. Crear el archivo de seguridad `config/security.php` ---
try {
    $encryption_key = bin2hex(random_bytes(32));
    $encryption_iv = bin2hex(random_bytes(16)); // IV de 16 bytes para aes-256-cbc

    $security_content = "<?php

// --- Claves de Seguridad ---
// Generadas automáticamente por el instalador.
define('ENCRYPTION_KEY', '{$encryption_key}');
define('ENCRYPTION_IV', '{$encryption_iv}');
define('ENCRYPTION_METHOD', 'aes-256-cbc');

function encrypt_token(\$string) {
    \$key = hex2bin(ENCRYPTION_KEY);
    \$iv = hex2bin(ENCRYPTION_IV);
    \$encrypted = openssl_encrypt(\$string, ENCRYPTION_METHOD, \$key, 0, \$iv);
    return base64_encode(\$encrypted);
}

function decrypt_token(\$encrypted_string) {
    \$key = hex2bin(ENCRYPTION_KEY);
    \$iv = hex2bin(ENCRYPTION_IV);
    \$decrypted = openssl_decrypt(base64_decode(\$encrypted_string), ENCRYPTION_METHOD, \$key, 0, \$iv);
    return \$decrypted;
}
?>";

    $security_path = __DIR__ . '/../config/security.php';
    if (!file_put_contents($security_path, $security_content)) {
        throw new Exception('No se pudo escribir el archivo de seguridad en config/security.php. Verifique los permisos.');
    }
} catch (Exception $e) {
    echo json_encode(['success' => false, 'message' => 'Error al crear archivo de seguridad: ' . $e->getMessage()]);
    exit;
}

// --- 3.5 Limpiar tablas existentes para una instalación limpia ---
try {
    // Obtener todas las tablas de la base de datos para una limpieza completa
    $tables_result = $pdo->query("SHOW TABLES");
    $tables_to_drop = $tables_result->fetchAll(PDO::FETCH_COLUMN);

    $pdo->exec("SET FOREIGN_KEY_CHECKS = 0;");
    if (!empty($tables_to_drop)) {
        foreach ($tables_to_drop as $table) {
            $pdo->exec("DROP TABLE IF EXISTS `$table`;");
        }
    }
    $pdo->exec("SET FOREIGN_KEY_CHECKS = 1;");
} catch (PDOException $e) {
    if (file_exists($security_path)) unlink($security_path);
    unlink($config_path);
    echo json_encode(['success' => false, 'message' => 'Error al limpiar las tablas existentes: ' . $e->getMessage()]);
    exit;
}

// --- 4. Ejecutar el script SQL para crear las tablas ---
try {
    $sql_schema = "
    CREATE TABLE `settings` (
      `setting_key` VARCHAR(100) NOT NULL PRIMARY KEY,
      `setting_value` JSON NOT NULL
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

    CREATE TABLE `professionals` (
      `id` INT AUTO_INCREMENT PRIMARY KEY,
      `nombre_completo` VARCHAR(255) NOT NULL,
      `puesto` VARCHAR(100),
      `tipo_contrato` VARCHAR(100),
      `fecha_ingreso` DATE,
      `salario_mensual` DECIMAL(10, 2),
      `email` VARCHAR(255) NOT NULL UNIQUE,
      `telefono` VARCHAR(50),
      `estado` VARCHAR(50) DEFAULT 'Activo',
      `password` VARCHAR(255) NOT NULL,
      `foto_url` VARCHAR(255)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

    CREATE TABLE `clients` (
      `id` INT AUTO_INCREMENT PRIMARY KEY,
      `nombre_empresa` VARCHAR(255) NOT NULL,
      `razon_social` VARCHAR(255),
      `ruc` VARCHAR(50),
      `telefono` VARCHAR(50),
      `correo` VARCHAR(255),
      `web` VARCHAR(255),
      `direccion` TEXT,
      `ciudad` VARCHAR(100),
      `pais` VARCHAR(100),
      `estado` VARCHAR(50) DEFAULT 'Activo',
      `link_fb` VARCHAR(255), `link_ig` VARCHAR(255), `link_in` VARCHAR(255), `link_ytb` VARCHAR(255), `link_x` VARCHAR(255), `link_tiktok` VARCHAR(255)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

    CREATE TABLE `contacts` (
      `id` INT AUTO_INCREMENT PRIMARY KEY,
      `empresa_id` INT,
      `nombre_completo` VARCHAR(255) NOT NULL,
      `cargo` VARCHAR(100),
      `correo` VARCHAR(255),
      `telefono` VARCHAR(50),
      FOREIGN KEY (`empresa_id`) REFERENCES `clients`(`id`) ON DELETE CASCADE
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

    CREATE TABLE `projects` (
      `id` INT AUTO_INCREMENT PRIMARY KEY,
      `nombre_proyecto` VARCHAR(255) NOT NULL,
      `cliente_id` INT NOT NULL,
      `descripcion` TEXT,
      `fecha_inicio` DATE,
      `fecha_fin_estimada` DATE,
      `fecha_fin_real` DATE,
      `tipo_marketing` VARCHAR(100),
      `estado` VARCHAR(50),
      `estrategia_estrategico` TEXT,
      `estrategia_tactico` TEXT,
      `estrategia_operativo` TEXT,
      `budget_total` DECIMAL(15, 2),
      FOREIGN KEY (`cliente_id`) REFERENCES `clients`(`id`) ON DELETE RESTRICT
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

    CREATE TABLE `project_professionals` (
      `project_id` INT NOT NULL,
      `professional_id` INT NOT NULL,
      PRIMARY KEY (`project_id`, `professional_id`),
      FOREIGN KEY (`project_id`) REFERENCES `projects`(`id`) ON DELETE CASCADE,
      FOREIGN KEY (`professional_id`) REFERENCES `professionals`(`id`) ON DELETE CASCADE
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

    CREATE TABLE `tasks` (
      `id` INT AUTO_INCREMENT PRIMARY KEY,
      `project_id` INT NOT NULL,
      `nombre_tarea` VARCHAR(255) NOT NULL,
      `asignado_a` INT,
      `fecha_inicio` DATE,
      `fecha_fin` DATE,
      `fecha_fin_real` DATE,
      `porcentaje` INT DEFAULT 0,
      `prioridad` VARCHAR(50),
      `comentarios` JSON,
      `archivos` JSON,
      `links` JSON,
      `tags` JSON,
      FOREIGN KEY (`project_id`) REFERENCES `projects`(`id`) ON DELETE CASCADE,
      FOREIGN KEY (`asignado_a`) REFERENCES `professionals`(`id`) ON DELETE SET NULL
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

    CREATE TABLE `content_grids` (
      `id` INT AUTO_INCREMENT PRIMARY KEY,
      `project_id` INT NOT NULL,
      `name` VARCHAR(255) NOT NULL,
      `start_date` DATE,
      `end_date` DATE,
      `networks` JSON,
      `responsables` JSON,
      FOREIGN KEY (`project_id`) REFERENCES `projects`(`id`) ON DELETE CASCADE
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

    CREATE TABLE `grid_posts` (
      `id` INT AUTO_INCREMENT PRIMARY KEY,
      `grid_id` INT NOT NULL,
      `fecha_posteo` DATE,
      `hora_posteo` TIME,
      `objetivo` TEXT,
      `redes` JSON,
      `tipo_contenido` VARCHAR(100),
      `hook` TEXT,
      `copy` TEXT,
      `hashtags` TEXT,
      `cta` TEXT,
      `publico_objetivo` TEXT,
      `resultado_esperado` TEXT,
      `realizado` BOOLEAN DEFAULT FALSE,
      `fecha_realizado` DATETIME,
      FOREIGN KEY (`grid_id`) REFERENCES `content_grids`(`id`) ON DELETE CASCADE
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

    CREATE TABLE `media_plans` (
      `id` INT AUTO_INCREMENT PRIMARY KEY,
      `project_id` INT NOT NULL,
      `name` VARCHAR(255) NOT NULL,
      `objective` TEXT,
      `audience` TEXT,
      `budget` DECIMAL(15, 2),
      `start_date` DATE,
      `end_date` DATE,
      `items` JSON,
      `responsables` JSON,
      FOREIGN KEY (`project_id`) REFERENCES `projects`(`id`) ON DELETE CASCADE
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

    CREATE TABLE `invoices` (
      `id` INT AUTO_INCREMENT PRIMARY KEY,
      `client_id` INT NOT NULL,
      `invoice_number` VARCHAR(100) NOT NULL,
      `invoice_date` DATE NOT NULL,
      `due_date` DATE,
      `status` VARCHAR(50) DEFAULT 'Pendiente',
      `totals` JSON,
      FOREIGN KEY (`client_id`) REFERENCES `clients`(`id`) ON DELETE RESTRICT
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

    CREATE TABLE `expenses` (
      `id` INT AUTO_INCREMENT PRIMARY KEY,
      `provider` VARCHAR(255) NOT NULL,
      `expense_number` VARCHAR(100),
      `expense_date` DATE NOT NULL,
      `due_date` DATE,
      `status` VARCHAR(50) DEFAULT 'Pendiente',
      `totals` JSON
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
    ";
    $pdo->exec($sql_schema);
} catch (PDOException $e) {
    if (file_exists($security_path)) unlink($security_path);
    unlink($config_path);
    echo json_encode(['success' => false, 'message' => 'Error al crear las tablas: ' . $e->getMessage()]);
    exit;
}

// --- 5. Insertar el primer usuario administrador ---
try {
    $hashed_password = password_hash($admin_password, PASSWORD_DEFAULT);
    $stmt = $pdo->prepare("INSERT INTO professionals (nombre_completo, email, password, puesto, estado) VALUES (:name, :email, :password, :puesto, 'Activo')");
    $stmt->execute(['name' => $admin_name, 'email' => $admin_email, 'password' => $hashed_password, 'puesto' => $admin_puesto]);
} catch (PDOException $e) {
    if (file_exists($security_path)) unlink($security_path);
    unlink($config_path);
    echo json_encode(['success' => false, 'message' => 'Error al crear el usuario administrador: ' . $e->getMessage()]);
    exit;
}

// --- 6. Crear tablas del chat ---
try {
    $chat_schema = "
    CREATE TABLE IF NOT EXISTS `chat_conversations` (
      `id` INT AUTO_INCREMENT PRIMARY KEY,
      `name` VARCHAR(255),
      `group_photo_url` VARCHAR(255),
      `is_group_chat` BOOLEAN NOT NULL DEFAULT FALSE,
      `created_by` INT,
      `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
      FOREIGN KEY (`created_by`) REFERENCES `professionals`(`id`) ON DELETE SET NULL
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

    CREATE TABLE IF NOT EXISTS `chat_conversation_members` (
      `conversation_id` INT NOT NULL,
      `user_id` INT NOT NULL,
      PRIMARY KEY (`conversation_id`, `user_id`),
      FOREIGN KEY (`conversation_id`) REFERENCES `chat_conversations`(`id`) ON DELETE CASCADE,
      FOREIGN KEY (`user_id`) REFERENCES `professionals`(`id`) ON DELETE CASCADE
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

    CREATE TABLE IF NOT EXISTS `chat_messages` (
      `id` INT AUTO_INCREMENT PRIMARY KEY,
      `conversation_id` INT NOT NULL,
      `sender_id` INT NOT NULL,
      `message_content` TEXT NOT NULL,
      `timestamp` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
      FOREIGN KEY (`conversation_id`) REFERENCES `chat_conversations`(`id`) ON DELETE CASCADE,
      FOREIGN KEY (`sender_id`) REFERENCES `professionals`(`id`) ON DELETE CASCADE
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
    ";
    $pdo->exec($chat_schema);
} catch (PDOException $e) {
    if (file_exists($security_path)) unlink($security_path);
    unlink($config_path);
    echo json_encode(['success' => false, 'message' => 'Error al crear las tablas del chat: ' . $e->getMessage()]);
    exit;
}

echo json_encode(['success' => true, 'message' => '¡Instalación completada!']);
?>