<?php
// Backend para gestionar los proyectos (CRUD)
header("Content-Type: application/json");
require_once __DIR__ . '/../config/db.php';

$method = $_SERVER['REQUEST_METHOD'];
$id = isset($_GET['id']) ? (int)$_GET['id'] : null;
$pdo = get_db_connection();

if ($method === 'GET') {
    if ($id !== null) {
        // Obtener un solo proyecto con sus profesionales asignados
        $stmt = $pdo->prepare("SELECT p.*, c.nombre_empresa as cliente_nombre FROM projects p JOIN clients c ON p.cliente_id = c.id WHERE p.id = ?");
        $stmt->execute([$id]);
        $project = $stmt->fetch();

        if ($project) {
            // Obtener IDs de profesionales asignados
            $stmt_prof = $pdo->prepare("SELECT professional_id FROM project_professionals WHERE project_id = ?");
            $stmt_prof->execute([$id]);
            $professional_ids = $stmt_prof->fetchAll(PDO::FETCH_COLUMN);
            $project['profesionales_asignados'] = $professional_ids;
            echo json_encode($project);
        } else {
            http_response_code(404);
            echo json_encode(['error' => 'Proyecto no encontrado']);
        }
    } else {
        // Obtener todos los proyectos
        $stmt = $pdo->query("SELECT p.*, c.nombre_empresa as cliente_nombre FROM projects p JOIN clients c ON p.cliente_id = c.id ORDER BY p.fecha_inicio DESC");
        $projects = $stmt->fetchAll();
        echo json_encode($projects);
    }
} elseif ($method === 'POST') {
    $input = json_decode(file_get_contents('php://input'), true);

    if (empty($input['nombre_proyecto']) || empty($input['cliente_id'])) {
        http_response_code(400);
        echo json_encode(['error' => 'El nombre del proyecto y el cliente son obligatorios.']);
        exit;
    }

    try {
        $pdo->beginTransaction();
        try {
            $sql = "INSERT INTO projects (nombre_proyecto, cliente_id, descripcion, fecha_inicio, fecha_fin_estimada, tipo_marketing, estado, estrategia_estrategico, estrategia_tactico, estrategia_operativo, budget_total) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
            $stmt = $pdo->prepare($sql);
            $stmt->execute([
                $input['nombre_proyecto'],
                $input['cliente_id'],
                $input['descripcion'] ?? '',
                empty($input['fecha_inicio']) ? null : $input['fecha_inicio'],
                empty($input['fecha_fin_estimada']) ? null : $input['fecha_fin_estimada'],
                $input['tipo_marketing'] ?? 'Mixto',
                $input['estado'] ?? 'Planificación',
                $input['estrategia_estrategico'] ?? '',
                $input['estrategia_tactico'] ?? '',
                $input['estrategia_operativo'] ?? '',
                $input['budget_total'] ?? 0
            ]);
            $new_id = $pdo->lastInsertId();

            // Insertar profesionales asignados en la tabla pivote
            if (!empty($input['profesionales_asignados'])) {
                $stmt_prof = $pdo->prepare("INSERT INTO project_professionals (project_id, professional_id) VALUES (?, ?)");
                foreach ($input['profesionales_asignados'] as $prof_id) {
                    $stmt_prof->execute([$new_id, $prof_id]);
                }
            }

            $pdo->commit();

            // Devolver el proyecto recién creado
            $stmt = $pdo->prepare("SELECT * FROM projects WHERE id = ?");
            $stmt->execute([$new_id]);
            $new_project = $stmt->fetch();
            $new_project['profesionales_asignados'] = $input['profesionales_asignados'] ?? [];

            http_response_code(201);
            echo json_encode($new_project);

        } catch (Exception $e) {
            $pdo->rollBack();
            throw $e; // Relanzar para que el catch externo lo maneje
        }
    } catch (PDOException $e) {
        $pdo->rollBack();
        http_response_code(500);
        echo json_encode(['error' => 'Error al crear el proyecto: ' . $e->getMessage()]);
    }

} elseif ($method === 'PUT' && $id !== null) {
    $input = json_decode(file_get_contents('php://input'), true);

    // Lógica para la fecha de fin real
    if (isset($input['estado']) && $input['estado'] === 'Completado') {
        // Obtener el estado actual para evitar actualizar la fecha si ya estaba completado
        $stmt_check = $pdo->prepare("SELECT estado FROM projects WHERE id = ?");
        $stmt_check->execute([$id]);
        $current_status = $stmt_check->fetchColumn();
        if ($current_status !== 'Completado') {
            $input['fecha_fin_real'] = date('Y-m-d');
        }
    } elseif (isset($input['estado']) && $input['estado'] !== 'Completado') {
        $input['fecha_fin_real'] = null; // Si se revierte el estado, limpiamos la fecha.
    }

    $pdo->beginTransaction();
    try {
        $sql = "UPDATE projects SET 
                    nombre_proyecto = :nombre_proyecto, 
                    cliente_id = :cliente_id, 
                    descripcion = :descripcion, 
                    fecha_inicio = :fecha_inicio, 
                    fecha_fin_estimada = :fecha_fin_estimada, 
                    fecha_fin_real = :fecha_fin_real,
                    tipo_marketing = :tipo_marketing, 
                    estado = :estado, 
                    estrategia_estrategico = :estrategia_estrategico, 
                    estrategia_tactico = :estrategia_tactico, 
                    estrategia_operativo = :estrategia_operativo, 
                    budget_total = :budget_total 
                WHERE id = :id";
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            ':nombre_proyecto' => $input['nombre_proyecto'],
            ':cliente_id' => $input['cliente_id'],
            ':descripcion' => $input['descripcion'],
            ':fecha_inicio' => empty($input['fecha_inicio']) ? null : $input['fecha_inicio'],
            ':fecha_fin_estimada' => empty($input['fecha_fin_estimada']) ? null : $input['fecha_fin_estimada'],
            ':fecha_fin_real' => $input['fecha_fin_real'] ?? null,
            ':tipo_marketing' => $input['tipo_marketing'],
            ':estado' => $input['estado'],
            ':estrategia_estrategico' => $input['estrategia_estrategico'],
            ':estrategia_tactico' => $input['estrategia_tactico'],
            ':estrategia_operativo' => $input['estrategia_operativo'],
            ':budget_total' => $input['budget_total'] ?? 0,
            ':id' => $id
        ]);

        // Actualizar profesionales asignados
        if (isset($input['profesionales_asignados'])) {
            // 1. Borrar asignaciones existentes
            $stmt_del = $pdo->prepare("DELETE FROM project_professionals WHERE project_id = ?");
            $stmt_del->execute([$id]);
            // 2. Insertar nuevas asignaciones
            if (!empty($input['profesionales_asignados'])) {
                $stmt_add = $pdo->prepare("INSERT INTO project_professionals (project_id, professional_id) VALUES (?, ?)");
                foreach ($input['profesionales_asignados'] as $prof_id) {
                    $stmt_add->execute([$id, $prof_id]);
                }
            }
        }

        $pdo->commit();

        // Devolver el proyecto actualizado
        $stmt = $pdo->prepare("SELECT * FROM projects WHERE id = ?");
        $stmt->execute([$id]);
        $updated_project = $stmt->fetch();
        $updated_project['profesionales_asignados'] = $input['profesionales_asignados'] ?? [];

        echo json_encode($updated_project);

    } catch (Exception $e) {
        $pdo->rollBack();
        http_response_code(500);
        echo json_encode(['error' => 'Error al actualizar el proyecto: ' . $e->getMessage()]);
    }

} elseif ($method === 'DELETE' && $id !== null) {
    // La eliminación en cascada (`ON DELETE CASCADE`) en las tablas `tasks` y `project_professionals`
    // se encargará de borrar los registros relacionados automáticamente.
    $stmt = $pdo->prepare("DELETE FROM projects WHERE id = ?");
    $stmt->execute([$id]);

    if ($stmt->rowCount()) {
        http_response_code(204); // No Content
    } else {
        http_response_code(404);
        echo json_encode(['error' => 'Proyecto no encontrado']);
    }

} else {
    http_response_code(405);
    echo json_encode(['error' => 'Método no permitido o ID de proyecto no proporcionado']);
}
?>