<?php
// Backend para gestionar los profesionales (CRUD)
header("Content-Type: application/json");
ini_set('display_errors', 1); // Mostrar errores para depuración
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

require_once __DIR__ . '/../api/check_session.php';

/**
 * Maneja la subida de un archivo de imagen.
 * @param array $file - El array del archivo de $_FILES.
 * @return string|null - La URL del archivo guardado o null si hay error.
 */
function handle_file_upload($file) {
    if (isset($file) && $file['error'] === UPLOAD_ERR_OK) {
        $upload_dir = '../uploads/professionals/';
        if (!is_dir($upload_dir)) {
            mkdir($upload_dir, 0777, true);
        }
        $file_extension = pathinfo($file['name'], PATHINFO_EXTENSION);
        $file_name = 'prof_' . uniqid() . '.' . $file_extension;
        $target_path = $upload_dir . $file_name;

        if (move_uploaded_file($file['tmp_name'], $target_path)) {
            // Devuelve la URL relativa accesible desde el cliente
            return 'uploads/professionals/' . $file_name;
        }
    }
    return null;
}

$method = $_SERVER['REQUEST_METHOD'];
$id = isset($_GET['id']) ? (int)$_GET['id'] : null;
$id = $id ?: (isset($_POST['id']) ? (int)$_POST['id'] : null); // También buscar ID en POST para FormData

$pdo = get_db_connection();

if ($method === 'GET') {
    if ($id !== null) {
        $stmt = $pdo->prepare("SELECT * FROM professionals WHERE id = ?");
        $stmt->execute([$id]);
        $professional = $stmt->fetch();
        if ($professional) {
            echo json_encode($professional);
        } else {
            http_response_code(404);
            echo json_encode(['error' => 'Profesional no encontrado']);
        }
    } else {
        $stmt = $pdo->query("SELECT * FROM professionals ORDER BY nombre_completo");
        $professionals = $stmt->fetchAll();
        echo json_encode($professionals);
    }
} elseif ($method === 'POST') {
    // Solo roles autorizados pueden crear o editar
    if (!check_permission('profesionales')) {
        http_response_code(403); // Forbidden
        echo json_encode(['error' => 'No tienes permiso para gestionar profesionales.']);
        exit;
    }

    $input = $_POST;

    if ($id) { // --- MODO EDICIÓN ---
        // Primero, obtener los datos actuales para la foto
        $stmt = $pdo->prepare("SELECT foto_url FROM professionals WHERE id = ?");
        $stmt->execute([$id]);
        $current_professional = $stmt->fetch();

        if (!$current_professional) {
            http_response_code(404);
            echo json_encode(['success' => false, 'error' => 'Profesional no encontrado para actualizar']);
            exit;
        }

        // Construcción dinámica de la consulta UPDATE para manejar solo los campos presentes
        $fields = [];
        $params = [];

        // Campos que pueden ser actualizados
        $update_data = [
            'nombre_completo' => $input['nombre_completo'] ?? $current_professional['nombre_completo'],
            'puesto' => $input['puesto'] ?? $current_professional['puesto'],
            'tipo_contrato' => $input['tipo_contrato'] ?? $current_professional['tipo_contrato'],
            'fecha_ingreso' => empty($input['fecha_ingreso']) ? null : ($input['fecha_ingreso'] ?? $current_professional['fecha_ingreso']),
            'salario_mensual' => empty($input['salario_mensual']) ? null : ($input['salario_mensual'] ?? $current_professional['salario_mensual']),
            'email' => $input['email'] ?? $current_professional['email'], // Email es NOT NULL, siempre debe tener un valor
            'telefono' => $input['telefono'] ?? $current_professional['telefono'],
            'estado' => $input['estado'] ?? $current_professional['estado'],
        ];

        foreach ($update_data as $field => $value) {
            $fields[] = "$field = ?";
            $params[] = $value;
        }

        // Contraseña solo si se proporciona una nueva
        if (!empty($input['password'])) {
            $fields[] = "password = ?";
            $params[] = password_hash($input['password'], PASSWORD_DEFAULT);
        }

        // Manejar subida de foto (si hay una nueva)
        $new_foto_url = null;
        if (isset($_FILES['foto']) && $_FILES['foto']['error'] === UPLOAD_ERR_OK) {
            $new_foto_url = handle_file_upload($_FILES['foto']);
            if ($new_foto_url) {
                // Borrar la foto antigua si existe y es diferente a la nueva
                if (!empty($current_professional['foto_url']) && $current_professional['foto_url'] !== $new_foto_url && file_exists(__DIR__ . '/../' . $current_professional['foto_url'])) {
                    @unlink(__DIR__ . '/../' . $current_professional['foto_url']);
                }
                $fields[] = "foto_url = ?";
                $params[] = $new_foto_url;
            }
        }

        try {
            if (count($fields) > 0) {
                $sql = "UPDATE professionals SET " . implode(', ', $fields) . " WHERE id = ?";
                $params[] = $id;

                $stmt = $pdo->prepare($sql);
                $stmt->execute($params);
            }

            $stmt = $pdo->prepare("SELECT * FROM professionals WHERE id = ?");
            $stmt->execute([$id]);
            $updated_professional = $stmt->fetch();

            echo json_encode(['success' => true, 'professional' => $updated_professional]);
        } catch (PDOException $e) {
            http_response_code(500);
            echo json_encode(['success' => false, 'error' => 'Error al actualizar profesional: ' . $e->getMessage()]);
        }

    } else { // --- MODO CREACIÓN ---
        // Solo roles autorizados pueden crear
        if (!check_permission('profesionales')) {
            http_response_code(403); // Forbidden
            echo json_encode(['error' => 'No tienes permiso para gestionar profesionales.']);
            exit;
        }

        try {
            $sql = "INSERT INTO professionals (nombre_completo, puesto, tipo_contrato, fecha_ingreso, salario_mensual, email, telefono, estado, password, foto_url) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
            $params = [
                $input['nombre_completo'] ?? '',
                $input['puesto'] ?? '',
                $input['tipo_contrato'] ?? '',
                empty($input['fecha_ingreso']) ? null : $input['fecha_ingreso'],
                empty($input['salario_mensual']) ? null : $input['salario_mensual'],
                $input['email'] ?? '', // Email es NOT NULL, el frontend debe asegurar que no esté vacío
                $input['telefono'] ?? '',
                $input['estado'] ?? 'Activo',
                password_hash($input['password'] ?: 'password123', PASSWORD_DEFAULT), // Contraseña por defecto si está vacía
                handle_file_upload($_FILES['foto'] ?? null)
            ];

            $stmt = $pdo->prepare($sql);
            $stmt->execute($params);
            $new_id = $pdo->lastInsertId();

            $stmt = $pdo->prepare("SELECT * FROM professionals WHERE id = ?");
            $stmt->execute([$new_id]);
            $new_professional = $stmt->fetch();

            http_response_code(201);
            echo json_encode(['success' => true, 'professional' => $new_professional]);
        } catch (PDOException $e) {
            http_response_code(500);
            echo json_encode(['success' => false, 'error' => 'Error al crear profesional: ' . $e->getMessage()]);
        }
    }
} elseif ($method === 'DELETE' && $id !== null) {
    // Solo roles autorizados pueden eliminar
    if (!check_permission('profesionales')) {
        http_response_code(403); // Forbidden
        echo json_encode(['error' => 'No tienes permiso para gestionar profesionales.']);
        exit;
    }

    try {
        // Verificar si el profesional está asignado a algún proyecto o tarea
        $stmt_check = $pdo->prepare("SELECT COUNT(*) FROM project_professionals WHERE professional_id = ?");
        $stmt_check->execute([$id]);
        if ($stmt_check->fetchColumn() > 0) {
            http_response_code(409); // Conflict
            echo json_encode(['success' => false, 'error' => 'No se puede eliminar el profesional porque está asignado a uno o más proyectos.']);
            exit;
        }

        $stmt = $pdo->prepare("DELETE FROM professionals WHERE id = ?");
        $deleted = $stmt->execute([$id]);
        if ($deleted) {
            echo json_encode(['success' => true]);
        }
    } else {
        http_response_code(404);
        echo json_encode(['success' => false, 'error' => 'Profesional no encontrado']);
    }

} else {
    http_response_code(405);
    echo json_encode(['error' => 'Método no permitido o ID no proporcionado']);
}

?>