<?php
header("Content-Type: application/json");
require_once __DIR__ . '/../config/db.php';
ini_set('display_errors', 1); // Mostrar errores para depuración
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);
require_once __DIR__ . '/../config/security.php';

/**
 * Lee la configuración de Meta desde la base de datos.
 * @param PDO $pdo
 * @return array|null
 */
function get_meta_settings($pdo) {
    $stmt = $pdo->prepare("SELECT setting_value FROM settings WHERE setting_key = 'meta_settings'");
    $stmt->execute();
    $result = $stmt->fetchColumn();
    if ($result) {
        return json_decode($result, true);
    }
    return null;
}

function make_curl_request($url) {
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    $response = curl_exec($ch);
    curl_close($ch);
    return json_decode($response, true);
}

$method = $_SERVER['REQUEST_METHOD'];

if ($method === 'GET') {
    // 1. Obtener credenciales de la base de datos SQL
    $pdo = get_db_connection();
    $meta_settings = get_meta_settings($pdo);
    $pageId = $meta_settings['page_id'] ?? null;
    $encryptedToken = $meta_settings['page_access_token'] ?? null;
    $pageAccessToken = $encryptedToken ? decrypt_token($encryptedToken) : null;

    if (!$pageId || !$pageAccessToken) {
        http_response_code(409); // Conflict
        echo json_encode(['error' => 'La cuenta de Meta no está conectada.']);
        exit;
    }

    // 2. Definir métricas y rangos de fechas
    $metrics = [
        'page_fans', // Total de "Me gusta" (no es una serie de tiempo)
        'page_fans_country', // Fans por país (lifetime)
        'page_post_engagements', // Interacciones por día
        'page_impressions', // Impresiones por día
        'page_reach_unique' // Alcance por día
    ];
    $metrics_string = implode(',', $metrics);

    // Leer fechas desde los parámetros GET, o usar valores por defecto (últimos 30 días)
    $since = isset($_GET['since']) ? strtotime($_GET['since']) : strtotime('-30 days');
    $until = isset($_GET['until']) ? strtotime($_GET['until'] . ' 23:59:59') : time(); // Incluir el día completo

    // Calcular el período anterior
    $duration = $until - $since;
    $previous_period_until = $since - 1;
    $previous_period_since = $previous_period_until - $duration;

    // 3. Petición para el período ACTUAL
    $url = "https://graph.facebook.com/v20.0/{$pageId}/insights"
         . "?metric={$metrics_string}"
         . "&period=day"
         . "&since={$since}"
         . "&until={$until}"
         . "&access_token={$pageAccessToken}";

    $current_period_response = make_curl_request($url);

    if (isset($current_period_response['error'])) {
        http_response_code(500);
        echo json_encode(['error' => 'Error al obtener datos del período actual: ' . $current_period_response['error']['message']]);
        exit;
    }

    // Petición para el período ANTERIOR
    $previous_period_url = "https://graph.facebook.com/v20.0/{$pageId}/insights"
         . "?metric={$metrics_string}"
         . "&period=day"
         . "&since={$previous_period_since}"
         . "&until={$previous_period_until}"
         . "&access_token={$pageAccessToken}";
    
    $previous_period_response = make_curl_request($previous_period_url);
    // No consideramos un error aquí si falla, simplemente no habrá datos de comparación.

    // 4. Procesar y formatear la respuesta para el frontend
    $formatted_response = [];
    if (isset($current_period_response['data'])) {
        foreach ($current_period_response['data'] as $metric_data) {
            $metric_name = $metric_data['name'];
            $formatted_response[$metric_name] = [
                'title' => $metric_data['title'],
                'description' => $metric_data['description'],
                'current_values' => $metric_data['values'],
                'previous_values' => [] // Inicializar
            ];
        }
    }

    // Añadir los datos del período anterior si existen
    if (isset($previous_period_response['data'])) {
        foreach ($previous_period_response['data'] as $metric_data) {
            $metric_name = $metric_data['name'];
            if (isset($formatted_response[$metric_name])) {
                $formatted_response[$metric_name]['previous_values'] = $metric_data['values'];
            }
        }
    }

    // La métrica 'page_fans' solo devuelve el último valor, no una serie de tiempo.
    // Lo extraemos para mostrarlo en una tarjeta.
    $total_fans = end($formatted_response['page_fans']['current_values'])['value'] ?? 0;
    $formatted_response['total_fans'] = $total_fans;

    // 5. Obtener los 5 posts más recientes y sus métricas
    $post_metrics = [
        'post_engagements',
        'post_impressions'
    ];
    $post_metrics_string = implode(',', $post_metrics);
    $posts_url = "https://graph.facebook.com/v20.0/{$pageId}/posts"
               . "?fields=id,message,created_time"
               . "&limit=5"
               . "&access_token={$pageAccessToken}";
    
    $posts_response = make_curl_request($posts_url);
    $top_posts_data = [];

    if (isset($posts_response['data'])) {
        foreach ($posts_response['data'] as $post) {
            $post_id = $post['id'];
            $post_insights_url = "https://graph.facebook.com/v20.0/{$post_id}/insights"
                               . "?metric={$post_metrics_string}"
                               . "&period=lifetime"
                               . "&access_token={$pageAccessToken}";
            
            $post_insights_response = make_curl_request($post_insights_url);

            $engagements = 0;
            if (isset($post_insights_response['data'])) {
                foreach ($post_insights_response['data'] as $insight) {
                    if ($insight['name'] === 'post_engagements') {
                        $engagements = $insight['values'][0]['value'] ?? 0;
                    }
                }
            }
            
            $top_posts_data[] = [
                'message' => substr($post['message'] ?? 'Sin mensaje', 0, 50) . '...',
                'engagements' => $engagements
            ];
        }
    }
    $formatted_response['top_posts'] = $top_posts_data;

    echo json_encode($formatted_response);

} else {
    http_response_code(405);
    echo json_encode(['error' => 'Método no permitido.']);
}
?>