document.addEventListener('DOMContentLoaded', function () {

    // --- ELEMENTOS DEL DOM ---
    const loader = document.getElementById('loader');
    const errorContainer = document.getElementById('error-container');
    const dashboardContent = document.getElementById('dashboard-content');
    const sinceDateInput = document.getElementById('since-date');
    const untilDateInput = document.getElementById('until-date');
    const filterBtn = document.getElementById('filter-btn');

    // --- URLs DE LA API ---
    const META_INSIGHTS_API_URL = 'api/meta_insights.php';

    // --- ESTADO Y GRÁFICOS ---
    let chartInstances = {
        performance: null,
        country: null,
        topPosts: null
    };
    let fullData = null;

    // --- FUNCIONES DE UTILIDAD ---
    const formatNumber = (num) => new Intl.NumberFormat('es-ES').format(num);

    // --- FUNCIONES PRINCIPALES ---

    async function fetchDashboardData(since, until) {
        loader.style.display = 'block';
        errorContainer.style.display = 'none';
        dashboardContent.style.display = 'none';

        try {
            const url = `${META_INSIGHTS_API_URL}?since=${since}&until=${until}`;
            const response = await fetch(url);
            const data = await response.json();

            if (!response.ok) {
                throw new Error(data.error || 'Error desconocido al obtener los datos de Meta.');
            }

            fullData = data; // Guardar los datos completos
            renderDashboard(data);

            loader.style.display = 'none';
            dashboardContent.style.display = 'block';

        } catch (error) {
            console.error('Error fetching Meta data:', error);
            loader.style.display = 'none';
            errorContainer.textContent = `Error: ${error.message}`;
            errorContainer.style.display = 'block';
        }
    }

    function renderDashboard(data) {
        renderKpiCards(data);
        renderPerformanceChart(data);
        renderCountryChart(data.page_fans_country);
        renderTopPostsChart(data.top_posts);
    }

    // --- FUNCIONES DE RENDERIZADO DE COMPONENTES ---

    function renderKpiCards(data) {
        // Función para calcular totales y comparaciones
        const calculateKpi = (currentMetric, previousMetric) => {
            const currentTotal = currentMetric.reduce((sum, item) => sum + item.value, 0);
            const previousTotal = previousMetric.reduce((sum, item) => sum + item.value, 0);
            
            let comparison = 0;
            if (previousTotal > 0) {
                comparison = ((currentTotal - previousTotal) / previousTotal) * 100;
            } else if (currentTotal > 0) {
                comparison = 100; // Si antes era 0 y ahora es > 0, es un 100% de aumento
            }

            const comparisonText = comparison >= 0 
                ? `<span class="text-success"><i class="bi bi-arrow-up"></i> ${comparison.toFixed(1)}%</span>`
                : `<span class="text-danger"><i class="bi bi-arrow-down"></i> ${comparison.toFixed(1)}%</span>`;

            return { total: currentTotal, comparisonHtml: `${comparisonText} vs período anterior` };
        };

        // Total Fans (métrica de vida, no de período)
        const totalFans = data.total_fans || 0;
        document.getElementById('total-fans-kpi').textContent = formatNumber(totalFans);
        document.getElementById('total-fans-comparison').innerHTML = 'Total histórico';

        // Interacciones
        const engagements = calculateKpi(data.page_post_engagements?.current_values || [], data.page_post_engagements?.previous_values || []);
        document.getElementById('total-engagements-kpi').textContent = formatNumber(engagements.total);
        document.getElementById('total-engagements-comparison').innerHTML = engagements.comparisonHtml;

        // Impresiones
        const impressions = calculateKpi(data.page_impressions?.current_values || [], data.page_impressions?.previous_values || []);
        document.getElementById('total-impressions-kpi').textContent = formatNumber(impressions.total);
        document.getElementById('total-impressions-comparison').innerHTML = impressions.comparisonHtml;
    }

    function renderPerformanceChart(data) {
        const ctx = document.getElementById('performance-chart').getContext('2d');
        if (chartInstances.performance) {
            chartInstances.performance.destroy();
        }

        const metricCheckboxes = document.querySelectorAll('#performance-chart-controls input[type="checkbox"]');
        const activeMetrics = Array.from(metricCheckboxes).filter(cb => cb.checked).map(cb => cb.value);

        const datasets = [];
        const colors = {
            page_impressions: 'rgba(54, 162, 235, 1)',
            page_reach_unique: 'rgba(75, 192, 192, 1)',
            page_post_engagements: 'rgba(255, 99, 132, 1)'
        };
        const labels = data.page_impressions?.current_values.map(v => new Date(v.end_time).toLocaleDateString('es-ES', { day: '2-digit', month: 'short' })) || [];

        activeMetrics.forEach(metricKey => {
            if (data[metricKey]) {
                datasets.push({
                    label: data[metricKey].title,
                    data: data[metricKey].current_values.map(v => v.value),
                    borderColor: colors[metricKey],
                    tension: 0.1,
                    fill: false
                });
            }
        });

        chartInstances.performance = new Chart(ctx, {
            type: 'line',
            data: {
                labels: labels,
                datasets: datasets
            },
            options: {
                responsive: true,
                plugins: {
                    title: { display: true, text: 'Rendimiento a lo largo del tiempo' },
                    legend: { position: 'top' }
                },
                scales: {
                    y: { beginAtZero: true }
                }
            }
        });
    }

    function renderCountryChart(countryData) {
        const ctx = document.getElementById('country-chart').getContext('2d');
        if (chartInstances.country) {
            chartInstances.country.destroy();
        }

        if (!countryData || !countryData.current_values || countryData.current_values.length === 0) {
            return;
        }

        const data = countryData.current_values[0].value;
        const sortedCountries = Object.entries(data).sort(([, a], [, b]) => b - a);
        
        const topCountries = sortedCountries.slice(0, 7);
        const otherCount = sortedCountries.slice(7).reduce((sum, [, count]) => sum + count, 0);

        const labels = topCountries.map(([countryCode]) => countryCode);
        const values = topCountries.map(([, count]) => count);

        if (otherCount > 0) {
            labels.push('Otros');
            values.push(otherCount);
        }

        chartInstances.country = new Chart(ctx, {
            type: 'doughnut',
            data: {
                labels: labels,
                datasets: [{
                    label: 'Fans por País',
                    data: values,
                    backgroundColor: [
                        '#0d6efd', '#6f42c1', '#d63384', '#fd7e14', '#ffc107', '#198754', '#20c997', '#6c757d'
                    ]
                }]
            },
            options: {
                responsive: true,
                plugins: {
                    title: { display: true, text: 'Top Países por Fans' },
                    legend: { position: 'right' }
                }
            }
        });
    }

    function renderTopPostsChart(topPostsData) {
        const ctx = document.getElementById('top-posts-chart').getContext('2d');
        if (chartInstances.topPosts) {
            chartInstances.topPosts.destroy();
        }

        if (!topPostsData || topPostsData.length === 0) return;

        const labels = topPostsData.map(post => post.message);
        const data = topPostsData.map(post => post.engagements);

        chartInstances.topPosts = new Chart(ctx, {
            type: 'bar',
            data: {
                labels: labels,
                datasets: [{
                    label: 'Interacciones',
                    data: data,
                    backgroundColor: 'rgba(255, 159, 64, 0.7)',
                    borderColor: 'rgba(255, 159, 64, 1)',
                    borderWidth: 1
                }]
            },
            options: {
                indexAxis: 'y', // Gráfico de barras horizontales
                responsive: true,
                plugins: {
                    title: { display: true, text: 'Top 5 Publicaciones Recientes por Interacción' },
                    legend: { display: false }
                },
                scales: {
                    x: { beginAtZero: true }
                }
            }
        });
    }

    // --- EVENT LISTENERS ---

    filterBtn.addEventListener('click', () => {
        const since = sinceDateInput.value;
        const until = untilDateInput.value;
        if (since && until) {
            fetchDashboardData(since, until);
        } else {
            alert('Por favor, seleccione ambas fechas, "Desde" y "Hasta".');
        }
    });

    document.querySelectorAll('#performance-chart-controls input').forEach(checkbox => {
        checkbox.addEventListener('change', () => {
            if (fullData) {
                renderPerformanceChart(fullData);
            }
        });
    });

    // --- CARGA INICIAL ---

    function setDefaultDatesAndLoad() {
        const today = new Date();
        const thirtyDaysAgo = new Date();
        thirtyDaysAgo.setDate(today.getDate() - 30);

        // Formato YYYY-MM-DD
        untilDateInput.value = today.toISOString().split('T')[0];
        sinceDateInput.value = thirtyDaysAgo.toISOString().split('T')[0];

        fetchDashboardData(sinceDateInput.value, untilDateInput.value);
    }

    setDefaultDatesAndLoad();
});