<?php
header("Content-Type: application/json");
require_once __DIR__ . '/../config/db.php';

$method = $_SERVER['REQUEST_METHOD'];
$project_id = isset($_GET['project_id']) ? (int)$_GET['project_id'] : null;
$plan_id = isset($_GET['id']) ? (int)$_GET['id'] : null;
$pdo = get_db_connection();

/**
 * Decodifica los campos JSON de un plan de medios.
 * @param array $plan El plan de la BD.
 * @return array El plan con los campos JSON decodificados.
 */
function decode_plan_json_fields($plan) {
    if ($plan) {
        $json_fields = ['items', 'responsables'];
        foreach ($json_fields as $field) {
            $plan[$field] = $plan[$field] ? json_decode($plan[$field], true) : [];
        }
    }
    return $plan;
}

if ($method === 'GET') {
    if ($plan_id !== null) {
        $stmt = $pdo->prepare("SELECT * FROM media_plans WHERE id = ?");
        $stmt->execute([$plan_id]);
        $plan = $stmt->fetch();
        if ($plan) {
            echo json_encode(decode_plan_json_fields($plan));
        } else {
            http_response_code(404);
            echo json_encode(['error' => 'Plan de medios no encontrado']);
        }
    } elseif ($project_id !== null) {
        $stmt = $pdo->prepare("SELECT * FROM media_plans WHERE project_id = ? ORDER BY start_date DESC");
        $stmt->execute([$project_id]);
        $project_plans = $stmt->fetchAll();
        $project_plans = array_map('decode_plan_json_fields', $project_plans);
        echo json_encode($project_plans);
    } else {
        $stmt = $pdo->query("SELECT * FROM media_plans ORDER BY start_date DESC");
        $plans = $stmt->fetchAll();
        $plans = array_map('decode_plan_json_fields', $plans);
        echo json_encode($plans);
    }

} elseif ($method === 'POST') {
    $input = json_decode(file_get_contents('php://input'), true);
    if (empty($input['project_id']) || empty($input['name']) || empty($input['start_date']) || empty($input['end_date'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Nombre, fechas de inicio/fin y ID de proyecto son obligatorios.']);
        exit;
    }

    try {
        $sql = "INSERT INTO media_plans (project_id, name, objective, audience, budget, start_date, end_date, items, responsables) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            (int)$input['project_id'],
            $input['name'],
            $input['objective'] ?? '',
            $input['audience'] ?? '',
            (float)($input['budget'] ?? 0),
            $input['start_date'],
            $input['end_date'],
            json_encode($input['items'] ?? []),
            json_encode($input['responsables'] ?? [])
        ]);

        $new_id = $pdo->lastInsertId();
        $stmt = $pdo->prepare("SELECT * FROM media_plans WHERE id = ?");
        $stmt->execute([$new_id]);
        $new_plan = decode_plan_json_fields($stmt->fetch());

        http_response_code(201);
        echo json_encode($new_plan);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Error al crear plan de medios: ' . $e->getMessage()]);
    }

} elseif ($method === 'PUT' && $plan_id !== null) {
    try {
        $input = json_decode(file_get_contents('php://input'), true);

        $sql = "UPDATE media_plans SET name = ?, objective = ?, audience = ?, budget = ?, start_date = ?, end_date = ?, items = ?, responsables = ? WHERE id = ?";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            $input['name'],
            $input['objective'] ?? '',
            $input['audience'] ?? '',
            (float)($input['budget'] ?? 0),
            $input['start_date'],
            $input['end_date'],
            json_encode($input['items'] ?? []),
            json_encode($input['responsables'] ?? []),
            $plan_id
        ]);

        if ($stmt->rowCount()) {
            $stmt = $pdo->prepare("SELECT * FROM media_plans WHERE id = ?");
            $stmt->execute([$plan_id]);
            $updated_plan = decode_plan_json_fields($stmt->fetch());
            echo json_encode($updated_plan);
        } else {
            http_response_code(404);
            echo json_encode(['error' => 'Plan de medios no encontrado para actualizar']);
        }
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Error al actualizar plan de medios: ' . $e->getMessage()]);
    }
} elseif ($method === 'DELETE' && $plan_id !== null) {
    try {
        $stmt = $pdo->prepare("DELETE FROM media_plans WHERE id = ?");
        $stmt->execute([$plan_id]);

        if ($stmt->rowCount()) {
            http_response_code(204);
        } else {
            http_response_code(404);
            echo json_encode(['error' => 'Plan de medios no encontrado']);
        }
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Error al eliminar plan de medios: ' . $e->getMessage()]);
    }
} else {
    http_response_code(405);
    echo json_encode(['error' => 'Método no permitido o ID de plan no proporcionado']);
}
?>