<?php
header("Content-Type: application/json");
require_once __DIR__ . '/../config/db.php';

$method = $_SERVER['REQUEST_METHOD'];
$project_id = isset($_GET['project_id']) ? (int)$_GET['project_id'] : null;
$grid_id = isset($_GET['id']) ? (int)$_GET['id'] : null;
$pdo = get_db_connection();

/**
 * Decodifica los campos JSON de una grilla.
 * @param array $grid La grilla de la BD.
 * @return array La grilla con los campos JSON decodificados.
 */
function decode_grid_json_fields($grid) {
    if ($grid) {
        $json_fields = ['networks', 'responsables'];
        foreach ($json_fields as $field) {
            $grid[$field] = $grid[$field] ? json_decode($grid[$field], true) : [];
        }
    }
    return $grid;
}

if ($method === 'GET') {
    if ($grid_id !== null) {
        // Obtener una grilla específica por su ID
        $stmt = $pdo->prepare("SELECT * FROM content_grids WHERE id = ?");
        $stmt->execute([$grid_id]);
        $grid = $stmt->fetch();
        if ($grid) {
            echo json_encode(decode_grid_json_fields($grid));
        } else {
            http_response_code(404);
            echo json_encode(['error' => 'Grilla no encontrada']);
        }
    } elseif ($project_id !== null) {
        // Obtener todas las grillas de un proyecto
        $stmt = $pdo->prepare("SELECT * FROM content_grids WHERE project_id = ? ORDER BY start_date DESC");
        $stmt->execute([$project_id]);
        $grids = $stmt->fetchAll();
        $grids = array_map('decode_grid_json_fields', $grids);
        echo json_encode($grids);
    } else {
        // Si no se especifica un ID de proyecto o grilla, devolver todas las grillas.
        $stmt = $pdo->query("SELECT * FROM content_grids ORDER BY start_date DESC");
        $grids = $stmt->fetchAll();
        $grids = array_map('decode_grid_json_fields', $grids);
        echo json_encode($grids);
    }

} elseif ($method === 'POST') {
    $input = json_decode(file_get_contents('php://input'), true);
    if (empty($input['project_id']) || empty($input['name']) || empty($input['start_date']) || empty($input['end_date'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Nombre, fechas de inicio/fin y ID de proyecto son obligatorios.']);
        exit;
    }

    try {
        $sql = "INSERT INTO content_grids (project_id, name, start_date, end_date, networks, responsables) VALUES (?, ?, ?, ?, ?, ?)";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            (int)$input['project_id'],
            $input['name'],
            $input['start_date'],
            $input['end_date'],
            json_encode($input['networks'] ?? []),
            json_encode($input['responsables'] ?? [])
        ]);

        $new_id = $pdo->lastInsertId();
        $stmt = $pdo->prepare("SELECT * FROM content_grids WHERE id = ?");
        $stmt->execute([$new_id]);
        $new_grid = $stmt->fetch();

        http_response_code(201);
        echo json_encode(decode_grid_json_fields($new_grid));
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Error al crear grilla: ' . $e->getMessage()]);
    }

} elseif ($method === 'PUT' && $grid_id !== null) {
    try {
        $input = json_decode(file_get_contents('php://input'), true);

        $sql = "UPDATE content_grids SET name = ?, start_date = ?, end_date = ?, networks = ?, responsables = ? WHERE id = ?";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            $input['name'],
            $input['start_date'],
            $input['end_date'],
            json_encode($input['networks'] ?? []),
            json_encode($input['responsables'] ?? []),
            $grid_id
        ]);

        if ($stmt->rowCount()) {
            $stmt = $pdo->prepare("SELECT * FROM content_grids WHERE id = ?");
            $stmt->execute([$grid_id]);
            $updated_grid = decode_grid_json_fields($stmt->fetch());
            echo json_encode($updated_grid);
        } else {
            http_response_code(404);
            echo json_encode(['error' => 'Grilla no encontrada para actualizar']);
        }
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Error al actualizar grilla: ' . $e->getMessage()]);
    }
} elseif ($method === 'DELETE' && $grid_id !== null) {
    try {
        // La FK en `grid_posts` tiene `ON DELETE CASCADE`, por lo que los posteos
        // asociados a esta grilla se borrarán automáticamente.
        $stmt = $pdo->prepare("DELETE FROM content_grids WHERE id = ?");
        $stmt->execute([$grid_id]);
        
        if ($stmt->rowCount()) {
            http_response_code(204);
        } else {
            http_response_code(404);
            echo json_encode(['error' => 'Grilla no encontrada']);
        }
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Error al eliminar grilla: ' . $e->getMessage()]);
    }
} else {
    http_response_code(405);
    echo json_encode(['error' => 'Método no permitido o ID de grilla no proporcionado']);
}
?>