<?php
header("Content-Type: application/json");
require_once __DIR__ . '/../config/db.php';

$method = $_SERVER['REQUEST_METHOD'];
$grid_id = isset($_GET['grid_id']) ? (int)$_GET['grid_id'] : null;
$id = isset($_GET['id']) ? (int)$_GET['id'] : null;
$pdo = get_db_connection();

/**
 * Decodifica los campos JSON de un posteo.
 * @param array $post El posteo de la BD.
 * @return array El posteo con los campos JSON decodificados.
 */
function decode_post_json_fields($post) {
    if ($post) {
        $post['redes'] = $post['redes'] ? json_decode($post['redes'], true) : [];
        // El campo 'realizado' en MySQL/MariaDB puede ser 0 o 1, lo convertimos a booleano.
        $post['realizado'] = (bool)$post['realizado'];
    }
    return $post;
}

if ($method === 'GET') {
    if ($grid_id !== null) {
        $stmt = $pdo->prepare("SELECT * FROM grid_posts WHERE grid_id = ? ORDER BY fecha_posteo, hora_posteo ASC");
        $stmt->execute([$grid_id]);
        $grid_items = $stmt->fetchAll();
        $grid_items = array_map('decode_post_json_fields', $grid_items);
        echo json_encode($grid_items);
    } else {
        // Devolver todos los posts si no hay grid_id (útil para la agenda general)
        $stmt = $pdo->query("SELECT * FROM grid_posts ORDER BY fecha_posteo, hora_posteo ASC");
        $posts = $stmt->fetchAll();
        $posts = array_map('decode_post_json_fields', $posts);
        echo json_encode($posts);
    }

} elseif ($method === 'POST') {
    $input = json_decode(file_get_contents('php://input'), true);
    if (empty($input['grid_id'])) {
        http_response_code(400);
        echo json_encode(['error' => 'El ID de la grilla es obligatorio']);
        exit;
    }

    try {
        $sql = "INSERT INTO grid_posts (grid_id, fecha_posteo, hora_posteo, objetivo, redes, tipo_contenido, hook, copy, hashtags, cta, publico_objetivo, resultado_esperado, realizado) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            (int)$input['grid_id'],
            empty($input['fecha_posteo']) ? null : $input['fecha_posteo'],
            empty($input['hora_posteo']) ? null : $input['hora_posteo'],
            $input['objetivo'] ?? '',
            json_encode($input['redes'] ?? []),
            $input['tipo_contenido'] ?? '',
            $input['hook'] ?? '',
            $input['copy'] ?? '',
            $input['hashtags'] ?? '',
            $input['cta'] ?? '',
            $input['publico_objetivo'] ?? '',
            $input['resultado_esperado'] ?? '',
            false // `realizado` por defecto es false
        ]);

        $new_id = $pdo->lastInsertId();
        $stmt = $pdo->prepare("SELECT * FROM grid_posts WHERE id = ?");
        $stmt->execute([$new_id]);
        $new_item = decode_post_json_fields($stmt->fetch());

        http_response_code(201);
        echo json_encode($new_item);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Error al crear posteo: ' . $e->getMessage()]);
    }

} elseif ($method === 'PUT' && $id !== null) {
    $input = json_decode(file_get_contents('php://input'), true);

    // Obtener el posteo actual para comparar
    $stmt = $pdo->prepare("SELECT * FROM grid_posts WHERE id = ?");
    $stmt->execute([$id]);
    $current_post = $stmt->fetch();

    if (!$current_post) {
        http_response_code(404);
        echo json_encode(['error' => 'Posteo no encontrado']);
        exit;
    }

    try {
        // Lógica especial para el check de "realizado"
        $fecha_realizado = $current_post['fecha_realizado'];
        if (isset($input['realizado']) && $input['realizado'] === true && !$current_post['realizado']) {
            $fecha_realizado = date('Y-m-d H:i:s');
        } elseif (isset($input['realizado']) && $input['realizado'] === false) {
            $fecha_realizado = null;
        }

        $sql = "UPDATE grid_posts SET 
                    fecha_posteo = ?, hora_posteo = ?, objetivo = ?, redes = ?, tipo_contenido = ?, 
                    hook = ?, copy = ?, hashtags = ?, cta = ?, publico_objetivo = ?, 
                    resultado_esperado = ?, realizado = ?, fecha_realizado = ?
                WHERE id = ?";
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            $input['fecha_posteo'] ?? $current_post['fecha_posteo'],
            $input['hora_posteo'] ?? $current_post['hora_posteo'],
            $input['objetivo'] ?? $current_post['objetivo'],
            isset($input['redes']) ? json_encode($input['redes']) : $current_post['redes'],
            $input['tipo_contenido'] ?? $current_post['tipo_contenido'],
            $input['hook'] ?? $current_post['hook'],
            $input['copy'] ?? $current_post['copy'],
            $input['hashtags'] ?? $current_post['hashtags'],
            $input['cta'] ?? $current_post['cta'],
            $input['publico_objetivo'] ?? $current_post['publico_objetivo'],
            $input['resultado_esperado'] ?? $current_post['resultado_esperado'],
            isset($input['realizado']) ? (bool)$input['realizado'] : (bool)$current_post['realizado'],
            $fecha_realizado,
            $id
        ]);

        if ($stmt->rowCount()) {
            $stmt = $pdo->prepare("SELECT * FROM grid_posts WHERE id = ?");
            $stmt->execute([$id]);
            $updated_item = decode_post_json_fields($stmt->fetch());
            echo json_encode($updated_item);
        } else {
            // Si no se afectaron filas, puede ser porque los datos eran idénticos.
            // Devolvemos el post actual para mantener consistencia en el frontend.
            echo json_encode(decode_post_json_fields($current_post));
        }
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Error al actualizar posteo: ' . $e->getMessage()]);
    }

} elseif ($method === 'DELETE' && $id !== null) {
    try {
        $stmt = $pdo->prepare("DELETE FROM grid_posts WHERE id = ?");
        $stmt->execute([$id]);

        if ($stmt->rowCount()) {
            http_response_code(204);
        } else {
            http_response_code(404);
            echo json_encode(['error' => 'Posteo no encontrado']);
        }
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Error al eliminar posteo: ' . $e->getMessage()]);
    }

} else {
    http_response_code(405);
    echo json_encode(['error' => 'Método no permitido o ID no proporcionado']);
}
?>