<?php
session_start();
header('Content-Type: application/json');
require_once __DIR__ . '/../config/db.php';


/**
 * Maneja la subida de un archivo de imagen para el chat.
 * @param array $file - El array del archivo de $_FILES.
 * @return string|null - La URL del archivo guardado o null si hay error.
 */
function handleChatPhotoUpload($file) {
    if (isset($file) && $file['error'] === UPLOAD_ERR_OK) {
        $upload_dir = __DIR__ . '/../uploads/chat_photos/';
        if (!is_dir($upload_dir)) {
            mkdir($upload_dir, 0777, true);
        }
        $file_extension = pathinfo($file['name'], PATHINFO_EXTENSION);
        $file_name = 'group_' . uniqid() . '.' . $file_extension;
        $target_path = $upload_dir . $file_name;

        if (move_uploaded_file($file['tmp_name'], $target_path)) {
            // Devuelve la URL relativa accesible desde el cliente
            return 'uploads/chat_photos/' . $file_name;
        }
    }
    return null;
}

// --- Lógica Principal ---

$action = $_GET['action'] ?? '';

if (!isset($_SESSION['user_id'])) {
    http_response_code(401);
    echo json_encode(['error' => 'No autenticado. Por favor, inicie sesión.']);
    exit();
}
$currentUser = $_SESSION['user_id'];
$pdo = get_db_connection();

switch ($action) {
    case 'get_session_user':
        echo json_encode(['user_id' => $currentUser]);
        break;

    case 'get_professionals':
        $stmt = $pdo->query("SELECT id, nombre_completo as name, foto_url as photo FROM professionals");
        $users = $stmt->fetchAll();
        echo json_encode($users);
        break;

    case 'get_conversations':
        // Esta consulta es compleja, une varias tablas para obtener toda la información necesaria de una vez.
        $sql = "
            SELECT 
                c.id,
                c.is_group_chat,
                IF(c.is_group_chat, c.name, p.nombre_completo) AS name,
                IF(c.is_group_chat, c.group_photo_url, p.foto_url) AS photo,
                (SELECT message_content FROM chat_messages WHERE conversation_id = c.id ORDER BY timestamp DESC LIMIT 1) as last_message,
                (SELECT timestamp FROM chat_messages WHERE conversation_id = c.id ORDER BY timestamp DESC LIMIT 1) as last_message_time
            FROM chat_conversations c
            JOIN chat_conversation_members m ON c.id = m.conversation_id
            -- Para chats individuales, encontrar al otro usuario
            LEFT JOIN chat_conversation_members m2 ON c.id = m2.conversation_id AND m2.user_id != ?
            LEFT JOIN professionals p ON m2.user_id = p.id AND c.is_group_chat = 0
            WHERE m.user_id = ?
            GROUP BY c.id
            ORDER BY last_message_time DESC
        ";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([$currentUser, $currentUser]);
        $conversations = $stmt->fetchAll();

        // Rellenar mensajes vacíos
        foreach ($conversations as &$conv) {
            if (empty($conv['last_message'])) {
                $conv['last_message'] = 'No hay mensajes aún.';
            }
        }

        echo json_encode($conversations);
        break;

    case 'check_new_messages':
        $last_check_timestamp = $_GET['since'] ?? '1970-01-01 00:00:00';
        $sql = "
            SELECT 1 FROM chat_messages cm
            JOIN chat_conversation_members ccm ON cm.conversation_id = ccm.conversation_id
            WHERE ccm.user_id = ? 
              AND cm.sender_id != ?
              AND cm.timestamp > ?
            LIMIT 1
        ";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([$currentUser, $currentUser, $last_check_timestamp]);
        $has_new = $stmt->fetchColumn();

        echo json_encode(['has_new_messages' => (bool)$has_new]);
        break;

    case 'send_message':
        $input = json_decode(file_get_contents('php://input'), true);
        $conversationId = $input['conversation_id'] ?? 0;
        $messageContent = trim($input['message_content'] ?? '');

        if (empty($conversationId) || empty($messageContent)) {
            http_response_code(400);
            echo json_encode(['error' => 'Datos incompletos.']);
            break;
        }

        // Validar que el usuario pertenezca a la conversación
        $stmt = $pdo->prepare("SELECT 1 FROM chat_conversation_members WHERE conversation_id = ? AND user_id = ?");
        $stmt->execute([$conversationId, $currentUser]);
        $is_member = $stmt->fetchColumn();

        if (!$is_member) {
            http_response_code(403);
            echo json_encode(['error' => 'No tienes permiso para enviar mensajes a esta conversación.']);
            break;
        }

        // Crear nuevo mensaje
        $stmt = $pdo->prepare("INSERT INTO chat_messages (conversation_id, sender_id, message_content) VALUES (?, ?, ?)");
        $stmt->execute([$conversationId, $currentUser, $messageContent]);
        $new_message_id = $pdo->lastInsertId();

        $stmt = $pdo->prepare("SELECT * FROM chat_messages WHERE id = ?");
        $stmt->execute([$new_message_id]);
        $newMessage = $stmt->fetch();

        echo json_encode($newMessage);
        break;

    case 'get_messages':
        $conversationId = (int)($_GET['conversation_id'] ?? 0);
        if ($conversationId === 0) {
            http_response_code(400);
            echo json_encode(['error' => 'ID de conversación no válido.']);
            break;
        }

        // Validar que el usuario pertenezca a la conversación
        $stmt = $pdo->prepare("SELECT 1 FROM chat_conversation_members WHERE conversation_id = ? AND user_id = ?");
        $stmt->execute([$conversationId, $currentUser]);
        $is_member = $stmt->fetchColumn();

        if (!$is_member) {
            http_response_code(403);
            echo json_encode(['error' => 'No tienes acceso a esta conversación.']);
            break;
        }

        // Obtener mensajes y detalles de la conversación
        $stmt = $pdo->prepare("SELECT * FROM chat_messages WHERE conversation_id = ? ORDER BY timestamp ASC");
        $stmt->execute([$conversationId]);
        $messages = $stmt->fetchAll();

        $sql = "
            SELECT 
                c.is_group_chat,
                IF(c.is_group_chat, c.name, p.nombre_completo) AS name,
                IF(c.is_group_chat, c.group_photo_url, p.foto_url) AS photo
            FROM chat_conversations c
            LEFT JOIN chat_conversation_members m ON c.id = m.conversation_id AND m.user_id != ?
            LEFT JOIN professionals p ON m.user_id = p.id
            WHERE c.id = ?
            LIMIT 1
        ";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([$currentUser, $conversationId]);
        $conv_details = $stmt->fetch();

        echo json_encode([
            'conversation' => $conv_details,
            'messages' => $messages
        ]);
        break;

    case 'get_or_create_individual_conversation':
        $otherUserId = (int)($_GET['user_id'] ?? 0);
        if (empty($otherUserId) || $otherUserId == $currentUser) {
            http_response_code(400);
            echo json_encode(['error' => 'ID de usuario no válido.']);
            break;
        }

        // Buscar si ya existe una conversación individual
        $sql = "
            SELECT c.id FROM chat_conversations c
            JOIN chat_conversation_members m1 ON c.id = m1.conversation_id
            JOIN chat_conversation_members m2 ON c.id = m2.conversation_id
            WHERE c.is_group_chat = 0
              AND m1.user_id = ?
              AND m2.user_id = ?
            GROUP BY c.id
            HAVING COUNT(c.id) = 1
        ";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([$currentUser, $otherUserId]);
        $found_conversation_id = $stmt->fetchColumn();

        if ($found_conversation_id) {
            echo json_encode(['conversation_id' => $found_conversation_id]);
        } else {
            // Crear nueva conversación
            $pdo->beginTransaction();
            $stmt = $pdo->prepare("INSERT INTO chat_conversations (is_group_chat, created_by) VALUES (0, ?)");
            $stmt->execute([$currentUser]);
            $new_conv_id = $pdo->lastInsertId();

            // Añadir miembros
            $stmt = $pdo->prepare("INSERT INTO chat_conversation_members (conversation_id, user_id) VALUES (?, ?), (?, ?)");
            $stmt->execute([$new_conv_id, $currentUser, $new_conv_id, $otherUserId]);
            $pdo->commit();

            echo json_encode(['conversation_id' => $new_conv_id]);
        }
        break;

    case 'create_group':
        // Ahora leemos desde $_POST y $_FILES porque usamos FormData
        $groupName = trim($_POST['name'] ?? '');
        // Los IDs de miembros vienen como un string JSON, hay que decodificarlo
        $memberIds = json_decode($_POST['members'] ?? '[]', true);
        $groupPhotoFile = $_FILES['group_photo'] ?? null;

        if (empty($groupName) || empty($memberIds)) {
            http_response_code(400);
            echo json_encode(['error' => 'El nombre del grupo y los miembros son requeridos.']);
            break;
        }

        // Manejar la subida de la foto
        $photo_url = handleChatPhotoUpload($groupPhotoFile);

        $pdo->beginTransaction();

        // Crear la nueva conversación de grupo
        $stmt = $pdo->prepare("INSERT INTO chat_conversations (name, group_photo_url, is_group_chat, created_by) VALUES (?, ?, 1, ?)");
        $stmt->execute([$groupName, $photo_url, $currentUser]);
        $new_conv_id = $pdo->lastInsertId();

        // Añadir al creador y a los miembros a la conversación
        $all_member_ids = array_unique(array_merge([$currentUser], $memberIds));

        $placeholders = implode(',', array_fill(0, count($all_member_ids), '(?, ?)'));
        $sql = "INSERT INTO chat_conversation_members (conversation_id, user_id) VALUES " . $placeholders;
        $params = [];
        foreach ($all_member_ids as $user_id) {
            array_push($params, $new_conv_id, (int)$user_id);
        }
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        $pdo->commit();
        echo json_encode(['conversation_id' => $new_conv_id]);
        break;

    case 'edit_group':
        $groupId = (int)($_GET['id'] ?? 0);
        if ($groupId === 0) {
            http_response_code(400);
            echo json_encode(['error' => 'ID de grupo no válido.']);
            break;
        }

        // Validar que el usuario sea el creador del grupo para poder editarlo
        $stmt = $pdo->prepare("SELECT created_by, group_photo_url FROM chat_conversations WHERE id = ?");
        $stmt->execute([$groupId]);
        $group = $stmt->fetch();

        if (!$group || $group['created_by'] != $currentUser) {
            http_response_code(403);
            echo json_encode(['error' => 'No tienes permiso para editar este grupo.']);
            break;
        }

        $groupName = trim($_POST['name'] ?? '');
        $memberIds = json_decode($_POST['members'] ?? '[]', true);
        $groupPhotoFile = $_FILES['group_photo'] ?? null;

        if (empty($groupName)) {
            http_response_code(400);
            echo json_encode(['error' => 'El nombre del grupo es requerido.']);
            break;
        }

        $pdo->beginTransaction();

        // Actualizar nombre
        $stmt = $pdo->prepare("UPDATE chat_conversations SET name = ? WHERE id = ?");
        $stmt->execute([$groupName, $groupId]);

        // Actualizar foto si se subió una nueva
        if ($groupPhotoFile) {
            $photo_url = handleChatPhotoUpload($groupPhotoFile);
            if ($photo_url) {
                // Opcional: borrar foto antigua
                $old_photo = $group['group_photo_url'];
                if ($old_photo && file_exists(__DIR__ . '/../' . $old_photo)) {
                    unlink(__DIR__ . '/../' . $old_photo);
                }
                $stmt = $pdo->prepare("UPDATE chat_conversations SET group_photo_url = ? WHERE id = ?");
                $stmt->execute([$photo_url, $groupId]);
            }
        }

        // Actualizar miembros: borrar los existentes y añadir los nuevos
        $stmt = $pdo->prepare("DELETE FROM chat_conversation_members WHERE conversation_id = ?");
        $stmt->execute([$groupId]);

        $all_member_ids = array_unique(array_merge([$currentUser], $memberIds));
        $placeholders = implode(',', array_fill(0, count($all_member_ids), '(?, ?)'));
        $sql = "INSERT INTO chat_conversation_members (conversation_id, user_id) VALUES " . $placeholders;
        $params = [];
        foreach ($all_member_ids as $user_id) {
            array_push($params, $groupId, (int)$user_id);
        }
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);

        $pdo->commit();
        echo json_encode(['conversation_id' => $groupId]);
        break;

    case 'delete_group':
        $groupId = (int)($_GET['id'] ?? 0);
        $stmt = $pdo->prepare("SELECT created_by FROM chat_conversations WHERE id = ?");
        $stmt->execute([$groupId]);
        $group_creator = $stmt->fetchColumn();

        if (!$group_creator || $group_creator != $currentUser) {
            http_response_code(403);
            echo json_encode(['error' => 'No tienes permiso para eliminar este grupo.']);
            break;
        }

        // La eliminación en cascada (ON DELETE CASCADE) se encargará de miembros y mensajes.
        $stmt = $pdo->prepare("DELETE FROM chat_conversations WHERE id = ?");
        $stmt->execute([$groupId]);
        echo json_encode(['success' => true]);
        break;

    case 'edit_message':
        $input = json_decode(file_get_contents('php://input'), true);
        $messageId = $input['message_id'] ?? 0;
        $content = trim($input['content'] ?? '');

        if (empty($messageId) || empty($content)) {
            http_response_code(400);
            echo json_encode(['success' => false, 'error' => 'Datos incompletos.']);
            break;
        }

        $stmt = $pdo->prepare("UPDATE chat_messages SET message_content = ? WHERE id = ? AND sender_id = ?");
        $stmt->execute([$content, $messageId, $currentUser]);

        if ($stmt->rowCount() > 0) {
            echo json_encode(['success' => true]);
        } else {
            echo json_encode(['success' => false, 'error' => 'No se pudo editar el mensaje o no tienes permiso.']);
        }
        break;

    case 'delete_message':
        $messageId = (int)($_GET['id'] ?? 0);
        $stmt = $pdo->prepare("DELETE FROM chat_messages WHERE id = ? AND sender_id = ?");
        $stmt->execute([$messageId, $currentUser]);
        if ($stmt->rowCount() > 0) {
            echo json_encode(['success' => true]);
        } else { echo json_encode(['success' => false, 'error' => 'No se pudo eliminar o no tienes permiso.']); }
        break;

    default:
        http_response_code(400);
        echo json_encode(['error' => 'Acción no válida o no especificada.']);
        break;
}
?>