<?php
header("Content-Type: application/json");

ini_set('display_errors', 1); // Mostrar errores para depuración
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);
require_once __DIR__ . '/../config/db.php';
require_once __DIR__ . '/../config/security.php';

// --- Función para obtener la configuración de la IA ---
function get_ai_settings($pdo) {
    $stmt = $pdo->prepare("SELECT setting_value FROM settings WHERE setting_key = 'ai_settings'");
    $stmt->execute();
    $result = $stmt->fetchColumn();
    if ($result) {
        return json_decode($result, true);
    }
    return null;
}

$method = $_SERVER['REQUEST_METHOD'];

if ($method !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'Método no permitido']);
    exit;
}

// La data puede venir como JSON o como FormData (para archivos)
$task_type = $_POST['task_type'] ?? json_decode(file_get_contents('php://input'), true)['task_type'] ?? null;
$context_json = $_POST['context'] ?? json_decode(file_get_contents('php://input'), true)['context'] ?? null;
$context = is_string($context_json) ? json_decode($context_json, true) : ($context_json ?? []);

if (!$task_type) {
    http_response_code(400);
    echo json_encode(['error' => 'El tipo de tarea de IA es requerido.']);
    exit;
}

// Obtener configuración de la IA desde la base de datos
$pdo = get_db_connection();
$ai_settings = get_ai_settings($pdo);

if (!$ai_settings || empty($ai_settings['api_key']) || empty($ai_settings['ai_model'])) {
    http_response_code(409); // Conflict
    echo json_encode(['error' => 'La configuración de la IA no está completa. Por favor, configure el proveedor, la API Key y el modelo en la página de Configuración.']);
    exit;
}

// --- CONSTRUCCIÓN DE PROMPTS ---
$request_parts = [];
$prompt = '';
switch ($task_type) {
    case 'generate_strategy':
        $prompt = "Actúa como un Director de Estrategia de Marketing. Para la empresa '{$context['clientName']}', y su proyecto '{$context['projectName']}', " .
                  "cuyos objetivos son: '{$context['projectDescription']}', y con un enfoque en marketing '{$context['marketingType']}', ".
                  "desarrolla una propuesta de marketing. Responde únicamente con un JSON que contenga tres claves: 'estrategico', 'tactico' y 'operativo'.";
        break;
 
    case 'generate_grid_posts':
        $prompt = "Actúa como un experto Community Manager. Basado en la siguiente estrategia de marketing:\n" .
                  "- Estratégico: {$context['estrategico']}\n- Táctico: {$context['tactico']}\n- Operativo: {$context['operativo']}\n" .
                  "Genera una lista de 3 a 5 ideas de posteos para redes sociales para los próximos 10 días. " .
                  "Responde únicamente con un array JSON de objetos. Cada objeto debe tener las siguientes claves: " .
                  "'fecha_posteo' (YYYY-MM-DD), 'hora_posteo' (HH:MM), 'tipo_contenido', 'redes' (array), 'objetivo', 'hook', 'copy', 'hashtags', 'cta', 'publico_objetivo' y 'resultado_esperado'.";
        break;
 
    case 'generate_creative_asset':
        $prompt = "Actúa como un Director Creativo. Basado en la siguiente información de un posteo:\n" .
                  "- Tipo: {$context['tipo_contenido']}\n- Objetivo: {$context['objetivo']}\n- Copy: {$context['copy']}\n" .
                  "Genera una descripción detallada para un activo creativo visual (prompt para DALL-E, Midjourney, etc.) o un guion corto para un video. " .
                  "Responde únicamente con un JSON que contenga una clave: 'creative_asset'.";
        break;
 
    case 'analyze_data':
        $system_prompt = "Eres un analista de datos experto. Tu tarea es analizar los datos y el prompt proporcionado por el usuario para generar un reporte. " .
                         "El reporte final debe ser un código HTML interactivo y visualmente atractivo, utilizando exclusivamente Bootstrap 5. " .
                         "NO incluyas las etiquetas `<html>`, `<head>`, o `<body>`. El resultado debe ser solo el contenido que se pueda insertar directamente dentro de un `div`. " .
                         "Si el usuario pide gráficos, usa Chart.js e incluye el tag `<script>` necesario con el código de inicialización.";
 
        $user_prompt = "Solicitud del usuario: \n" . ($context['prompt'] ?? 'Analiza los datos adjuntos y genera un resumen.');
 
        $file_contents_prompt = "\n\nDatos de los archivos adjuntos:\n";
        $has_files = !empty($_FILES['data_files']['name'][0]);
 
        if ($has_files) {
            for ($i = 0; $i < count($_FILES['data_files']['name']); $i++) {
                $request_parts[] = [
                    'inline_data' => ['mimeType' => $_FILES['data_files']['type'][$i], 'data' => base64_encode(file_get_contents($_FILES['data_files']['tmp_name'][$i]))]
                ];
                $file_contents_prompt .= "- Archivo '{$_FILES['data_files']['name'][$i]}'\n";
            }
        }
        $prompt = $system_prompt . $user_prompt . ($has_files ? $file_contents_prompt : "");
        break;
 
    default:
        http_response_code(400);
        echo json_encode(['error' => 'Tipo de tarea de IA no válido.']);
        exit;
}
 
// --- LLAMADA A LA API DE IA ---
$request_parts[] = ['text' => $prompt];
$api_provider = $ai_settings['ai_provider'] ?? 'gemini';
$response_data = null;
 
if ($api_provider === 'gemini') {
    $api_key = decrypt_token($ai_settings['api_key']);
    $model = $ai_settings['ai_model'];
    $url = "https://generativelanguage.googleapis.com/v1beta/models/{$model}:generateContent?key={$api_key}";
 
    $data = ['contents' => [['parts' => $request_parts]]];
 
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
    curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false); // Considerar remover en producción
 
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curl_error = curl_error($ch);
    curl_close($ch);
 
    if ($curl_error) {
        http_response_code(500);
        echo json_encode(['error' => 'Error de cURL: ' . $curl_error]);
        exit;
    }
 
    if ($http_code !== 200) {
        http_response_code($http_code);
        $error_response = json_decode($response, true);
        $error_message = $error_response['error']['message'] ?? 'Error desconocido al contactar la API de IA.';
        echo json_encode(['error' => "Error de la API de IA: " . $error_message]);
        exit;
    }
 
    $result = json_decode($response, true);
    $text_content = $result['candidates'][0]['content']['parts'][0]['text'] ?? null;
 
    if ($text_content === null) {
        http_response_code(500);
        echo json_encode(['error' => 'La IA devolvió una respuesta vacía o en un formato inesperado.']);
        exit;
    }
 
    // Para el análisis de datos, la respuesta es HTML directo. Para otros, es JSON.
    if ($task_type === 'analyze_data') {
        $response_data = ['report_html' => $text_content];
    } else {
        // Intentamos decodificar el JSON que la IA debería haber devuelto.
        $response_data = json_decode(trim($text_content), true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            // Si falla, intentamos extraer el JSON del texto por si la IA añadió texto extra.
            preg_match('/\{.*\}|\[.*\]/s', $text_content, $matches);
            if (isset($matches[0])) {
                $response_data = json_decode($matches[0], true);
            }
        }
    }
}
// else if ($api_provider === 'openai') {
//     // Aquí iría la lógica para llamar a la API de OpenAI en el futuro
// }
 
if ($response_data === null) {
    http_response_code(500);
    echo json_encode(['error' => 'La IA devolvió una respuesta, pero no se pudo interpretar como un JSON o HTML válido.']);
    exit;
}
 
echo json_encode($response_data);
?>