document.addEventListener('DOMContentLoaded', function() {

    // --- ELEMENTOS DEL DOM ---
    const calendarEl = document.getElementById('projects-calendar');

    // --- URLs DE LA API ---
    const PROJECTS_API_URL = 'api/projects.php';
    const CLIENTS_API_URL = 'api/clients.php';

    // --- INICIALIZACIÓN ---
    async function initialLoad() {
        try {
            const [projectsRes, clientsRes] = await Promise.all([
                fetch(PROJECTS_API_URL),
                fetch(CLIENTS_API_URL)
            ]);

            if (!projectsRes.ok) throw new Error('No se pudieron cargar los proyectos.');
            if (!clientsRes.ok) throw new Error('No se pudieron cargar los clientes.');
            
            const projects = await projectsRes.json();
            const clients = await clientsRes.json();
            renderCalendar(projects, clients);

        } catch (error) {
            console.error("Error en la carga de la agenda de proyectos:", error);
            if (calendarEl) {
                calendarEl.innerHTML = `<div class="alert alert-danger">Error al cargar el calendario: ${error.message}</div>`;
            }
        }
    }

    function renderCalendar(projects, clients) {
        if (!calendarEl) return;

        const clientMap = new Map(clients.map(c => [c.id, c.nombre_empresa]));

        const projectEvents = projects.filter(p => p.fecha_inicio && p.fecha_fin_estimada).map(project => {
            // FullCalendar's end date is exclusive, so add one day to make it inclusive.
            const endDate = new Date(project.fecha_fin_estimada);
            endDate.setDate(endDate.getDate() + 1);

            return {
                title: project.nombre_proyecto,
                start: project.fecha_inicio,
                end: endDate.toISOString().split('T')[0],
                url: `project-detail.html?id=${project.id}`,
                backgroundColor: '#6f42c1', // Púrpura para proyectos
                borderColor: '#6f42c1',
                extendedProps: {
                    client: clientMap.get(project.cliente_id) || 'N/A',
                    status: project.estado
                }
            };
        });

        const calendar = new FullCalendar.Calendar(calendarEl, {
            initialView: 'dayGridMonth',
            locale: 'es',
            headerToolbar: {
                left: 'prev,next today',
                center: 'title',
                right: 'dayGridMonth,timeGridWeek,listWeek'
            },
            buttonText: { today: 'Hoy', month: 'Mes', week: 'Semana', list: 'Lista' },
            events: projectEvents,
            eventClick: function(info) {
                info.jsEvent.preventDefault();
                if (info.event.url) window.location.href = info.event.url;
            },
            eventDidMount: function(info) {
                new bootstrap.Popover(info.el, {
                    title: info.event.title,
                    content: `Cliente: ${info.event.extendedProps.client}<br>Estado: ${info.event.extendedProps.status}`,
                    trigger: 'hover', placement: 'top', container: 'body', html: true
                });
            }
        });
        calendar.render();
    }

    initialLoad();
});