document.addEventListener('DOMContentLoaded', function() {

    // --- ELEMENTOS DEL DOM ---
    const calendarEl = document.getElementById('media-plan-calendar');
    const projectFilterEl = document.getElementById('project-filter');

    // --- URLs DE LA API ---
    const MEDIA_PLANS_API_URL = 'api/media_plans.php';
    const PROJECTS_API_URL = 'api/projects.php';

    // --- ESTADO ---
    let appState = {
        mediaPlans: [],
        projects: [],
        calendar: null
    };

    // --- INICIALIZACIÓN ---
    async function initialLoad() {
        try {
            const [plansRes, projectsRes] = await Promise.all([
                fetch(MEDIA_PLANS_API_URL),
                fetch(PROJECTS_API_URL)
            ]);

            if (!plansRes.ok) throw new Error('No se pudieron cargar los planes de medios.');
            if (!projectsRes.ok) throw new Error('No se pudieron cargar los proyectos.');

            appState.mediaPlans = await plansRes.json();
            appState.projects = await projectsRes.json();

            populateFilters();
            initializeCalendar();
            
        } catch (error) {
            console.error("Error en la carga de la agenda de plan de medios:", error);
            if (calendarEl) {
                calendarEl.innerHTML = `<div class="alert alert-danger">Error al cargar el calendario: ${error.message}</div>`;
            }
        }
    }

    function initializeCalendar() {
        if (!calendarEl) return;

        appState.calendar = new FullCalendar.Calendar(calendarEl, {
            initialView: 'dayGridMonth',
            locale: 'es',
            headerToolbar: {
                left: 'prev,next today',
                center: 'title',
                right: 'dayGridMonth,timeGridWeek,listWeek'
            },
            buttonText: { today: 'Hoy', month: 'Mes', week: 'Semana', list: 'Lista' },
            events: getFilteredEvents(),
            eventClick: function(info) {
                info.jsEvent.preventDefault();
                if (info.event.url) window.location.href = info.event.url;
            },
            eventDidMount: function(info) {
                new bootstrap.Popover(info.el, {
                    title: info.event.title,
                    content: info.event.extendedProps.description,
                    trigger: 'hover', placement: 'top', container: 'body', html: true
                });
            }
        });
        appState.calendar.render();
    }

    // --- FILTROS Y EVENTOS ---

    function populateFilters() {
        projectFilterEl.innerHTML = '<option value="all">Todos los Proyectos</option>';
        appState.projects.forEach(p => {
            projectFilterEl.innerHTML += `<option value="${p.id}">${p.nombre_proyecto}</option>`;
        });
    }

    function getFilteredEvents() {
        const selectedProject = projectFilterEl.value;
        const projectMap = new Map(appState.projects.map(p => [p.id, p.nombre_proyecto]));

        const filteredPlans = appState.mediaPlans.filter(plan => {
            return selectedProject === 'all' || plan.project_id == selectedProject;
        });

        let events = [];
        filteredPlans.forEach(plan => {
            // Evento principal para el plan
            if (plan.start_date && plan.end_date) {
                const endDate = new Date(plan.end_date);
                endDate.setDate(endDate.getDate() + 1); // Hacer inclusivo
                events.push({
                    title: `Plan: ${plan.name}`,
                    start: plan.start_date,
                    end: endDate.toISOString().split('T')[0],
                    url: `media-plan-editor.html?plan_id=${plan.id}`,
                    backgroundColor: '#20c997', // Teal
                    borderColor: '#20c997',
                    extendedProps: { description: `Proyecto: ${projectMap.get(plan.project_id)}` }
                });
            }

            // Eventos para cada línea de pauta (item)
            (plan.items || []).forEach(item => {
                if (item.startDate && item.endDate) {
                    const itemEndDate = new Date(item.endDate);
                    itemEndDate.setDate(itemEndDate.getDate() + 1); // Hacer inclusivo
                    events.push({
                        title: `Pauta: ${item.media}`,
                        start: item.startDate,
                        end: itemEndDate.toISOString().split('T')[0],
                        url: `media-plan-editor.html?plan_id=${plan.id}`,
                        backgroundColor: '#6610f2', // Indigo
                        borderColor: '#6610f2',
                        extendedProps: { description: `Formato: ${item.format}` }
                    });
                }
            });
        });
        return events;
    }

    function handleFilterChange() {
        if (!appState.calendar) return;
        appState.calendar.removeAllEvents();
        appState.calendar.addEventSource(getFilteredEvents());
    }

    // --- LISTENERS ---
    projectFilterEl.addEventListener('change', handleFilterChange);

    // --- INICIO ---
    initialLoad();
});