document.addEventListener('DOMContentLoaded', function() {

    // --- ELEMENTOS DEL DOM ---
    const calendarEl = document.getElementById('editorial-calendar');
    const projectFilterEl = document.getElementById('project-filter');
    const networkFilterEl = document.getElementById('network-filter');

    // --- URLs DE LA API ---
    const GRID_POSTS_API_URL = 'api/grid_posts.php';
    const GRIDS_API_URL = 'api/grids.php';
    const PROJECTS_API_URL = 'api/projects.php';

    // --- ESTADO ---
    let appState = {
        posts: [],
        grids: [],
        projects: [],
        calendar: null
    };

    // --- INICIALIZACIÓN ---
    async function initialLoad() {
        try {
            const [postsRes, gridsRes, projectsRes] = await Promise.all([
                fetch(GRID_POSTS_API_URL),
                fetch(GRIDS_API_URL),
                fetch(PROJECTS_API_URL)
            ]);

            if (!postsRes.ok) throw new Error('No se pudieron cargar los posteos.');
            if (!gridsRes.ok) throw new Error('No se pudieron cargar las grillas.');
            if (!projectsRes.ok) throw new Error('No se pudieron cargar los proyectos.');

            appState.posts = await postsRes.json();
            appState.grids = await gridsRes.json();
            appState.projects = await projectsRes.json();

            populateFilters();
            initializeCalendar();
            
        } catch (error) {
            console.error("Error en la carga inicial de la agenda editorial:", error);
            if (calendarEl) {
                calendarEl.innerHTML = `<div class="alert alert-danger">No se pudo cargar la agenda. ${error.message}</div>`;
            }
        }
    }

    function initializeCalendar() {
        if (!calendarEl) return;

        appState.calendar = new FullCalendar.Calendar(calendarEl, {
            initialView: 'dayGridMonth',
            locale: 'es',
            headerToolbar: {
                left: 'prev,next today',
                center: 'title',
                right: 'dayGridMonth,timeGridWeek,listWeek'
            },
            buttonText: { today: 'Hoy', month: 'Mes', week: 'Semana', list: 'Lista' },
            events: getFilteredEvents(),
            eventClick: function(info) {
                info.jsEvent.preventDefault();
                if (info.event.url) window.location.href = info.event.url;
            },
            eventDidMount: function(info) {
                new bootstrap.Popover(info.el, {
                    title: info.event.title,
                    content: info.event.extendedProps.copy || 'Sin copy definido.',
                    trigger: 'hover',
                    placement: 'top',
                    container: 'body',
                    html: true
                });
            }
        });
        appState.calendar.render();
    }

    // --- FILTROS Y EVENTOS ---

    function populateFilters() {
        // Proyectos
        projectFilterEl.innerHTML = '<option value="all">Todos los Proyectos</option>';
        appState.projects.forEach(p => {
            projectFilterEl.innerHTML += `<option value="${p.id}">${p.nombre_proyecto}</option>`;
        });

        // Redes Sociales
        const allNetworks = new Set(appState.grids.flatMap(g => g.networks || []));
        networkFilterEl.innerHTML = '<option value="all">Todas las Redes</option>';
        allNetworks.forEach(net => {
            networkFilterEl.innerHTML += `<option value="${net}">${net}</option>`;
        });
    }

    function getFilteredEvents() {
        const selectedProject = projectFilterEl.value;
        const selectedNetwork = networkFilterEl.value;

        // Crear un mapa de grid_id a project_id para una búsqueda rápida
        const gridProjectMap = new Map(appState.grids.map(g => [g.id, g.project_id]));

        const filteredPosts = appState.posts.filter(post => {
            const projectId = gridProjectMap.get(post.grid_id);
            const projectMatch = selectedProject === 'all' || projectId == selectedProject;
            const networkMatch = selectedNetwork === 'all' || (post.redes && post.redes.includes(selectedNetwork));
            return projectMatch && networkMatch;
        });

        // Mapear posteos a eventos del calendario
        return filteredPosts.filter(post => post.fecha_posteo).map(post => {
            const projectName = appState.projects.find(p => p.id == gridProjectMap.get(post.grid_id))?.nombre_proyecto || 'N/A';
            
            return {
                title: `${projectName}: ${post.tipo_contenido}`,
                start: `${post.fecha_posteo}T${post.hora_posteo || '12:00:00'}`,
                allDay: !post.hora_posteo,
                url: `grid-editor.html?grid_id=${post.grid_id}`,
                backgroundColor: '#fd7e14', // Naranja para posteos
                borderColor: '#fd7e14',
                extendedProps: { 
                    copy: post.copy 
                }
            };
        });
    }

    function handleFilterChange() {
        if (!appState.calendar) return;
        appState.calendar.removeAllEvents();
        appState.calendar.addEventSource(getFilteredEvents());
    }

    // --- LISTENERS ---
    projectFilterEl.addEventListener('change', handleFilterChange);
    networkFilterEl.addEventListener('change', handleFilterChange);

    // --- INICIO ---
    initialLoad();
});