<?php
/**
 * Funciones auxiliares del sistema LexiPro
 */

/**
 * Escapa HTML para prevenir XSS
 */
function e($string) {
    return htmlspecialchars($string ?? '', ENT_QUOTES, 'UTF-8');
}

/**
 * Formatea una fecha al formato dd/mm/yyyy
 */
function formatDate($date, $format = 'd/m/Y') {
    if (empty($date) || $date === '0000-00-00') {
        return '';
    }
    
    try {
        $d = new DateTime($date);
        return $d->format($format);
    } catch (Exception $e) {
        return $date;
    }
}

/**
 * Formatea fecha y hora
 */
function formatDateTime($datetime, $format = 'd/m/Y H:i') {
    if (empty($datetime) || $datetime === '0000-00-00 00:00:00') {
        return '';
    }
    
    try {
        $d = new DateTime($datetime);
        return $d->format($format);
    } catch (Exception $e) {
        return $datetime;
    }
}

/**
 * Formatea moneda paraguaya
 */
function formatMoney($amount, $currency = 'Gs.') {
    if ($amount === null || $amount === '') {
        return '';
    }
    
    return $currency . ' ' . number_format($amount, 0, ',', '.');
}

/**
 * Formatea número
 */
function formatNumber($number, $decimals = 0) {
    if ($number === null || $number === '') {
        return '';
    }
    
    return number_format($number, $decimals, ',', '.');
}

/**
 * Valida email
 */
function validateEmail($email) {
    return filter_var($email, FILTER_VALIDATE_EMAIL) !== false;
}

/**
 * Valida RUC paraguayo (formato: 12345678-9)
 */
function validateRUC($ruc) {
    $ruc = preg_replace('/[^0-9]/', '', $ruc);
    return strlen($ruc) >= 6 && strlen($ruc) <= 9;
}

/**
 * Valida CI paraguayo
 */
function validateCedula($ci) {
    $ci = preg_replace('/[^0-9]/', '', $ci);
    return strlen($ci) >= 6 && strlen($ci) <= 8;
}

/**
 * Sanitiza string
 */
function sanitizeString($string) {
    return trim(strip_tags($string));
}

/**
 * Genera un slug amigable para URLs
 */
function generateSlug($string) {
    $string = strtolower($string);
    $string = preg_replace('/[^a-z0-9\s-]/', '', $string);
    $string = preg_replace('/[\s-]+/', '-', $string);
    $string = trim($string, '-');
    return $string;
}

/**
 * Genera una contraseña aleatoria
 */
function generatePassword($length = 12) {
    $characters = '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ!@#$%^&*';
    $password = '';
    $max = strlen($characters) - 1;
    
    for ($i = 0; $i < $length; $i++) {
        $password .= $characters[random_int(0, $max)];
    }
    
    return $password;
}

/**
 * Sube un archivo
 */
function uploadFile($file, $directory, $allowedTypes = null) {
    if (!isset($file['error']) || is_array($file['error'])) {
        return ['success' => false, 'message' => 'Parámetros inválidos'];
    }
    
    if ($file['error'] !== UPLOAD_ERR_OK) {
        return ['success' => false, 'message' => 'Error al subir el archivo'];
    }
    
    // Verificar tamaño
    if (defined('UPLOAD_MAX_SIZE') && $file['size'] > UPLOAD_MAX_SIZE) {
        return ['success' => false, 'message' => 'El archivo excede el tamaño máximo permitido'];
    }
    
    // Verificar tipo
    $finfo = new finfo(FILEINFO_MIME_TYPE);
    $mimeType = $finfo->file($file['tmp_name']);
    $extension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
    
    if ($allowedTypes && !in_array($extension, $allowedTypes)) {
        return ['success' => false, 'message' => 'Tipo de archivo no permitido'];
    }
    
    // Generar nombre único
    $filename = uniqid() . '_' . time() . '.' . $extension;
    $filepath = rtrim($directory, '/') . '/' . $filename;
    
    // Crear directorio si no existe
    if (!is_dir($directory)) {
        mkdir($directory, 0755, true);
    }
    
    // Mover archivo
    if (!move_uploaded_file($file['tmp_name'], $filepath)) {
        return ['success' => false, 'message' => 'Error al guardar el archivo'];
    }
    
    return [
        'success' => true,
        'filename' => $filename,
        'filepath' => $filepath,
        'size' => $file['size'],
        'mime_type' => $mimeType
    ];
}

/**
 * Elimina un archivo
 */
function deleteFile($filepath) {
    if (file_exists($filepath)) {
        return unlink($filepath);
    }
    return false;
}

/**
 * Redirecciona con mensaje
 */
function redirect($url, $message = null, $type = 'success') {
    if ($message) {
        $_SESSION['flash_message'] = $message;
        $_SESSION['flash_type'] = $type;
    }
    header("Location: $url");
    exit;
}

/**
 * Muestra mensaje flash
 */
function getFlashMessage() {
    if (isset($_SESSION['flash_message'])) {
        $message = $_SESSION['flash_message'];
        $type = $_SESSION['flash_type'] ?? 'info';
        unset($_SESSION['flash_message']);
        unset($_SESSION['flash_type']);
        return ['message' => $message, 'type' => $type];
    }
    return null;
}

/**
 * Respuesta JSON exitosa
 */
function jsonSuccess($data = null, $message = 'Operación exitosa') {
    header('Content-Type: application/json');
    echo json_encode([
        'success' => true,
        'message' => $message,
        'data' => $data
    ]);
    exit;
}

/**
 * Respuesta JSON con error
 */
function jsonError($message = 'Error en la operación', $statusCode = 400) {
    http_response_code($statusCode);
    header('Content-Type: application/json');
    echo json_encode([
        'success' => false,
        'message' => $message
    ]);
    exit;
}

/**
 * Crea una notificación para un usuario
 */
function createNotification($userId, $tipo, $titulo, $mensaje = null, $url = null) {
    try {
        $db = Database::getInstance();
        $db->insert('notificaciones', [
            'usuario_id' => $userId,
            'tipo' => $tipo,
            'titulo' => $titulo,
            'mensaje' => $mensaje,
            'url' => $url
        ]);
        return true;
    } catch (Exception $e) {
        return false;
    }
}

/**
 * Obtiene notificaciones no leídas
 */
function getUnreadNotifications($userId) {
    try {
        $db = Database::getInstance();
        return $db->fetchAll(
            "SELECT * FROM notificaciones 
             WHERE usuario_id = ? AND leida = 0 
             ORDER BY created_at DESC 
             LIMIT 10",
            [$userId]
        );
    } catch (Exception $e) {
        return [];
    }
}

/**
 * Registra una actividad
 */
function logActivity($userId, $accion, $modulo, $moduloId = null, $descripcion = null) {
    try {
        $db = Database::getInstance();
        $db->insert('actividades', [
            'usuario_id' => $userId,
            'accion' => $accion,
            'modulo' => $modulo,
            'modulo_id' => $moduloId,
            'descripcion' => $descripcion,
            'ip_address' => $_SERVER['REMOTE_ADDR'] ?? null,
            'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? null
        ]);
        return true;
    } catch (Exception $e) {
        return false;
    }
}

/**
 * Genera número de caso único
 */
function generateCaseNumber($prefix = 'CASO') {
    $year = date('Y');
    $random = rand(1000, 9999);
    return "$prefix-$year-$random";
}

/**
 * Obtiene días hábiles entre dos fechas
 */
function getBusinessDays($startDate, $endDate) {
    $begin = new DateTime($startDate);
    $end = new DateTime($endDate);
    $interval = new DateInterval('P1D');
    $dateRange = new DatePeriod($begin, $interval, $end);
    
    $days = 0;
    foreach ($dateRange as $date) {
        if ($date->format('N') < 6) { // 1-5 = Lunes a Viernes
            $days++;
        }
    }
    
    return $days;
}

/**
 * Calcula el IVA
 */
function calculateIVA($amount, $rate = 10) {
    return $amount * ($rate / 100);
}

/**
 * Trunca texto
 */
function truncateText($text, $length = 100, $suffix = '...') {
    if (mb_strlen($text) <= $length) {
        return $text;
    }
    
    return mb_substr($text, 0, $length) . $suffix;
}

/**
 * Obtiene el tamaño legible de un archivo
 */
function formatFileSize($bytes) {
    $units = ['B', 'KB', 'MB', 'GB', 'TB'];
    $bytes = max($bytes, 0);
    $pow = floor(($bytes ? log($bytes) : 0) / log(1024));
    $pow = min($pow, count($units) - 1);
    $bytes /= pow(1024, $pow);
    
    return round($bytes, 2) . ' ' . $units[$pow];
}

/**
 * Verifica si una fecha es un día hábil
 */
function isBusinessDay($date) {
    $d = new DateTime($date);
    $dayOfWeek = $d->format('N');
    return $dayOfWeek < 6; // 1-5 = Lunes a Viernes
}

/**
 * Obtiene la edad desde una fecha de nacimiento
 */
function getAge($birthdate) {
    if (empty($birthdate) || $birthdate === '0000-00-00') {
        return null;
    }
    
    $birth = new DateTime($birthdate);
    $today = new DateTime('today');
    $age = $birth->diff($today)->y;
    
    return $age;
}

/**
 * Genera un token seguro
 */
function generateToken($length = 32) {
    return bin2hex(random_bytes($length));
}

/**
 * Verifica si es una petición AJAX
 */
function isAjax() {
    return !empty($_SERVER['HTTP_X_REQUESTED_WITH']) && 
           strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest';
}

/**
 * Obtiene el método HTTP de la petición
 */
function getRequestMethod() {
    return $_SERVER['REQUEST_METHOD'];
}

/**
 * Verifica si es POST
 */
function isPost() {
    return getRequestMethod() === 'POST';
}

/**
 * Verifica si es GET
 */
function isGet() {
    return getRequestMethod() === 'GET';
}