<?php
/**
 * Auth Class - Manejo de Autenticación y Sesiones
 */

require_once __DIR__ . '/../config/database.php';

class Auth {
    private $db;
    
    public function __construct() {
        $this->db = Database::getInstance();
        
        // Configurar sesión si no está iniciada
        if (session_status() === PHP_SESSION_NONE) {
            $this->configureSession();
            session_start();
        }
    }
    
    /**
     * Configuración de sesión segura
     */
    private function configureSession() {
        ini_set('session.cookie_httponly', 1);
        ini_set('session.use_only_cookies', 1);
        ini_set('session.cookie_secure', isset($_SERVER['HTTPS']));
        
        if (defined('SESSION_LIFETIME')) {
            ini_set('session.gc_maxlifetime', SESSION_LIFETIME);
        }
    }
    
    /**
     * Intenta autenticar a un usuario
     */
    public function login($usuario, $password, $remember = false) {
        try {
            $user = $this->db->fetchOne(
                "SELECT u.*, r.nombre as rol_nombre, r.nivel as rol_nivel 
                 FROM usuarios u
                 INNER JOIN roles r ON u.rol_id = r.id
                 WHERE u.usuario = ? AND u.activo = 1",
                [$usuario]
            );
            
            if (!$user) {
                return ['success' => false, 'message' => 'Usuario no encontrado o inactivo'];
            }
            
            if (!password_verify($password, $user['password'])) {
                return ['success' => false, 'message' => 'Contraseña incorrecta'];
            }
            
            // Actualizar último acceso
            $this->db->update('usuarios', 
                ['ultimo_acceso' => date('Y-m-d H:i:s')],
                'id = ?',
                [$user['id']]
            );
            
            // Establecer sesión
            $_SESSION['logged_in'] = true;
            $_SESSION['user_id'] = $user['id'];
            $_SESSION['user_nombre'] = $user['nombre'];
            $_SESSION['user_apellido'] = $user['apellido'];
            $_SESSION['user_email'] = $user['email'];
            $_SESSION['user_rol_id'] = $user['rol_id'];
            $_SESSION['user_rol_nombre'] = $user['rol_nombre'];
            $_SESSION['user_rol_nivel'] = $user['rol_nivel'];
            $_SESSION['user_foto'] = $user['foto'];
            $_SESSION['login_time'] = time();
            
            // Regenerar ID de sesión por seguridad
            session_regenerate_id(true);
            
            // Cookie de "Recordarme" (opcional)
            if ($remember) {
                $token = bin2hex(random_bytes(32));
                setcookie('remember_token', $token, time() + (86400 * 30), '/', '', isset($_SERVER['HTTPS']), true);
                // Guardar token en BD (implementar tabla de tokens si es necesario)
            }
            
            // Log de actividad
            $this->logActivity($user['id'], 'login', 'usuarios', $user['id'], 'Inicio de sesión exitoso');
            
            return ['success' => true, 'user' => $user];
            
        } catch (Exception $e) {
            return ['success' => false, 'message' => 'Error en el sistema de autenticación'];
        }
    }
    
    /**
     * Cierra la sesión del usuario
     */
    public function logout() {
        if (isset($_SESSION['user_id'])) {
            $this->logActivity($_SESSION['user_id'], 'logout', 'usuarios', $_SESSION['user_id'], 'Cierre de sesión');
        }
        
        // Limpiar sesión
        $_SESSION = [];
        
        // Destruir cookie de sesión
        if (isset($_COOKIE[session_name()])) {
            setcookie(session_name(), '', time() - 3600, '/');
        }
        
        // Destruir sesión
        session_destroy();
        
        // Limpiar cookie de "Recordarme"
        if (isset($_COOKIE['remember_token'])) {
            setcookie('remember_token', '', time() - 3600, '/');
        }
    }
    
    /**
     * Verifica si el usuario está autenticado
     */
    public function isLoggedIn() {
        if (!isset($_SESSION['logged_in']) || $_SESSION['logged_in'] !== true) {
            return false;
        }
        
        // Verificar timeout de sesión
        if (defined('SESSION_LIFETIME')) {
            if (isset($_SESSION['login_time']) && (time() - $_SESSION['login_time']) > SESSION_LIFETIME) {
                $this->logout();
                return false;
            }
        }
        
        return true;
    }
    
    /**
     * Requiere que el usuario esté autenticado
     */
    public function requireLogin() {
        if (!$this->isLoggedIn()) {
            header('Location: ' . $this->getBaseUrl() . 'login.php');
            exit;
        }
    }
    
    /**
     * Obtiene el usuario actual de la sesión
     */
    public function getCurrentUser() {
        if (!$this->isLoggedIn()) {
            return null;
        }
        
        return [
            'id' => $_SESSION['user_id'],
            'nombre' => $_SESSION['user_nombre'],
            'apellido' => $_SESSION['user_apellido'],
            'email' => $_SESSION['user_email'],
            'rol_id' => $_SESSION['user_rol_id'],
            'rol_nombre' => $_SESSION['user_rol_nombre'],
            'rol_nivel' => $_SESSION['user_rol_nivel'],
            'foto' => $_SESSION['user_foto']
        ];
    }
    
    /**
     * Verifica si el usuario tiene un permiso específico
     */
    public function hasPermission($modulo, $accion = 'ver') {
        if (!$this->isLoggedIn()) {
            return false;
        }
        
        // Administradores tienen todos los permisos
        if ($_SESSION['user_rol_nivel'] >= 100) {
            return true;
        }
        
        try {
            $permiso = $this->db->fetchOne(
                "SELECT p.* FROM permisos p
                 INNER JOIN modulos m ON p.modulo_id = m.id
                 WHERE p.rol_id = ? AND m.slug = ?",
                [$_SESSION['user_rol_id'], $modulo]
            );
            
            if (!$permiso) {
                return false;
            }
            
            switch ($accion) {
                case 'ver':
                    return $permiso['ver'] == 1;
                case 'crear':
                    return $permiso['crear'] == 1;
                case 'editar':
                    return $permiso['editar'] == 1;
                case 'eliminar':
                    return $permiso['eliminar'] == 1;
                default:
                    return false;
            }
        } catch (Exception $e) {
            return false;
        }
    }
    
    /**
     * Requiere un permiso específico
     */
    public function requirePermission($modulo, $accion = 'ver') {
        if (!$this->hasPermission($modulo, $accion)) {
            http_response_code(403);
            die('No tienes permisos para acceder a este recurso.');
        }
    }
    
    /**
     * Verifica si el usuario tiene un nivel de rol mínimo
     */
    public function hasRoleLevel($minLevel) {
        if (!$this->isLoggedIn()) {
            return false;
        }
        
        return $_SESSION['user_rol_nivel'] >= $minLevel;
    }
    
    /**
     * Registra una actividad del usuario
     */
    private function logActivity($userId, $accion, $modulo, $moduloId = null, $descripcion = null) {
        try {
            $this->db->insert('actividades', [
                'usuario_id' => $userId,
                'accion' => $accion,
                'modulo' => $modulo,
                'modulo_id' => $moduloId,
                'descripcion' => $descripcion,
                'ip_address' => $_SERVER['REMOTE_ADDR'] ?? null,
                'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? null
            ]);
        } catch (Exception $e) {
            // Log error pero no interrumpir el flujo
        }
    }
    
    /**
     * Obtiene la URL base del sistema
     */
    private function getBaseUrl() {
        $protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https://' : 'http://';
        $host = $_SERVER['HTTP_HOST'];
        $script = $_SERVER['SCRIPT_NAME'];
        $dir = dirname($script);
        
        return $protocol . $host . ($dir !== '/' ? $dir . '/' : '/');
    }
}