let state = {
    currentView: 'empresas',
    currentUser: null,
    userCategory: null,
    currentEmpresa: null, // { id, nombre, ruc, tipo }
    planDeCuentas: [],
    asientos: [] // To store the asientos
};

function formatNumber(num) {
    if (typeof num !== 'number') {
        return num;
    }
    let [integerPart, decimalPart] = num.toFixed(2).split('.');
    integerPart = integerPart.replace(/\B(?=(\d{3})+(?!\d))/g, '.');
    return `${integerPart},${decimalPart}`;
}

document.addEventListener('DOMContentLoaded', async () => {
    try {
        const response = await fetch('../api.php?action=check_session');
        const result = await response.json();
        if (result.loggedIn) {
            state.currentUser = result.username;
            state.userCategory = result.categoria;
            document.getElementById('username-display').textContent = `Usuario: ${state.currentUser}`;
            
            if (state.userCategory === 'Administrador') {
                document.getElementById('nav-gestion-usuarios').classList.remove('d-none');
            }
            if (state.userCategory !== 'Digitador') {
                document.getElementById('nav-vencimientos').classList.remove('d-none');
            }

            document.getElementById('btn-logout').addEventListener('click', () => {
                window.location.href = '../api.php?action=logout';
            });
            render();
        } else {
            window.location.href = 'login.php';
        }
    } catch (error) {
        console.error("Error checking session:", error);
        window.location.href = 'login.php';
    }
});

// --- RENDERIZADO PRINCIPAL ---
function render(planDeCuentas = []) {
    const contentView = document.getElementById('app-content');
    if (state.currentView === 'empresas') {
        contentView.innerHTML = getEmpresasView();
        attachEmpresasListeners();
        cargarEmpresas();
    } else if (state.currentView === 'dashboard') {
        contentView.innerHTML = getDashboardView(planDeCuentas);
        attachDashboardListeners();
    }
}

// --- VISTAS HTML ---
function getEmpresasView() {
    const createButtonHTML = state.userCategory !== 'Digitador' ? `
        <div class="mb-3">
            <button type="button" class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#modal-empresa">
                Crear Nueva Empresa
            </button>
        </div>
    ` : '';

    return `
        <div class="row">
            <div class="col-12">
                <!-- El botón de crear empresa ahora apunta al ID correcto del modal simple -->
                <h3 class="mb-3">Seleccionar Empresa</h3>
                ${createButtonHTML}
                <div class="list-group" id="lista-empresas"></div>
            </div>
        </div>
        ${getEmpresaModalHTML()}
    `;
}

function getEmpresaModalHTML() {
    if (state.userCategory === 'Digitador') return '';
    return `
    <div class="modal fade" id="modal-empresa" tabindex="-1" aria-labelledby="modalEmpresaLabel" aria-hidden="true">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="modalEmpresaLabel"></h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <form id="form-nueva-empresa" novalidate autocomplete="off">
                    <div class="modal-body">
                        <ul class="nav nav-tabs" id="empresa-tabs" role="tablist">
                            <li class="nav-item" role="presentation"><button class="nav-link active" id="generales-tab" data-bs-toggle="tab" data-bs-target="#generales-panel" type="button" role="tab">Datos Generales</button></li>
                            <li class="nav-item" role="presentation"><button class="nav-link" id="equipo-tab" data-bs-toggle="tab" data-bs-target="#equipo-panel" type="button" role="tab">Equipo de Trabajo</button></li>
                            <li class="nav-item" role="presentation"><button class="nav-link" id="timbrados-tab" data-bs-toggle="tab" data-bs-target="#timbrados-panel" type="button" role="tab">Timbrados</button></li>
                            <li class="nav-item" role="presentation"><button class="nav-link" id="presentaciones-tab" data-bs-toggle="tab" data-bs-target="#presentaciones-panel" type="button" role="tab">Presentaciones Fiscales</button></li>
                        </ul>
                        <div class="tab-content pt-3" id="empresa-tabs-content">
                            <!-- Pestaña Datos Generales -->
                            <div class="tab-pane fade show active" id="generales-panel" role="tabpanel">
                                <input type="hidden" id="empresa-id" name="id">
                                <div class="row">
                                    <div class="col-md-6 mb-3"><label for="nombre" class="form-label">Nombre o Razón Social</label><input type="text" class="form-control" id="nombre" name="nombre" required></div>
                                    <div class="col-md-6 mb-3"><label for="ruc" class="form-label">RUC (con DV)</label><input type="text" class="form-control" id="ruc" name="ruc" required></div>
                                </div>
                                <div class="row">
                                    <div class="col-md-6 mb-3"><label for="tipo" class="form-label">Tipo</label><select class="form-select" id="tipo" name="tipo"><option value="fisica">Persona Física</option><option value="juridica">Persona Jurídica</option></select></div>
                                    <div class="col-md-6 mb-3"><label for="responsable_contabilidad" class="form-label">Responsable Contabilidad</label><input type="text" class="form-control" id="responsable_contabilidad" name="responsable_contabilidad"></div>
                                </div>
                                <div class="row">
                                    <div class="col-md-4 mb-3"><label for="telefono" class="form-label">Teléfono</label><input type="text" class="form-control" id="telefono" name="telefono"></div>
                                    <div class="col-md-4 mb-3"><label for="ciudad" class="form-label">Ciudad</label><input type="text" class="form-control" id="ciudad" name="ciudad"></div>
                                    <div class="col-md-4 mb-3"><label for="pais" class="form-label">País</label><input type="text" class="form-control" id="pais" name="pais"></div>
                                </div>
                                <div class="mb-3"><label for="direccion" class="form-label">Dirección</label><input type="text" class="form-control" id="direccion" name="direccion"></div>
                                <div class="mb-3"><label for="actividad_economica" class="form-label">Actividad Económica</label><input type="text" class="form-control" id="actividad_economica" name="actividad_economica"></div>
                                <div class="mb-3"><label for="observaciones" class="form-label">Observaciones</label><textarea class="form-control" id="observaciones" name="observaciones"></textarea></div>
                                <div class="mb-3"><label for="logo" class="form-label">Logo de la Empresa</label><input class="form-control" type="file" id="logo" name="logo" accept="image/png, image/jpeg"><div id="logo-preview" class="mt-2"></div></div>
                            </div>
                            <!-- Pestaña Equipo de Trabajo -->
                            <div class="tab-pane fade" id="equipo-panel" role="tabpanel">
                                <div class="mb-3"><label for="contador" class="form-label">Asignar Contador</label><select class="form-select" id="contador" name="contador_id"></select></div>
                                <div class="mb-3"><label for="digitadores" class="form-label">Asignar Digitadores</label><select class="form-select" id="digitadores" name="digitadores[]" multiple></select></div>
                            </div>
                            <!-- Pestaña Timbrados -->
                            <div class="tab-pane fade" id="timbrados-panel" role="tabpanel">
                                <h6>Gestión de Timbrados</h6>
                                <div id="timbrados-container"></div>
                                <button type="button" class="btn btn-sm btn-outline-success mt-2" id="btn-add-timbrado">Añadir Timbrado</button>
                            </div>
                            <!-- Pestaña Presentaciones -->
                            <div class="tab-pane fade" id="presentaciones-panel" role="tabpanel">
                                <h6>Fechas de Presentación de Impuestos</h6>
                                <div class="input-group mb-3"><span class="input-group-text">IVA presenta el día</span><input type="number" class="form-control" name="presentaciones[iva_dia]" min="1" max="31"><span class="input-group-text">de cada mes</span></div>
                                <div class="input-group mb-3"><span class="input-group-text">IRP presenta el día</span><input type="number" class="form-control" name="presentaciones[irp_dia]" min="1" max="31"><span class="input-group-text">del mes de</span><select class="form-select" name="presentaciones[irp_mes]"><option value="1">Enero</option><option value="2">Febrero</option><option value="3">Marzo</option><option value="4">Abril</option><option value="5">Mayo</option><option value="6">Junio</option><option value="7">Julio</option><option value="8">Agosto</option><option value="9">Septiembre</option><option value="10">Octubre</option><option value="11">Noviembre</option><option value="12">Diciembre</option></select><span class="input-group-text">de cada</span><select class="form-select" name="presentaciones[irp_periodo]"><option value="mes">Mes</option><option value="bimestre">Bimestre</option><option value="trimestre">Trimestre</option><option value="cuatrimestre">Cuatrimestre</option><option value="semestre">Semestre</option><option value="año">Año</option></select></div>
                                <div class="input-group mb-3"><span class="input-group-text">IRE presenta el día</span><input type="number" class="form-control" name="presentaciones[ire_dia]" min="1" max="31"><span class="input-group-text">del mes de</span><select class="form-select" name="presentaciones[ire_mes]"><option value="1">Enero</option><option value="2">Febrero</option><option value="3">Marzo</option><option value="4">Abril</option><option value="5">Mayo</option><option value="6">Junio</option><option value="7">Julio</option><option value="8">Agosto</option><option value="9">Septiembre</option><option value="10">Octubre</option><option value="11">Noviembre</option><option value="12">Diciembre</option></select><span class="input-group-text">de cada</span><select class="form-select" name="presentaciones[ire_periodo]"><option value="mes">Mes</option><option value="bimestre">Bimestre</option><option value="trimestre">Trimestre</option><option value="cuatrimestre">Cuatrimestre</option><option value="semestre">Semestre</option><option value="año">Año</option></select></div>
                                <div class="input-group mb-3"><span class="input-group-text">Informes Financieros presentar el día</span><input type="number" class="form-control" name="presentaciones[inf_dia]" min="1" max="31"><span class="input-group-text">del mes de</span><select class="form-select" name="presentaciones[inf_mes]"><option value="1">Enero</option><option value="2">Febrero</option><option value="3">Marzo</option><option value="4">Abril</option><option value="5">Mayo</option><option value="6">Junio</option><option value="7">Julio</option><option value="8">Agosto</option><option value="9">Septiembre</option><option value="10">Octubre</option><option value="11">Noviembre</option><option value="12">Diciembre</option></select><span class="input-group-text">de cada año</span></div>
                            </div>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                        <button type="submit" class="btn btn-primary" id="btn-guardar-empresa">Guardar</button>
                    </div>
                </div>
            </div>
        </div>
    </div>
    <template id="timbrado-row-template">
        <div class="row gx-2 mb-2 align-items-center timbrado-row">
            <div class="col-md-4"><input type="text" class="form-control form-control-sm" name="timbrados[{index}][numero]" placeholder="Número de Timbrado" required></div>
            <div class="col-md-3"><input type="date" class="form-control form-control-sm" name="timbrados[{index}][inicio]" placeholder="Fecha Inicio" required></div>
            <div class="col-md-3"><input type="date" class="form-control form-control-sm" name="timbrados[{index}][fin]" placeholder="Fecha Fin" required></div>
            <div class="col-md-2"><button type="button" class="btn btn-sm btn-danger btn-remove-timbrado">Eliminar</button></div>
        </div>
    </template>
    `;
}

function getDashboardView(planDeCuentas) {
    return `
        <div class="d-flex justify-content-between align-items-center mb-3">
            <div class="d-flex align-items-center">
                ${state.currentEmpresa.logo ? `<img src="../${state.currentEmpresa.logo}" alt="Logo" class="me-3" style="height: 50px; border-radius: 4px;">` : ''}
                <h2 class="mb-0">Empresa: ${state.currentEmpresa.nombre}</h2>
            </div>
            <div>
                ${state.userCategory !== 'Digitador' ? `
                    <button id="btn-delete-empresa" class="btn btn-danger btn-sm">Eliminar Empresa</button>
                ` : ''}
                <button id="btn-volver" class="btn btn-secondary btn-sm">Volver</button>
            </div>
        </div>
        
        <ul class="nav nav-tabs" id="dashboard-tabs" role="tablist">
            <li class="nav-item" role="presentation">
                <button class="nav-link active" id="asientos-tab" data-bs-toggle="tab" data-bs-target="#asientos-panel" type="button" role="tab">Cargar Asiento</button>
            </li>
            <li class="nav-item" role="presentation">
                <button class="nav-link" id="plan-tab" data-bs-toggle="tab" data-bs-target="#plan-panel" type="button" role="tab">Plan de Cuentas</button>
            </li>
             <li class="nav-item" role="presentation">
                <button class="nav-link" id="diario-tab" data-bs-toggle="tab" data-bs-target="#diario-panel" type="button" role="tab">Libro Diario</button>
            </li>
            <li class="nav-item" role="presentation">
                <button class="nav-link" id="saldos-tab" data-bs-toggle="tab" data-bs-target="#saldos-panel" type="button" role="tab">Sumas y Saldos</button>
            </li>
            <li class="nav-item" role="presentation">
                <button class="nav-link" id="resultados-tab" data-bs-toggle="tab" data-bs-target="#resultados-panel" type="button" role="tab">Estado de Resultados</button>
            </li>
            <li class="nav-item" role="presentation">
                <button class="nav-link" id="balance-tab" data-bs-toggle="tab" data-bs-target="#balance-panel" type="button" role="tab">Balance General</button>
            </li>
            <li class="nav-item" role="presentation">
                <button class="nav-link" id="iva-tab" data-bs-toggle="tab" data-bs-target="#iva-panel" type="button" role="tab">Liquidación IVA</button>
            </li>
            <li class="nav-item" role="presentation">
                <button class="nav-link" id="ratios-tab" data-bs-toggle="tab" data-bs-target="#ratios-panel" type="button" role="tab">Ratios Financieros</button>
            </li>
            ${state.userCategory !== 'Digitador' ? `<li class="nav-item" role="presentation">
                <button class="nav-link" id="log-tab" data-bs-toggle="tab" data-bs-target="#log-panel" type="button" role="tab">Registro de Actividad</button>
            </li>` : ''}
        </ul>

        <div class="tab-content pt-4" id="dashboard-tab-content">
            <div class="tab-pane fade show active" id="asientos-panel" role="tabpanel">
                ${getAsientosFormView(planDeCuentas)}
            </div>
            <div class="tab-pane fade" id="plan-panel" role="tabpanel"></div>
            <div class="tab-pane fade" id="diario-panel" role="tabpanel"><div id="libro-diario-content"></div></div>
            <div class="tab-pane fade" id="saldos-panel" role="tabpanel"><div id="sumas-y-saldos-content"></div></div>
            <div class="tab-pane fade" id="resultados-panel" role="tabpanel"><div id="estado-resultados-content"></div></div>
            <div class="tab-pane fade" id="balance-panel" role="tabpanel"><div id="balance-general-content"></div></div>
            <div class="tab-pane fade" id="iva-panel" role="tabpanel"></div>
            <div class="tab-pane fade" id="ratios-panel" role="tabpanel"></div>
            <div class="tab-pane fade" id="log-panel" role="tabpanel"></div>
        </div>
    `;
}

function addTimbradoRow(timbrado = null) {
    const container = document.getElementById('timbrados-container');
    const index = container.children.length;
    const template = document.getElementById('timbrado-row-template').innerHTML;
    const newRowHTML = template.replace(/\{index\}/g, index);
    
    const rowDiv = document.createElement('div');
    rowDiv.innerHTML = newRowHTML;
    
    if (timbrado) {
        rowDiv.querySelector('[name$="[numero]"]').value = timbrado.numero || '';
        rowDiv.querySelector('[name$="[inicio]"]').value = timbrado.inicio || '';
        rowDiv.querySelector('[name$="[fin]"]').value = timbrado.fin || '';
    }

    container.appendChild(rowDiv.firstElementChild);
}

function attachEmpresaModalListeners() {
    const btnAddTimbrado = document.getElementById('btn-add-timbrado');
    if (btnAddTimbrado) {
        btnAddTimbrado.addEventListener('click', () => addTimbradoRow());
    }

    const timbradosContainer = document.getElementById('timbrados-container');
    if (timbradosContainer) {
        timbradosContainer.addEventListener('click', (e) => {
            if (e.target.classList.contains('btn-remove-timbrado')) {
                e.target.closest('.timbrado-row').remove();
            }
        });
    }
}

function getCuentasOptions(planDeCuentas, filterPrefix) {
    if (!planDeCuentas) return '';
    return planDeCuentas
        .filter(c => c.codigo.startsWith(filterPrefix) && c.tipo === 'detalle')
        .map(c => `<option value="${c.codigo}">${c.nombre}</option>`)
        .join('');
}

function getAsientosFormView() {
    const cuentasCajaBancos = getCuentasOptions(state.planDeCuentas, '1.1.1');
    const cuentasIngresos = getCuentasOptions(state.planDeCuentas, '4');
    const cuentasGastos = getCuentasOptions(state.planDeCuentas, '5');

    return `
        <div class="row">
            <div class="col-md-6">
                <h5>Cargar Factura de Venta</h5>
                <form id="form-nueva-venta" class="p-3 border rounded bg-light">
                    <input type="hidden" name="asiento_id">
                    <div class="row">
                        <div class="col-6 mb-2"><label class="form-label">Fecha</label><input type="date" class="form-control form-control-sm" name="fecha" required></div>
                        <div class="col-6 mb-2"><label class="form-label">Condición</label><select class="form-select form-select-sm" name="condicion_operacion"><option value="contado">Contado</option><option value="credito">Crédito</option></select></div>
                    </div>
                    <div class="row">
                        <div class="col-6 mb-2"><label class="form-label">Timbrado</label><input type="text" class="form-control form-control-sm" name="timbrado" required></div>
                        <div class="col-6 mb-2"><label class="form-label">Nro. Comprobante</label><input type="text" class="form-control form-control-sm" name="comprobante_nro" required></div>
                    </div>
                    <div class="form-check mb-2">
                        <input class="form-check-input" type="checkbox" name="es_electronica">
                        <label class="form-check-label">Factura Electrónica</label>
                    </div>
                     <div class="mb-2"><label class="form-label">Tipo Comprobante</label><input type="text" class="form-control form-control-sm" name="tipo_comprobante" value="Factura" required></div>
                    <div class="row">
                        <div class="col-6 mb-2"><label class="form-label">RUC Cliente</label><input type="text" class="form-control form-control-sm" name="ruc_cliente_proveedor" required></div>
                        <div class="col-6 mb-2"><label class="form-label">Nombre Cliente</label><input type="text" class="form-control form-control-sm" name="descripcion" required></div>
                    </div>
                    <div class="row">
                        <div class="col-4 mb-2"><label class="form-label">Monto Total</label><input type="number" class="form-control form-control-sm" name="monto" step="0.01" required></div>
                        <div class="col-4 mb-2"><label class="form-label">Moneda</label><input type="text" class="form-control form-control-sm" name="moneda" value="PYG" required></div>
                        <div class="col-4 mb-2"><label class="form-label">IVA</label><select class="form-select form-select-sm" name="iva_tipo"><option value="10">10%</option><option value="5">5%</option><option value="0">Exento</option></select></div>
                    </div>
                    <div class="row">
                        <div class="col-6 mb-2">
                            <label class="form-label">Cobrado con</label>
                            <select class="form-select form-select-sm" name="cuenta_pago_codigo" required>
                                ${cuentasCajaBancos}
                            </select>
                        </div>
                        <div class="col-6 mb-2">
                            <label class="form-label">Cuenta de Ingreso</label>
                            <select class="form-select form-select-sm" name="cuenta_ingreso_codigo" required>
                                ${cuentasIngresos}
                            </select>
                        </div>
                    </div>
                    <button type="submit" class="btn btn-primary btn-sm">Guardar Venta</button>
                    <button type="button" id="btn-cancelar-edicion-venta" class="btn btn-secondary btn-sm d-none">Cancelar Edición</button>
                </form>
            </div>
            <div class="col-md-6">
                <h5>Cargar Factura de Compra</h5>
                 <form id="form-nueva-compra" class="p-3 border rounded bg-light">
                    <input type="hidden" name="asiento_id">
                    <div class="row">
                        <div class="col-6 mb-2"><label class="form-label">Fecha</label><input type="date" class="form-control form-control-sm" name="fecha" required></div>
                        <div class="col-6 mb-2"><label class="form-label">Condición</label><select class="form-select form-select-sm" name="condicion_operacion"><option value="contado">Contado</option><option value="credito">Crédito</option></select></div>
                    </div>
                    <div class="row">
                        <div class="col-6 mb-2"><label class="form-label">Timbrado</label><input type="text" class="form-control form-control-sm" name="timbrado" required></div>
                        <div class="col-6 mb-2"><label class="form-label">Nro. Comprobante</label><input type="text" class="form-control form-control-sm" name="comprobante_nro" required></div>
                    </div>
                    <div class="form-check mb-2">
                        <input class="form-check-input" type="checkbox" name="es_electronica">
                        <label class="form-check-label">Factura Electrónica</label>
                    </div>
                     <div class="mb-2"><label class="form-label">Tipo Comprobante</label><input type="text" class="form-control form-control-sm" name="tipo_comprobante" value="Factura" required></div>
                    <div class="row">
                        <div class="col-6 mb-2"><label class="form-label">RUC Proveedor</label><input type="text" class="form-control form-control-sm" name="ruc_cliente_proveedor" required></div>
                        <div class="col-6 mb-2"><label class="form-label">Nombre Proveedor</label><input type="text" class="form-control form-control-sm" name="descripcion" required></div>
                    </div>
                    <div class="row">
                        <div class="col-4 mb-2"><label class="form-label">Monto Total</label><input type="number" class="form-control form-control-sm" name="monto" step="0.01" required></div>
                        <div class="col-4 mb-2"><label class="form-label">Moneda</label><input type="text" class="form-control form-control-sm" name="moneda" value="PYG" required></div>
                        <div class="col-4 mb-2"><label class="form-label">IVA</label><select class="form-select form-select-sm" name="iva_tipo"><option value="10">10%</option><option value="5">5%</option><option value="0">Exento</option></select></div>
                    </div>
                    <div class="row">
                        <div class="col-6 mb-2">
                            <label class="form-label">Pagado con</label>
                            <select class="form-select form-select-sm" name="cuenta_pago_codigo" required>
                                ${cuentasCajaBancos}
                            </select>
                        </div>
                        <div class="col-6 mb-2">
                            <label class="form-label">Cuenta de Gasto</label>
                            <select class="form-select form-select-sm" name="cuenta_gasto_codigo" required>
                                ${cuentasGastos}
                            </select>
                        </div>
                    </div>
                    <button type="submit" class="btn btn-success btn-sm">Guardar Compra</button>
                    <button type="button" id="btn-cancelar-edicion-compra" class="btn btn-secondary btn-sm d-none">Cancelar Edición</button>
                </form>
            </div>
        </div>
        <div class="row mt-4">
            <div class="col-12">
                <div class="card">
                    <div class="card-header">
                        <div class="d-flex justify-content-between align-items-center">
                            <h5>Asientos Registrados</h5>
                            <div class="btn-group" role="group">
                                <button type="button" class="btn btn-sm btn-outline-success" id="btn-export-ventas-csv">Exportar Ventas (CSV)</button>
                                <button type="button" class="btn btn-sm btn-outline-success" id="btn-export-ventas-excel">Exportar Ventas (Excel)</button>
                                <button type="button" class="btn btn-sm btn-outline-info" id="btn-export-compras-csv">Exportar Compras (CSV)</button>
                                <button type="button" class="btn btn-sm btn-outline-info" id="btn-export-compras-excel">Exportar Compras (Excel)</button>
                            </div>
                        </div>
                    </div>
                    <div class="card-body">
                        <div class="row g-2 mb-3" id="filtros-asientos">
                            <div class="col-md-2"><label class="form-label">Desde</label><input type="date" class="form-control form-control-sm" id="filtro-fecha-desde"></div>
                            <div class="col-md-2"><label class="form-label">Hasta</label><input type="date" class="form-control form-control-sm" id="filtro-fecha-hasta"></div>
                            <div class="col-md-5"><label class="form-label">Buscar (Nro, Desc, Monto)</label><input type="text" class="form-control form-control-sm" id="filtro-texto" placeholder="Buscar..."></div>
                            <div class="col-md-3 d-flex align-items-end">
                                <button class="btn btn-primary btn-sm me-2" id="btn-filtrar-asientos">Filtrar</button>
                                <button class="btn btn-secondary btn-sm" id="btn-limpiar-filtros-asientos">Limpiar</button>
                            </div>
                        </div>
                        <div class="table-responsive">
                            <table class="table table-sm table-bordered table-hover">
                                <thead><tr><th>Fecha</th><th>Nro. Comp.</th><th>Descripción</th><th class="text-end">Monto</th><th>Acciones</th></tr></thead>
                                <tbody id="tabla-asientos-registrados"></tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    `;
}

function getPlanCuentasView() {
    return `
        <div class="card-header d-flex justify-content-between align-items-center">
            <h4 class="mb-0">Plan de Cuentas</h4>
            <button id="btn-imprimir-plan" class="btn btn-sm btn-outline-primary">Imprimir Reporte</button>
        </div>
        <div class="row">
            <div class="col-md-8">
                <table class="table table-sm table-hover">
                    <thead><tr><th>Código</th><th>Nombre</th><th>Acciones</th></tr></thead>
                    <tbody id="tabla-plan-de-cuentas"></tbody>
                </table>
                
            </div>
            <div class="col-md-4">
                <h5 id="form-cuenta-titulo">Añadir Cuenta</h5>
                <form id="form-nueva-cuenta">
                    <input type="hidden" id="cuenta-original-codigo">
                    <div class="mb-2"><label for="cuenta-codigo" class="form-label">Código</label><input type="text" id="cuenta-codigo" class="form-control form-control-sm" required></div>
                    <div class="mb-2"><label for="cuenta-nombre" class="form-label">Nombre</label><input type="text" id="cuenta-nombre" class="form-control form-control-sm" required></div>
                    <button type="submit" class="btn btn-success btn-sm">Guardar</button>
                    <button type="button" id="btn-cancelar-edicion" class="btn btn-secondary btn-sm d-none">Cancelar</button>
                </form>
            </div>
        </div>
    `;
}

function attachEmpresasListeners() {
    if (state.userCategory !== 'Digitador') {
        const form = document.getElementById('form-nueva-empresa');
        const modalEl = document.getElementById('modal-empresa');
        const createButton = document.querySelector('[data-bs-target="#modal-nueva-empresa"]');

        if (createButton) {
            createButton.addEventListener('click', handleNewEmpresa);
        }

        if (form) {
            form.addEventListener('submit', saveEmpresa);
        }
        
        if (modalEl) {
            // Limpiar el formulario del modal cada vez que se cierra
            attachEmpresaModalListeners();
            modalEl.addEventListener('hidden.bs.modal', () => form.reset());
        }
    }

    attachEmpresaListListeners();
}

function getDateFilterView() {
    return `
        <div class="row mb-3 date-filter-container">
            <div class="col-md-3">
                <label class="form-label">Desde</label>
                <input type="date" class="form-control form-control-sm fecha-desde">
            </div>
            <div class="col-md-3">
                <label class="form-label">Hasta</label>
                <input type="date" class="form-control form-control-sm fecha-hasta">
            </div>
            <div class="col-md-2 d-flex align-items-end">
                <button class="btn btn-primary btn-sm btn-filtrar">Filtrar</button>
            </div>
        </div>
    `;
}

function attachEmpresaListListeners() {
    const lista = document.getElementById('lista-empresas');
    if (!lista) return;

    lista.addEventListener('click', (event) => {
        const selectButton = event.target.closest('.btn-seleccionar-empresa');
        if (selectButton) {
            seleccionarEmpresa(selectButton.dataset.id);
        }

        const editButton = event.target.closest('.btn-editar-empresa-lista');
        if (editButton) {
            handleEditEmpresa(editButton.dataset.id);
        }

        const deleteButton = event.target.closest('.btn-eliminar-empresa-lista');
        if (deleteButton) {
            // Reutilizamos la función handleDeleteEmpresa, pero pasándole el ID
            handleDeleteEmpresa(deleteButton.dataset.id);
        }
    });
}

function attachDashboardListeners() {
    document.getElementById('btn-volver').addEventListener('click', () => {
        state.currentView = 'empresas';
        state.currentEmpresa = null;
        render();
    });

    if (state.userCategory !== 'Digitador') {
        document.getElementById('btn-delete-empresa').addEventListener('click', handleDeleteEmpresa);
    }

    document.getElementById('asientos-tab').addEventListener('shown.bs.tab', cargarYRenderizarAsientos);

    document.getElementById('plan-tab').addEventListener('shown.bs.tab', () => {
        const planPanel = document.getElementById('plan-panel');
        if (!planPanel.querySelector('#form-nueva-cuenta')) {
            planPanel.innerHTML = `<div class="card"><div class="card-body">${getPlanCuentasView()}</div></div>`;
            document.getElementById('form-nueva-cuenta').addEventListener('submit', guardarCuenta);
            document.getElementById('tabla-plan-de-cuentas').addEventListener('click', handlePlanCuentasActions);
            document.getElementById('btn-cancelar-edicion').addEventListener('click', cancelarEdicion);
        }
        cargarPlanDeCuentas();
    });

    document.getElementById('diario-tab').addEventListener('shown.bs.tab', () => {
        const panel = document.getElementById('diario-panel');
        if (!panel.querySelector('.date-filter-container')) {
            panel.innerHTML = `<div id="libro-diario-content"></div>`; // Limpiar por si acaso
            panel.insertAdjacentHTML('afterbegin', getDateFilterView());
            panel.querySelector('.btn-filtrar').addEventListener('click', cargarLibroDiario);
        }
        cargarLibroDiario();
    });

    document.getElementById('saldos-tab').addEventListener('shown.bs.tab', () => {
        const panel = document.getElementById('saldos-panel');
        if (!panel.querySelector('.date-filter-container')) {
            panel.insertAdjacentHTML('afterbegin', getDateFilterView());
            panel.querySelector('.btn-filtrar').addEventListener('click', cargarSumasySaldos);
        }
        cargarSumasySaldos();
    });

    document.getElementById('resultados-tab').addEventListener('shown.bs.tab', () => {
        const panel = document.getElementById('resultados-panel');
        if (!panel.querySelector('.date-filter-container')) {
            panel.insertAdjacentHTML('afterbegin', getDateFilterView());
            panel.querySelector('.btn-filtrar').addEventListener('click', cargarEstadoResultados);
        }
        cargarEstadoResultados();
    });

    document.getElementById('balance-tab').addEventListener('shown.bs.tab', () => {
        const panel = document.getElementById('balance-panel');
        if (!panel.querySelector('.date-filter-container')) {
            panel.insertAdjacentHTML('afterbegin', getDateFilterView());
            panel.querySelector('.btn-filtrar').addEventListener('click', cargarBalanceGeneral);
        }
        cargarBalanceGeneral();
    });

    document.getElementById('iva-tab').addEventListener('shown.bs.tab', () => {
        const panel = document.getElementById('iva-panel');
        if (!panel.querySelector('.date-filter-container')) {
            panel.insertAdjacentHTML('afterbegin', getDateFilterView());
            panel.querySelector('.btn-filtrar').addEventListener('click', cargarReporteIVA);
        }
        cargarReporteIVA();
    });

    document.getElementById('ratios-tab').addEventListener('shown.bs.tab', () => {
        const panel = document.getElementById('ratios-panel');
        if (!panel.querySelector('.date-filter-container')) {
            panel.insertAdjacentHTML('afterbegin', getDateFilterView());
            panel.querySelector('.btn-filtrar').addEventListener('click', cargarRatiosFinancieros);
        }
        cargarRatiosFinancieros();
    });

    if (state.userCategory !== 'Digitador') {
        document.getElementById('log-tab').addEventListener('shown.bs.tab', () => {
            const panel = document.getElementById('log-panel');
            cargarActividadLog(panel);
        });
    }
}

async function cargarYRenderizarAsientos() {
    const tablaBody = document.getElementById('tabla-asientos-registrados');
    tablaBody.innerHTML = '<tr><td colspan="5">Cargando...</td></tr>';

    try {
        const response = await fetch(`../api.php?action=get_asientos&empresa_id=${state.currentEmpresa.id}`);
        if (!response.ok) throw new Error('No se pudieron cargar los asientos.');
        state.asientos = await response.json();
        
        // Aplicar filtros
        const fechaDesde = document.getElementById('filtro-fecha-desde').value;
        const fechaHasta = document.getElementById('filtro-fecha-hasta').value;
        const texto = document.getElementById('filtro-texto').value.toLowerCase();

        let asientosFiltrados = state.asientos;

        if (fechaDesde) {
            asientosFiltrados = asientosFiltrados.filter(a => a.fecha >= fechaDesde);
        }
        if (fechaHasta) {
            asientosFiltrados = asientosFiltrados.filter(a => a.fecha <= fechaHasta);
        }
        if (texto) {
            asientosFiltrados = asientosFiltrados.filter(a => {
                const monto = a.movimientos.reduce((acc, m) => acc + (parseFloat(m.debe) || 0), 0);
                return a.comprobante_nro.toLowerCase().includes(texto) ||
                       a.descripcion.toLowerCase().includes(texto) ||
                       monto.toString().includes(texto);
            });
        }

        if (asientosFiltrados.length === 0) {
            tablaBody.innerHTML = '<tr><td colspan="5" class="text-center text-muted">No hay asientos registrados.</td></tr>';
            return;
        }

        tablaBody.innerHTML = '';
        asientosFiltrados.forEach(asiento => {
            const tr = document.createElement('tr');
            const monto = asiento.movimientos.reduce((acc, m) => acc + (parseFloat(m.debe) || 0), 0);
            tr.innerHTML = `
                <td>${asiento.fecha}</td>
                <td>${asiento.comprobante_nro}</td>
                <td>${asiento.descripcion}</td>
                <td class="text-end">${formatNumber(monto)}</td>
                <td>
                    <button class="btn btn-sm btn-warning btn-edit-asiento" data-id="${asiento.id}">Editar</button>
                    <button class="btn btn-sm btn-danger btn-delete-asiento" data-id="${asiento.id}">Eliminar</button>
                </td>
            `;
            tablaBody.appendChild(tr);
        });

        // Add event listeners for the new buttons
        tablaBody.querySelectorAll('.btn-edit-asiento').forEach(btn => {
            btn.addEventListener('click', (e) => handleEditAsiento(e.target.dataset.id));
        });
        tablaBody.querySelectorAll('.btn-delete-asiento').forEach(btn => {
            btn.addEventListener('click', (e) => handleDeleteAsiento(e.target.dataset.id));
        });

    } catch (error) {
        tablaBody.innerHTML = `<tr><td colspan="5" class="text-danger">${error.message}</td></tr>`;
        console.error('Error al cargar asientos:', error);
    }
}

function limpiarFiltrosYRecargarAsientos() {
    document.getElementById('filtro-fecha-desde').value = '';
    document.getElementById('filtro-fecha-hasta').value = '';
    document.getElementById('filtro-texto').value = '';
    cargarYRenderizarAsientos();
}

function handleEditAsiento(asientoId) {
    const asiento = state.asientos.find(a => a.id === asientoId);
    if (!asiento) return;

    const tipoFactura = asiento.tipo_factura;
    const form = document.getElementById(`form-nueva-${tipoFactura}`);
    
    form.elements.asiento_id.value = asiento.id;
    form.elements.fecha.value = asiento.fecha;
    form.elements.condicion_operacion.value = asiento.condicion_operacion;
    form.elements.timbrado.value = asiento.timbrado;
    form.elements.comprobante_nro.value = asiento.comprobante_nro;
    form.elements.tipo_comprobante.value = asiento.tipo_comprobante;
    form.elements.es_electronica.checked = asiento.es_electronica || false;
    form.elements.ruc_cliente_proveedor.value = asiento.ruc_cliente_proveedor;
    form.elements.descripcion.value = asiento.descripcion.replace(`Factura ${tipoFactura} - `, '');

    // --- Corrección de redondeo: Obtener el monto total directamente del movimiento correcto ---
    let monto = 0;
    if (tipoFactura === 'venta') {
        // En ventas, el total está en el DEBE de la cuenta de cobro (Caja, Bancos, Cuentas por Cobrar)
        const totalMovement = asiento.movimientos.find(m => m.cuenta_codigo.startsWith('1.1.1') || m.cuenta_codigo.startsWith('1.1.2'));
        monto = totalMovement ? parseFloat(totalMovement.debe) : 0;
    } else { // compra
        // En compras, el total está en el HABER de la cuenta de pago (Caja, Bancos, Cuentas por Pagar)
        const totalMovement = asiento.movimientos.find(m => m.cuenta_codigo.startsWith('1.1.1') || m.cuenta_codigo.startsWith('2.1.1'));
        monto = totalMovement ? parseFloat(totalMovement.haber) : 0;
    }
    form.elements.monto.value = monto;
    form.elements.moneda.value = asiento.moneda;

    // Determine IVA type from movements
    const ivaDebitMovement = asiento.movimientos.find(m => m.cuenta_codigo === '2.1.3');
    const ivaCreditMovement = asiento.movimientos.find(m => m.cuenta_codigo === '1.1.4');
    const baseMovementVenta = asiento.movimientos.find(m => m.cuenta_codigo.startsWith('4'));
    const baseMovementCompra = asiento.movimientos.find(m => m.cuenta_codigo.startsWith('5'));
    
    if (ivaDebitMovement && baseMovementVenta) {
        const ivaRate = (ivaDebitMovement.haber / baseMovementVenta.haber) * 100;
        form.elements.iva_tipo.value = Math.round(ivaRate);
    } else if (ivaCreditMovement && baseMovementCompra) {
        const ivaRate = (ivaCreditMovement.debe / baseMovementCompra.debe) * 100;
        form.elements.iva_tipo.value = Math.round(ivaRate);
    } else {
        form.elements.iva_tipo.value = 0;
    }


    const pagoMovement = asiento.movimientos.find(m => m.cuenta_codigo.startsWith('1.1.1'));
    if(pagoMovement) {
        form.elements.cuenta_pago_codigo.value = pagoMovement.cuenta_codigo;
    }

    if (tipoFactura === 'venta') {
        const ingresoMovement = asiento.movimientos.find(m => m.cuenta_codigo.startsWith('4'));
        if(ingresoMovement) {
            form.elements.cuenta_ingreso_codigo.value = ingresoMovement.cuenta_codigo;
        }
    } else { // compra
        const gastoMovement = asiento.movimientos.find(m => m.cuenta_codigo.startsWith('5'));
        if(gastoMovement) {
            form.elements.cuenta_gasto_codigo.value = gastoMovement.cuenta_codigo;
        }
    }


    document.getElementById(`btn-cancelar-edicion-${tipoFactura}`).classList.remove('d-none');
}

function cancelarEdicionAsiento(tipoFactura) {
    const form = document.getElementById(`form-nueva-${tipoFactura}`);
    form.reset();
    form.elements.asiento_id.value = '';
    document.getElementById(`btn-cancelar-edicion-${tipoFactura}`).classList.add('d-none');
}


async function handleDeleteAsiento(asientoId) {
    if (confirm('¿Está seguro de que desea eliminar este asiento?')) {
        try {
            const response = await fetch('../api.php?action=delete_asiento', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    empresa_id: state.currentEmpresa.id, 
                    asiento_id: asientoId 
                })
            });
            if (!response.ok) {
                const err = await response.json();
                throw new Error(err.error || 'Error al eliminar el asiento');
            }
            await response.json();
            cargarYRenderizarAsientos(); // Recargar la tabla
        } catch (error) {
            console.error('Error:', error);
            alert(`No se pudo eliminar el asiento.\n${error.message}`);
        }
    }
}

function exportarAsientos(tipo, format) {
    const empresaId = state.currentEmpresa.id;
    const action = tipo === 'ventas' ? 'export_ventas' : 'export_compras';
    const url = `../api.php?action=${action}&empresa_id=${empresaId}&format=${format}`;
    window.open(url, '_blank');
}

async function getProcessedMayorData(fechaDesde, fechaHasta) {
    let asientosUrl = `../api.php?action=get_asientos&empresa_id=${state.currentEmpresa.id}`;
    if (fechaDesde) asientosUrl += `&fecha_desde=${fechaDesde}`;
    if (fechaHasta) asientosUrl += `&fecha_hasta=${fechaHasta}`;

    const [asientosRes, planRes] = await Promise.all([
        fetch(asientosUrl),
        fetch(`../api.php?action=get_plan_de_cuentas&empresa_id=${state.currentEmpresa.id}`)
    ]);

    if (!asientosRes.ok || !planRes.ok) throw new Error('No se pudieron cargar los datos.');

    const asientos = await asientosRes.json();
    const planDeCuentas = await planRes.json();

    const mayor = {};
    planDeCuentas.forEach(c => {
        mayor[c.codigo] = { nombre: c.nombre, tipo: c.tipo, totalDebe: 0, totalHaber: 0 };
    });

    asientos.forEach(asiento => {
        asiento.movimientos.forEach(mov => {
            if (mayor[mov.cuenta_codigo]) {
                mayor[mov.cuenta_codigo].totalDebe += parseFloat(mov.debe) || 0;
                mayor[mov.cuenta_codigo].totalHaber += parseFloat(mov.haber) || 0;
            }
        });
    });

    return { data: mayor, asientos };
}

async function cargarLibroDiario() {
    const contentDiv = document.getElementById('libro-diario-content');
    contentDiv.innerHTML = '<p>Cargando...</p>';

    const panel = document.getElementById('diario-panel');
    const fechaDesde = panel.querySelector('.fecha-desde')?.value;
    const fechaHasta = panel.querySelector('.fecha-hasta')?.value;

    let url = `../api.php?action=get_asientos&empresa_id=${state.currentEmpresa.id}`;
    if (fechaDesde) url += `&fecha_desde=${fechaDesde}`;
    if (fechaHasta) url += `&fecha_hasta=${fechaHasta}`;

    try {
        const response = await fetch(url);
        if (!response.ok) throw new Error('No se pudieron cargar los asientos.');
        const asientos = await response.json();

        if (asientos.length === 0) {
            contentDiv.innerHTML = '<p class="text-muted">No hay asientos registrados para el período seleccionado.</p>';
            return;
        }

        let totalDebe = 0;
        let totalHaber = 0;
        let headerHTML = `
            <div class="card-header">
                <div class="d-flex justify-content-between align-items-center">
                    <h4 class="mb-0">Libro Diario</h4>
                    <div>
                        <button id="btn-crear-asiento-manual" class="btn btn-sm btn-success">Crear Asiento Manual</button>
                        <button id="btn-imprimir-diario" class="btn btn-sm btn-outline-primary">Imprimir Reporte</button>
                    </div>
                </div>
            </div>
        `;

        let tableHTML = `
            <table class="table table-sm table-bordered"><thead><tr><th>Fecha</th><th>Cuenta</th><th>Descripción</th><th class="text-end">Debe</th><th class="text-end">Haber</th></tr></thead><tbody>`;

        asientos.forEach(asiento => {
            tableHTML += `<tr class="table-secondary fw-bold"><td colspan="2">Asiento #${asiento.id.substring(0, 6)}</td><td>${asiento.descripcion} (${asiento.comprobante_nro || 'S/N'})</td><td colspan="2" class="text-end"><button class="btn btn-xs btn-warning py-0 px-1 btn-edit-manual-asiento" data-id="${asiento.id}">Editar</button></td></tr>`;
            asiento.movimientos.forEach(mov => {
                const debe = parseFloat(mov.debe) || 0;
                const haber = parseFloat(mov.haber) || 0;
                totalDebe += debe;
                totalHaber += haber;
                tableHTML += `
                    <tr>
                        <td>${asiento.fecha}</td>
                        <td>(${mov.cuenta_codigo}) ${mov.descripcion}</td>
                        <td></td>
                        <td class="text-end">${debe > 0 ? formatNumber(debe) : ''}</td>
                        <td class="text-end">${haber > 0 ? formatNumber(haber) : ''}</td>
                    </tr>
                `;
            });
        });

        tableHTML += `
            <tr class="table-dark fw-bold">
                <td colspan="3" class="text-end">TOTALES</td>
                <td class="text-end">${formatNumber(totalDebe)}</td>
                <td class="text-end">${formatNumber(totalHaber)}</td>
            </tr>
        `;

        tableHTML += '</tbody></table>';
        
        contentDiv.innerHTML = `<div class="card"><div class="card-body p-0">${tableHTML}</div></div>`;
        const cardElement = contentDiv.querySelector('.card');
        cardElement.insertAdjacentHTML('afterbegin', headerHTML);
        
        // Listeners para los nuevos botones
        cardElement.querySelector('#btn-imprimir-diario').addEventListener('click', imprimirLibroDiario);
        cardElement.querySelector('#btn-crear-asiento-manual').addEventListener('click', () => abrirEditorAsientoManual());
        cardElement.querySelectorAll('.btn-edit-manual-asiento').forEach(btn => {
            btn.addEventListener('click', (e) => abrirEditorAsientoManual(e.target.dataset.id));
        });

    } catch (error) {
        contentDiv.innerHTML = `<p class="text-danger">${error.message}</p>`;
        console.error('Error al cargar libro diario:', error);
    }
}

async function crearAsiento(event, tipoFactura) {
    event.preventDefault();
    const form = event.target;
    const asientoId = form.elements.asiento_id.value;
    const action = asientoId ? 'update_asiento' : 'create_asiento';

    const data = {
        empresa_id: state.currentEmpresa.id,
        tipo_factura: tipoFactura,
        fecha: form.elements.fecha.value,
        comprobante_nro: form.elements.comprobante_nro.value,
        descripcion: `Factura ${tipoFactura} - ${form.elements.descripcion.value}`,
        monto: form.elements.monto.value,
        iva_tipo: form.elements.iva_tipo.value,
        condicion_operacion: form.elements.condicion_operacion.value,
        timbrado: form.elements.timbrado.value,
        tipo_comprobante: form.elements.tipo_comprobante.value,
        es_electronica: form.elements.es_electronica.checked,
        ruc_cliente_proveedor: form.elements.ruc_cliente_proveedor.value,
        moneda: form.elements.moneda.value,
        cuenta_pago_codigo: form.elements.cuenta_pago_codigo.value,
        cuenta_ingreso_codigo: tipoFactura === 'venta' ? form.elements.cuenta_ingreso_codigo.value : null,
        cuenta_gasto_codigo: tipoFactura === 'compra' ? form.elements.cuenta_gasto_codigo.value : null,
    };

    if (asientoId) {
        data.asiento_id = asientoId;
    }

    try {
        const response = await fetch(`../api.php?action=${action}`, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify(data)
        });
        if (!response.ok) {
            const err = await response.json();
            throw new Error(err.error || `Error al guardar ${tipoFactura}`);
        }
        await response.json();
        alert(`${tipoFactura.charAt(0).toUpperCase() + tipoFactura.slice(1)} guardada con éxito.`);
        form.reset();
        cancelarEdicionAsiento(tipoFactura);
        cargarYRenderizarAsientos();
    } catch (error) {
        console.error('Error:', error);
        alert(`No se pudo guardar la ${tipoFactura}.\n${error.message}`);
    }
}

// --- FUNCIONES API (EMPRESAS) ---
async function cargarEmpresas() {
    try {
        const response = await fetch('../api.php?action=get_empresas');
        const empresas = await response.json();
        state.empresas = empresas;
        const lista = document.getElementById('lista-empresas');
        lista.innerHTML = '';
        if (empresas.length === 0) {
            lista.innerHTML = '<p class="text-muted">No hay empresas creadas.</p>';
            return;
        }
        empresas.forEach(empresa => {
            const item = document.createElement('div');
            item.className = 'list-group-item d-flex justify-content-between align-items-center';
            
            const editButtonHTML = state.userCategory !== 'Digitador' ? `
                <button class="btn btn-sm btn-warning me-2 btn-editar-empresa-lista" data-id="${empresa.id}">
                    Editar
                </button>
                <button class="btn btn-sm btn-danger me-2 btn-eliminar-empresa-lista" data-id="${empresa.id}">
                    Eliminar
                </button>
            ` : '';

            item.innerHTML = `
                <div>
                    <h5 class="mb-1">${empresa.nombre}</h5>
                    <small class="text-muted">RUC: ${empresa.ruc}</small>
                </div>
                <div>
                    ${editButtonHTML}
                    <button class="btn btn-sm btn-primary btn-seleccionar-empresa" data-id="${empresa.id}">
                        Seleccionar
                    </button>
                </div>
            `;
            lista.appendChild(item);
        });
    } catch (error) {
        console.error('Error al cargar empresas:', error);
    }
}

async function cargarDigitadoresParaSelect(selectId = 'digitadores') {
    try {
        const response = await fetch('../api.php?action=get_digitadores');
        if (!response.ok) throw new Error('No se pudieron cargar los digitadores.');
        const digitadores = await response.json();
        const select = document.getElementById(selectId);
        if (select) {
            select.innerHTML = digitadores.map(d => `<option value="${d.id}">${d.nombrecompleto}</option>`).join('');
        }
    } catch (error) {
        console.error('Error al cargar digitadores:', error);
    }
}

async function cargarContadoresParaSelect(selectId = 'contador') {
    try {
        const response = await fetch('../api.php?action=get_contadores');
        if (!response.ok) throw new Error('No se pudieron cargar los contadores.');
        const contadores = await response.json();
        const select = document.getElementById(selectId);
        if (select) {
            select.innerHTML = '<option value="">-- Seleccione un Contador --</option>'; // Opción por defecto
            select.innerHTML += contadores.map(c => `<option value="${c.id}">${c.nombrecompleto}</option>`).join('');
        }
    } catch (error) {
        console.error('Error al cargar contadores:', error);
    }
}

function handleNewEmpresa() {
    const form = document.getElementById('form-nueva-empresa');
    form.reset();
    document.getElementById('empresa-id').value = ''; // Asegurarse que el ID esté vacío
    document.getElementById('modalEmpresaLabel').textContent = 'Crear Nueva Empresa';
    document.getElementById('btn-guardar-empresa').textContent = 'Crear Empresa';
    document.getElementById('logo-preview').innerHTML = '';
    document.getElementById('timbrados-container').innerHTML = '';
    cargarContadoresParaSelect();
    cargarDigitadoresParaSelect();
    const modal = new bootstrap.Modal(document.getElementById('modal-empresa'));
    modal.show();
}

async function saveEmpresa(event) {
    event.preventDefault();
    const form = event.target;
    const formData = new FormData(form);
    const empresaId = formData.get('id');
    const action = empresaId ? 'update_empresa' : 'create_empresa';

    // Recolectar datos de timbrados dinámicos
    const timbrados = [];
    document.querySelectorAll('#timbrados-container .timbrado-row').forEach((row) => {
        const numero = row.querySelector('[name$="[numero]"]').value;
        const inicio = row.querySelector('[name$="[inicio]"]').value;
        const fin = row.querySelector('[name$="[fin]"]').value;
        if (numero && inicio && fin) {
            timbrados.push({ numero, inicio, fin });
        }
    });
    formData.append('timbrados_json', JSON.stringify(timbrados));

    try {
        const response = await fetch(`../api.php?action=${action}`, {
            method: 'POST',
            body: formData
        });
        const result = await response.json();
        if (!response.ok) throw new Error(result.error || 'Error al guardar la empresa');
        
        // Cerrar el modal y recargar
        const modal = bootstrap.Modal.getInstance(document.getElementById('modal-empresa'));
        modal.hide();

        if (state.currentView === 'empresas') {
            cargarEmpresas();
        } else if (state.currentView === 'dashboard' && empresaId) {
            state.currentEmpresa = result; // Actualizar la empresa en el estado
            render(); // Re-renderizar el dashboard con la info actualizada
        }
    } catch (error) {
        console.error('Error:', error);
        alert(`No se pudo guardar la empresa.\n${error.message}`);
    }
}
async function seleccionarEmpresa(empresaId) {
    if (!empresaId) return;

    const empresa = state.empresas.find(e => e.id === empresaId);
    if (empresa) {
        state.currentView = 'dashboard';
        state.currentEmpresa = empresa;

        // 1. Cargar el plan de cuentas, que es esencial para los formularios de asientos.
        await cargarPlanDeCuentas();

        // 2. Renderizar la estructura principal del dashboard, pasándole el plan de cuentas.
        render();

        // 3. Adjuntar todos los listeners necesarios para la pestaña de asientos.
        document.getElementById('form-nueva-venta').addEventListener('submit', (e) => crearAsiento(e, 'venta'));
        document.getElementById('form-nueva-compra').addEventListener('submit', (e) => crearAsiento(e, 'compra'));
        document.getElementById('btn-cancelar-edicion-venta').addEventListener('click', () => cancelarEdicionAsiento('venta'));
        document.getElementById('btn-cancelar-edicion-compra').addEventListener('click', () => cancelarEdicionAsiento('compra'));
        document.getElementById('btn-filtrar-asientos').addEventListener('click', cargarYRenderizarAsientos);
        document.getElementById('btn-limpiar-filtros-asientos').addEventListener('click', limpiarFiltrosYRecargarAsientos);
        attachExportListeners();
        crearModalAsientoManual();
        cargarYRenderizarAsientos();
    }
}

function handleEditEmpresa(empresaId) {
    const empresa = state.empresas.find(e => e.id === empresaId);
    if (!empresa) return;

    // Cambiar título y botón del modal
    document.getElementById('modalEmpresaLabel').textContent = 'Editar Empresa';
    document.getElementById('btn-guardar-empresa').textContent = 'Guardar Cambios';

    // Rellenar el formulario del modal de edición
    document.getElementById('empresa-id').value = empresa.id;
    document.getElementById('nombre').value = empresa.nombre;
    document.getElementById('ruc').value = empresa.ruc;
    document.getElementById('tipo').value = empresa.tipo;
    document.getElementById('responsable_contabilidad').value = empresa.responsable_contabilidad || '';
    document.getElementById('telefono').value = empresa.telefono || '';
    document.getElementById('direccion').value = empresa.direccion || '';
    document.getElementById('ciudad').value = empresa.ciudad || '';
    document.getElementById('pais').value = empresa.pais || '';
    document.getElementById('actividad_economica').value = empresa.actividad_economica || '';
    document.getElementById('observaciones').value = empresa.observaciones || '';

    const logoPreview = document.getElementById('logo-preview');
    logoPreview.innerHTML = empresa.logo ? `<img src="../${empresa.logo}" alt="Logo Actual" style="max-height: 60px; border-radius: 4px;">` : 'No hay logo actual.';

    // Rellenar timbrados
    const timbradosContainer = document.getElementById('timbrados-container');
    timbradosContainer.innerHTML = '';
    if (empresa.timbrados && Array.isArray(empresa.timbrados)) {
        empresa.timbrados.forEach(timbrado => addTimbradoRow(timbrado));
    }

    // Rellenar presentaciones
    const form = document.getElementById('form-nueva-empresa');
    if (empresa.presentaciones) {
        form.querySelector('[name="presentaciones[iva_dia]"]').value = empresa.presentaciones.iva_dia || '';
        form.querySelector('[name="presentaciones[irp_dia]"]').value = empresa.presentaciones.irp_dia || '';
        form.querySelector('[name="presentaciones[irp_mes]"]').value = empresa.presentaciones.irp_mes || '1';
        form.querySelector('[name="presentaciones[irp_periodo]"]').value = empresa.presentaciones.irp_periodo || 'mes';
        form.querySelector('[name="presentaciones[ire_dia]"]').value = empresa.presentaciones.ire_dia || '';
        form.querySelector('[name="presentaciones[ire_mes]"]').value = empresa.presentaciones.ire_mes || '1';
        form.querySelector('[name="presentaciones[ire_periodo]"]').value = empresa.presentaciones.ire_periodo || 'mes';
        form.querySelector('[name="presentaciones[inf_dia]"]').value = empresa.presentaciones.inf_dia || '';
        form.querySelector('[name="presentaciones[inf_mes]"]').value = empresa.presentaciones.inf_mes || '1';
    }

    cargarContadoresParaSelect().then(() => {
        const contadorSelect = document.getElementById('contador');
        if (empresa.contador_id) contadorSelect.value = empresa.contador_id;
    });

    cargarDigitadoresParaSelect('digitadores').then(() => {
        const digitadoresSelect = document.getElementById('digitadores');
        (empresa.digitadores || []).forEach(digitadorId => {
            const option = digitadoresSelect.querySelector(`option[value="${digitadorId}"]`);
            if (option) option.selected = true;
        });
    });

    // Mostrar el modal unificado
    const modal = new bootstrap.Modal(document.getElementById('modal-empresa'));
    modal.show();
}

async function handleDeleteEmpresa(empresaId) {
    // Si se llama sin ID (desde el dashboard), usa el ID del estado actual.
    const idParaEliminar = empresaId || state.currentEmpresa.id;
    if (!idParaEliminar) return;

    // Busca el nombre de la empresa para mostrarlo en la confirmación.
    const empresa = state.empresas.find(e => e.id === idParaEliminar);
    const nombreEmpresa = empresa ? empresa.nombre : 'desconocida';


    if (confirm(`¿Está seguro de que desea eliminar la empresa "${nombreEmpresa}"? Esta acción no se puede deshacer.`)) {
        try {
            const response = await fetch('../api.php?action=delete_empresa', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ id: idParaEliminar })
            });
            if (!response.ok) throw new Error('Error al eliminar la empresa');
            await response.json();
            state.currentView = 'empresas';
            state.currentEmpresa = null;
            render();
        } catch (error) {
            console.error('Error:', error);
            alert('No se pudo eliminar la empresa.');
        }
    }
}


// --- FUNCIONES API (PLAN DE CUENTAS) ---
async function cargarPlanDeCuentas() {
    try {
        const response = await fetch(`../api.php?action=get_plan_de_cuentas&empresa_id=${state.currentEmpresa.id}`);
        if (!response.ok) throw new Error('Plan no encontrado');
        state.planDeCuentas = await response.json();
        if (document.getElementById('tabla-plan-de-cuentas')) {
            renderPlanDeCuentas();
        }
        return state.planDeCuentas;
    } catch (error) {
        console.error('Error cargando plan de cuentas:', error);
        return [];
    }
}

function attachExportListeners() {
    document.getElementById('btn-export-ventas-csv').addEventListener('click', () => exportarAsientos('ventas', 'csv'));
    document.getElementById('btn-export-ventas-excel').addEventListener('click', () => exportarAsientos('ventas', 'excel'));
    document.getElementById('btn-export-compras-csv').addEventListener('click', () => exportarAsientos('compras', 'csv'));
    document.getElementById('btn-export-compras-excel').addEventListener('click', () => exportarAsientos('compras', 'excel'));
}

function renderPlanDeCuentas() {
    const tablaBody = document.getElementById('tabla-plan-de-cuentas');
    if (!tablaBody) return; // Salir si la tabla no está en el DOM
    tablaBody.innerHTML = '';
    state.planDeCuentas.forEach(cuenta => {
        const tr = document.createElement('tr');
        if(cuenta.tipo === 'titulo') {
            tr.classList.add('table-light', 'fw-bold');
            tr.innerHTML = `<td colspan="3">${cuenta.nombre}</td>`;
        } else {
            tr.innerHTML = `
                <td>${cuenta.codigo}</td>
                <td>${cuenta.nombre}</td>
                <td>
                    <button class="btn btn-sm btn-warning btn-edit-cuenta" data-codigo="${cuenta.codigo}">Editar</button>
                    <button class="btn btn-sm btn-danger btn-delete-cuenta" data-codigo="${cuenta.codigo}">Eliminar</button>
                </td>
            `;
        }
        tablaBody.appendChild(tr);
    });
    document.getElementById('btn-imprimir-plan').addEventListener('click', imprimirPlanDeCuentas);
}

function handlePlanCuentasActions(event) {
    const target = event.target;
    if (target.classList.contains('btn-edit-cuenta')) {
        prepararEdicionCuenta(target.dataset.codigo);
    } else if (target.classList.contains('btn-delete-cuenta')) {
        eliminarCuenta(target.dataset.codigo);
    }
}

function prepararEdicionCuenta(codigo) {
    const cuenta = state.planDeCuentas.find(c => c.codigo === codigo);
    if (cuenta) {
        document.getElementById('form-cuenta-titulo').innerText = 'Editar Cuenta';
        document.getElementById('cuenta-original-codigo').value = cuenta.codigo;
        document.getElementById('cuenta-codigo').value = cuenta.codigo;
        document.getElementById('cuenta-codigo').readOnly = true; // No permitir editar el código
        document.getElementById('cuenta-nombre').value = cuenta.nombre;
        document.getElementById('btn-cancelar-edicion').classList.remove('d-none');
    }
}

function cancelarEdicion() {
    document.getElementById('form-cuenta-titulo').innerText = 'Añadir Cuenta';
    document.getElementById('form-nueva-cuenta').reset();
    document.getElementById('cuenta-original-codigo').value = '';
    document.getElementById('cuenta-codigo').readOnly = false;
    document.getElementById('btn-cancelar-edicion').classList.add('d-none');
}

async function eliminarCuenta(codigo) {
    if (confirm(`¿Está seguro de que desea eliminar la cuenta ${codigo}?`)) {
        state.planDeCuentas = state.planDeCuentas.filter(c => c.codigo !== codigo);
        await guardarPlanDeCuentas();
        renderPlanDeCuentas();
    }
}

async function guardarCuenta(event) {
    event.preventDefault();
    const form = event.target;
    const originalCodigo = document.getElementById('cuenta-original-codigo').value;
    const codigo = document.getElementById('cuenta-codigo').value;
    const nombre = document.getElementById('cuenta-nombre').value;

    if (originalCodigo) { // Editando
        const cuenta = state.planDeCuentas.find(c => c.codigo === originalCodigo);
        if (cuenta) {
            cuenta.nombre = nombre;
        }
    } else { // Creando
        if (state.planDeCuentas.some(c => c.codigo === codigo)) {
            alert('El código de cuenta ya existe.');
            return;
        }
        const nuevaCuenta = { codigo, nombre, tipo: 'detalle' };
        state.planDeCuentas.push(nuevaCuenta);
    }
    
    state.planDeCuentas.sort((a, b) => a.codigo.localeCompare(b.codigo));
    
    await guardarPlanDeCuentas();
    renderPlanDeCuentas();
    cancelarEdicion();
}

async function guardarPlanDeCuentas() {
    try {
        const response = await fetch('../api.php?action=update_plan_de_cuentas', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                empresa_id: state.currentEmpresa.id, 
                cuentas: state.planDeCuentas 
            })
        });
        if (!response.ok) throw new Error('Error al guardar el plan de cuentas');
        await response.json();
    } catch (error) {
        console.error('Error:', error);
        alert('No se pudo guardar el plan de cuentas.');
        // Opcional: revertir cambios en el state si falla el guardado
    }
}


async function cargarSumasySaldos() {
    const contentDiv = document.getElementById('sumas-y-saldos-content');
    contentDiv.innerHTML = '<p>Cargando...</p>';

    const panel = document.getElementById('saldos-panel');
    const fechaDesde = panel.querySelector('.fecha-desde')?.value;
    const fechaHasta = panel.querySelector('.fecha-hasta')?.value;

    try {
        const { data, asientos } = await getProcessedMayorData(fechaDesde, fechaHasta);

        if (asientos.length === 0) {
            contentDiv.innerHTML = '<p class="text-muted">No hay asientos para mostrar en el período seleccionado.</p>';
            return;
        }

        const headerHTML = `
            <div class="card-header d-flex justify-content-between align-items-center">
                <h4 class="mb-0">Balance de Sumas y Saldos</h4>
                <button id="btn-imprimir-saldos" class="btn btn-sm btn-outline-primary">Imprimir Reporte</button>
            </div>
        `;

        let tableHTML = `
            <table class="table table-sm table-bordered">
                <thead class="table-light">
                    <tr>
                        <th rowspan="2">Código</th>
                        <th rowspan="2">Cuenta</th>
                        <th colspan="2" class="text-center">Sumas</th>
                        <th colspan="2" class="text-center">Saldos</th>
                    </tr>
                    <tr>
                        <th class="text-end">Debe</th>
                        <th class="text-end">Haber</th>
                        <th class="text-end">Deudor</th>
                        <th class="text-end">Acreedor</th>
                    </tr>
                </thead>
                <tbody>
        `;

        let granTotalDebe = 0;
        let granTotalHaber = 0;
        let granTotalDeudor = 0;
        let granTotalAcreedor = 0;

        for (const codigo in data) {
            const cuenta = data[codigo];
            if (cuenta.totalDebe > 0 || cuenta.totalHaber > 0) {
                const saldo = cuenta.totalDebe - cuenta.totalHaber;
                granTotalDebe += cuenta.totalDebe;
                granTotalHaber += cuenta.totalHaber;
                
                let saldoDeudor = saldo > 0 ? saldo : 0;
                let saldoAcreedor = saldo < 0 ? -saldo : 0;
                granTotalDeudor += saldoDeudor;
                granTotalAcreedor += saldoAcreedor;

                tableHTML += `
                    <tr>
                        <td>${codigo}</td>
                        <td>${cuenta.nombre}</td>
                        <td class="text-end">${formatNumber(cuenta.totalDebe)}</td>
                        <td class="text-end">${formatNumber(cuenta.totalHaber)}</td>
                        <td class="text-end">${saldoDeudor > 0 ? formatNumber(saldoDeudor) : ''}</td>
                        <td class="text-end">${saldoAcreedor > 0 ? formatNumber(saldoAcreedor) : ''}</td>
                    </tr>
                `;
            }
        }

        tableHTML += `
                </tbody>
                <tfoot class="table-dark fw-bold">
                    <tr>
                        <td colspan="2" class="text-end">TOTALES</td>
                        <td class="text-end">${formatNumber(granTotalDebe)}</td>
                        <td class="text-end">${formatNumber(granTotalHaber)}</td>
                        <td class="text-end">${formatNumber(granTotalDeudor)}</td>
                        <td class="text-end">${formatNumber(granTotalAcreedor)}</td>
                    </tr>
                </tfoot>
            </table>
        `;

        contentDiv.innerHTML = `<div class="card"><div class="card-body p-0">${tableHTML}</div></div>`;
        contentDiv.querySelector('.card').insertAdjacentHTML('afterbegin', headerHTML);

        document.getElementById('btn-imprimir-saldos').addEventListener('click', imprimirSumasySaldos);
    } catch (error) {
        contentDiv.innerHTML = `<p class="text-danger">${error.message}</p>`;
        console.error('Error al cargar sumas y saldos:', error);
    }
}

async function cargarEstadoResultados() {
    const contentDiv = document.getElementById('estado-resultados-content');
    contentDiv.innerHTML = '<p>Cargando...</p>';

    const panel = document.getElementById('resultados-panel');
    const fechaDesde = panel.querySelector('.fecha-desde')?.value;
    const fechaHasta = panel.querySelector('.fecha-hasta')?.value;

    try {
        const { data, asientos } = await getProcessedMayorData(fechaDesde, fechaHasta);

        if (asientos.length === 0) {
            contentDiv.innerHTML = '<p class="text-muted">No hay asientos para mostrar en el período seleccionado.</p>';
            return;
        }

        let totalIngresos = 0;
        let totalGastos = 0;
        let ingresosHTML = '';
        let gastosHTML = '';

        for (const codigo in data) {
            const cuenta = data[codigo];
            const saldo = cuenta.totalDebe - cuenta.totalHaber;

            if (codigo.startsWith('4') && saldo !== 0) { // Ingresos
                const saldoAcreedor = -saldo;
                totalIngresos += saldoAcreedor;
                ingresosHTML += `<div class="d-flex justify-content-between"><span>  (${codigo}) ${cuenta.nombre}</span><span>${formatNumber(saldoAcreedor)}</span></div>`;
            }

            if (codigo.startsWith('5') && saldo !== 0) { // Gastos
                const saldoDeudor = saldo;
                totalGastos += saldoDeudor;
                gastosHTML += `<div class="d-flex justify-content-between"><span>  (${codigo}) ${cuenta.nombre}</span><span>${formatNumber(saldoDeudor)}</span></div>`;
            }
        }

        const resultado = totalIngresos - totalGastos;

        let finalHTML = `
            <div class="card">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <h4 class="mb-0">Estado de Resultados</h4><div><button id="btn-analizar-resultados" class="btn btn-sm btn-outline-info me-2">Analizar con IA ✨</button><button id="btn-imprimir-resultados" class="btn btn-sm btn-outline-primary">Imprimir Reporte</button></div>
                </div>
                <div class="card-body">
                    <h5>Ingresos</h5>
                    ${ingresosHTML || '<p class="text-muted">No hay ingresos registrados.</p>'}
                    <div class="d-flex justify-content-between fw-bold border-top mt-2 pt-2">
                        <span>Total Ingresos</span>
                        <span>${formatNumber(totalIngresos)}</span>
                    </div>
                    <hr>
                    <h5>Gastos</h5>
                    ${gastosHTML || '<p class="text-muted">No hay gastos registrados.</p>'}
                    <div class="d-flex justify-content-between fw-bold border-top mt-2 pt-2">
                        <span>Total Gastos</span>
                        <span>${formatNumber(totalGastos)}</span>
                    </div>
                    <hr>
                    <div class="d-flex justify-content-between fs-5 fw-bold ${resultado >= 0 ? 'text-success' : 'text-danger'}">
                        <span>${resultado >= 0 ? 'Utilidad del Ejercicio' : 'Pérdida del Ejercicio'}</span>
                        <span>${formatNumber(resultado)}</span>
                    </div>
                </div>
            </div>
        `;

        contentDiv.innerHTML = finalHTML;
        // Attach listener to the new button
        document.getElementById('btn-analizar-resultados').addEventListener('click', () => {
            const datosParaIA = { totalIngresos, totalGastos, resultado, detalleIngresos: ingresosHTML, detalleGastos: gastosHTML };
            analizarReporteConIA('Estado de Resultados', datosParaIA);
        });
        document.getElementById('btn-imprimir-resultados').addEventListener('click', imprimirEstadoResultados);

    } catch (error) {
        contentDiv.innerHTML = `<p class="text-danger">${error.message}</p>`;
        console.error('Error al cargar estado de resultados:', error);
    }
}

async function imprimirEstadoResultados() {
    const panel = document.getElementById('resultados-panel');
    const fechaDesde = panel.querySelector('.fecha-desde')?.value;
    const fechaHasta = panel.querySelector('.fecha-hasta')?.value;

    let url = `../api.php?action=generar_reporte_estado_resultados&empresa_id=${state.currentEmpresa.id}`;
    if (fechaDesde) url += `&fecha_desde=${fechaDesde}`;
    if (fechaHasta) url += `&fecha_hasta=${fechaHasta}`;

    try {
        const response = await fetch(url);
        if (!response.ok) throw new Error('No se pudo generar el reporte.');
        const reportHTML = await response.text();

        const newTab = window.open();
        newTab.document.open();
        newTab.document.write(reportHTML);
        newTab.document.close();
        setTimeout(() => newTab.print(), 500); // Wait a bit for content to render before printing

    } catch (error) {
        alert(error.message);
        console.error('Error al imprimir el estado de resultados:', error);
    }
}

async function imprimirSumasySaldos() {
    const panel = document.getElementById('saldos-panel');
    const fechaDesde = panel.querySelector('.fecha-desde')?.value;
    const fechaHasta = panel.querySelector('.fecha-hasta')?.value;

    let url = `../api.php?action=generar_reporte_sumas_y_saldos&empresa_id=${state.currentEmpresa.id}`;
    if (fechaDesde) url += `&fecha_desde=${fechaDesde}`;
    if (fechaHasta) url += `&fecha_hasta=${fechaHasta}`;

    try {
        const response = await fetch(url);
        if (!response.ok) throw new Error('No se pudo generar el reporte de Sumas y Saldos.');
        const reportHTML = await response.text();

        const newTab = window.open();
        newTab.document.open();
        newTab.document.write(reportHTML);
        newTab.document.close();
        setTimeout(() => newTab.print(), 500);
    } catch (error) {
        alert(error.message);
        console.error('Error al imprimir Sumas y Saldos:', error);
    }
}

async function imprimirLibroDiario() {
    const panel = document.getElementById('diario-panel');
    const fechaDesde = panel.querySelector('.fecha-desde')?.value;
    const fechaHasta = panel.querySelector('.fecha-hasta')?.value;

    let url = `../api.php?action=generar_reporte_libro_diario&empresa_id=${state.currentEmpresa.id}`;
    if (fechaDesde) url += `&fecha_desde=${fechaDesde}`;
    if (fechaHasta) url += `&fecha_hasta=${fechaHasta}`;

    try {
        const response = await fetch(url);
        if (!response.ok) throw new Error('No se pudo generar el reporte del Libro Diario.');
        const reportHTML = await response.text();

        const newTab = window.open();
        newTab.document.open();
        newTab.document.write(reportHTML);
        newTab.document.close();
        setTimeout(() => newTab.print(), 500);
    } catch (error) {
        alert(error.message);
        console.error('Error al imprimir Libro Diario:', error);
    }
}

async function imprimirPlanDeCuentas() {
    let url = `../api.php?action=generar_reporte_plan_de_cuentas&empresa_id=${state.currentEmpresa.id}`;

    try {
        const response = await fetch(url);
        if (!response.ok) throw new Error('No se pudo generar el reporte del Plan de Cuentas.');
        const reportHTML = await response.text();

        const newTab = window.open();
        newTab.document.open();
        newTab.document.write(reportHTML);
        newTab.document.close();
        setTimeout(() => newTab.print(), 500);
    } catch (error) {
        alert(error.message);
        console.error('Error al imprimir Plan de Cuentas:', error);
    }
}

async function cargarBalanceGeneral() {
    const contentDiv = document.getElementById('balance-general-content');
    contentDiv.innerHTML = '<p>Cargando...</p>';

    const panel = document.getElementById('balance-panel');
    const fechaDesde = panel.querySelector('.fecha-desde')?.value;
    const fechaHasta = panel.querySelector('.fecha-hasta')?.value;

    try {
        const { data, asientos } = await getProcessedMayorData(fechaDesde, fechaHasta);

        if (asientos.length === 0) {
            contentDiv.innerHTML = '<p class="text-muted">No hay asientos para mostrar en el período seleccionado.</p>';
            return;
        }

        // 1. Calcular resultado del ejercicio
        let totalIngresos = 0;
        let totalGastos = 0;
        for (const codigo in data) {
            const cuenta = data[codigo];
            const saldo = cuenta.totalDebe - cuenta.totalHaber;
            if (codigo.startsWith('4')) { // Ingresos
                totalIngresos += -saldo;
            }
            if (codigo.startsWith('5')) { // Gastos
                totalGastos += saldo;
            }
        }
        const resultadoEjercicio = totalIngresos - totalGastos;

        // 2. Clasificar cuentas para el Balance
        let activosHTML = '';
        let pasivosHTML = '';
        let patrimonioHTML = '';
        let totalActivo = 0;
        let totalPasivo = 0;
        let totalPatrimonio = 0;

        for (const codigo in data) {
            const cuenta = data[codigo];
            const saldo = cuenta.totalDebe - cuenta.totalHaber;

            if (saldo === 0 && cuenta.tipo === 'detalle') continue;

            const clase = cuenta.tipo === 'titulo' ? 'fw-bold' : 'ps-4';
            
            if (codigo.startsWith('1')) { // Activos
                if (cuenta.tipo === 'detalle') totalActivo += saldo;
                activosHTML += `<div class="d-flex justify-content-between ${clase}"><span>(${codigo}) ${cuenta.nombre}</span><span>${formatNumber(saldo)}</span></div>`;
            } else if (codigo.startsWith('2')) { // Pasivos
                const montoPasivo = -saldo;
                if (cuenta.tipo === 'detalle') totalPasivo += montoPasivo;
                pasivosHTML += `<div class="d-flex justify-content-between ${clase}"><span>(${codigo}) ${cuenta.nombre}</span><span>${formatNumber(montoPasivo)}</span></div>`;
            } else if (codigo.startsWith('3')) { // Patrimonio
                const montoPatrimonio = -saldo;
                if (cuenta.tipo === 'detalle') totalPatrimonio += montoPatrimonio;
                patrimonioHTML += `<div class="d-flex justify-content-between ${clase}"><span>(${codigo}) ${cuenta.nombre}</span><span>${formatNumber(montoPatrimonio)}</span></div>`;
            }
        }
        
        // Añadir resultado del ejercicio al patrimonio
        totalPatrimonio += resultadoEjercicio;
        const resultadoClass = resultadoEjercicio >= 0 ? 'text-success' : 'text-danger';
        patrimonioHTML += `
            <div class="d-flex justify-content-between ps-4 ${resultadoClass}">
                <span>Resultado del Ejercicio</span>
                <span>${formatNumber(resultadoEjercicio)}</span>
            </div>`;

        const totalPasivoPatrimonio = totalPasivo + totalPatrimonio;

        // 3. Renderizar el Balance
        const finalHTML = `
            <div class="card">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <h4 class="mb-0">Balance General</h4><div><button id="btn-analizar-balance" class="btn btn-sm btn-outline-info me-2">Analizar con IA ✨</button><button id="btn-imprimir-balance" class="btn btn-sm btn-outline-primary">Imprimir Reporte</button></div>
                </div>
                <div class="card-body">
                    <div class="row">
                        <div class="col-md-6">
                            <h5>Activo</h5>
                            ${activosHTML}
                            <div class="d-flex justify-content-between fw-bold border-top mt-2 pt-2">
                                <span>Total Activo</span>
                                <span>${formatNumber(totalActivo)}</span>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <h5>Pasivo</h5>
                            ${pasivosHTML}
                            <div class="d-flex justify-content-between fw-bold border-top mt-2 pt-2">
                                <span>Total Pasivo</span>
                                <span>${formatNumber(totalPasivo)}</span>
                            </div>
                            <hr>
                            <h5>Patrimonio Neto</h5>
                            ${patrimonioHTML}
                            <div class="d-flex justify-content-between fw-bold border-top mt-2 pt-2">
                                <span>Total Patrimonio Neto</span>
                                <span>${formatNumber(totalPatrimonio)}</span>
                            </div>
                            <hr>
                             <div class="d-flex justify-content-between fw-bold fs-5 mt-3">
                                <span>Total Pasivo y Patrimonio</span>
                                <span>${formatNumber(totalPasivoPatrimonio)}</span>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        `;

        contentDiv.innerHTML = finalHTML;
        document.getElementById('btn-analizar-balance').addEventListener('click', () => {
            const datosParaIA = { totalActivo, totalPasivo, totalPatrimonio, resultadoEjercicio, totalPasivoPatrimonio };
            analizarReporteConIA('Balance General', datosParaIA);
        });
        document.getElementById('btn-imprimir-balance').addEventListener('click', imprimirBalanceGeneral);

    } catch (error) {
        contentDiv.innerHTML = `<p class="text-danger">${error.message}</p>`;
        console.error('Error al cargar el Balance General:', error);
    }
}

async function imprimirBalanceGeneral() {
    const panel = document.getElementById('balance-panel');
    const fechaDesde = panel.querySelector('.fecha-desde')?.value;
    const fechaHasta = panel.querySelector('.fecha-hasta')?.value;

    let url = `../api.php?action=generar_reporte_balance_general&empresa_id=${state.currentEmpresa.id}`;
    if (fechaDesde) url += `&fecha_desde=${fechaDesde}`;
    if (fechaHasta) url += `&fecha_hasta=${fechaHasta}`;

    try {
        const response = await fetch(url);
        if (!response.ok) throw new Error('No se pudo generar el reporte.');
        const reportHTML = await response.text();

        const newTab = window.open();
        newTab.document.open();
        newTab.document.write(reportHTML);
        newTab.document.close();
        setTimeout(() => newTab.print(), 500); // Wait a bit for content to render before printing

    } catch (error) {
        alert(error.message);
        console.error('Error al imprimir el balance general:', error);
    }
}

async function cargarReporteIVA() {
    const contentDiv = document.getElementById('iva-panel');
    // No limpiar el filtro, solo el contenido del reporte
    const reportContent = contentDiv.querySelector('#reporte-iva-content') || document.createElement('div');
    if (!reportContent.id) {
        reportContent.id = 'reporte-iva-content';
        contentDiv.appendChild(reportContent);
    }
    reportContent.innerHTML = '<p>Calculando IVA...</p>';

    const fechaDesde = contentDiv.querySelector('.fecha-desde')?.value;
    const fechaHasta = contentDiv.querySelector('.fecha-hasta')?.value;

    let url = `../api.php?action=get_iva_report_data&empresa_id=${state.currentEmpresa.id}`;
    if (fechaDesde) url += `&fecha_desde=${fechaDesde}`;
    if (fechaHasta) url += `&fecha_hasta=${fechaHasta}`;

    try {
        const response = await fetch(url);
        if (!response.ok) throw new Error('No se pudo calcular el reporte de IVA.');
        const data = await response.json();

        const totalDebito = data.ventas['10'].iva + data.ventas['5'].iva + data.ventas['0'].iva;
        const totalCredito = data.compras['10'].iva + data.compras['5'].iva + data.compras['0'].iva;
        const resultado = totalDebito - totalCredito;
        const resultadoTexto = resultado >= 0 ? 'IVA a Pagar' : 'Saldo a Favor del Contribuyente';
        const resultadoClase = resultado >= 0 ? 'text-danger' : 'text-success';

        const renderTable = (title, tableData) => {
            let totalCount = 0, totalTotal = 0, totalBase = 0, totalIva = 0;
            const rows = Object.entries(tableData).map(([tipo, valores]) => {
                if (valores.count === 0) return '';
                totalCount += valores.count;
                totalTotal += valores.total;
                totalBase += valores.base;
                totalIva += valores.iva;
                return `<tr>
                    <td>IVA ${tipo}%</td>
                    <td class="text-center">${valores.count}</td>
                    <td class="text-end">${formatNumber(valores.total)}</td>
                    <td class="text-end">${formatNumber(valores.base)}</td>
                    <td class="text-end">${formatNumber(valores.iva)}</td>
                </tr>`;
            }).join('');

            return `
                <h5 class="mt-4">${title}</h5>
                <table class="table table-sm table-bordered">
                    <thead class="table-light"><tr><th>Tipo</th><th class="text-center">Cant. Fact.</th><th class="text-end">Total Fact.</th><th class="text-end">Base Imp.</th><th class="text-end">Total IVA</th></tr></thead>
                    <tbody>${rows || '<tr><td colspan="5" class="text-center text-muted">Sin datos</td></tr>'}</tbody>
                    <tfoot class="fw-bold">
                        <tr>
                            <td>Total</td>
                            <td class="text-center">${totalCount}</td>
                            <td class="text-end">${formatNumber(totalTotal)}</td>
                            <td class="text-end">${formatNumber(totalBase)}</td>
                            <td class="text-end">${formatNumber(totalIva)}</td>
                        </tr>
                    </tfoot>
                </table>
            `;
        };

        const finalHTML = `
            <div class="card mt-3">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <h4 class="mb-0">Reporte Detallado de IVA</h4>
                    <button id="btn-imprimir-iva" class="btn btn-sm btn-outline-primary">Imprimir Reporte</button>
                </div>
                <div class="card-body">
                    ${renderTable('IVA Débito Fiscal (Ventas)', data.ventas)}
                    ${renderTable('IVA Crédito Fiscal (Compras)', data.compras)}

                    <hr class="my-4">
                    <h5 class="text-center">Resumen de Liquidación</h5>
                    <div class="col-md-6 offset-md-3">
                        <ul class="list-group">
                            <li class="list-group-item d-flex justify-content-between align-items-center">(+) Total IVA Débito <span class="badge bg-primary rounded-pill fs-6">${formatNumber(totalDebito)}</span></li>
                            <li class="list-group-item d-flex justify-content-between align-items-center">(-) Total IVA Crédito <span class="badge bg-info rounded-pill fs-6">${formatNumber(totalCredito)}</span></li>
                            <li class="list-group-item d-flex justify-content-between align-items-center fw-bold ${resultadoClase}">
                                (=) ${resultadoTexto}
                                <span class="badge bg-light text-dark rounded-pill fs-5">${formatNumber(Math.abs(resultado))}</span>
                            </li>
                        </ul>
                    </div>
                </div>
            </div>
        `;

        reportContent.innerHTML = finalHTML;
        document.getElementById('btn-imprimir-iva').addEventListener('click', imprimirReporteIVA);

    } catch (error) {
        reportContent.innerHTML = `<p class="text-danger">${error.message}</p>`;
        console.error('Error al cargar el reporte de IVA:', error);
    }
}

async function imprimirReporteIVA() {
    const panel = document.getElementById('iva-panel');
    const fechaDesde = panel.querySelector('.fecha-desde')?.value;
    const fechaHasta = panel.querySelector('.fecha-hasta')?.value;

    let url = `../api.php?action=generar_reporte_iva&empresa_id=${state.currentEmpresa.id}`;
    if (fechaDesde) url += `&fecha_desde=${fechaDesde}`;
    if (fechaHasta) url += `&fecha_hasta=${fechaHasta}`;

    try {
        const response = await fetch(url);
        if (!response.ok) throw new Error('No se pudo generar el reporte de IVA.');
        const reportHTML = await response.text();
        const newTab = window.open();
        newTab.document.write(reportHTML);
        newTab.document.close();
        setTimeout(() => newTab.print(), 500);
    } catch (error) {
        alert(error.message);
    }
}

async function cargarRatiosFinancieros() {
    const contentDiv = document.getElementById('ratios-panel');
    contentDiv.innerHTML = '<p>Calculando ratios...</p>';

    const panel = document.getElementById('ratios-panel');
    const fechaDesde = panel.querySelector('.fecha-desde')?.value;
    const fechaHasta = panel.querySelector('.fecha-hasta')?.value;

    try {
        const { data, asientos } = await getProcessedMayorData(fechaDesde, fechaHasta);

        if (asientos.length === 0) {
            contentDiv.innerHTML = '<p class="text-muted">No hay datos suficientes para calcular los ratios en el período seleccionado.</p>';
            return;
        }

        // --- Cálculos ---
        let ventasNetas = 0;
        let costoVentas = 0;
        let beneficioNeto = 0;
        let activoCorriente = 0;
        let pasivoCorriente = 0;
        let activoTotal = 0;
        let deudaTotal = 0;
        let patrimonio = 0;
        let inventario = 0;
        let efectivoEquivalentes = 0;

        for (const codigo in data) {
            const cuenta = data[codigo];
            const saldo = cuenta.totalDebe - cuenta.totalHaber;

            if (codigo.startsWith('4')) ventasNetas += -saldo;
            if (codigo.startsWith('5.1')) costoVentas += saldo;
            if (codigo.startsWith('1.1')) activoCorriente += saldo;
            if (codigo.startsWith('2.1')) pasivoCorriente += -saldo;
            if (codigo.startsWith('1')) activoTotal += saldo;
            if (codigo.startsWith('2')) deudaTotal += -saldo;
            if (codigo.startsWith('3')) patrimonio += -saldo;
            if (codigo.startsWith('1.1.3')) inventario += saldo;
            if (codigo.startsWith('1.1.1')) efectivoEquivalentes += saldo;
        }

        beneficioNeto = ventasNetas - costoVentas - (data['5.2'] ? data['5.2'].totalDebe - data['5.2'].totalHaber : 0);
        const beneficioBruto = ventasNetas - costoVentas;
        patrimonio += beneficioNeto; // Incluir resultado del ejercicio en el patrimonio

        // --- Preparación de datos para gráficos ---
        const monthlyData = {};
        asientos.forEach(asiento => {
            const month = asiento.fecha.substring(0, 7); // YYYY-MM
            if (!monthlyData[month]) {
                monthlyData[month] = { income: 0, expense: 0 };
            }
            asiento.movimientos.forEach(mov => {
                if (mov.cuenta_codigo.startsWith('4')) monthlyData[month].income += parseFloat(mov.haber) || 0;
                if (mov.cuenta_codigo.startsWith('5')) monthlyData[month].expense += parseFloat(mov.debe) || 0;
            });
        });

        const sortedMonths = Object.keys(monthlyData).sort();
        const chartLabels = sortedMonths;
        const incomeData = sortedMonths.map(month => monthlyData[month].income);
        const expenseData = sortedMonths.map(month => monthlyData[month].expense);
        const debtEquityData = [deudaTotal, patrimonio];
        const liquidityData = [efectivoEquivalentes, (activoCorriente - inventario - efectivoEquivalentes), inventario];

        // --- Cálculo de Ratios ---
        const margenBruto = ventasNetas > 0 ? (beneficioBruto / ventasNetas) * 100 : 0;
        const margenNeto = ventasNetas > 0 ? (beneficioNeto / ventasNetas) * 100 : 0;
        const roa = activoTotal > 0 ? (beneficioNeto / activoTotal) * 100 : 0;
        const rotacionActivos = activoTotal > 0 ? ventasNetas / activoTotal : 0;
        const rotacionInventarios = inventario > 0 ? costoVentas / inventario : 0;
        const liquidezCorriente = pasivoCorriente > 0 ? activoCorriente / pasivoCorriente : 0;
        const pruebaAcida = pasivoCorriente > 0 ? (activoCorriente - inventario) / pasivoCorriente : 0;
        const endeudamiento = activoTotal > 0 ? (deudaTotal / activoTotal) * 100 : 0;
        const ratioPropiedad = activoTotal > 0 ? (patrimonio / activoTotal) * 100 : 0;

        // --- Renderizado ---
        const renderRatio = (titulo, valor, unidad, tooltip) => `
            <div class="col-md-4 mb-3">
                <div class="card h-100">
                    <div class="card-body">
                        <h6 class="card-title" data-bs-toggle="tooltip" title="${tooltip}">${titulo}</h6>
                        <p class="card-text fs-4">${valor.toFixed(2)} ${unidad}</p>
                    </div>
                </div>
            </div>
        `;

        const finalHTML = `
            <div class="card">
                <div class="card-header"><h4 class="mb-0">Ratios Financieros</h4></div>
                <div class="card-body">
                    <h5>Visualización Gráfica</h5>
                    <div class="row">
                        <div class="col-lg-8 mb-4 d-flex">
                            <div class="card w-100"><div class="card-body"><h6 class="card-title">Ingresos vs. Gastos (Mensual)</h6><div style="height: 300px;"><canvas id="incomeExpenseChart"></canvas></div></div></div>
                        </div>
                        <div class="col-lg-4 mb-4 d-flex">
                            <div class="card w-100"><div class="card-body"><h6 class="card-title">Estructura Financiera (Deuda vs. Patrimonio)</h6><div style="height: 300px;"><canvas id="debtEquityChart"></canvas></div></div></div>
                        </div>
                    </div>
                    <hr>
                    <h5>Rentabilidad y Gestión</h5>
                    <div class="row">
                        ${renderRatio('Margen Bruto', margenBruto, '%', 'Indica la rentabilidad directa del negocio tras cubrir el coste de los productos vendidos.')}
                        ${renderRatio('Margen Neto', margenNeto, '%', 'Muestra la rentabilidad después de deducir todos los gastos.')}
                        ${renderRatio('ROA (Retorno sobre Activos)', roa, '%', 'Mide qué tan eficazmente la empresa utiliza sus activos para generar beneficios.')}
                        ${renderRatio('Rotación de Activos', rotacionActivos, 'veces', 'Indica la eficiencia con la que la empresa utiliza sus activos para generar ventas.')}
                        ${renderRatio('Rotación de Inventarios', rotacionInventarios, 'veces', 'Muestra cuántas veces se renueva el inventario en un período.')}
                    </div>
                    <hr>
                    <h5>Liquidez y Endeudamiento</h5>
                    <div class="row">
                        ${renderRatio('Liquidez Corriente', liquidezCorriente, '', 'Compara activos corrientes con pasivos corrientes para medir la capacidad de pago a corto plazo.')}
                        ${renderRatio('Prueba Ácida', pruebaAcida, '', 'Similar a la liquidez corriente, pero excluyendo los inventarios por ser menos líquidos.')}
                        ${renderRatio('Ratio de Endeudamiento', endeudamiento, '%', 'Indica la proporción de la financiación que proviene de deuda sobre los activos totales.')}
                        ${renderRatio('Ratio de Propiedad', ratioPropiedad, '%', 'Muestra la proporción de los activos que pertenece a los accionistas.')}
                    </div>
                </div>
            </div>
        `;

        contentDiv.innerHTML = finalHTML;

        // Inicializar tooltips de Bootstrap
        const tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
        tooltipTriggerList.map(function (tooltipTriggerEl) {
            return new bootstrap.Tooltip(tooltipTriggerEl);
        });

        // --- Renderizar Gráficos ---
        new Chart(document.getElementById('incomeExpenseChart'), {
            type: 'bar',
            data: {
                labels: chartLabels,
                datasets: [
                    {
                        label: 'Ingresos',
                        data: incomeData,
                        backgroundColor: 'rgba(25, 135, 84, 0.6)',
                        borderColor: 'rgba(25, 135, 84, 1)',
                        borderWidth: 1
                    },
                    {
                        label: 'Gastos',
                        data: expenseData,
                        backgroundColor: 'rgba(220, 53, 69, 0.6)',
                        borderColor: 'rgba(220, 53, 69, 1)',
                        borderWidth: 1
                    }
                ]
            },
            options: {
                scales: {
                    y: { beginAtZero: true }
                },
                responsive: true,
                maintainAspectRatio: false
            }
        });

        new Chart(document.getElementById('debtEquityChart'), {
            type: 'doughnut',
            data: {
                labels: ['Deuda Total', 'Patrimonio Neto'],
                datasets: [{
                    label: 'Estructura Financiera',
                    data: debtEquityData,
                    backgroundColor: [
                        'rgba(220, 53, 69, 0.7)',
                        'rgba(25, 135, 84, 0.7)'
                    ],
                    hoverOffset: 4
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false
            }
        });

    } catch (error) {
        contentDiv.innerHTML = `<p class="text-danger">${error.message}</p>`;
        console.error('Error al cargar los ratios financieros:', error);
    }
}

async function cargarActividadLog(contentDiv) {
    contentDiv.innerHTML = '<p>Cargando registro de actividad...</p>';
    try {
        const response = await fetch('../api.php?action=get_activity_log');
        if (!response.ok) {
            const err = await response.json();
            throw new Error(err.error || 'No se pudo cargar el registro.');
        }
        const logs = await response.json();

        if (logs.length === 0) {
            contentDiv.innerHTML = '<p class="text-muted">No hay actividades registradas.</p>';
            return;
        }

        let tableHTML = `
            <div class="card">
                <div class="card-header"><h4 class="mb-0">Registro de Actividad</h4></div>
                <div class="card-body">
                    <table class="table table-sm table-striped">
                        <thead><tr><th>Fecha y Hora</th><th>Usuario</th><th>Acción</th></tr></thead>
                        <tbody>
                            ${logs.map(log => `
                                <tr><td>${log.timestamp}</td><td>${log.user}</td><td>${log.action}</td></tr>
                            `).join('')}
                        </tbody>
                    </table>
                </div>
            </div>`;
        contentDiv.innerHTML = tableHTML;
    } catch (error) {
        contentDiv.innerHTML = `<p class="text-danger">${error.message}</p>`;
        console.error('Error al cargar el registro de actividad:', error);
    }
}

async function analizarReporteConIA(tipoReporte, datos) {
    // Crear y mostrar un modal de Bootstrap para el análisis
    let modal = document.getElementById('ai-analysis-modal');
    if (!modal) {
        const modalHTML = `
            <div class="modal fade" id="ai-analysis-modal" tabindex="-1" aria-labelledby="aiAnalysisModalLabel" aria-hidden="true">
                <div class="modal-dialog modal-lg">
                    <div class="modal-content">
                        <div class="modal-header">
                            <h5 class="modal-title" id="aiAnalysisModalLabel">Análisis con IA: ${tipoReporte}</h5>
                            <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                        </div>
                        <div class="modal-body" id="ai-analysis-content">
                            <p class="text-center">🧠 Analizando datos, por favor espera...</p>
                            <div class="d-flex justify-content-center"><div class="spinner-border" role="status"><span class="visually-hidden">Loading...</span></div></div>
                        </div>
                        <div class="modal-footer">
                            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cerrar</button>
                        </div>
                    </div>
                </div>
            </div>`;
        document.body.insertAdjacentHTML('beforeend', modalHTML);
        modal = document.getElementById('ai-analysis-modal');
    }

    const modalInstance = new bootstrap.Modal(modal);
    modalInstance.show();
    
    try {
        const response = await fetch('../api.php?action=analizar_con_ia', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ tipo: tipoReporte, datos: JSON.stringify(datos, null, 2) })
        });
        const result = await response.json();
        document.getElementById('ai-analysis-content').innerHTML = result.analisis.replace(/\n/g, '<br>'); // Simple markdown conversion
    } catch (error) {
        document.getElementById('ai-analysis-content').innerHTML = `<div class="alert alert-danger">Error al realizar el análisis: ${error.message}</div>`;
    }
}

function crearModalAsientoManual() {
    if (document.getElementById('asiento-manual-modal')) return;

    const modalHTML = `
        <div class="modal fade" id="asiento-manual-modal" tabindex="-1" aria-labelledby="asientoManualModalLabel" aria-hidden="true">
            <div class="modal-dialog modal-xl">
                <div class="modal-content">
                    <div class="modal-header">
                        <h5 class="modal-title" id="asientoManualModalLabel">Editor de Asiento Manual</h5>
                        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                    </div>
                    <div class="modal-body">
                        <form id="form-asiento-manual">
                            <input type="hidden" id="manual-asiento-id">
                            <div class="row mb-3">
                                <div class="col-md-3"><label class="form-label">Fecha</label><input type="date" id="manual-asiento-fecha" class="form-control" required></div>
                                <div class="col-md-6"><label class="form-label">Descripción General</label><input type="text" id="manual-asiento-descripcion" class="form-control" required></div>
                                <div class="col-md-3"><label class="form-label">Nro. Comprobante</label><input type="text" id="manual-asiento-comprobante" class="form-control"></div>
                            </div>
                            
                            <table class="table table-sm">
                                <thead><tr><th>Cuenta</th><th>Descripción</th><th class="text-end">Debe</th><th class="text-end">Haber</th><th></th></tr></thead>
                                <tbody id="movimientos-body">
                                    <!-- Las filas de movimientos se insertarán aquí -->
                                </tbody>
                                <tfoot>
                                    <tr class="table-light">
                                        <td colspan="2" class="text-end fw-bold">TOTALES</td>
                                        <td class="text-end fw-bold" id="total-debe">0,00</td>
                                        <td class="text-end fw-bold" id="total-haber">0,00</td>
                                        <td><button type="button" class="btn btn-sm btn-success py-0 px-1" id="btn-add-movimiento">+</button></td>
                                    </tr>
                                </tfoot>
                            </table>
                            <div id="balance-alert" class="alert alert-danger d-none" role="alert">El asiento no está balanceado.</div>
                        </form>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                        <button type="button" class="btn btn-primary" id="btn-guardar-asiento-manual">Guardar Asiento</button>
                    </div>
                </div>
            </div>
        </div>
        <template id="movimiento-row-template">
            <tr>
                <td><select class="form-select form-select-sm cuenta-codigo"></select></td>
                <td><input type="text" class="form-control form-control-sm mov-descripcion"></td>
                <td><input type="number" step="0.01" class="form-control form-control-sm text-end mov-debe" value="0"></td>
                <td><input type="number" step="0.01" class="form-control form-control-sm text-end mov-haber" value="0"></td>
                <td><button type="button" class="btn btn-sm btn-danger py-0 px-1 btn-delete-movimiento">X</button></td>
            </tr>
        </template>
    `;
    document.body.insertAdjacentHTML('beforeend', modalHTML);

    // Listeners
    document.getElementById('btn-add-movimiento').addEventListener('click', addMovimientoRow);
    document.getElementById('movimientos-body').addEventListener('click', e => {
        if (e.target.classList.contains('btn-delete-movimiento')) {
            e.target.closest('tr').remove();
            updateMovimientoTotals();
        }
    });
    document.getElementById('movimientos-body').addEventListener('change', e => {
        if (e.target.classList.contains('cuenta-codigo')) {
            const selectedOption = e.target.options[e.target.selectedIndex];
            const cuentaNombre = selectedOption.text.replace(/^\(.*\)\s*/, ''); // Extraer solo el nombre
            e.target.closest('tr').querySelector('.mov-descripcion').value = cuentaNombre;
        }
    });
    document.getElementById('movimientos-body').addEventListener('change', updateMovimientoTotals);
    document.getElementById('btn-guardar-asiento-manual').addEventListener('click', guardarAsientoManual);
}

function addMovimientoRow(movimiento = null, cuentasOptions = '') {
    const template = document.getElementById('movimiento-row-template');
    const newRow = template.content.cloneNode(true);
    const selectCuenta = newRow.querySelector('.cuenta-codigo');
    
    // (Corrección Clave) Poblar el select con las opciones de cuentas
    selectCuenta.innerHTML = cuentasOptions;

    if (movimiento) {
        selectCuenta.value = movimiento.cuenta_codigo;
        newRow.querySelector('.mov-descripcion').value = movimiento.descripcion;
        newRow.querySelector('.mov-debe').value = movimiento.debe;
        newRow.querySelector('.mov-haber').value = movimiento.haber;
    }
    document.getElementById('movimientos-body').appendChild(newRow);
}

function updateMovimientoTotals() {
    const rows = document.querySelectorAll('#movimientos-body tr');
    let totalDebe = 0;
    let totalHaber = 0;
    rows.forEach(row => {
        totalDebe += parseFloat(row.querySelector('.mov-debe').value) || 0;
        totalHaber += parseFloat(row.querySelector('.mov-haber').value) || 0;
    });

    document.getElementById('total-debe').textContent = formatNumber(totalDebe);
    document.getElementById('total-haber').textContent = formatNumber(totalHaber);

    const balanceAlert = document.getElementById('balance-alert');
    const saveButton = document.getElementById('btn-guardar-asiento-manual');
    if (Math.abs(totalDebe - totalHaber) > 0.001) {
        balanceAlert.classList.remove('d-none');
        saveButton.disabled = true;
    } else {
        balanceAlert.classList.add('d-none');
        saveButton.disabled = false;
    }
}

async function abrirEditorAsientoManual(asientoId = null) {
    // 1. Asegurarse de que el modal exista en el DOM. Si no, lo crea.
    if (!document.getElementById('asiento-manual-modal')) {
        crearModalAsientoManual();
    }

    // 2. (Corrección Clave) Asegurarse de que el plan de cuentas esté cargado.
    if (state.planDeCuentas.length === 0) {
        await cargarPlanDeCuentas();
    }

    // 3. Generar las opciones de cuentas AHORA, con los datos más recientes.
    const cuentasOptions = state.planDeCuentas
        .filter(c => c.tipo === 'detalle')
        .map(c => `<option value="${c.codigo}">(${c.codigo}) ${c.nombre}</option>`)
        .join('');

    // 4. Limpiar y preparar el formulario.
    const form = document.getElementById('form-asiento-manual');
    form.reset();
    document.getElementById('movimientos-body').innerHTML = '';

    if (asientoId) { // Editando
        const asiento = state.asientos.find(a => a.id === asientoId);
        document.getElementById('manual-asiento-id').value = asiento.id;
        document.getElementById('manual-asiento-fecha').value = asiento.fecha;
        document.getElementById('manual-asiento-descripcion').value = asiento.descripcion;
        document.getElementById('manual-asiento-comprobante').value = asiento.comprobante_nro || '';
        asiento.movimientos.forEach(mov => addMovimientoRow(mov, cuentasOptions));
    } else { // Creando
        document.getElementById('manual-asiento-id').value = '';
        document.getElementById('manual-asiento-fecha').valueAsDate = new Date();
        addMovimientoRow(null, cuentasOptions);
        addMovimientoRow(null, cuentasOptions);
    }

    // 5. Actualizar totales y mostrar el modal.
    updateMovimientoTotals();
    const modal = new bootstrap.Modal(document.getElementById('asiento-manual-modal'));
    modal.show();
}

async function guardarAsientoManual() {
    const asiento = {
        id: document.getElementById('manual-asiento-id').value,
        fecha: document.getElementById('manual-asiento-fecha').value,
        descripcion: document.getElementById('manual-asiento-descripcion').value,
        comprobante_nro: document.getElementById('manual-asiento-comprobante').value,
        movimientos: Array.from(document.querySelectorAll('#movimientos-body tr')).map(row => ({
            cuenta_codigo: row.querySelector('.cuenta-codigo').value,
            descripcion: row.querySelector('.mov-descripcion').value,
            debe: parseFloat(row.querySelector('.mov-debe').value) || 0,
            haber: parseFloat(row.querySelector('.mov-haber').value) || 0,
        }))
    };

    try {
        const response = await fetch('../api.php?action=save_manual_asiento', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ empresa_id: state.currentEmpresa.id, asiento: asiento })
        });
        const result = await response.json();
        if (!response.ok) throw new Error(result.error || 'Error al guardar el asiento.');
        
        bootstrap.Modal.getInstance(document.getElementById('asiento-manual-modal')).hide();
        cargarLibroDiario(); // Recargar la vista del diario
        cargarYRenderizarAsientos(); // Recargar la lista de asientos en la pestaña principal
    } catch (error) {
        alert(`Error: ${error.message}`);
    }
}
