<?php
/**
 * API Principal de Bridgi - Convertida a MySQL
 * Sistema de Contabilidad - Dunamis Group 2025
 */

session_start();
require_once 'db.php';

define('MAX_LOG_ENTRIES', 500);
define('UPLOAD_DIR', 'uploads/');

// --- FUNCIONES DE AYUDA ---

/**
 * Obtiene el plan de cuentas por defecto
 */
function get_plan_de_cuentas_default() {
    return [
        ["codigo" => "1", "nombre" => "ACTIVO", "tipo" => "titulo"],
        ["codigo" => "1.1", "nombre" => "ACTIVO CORRIENTE", "tipo" => "titulo"],
        ["codigo" => "1.1.1", "nombre" => "Caja y Bancos", "tipo" => "detalle"],
        ["codigo" => "1.1.2", "nombre" => "Cuentas por Cobrar", "tipo" => "detalle"],
        ["codigo" => "1.1.3", "nombre" => "Inventarios", "tipo" => "detalle"],
        ["codigo" => "1.1.4", "nombre" => "IVA Crédito Fiscal", "tipo" => "detalle"],
        ["codigo" => "2", "nombre" => "PASIVO", "tipo" => "titulo"],
        ["codigo" => "2.1", "nombre" => "PASIVO CORRIENTE", "tipo" => "titulo"],
        ["codigo" => "2.1.1", "nombre" => "Cuentas por Pagar", "tipo" => "detalle"],
        ["codigo" => "2.1.2", "nombre" => "Impuestos por Pagar", "tipo" => "detalle"],
        ["codigo" => "2.1.3", "nombre" => "IVA Débito Fiscal", "tipo" => "detalle"],
        ["codigo" => "3", "nombre" => "PATRIMONIO NETO", "tipo" => "titulo"],
        ["codigo" => "3.1", "nombre" => "Capital Social", "tipo" => "detalle"],
        ["codigo" => "4", "nombre" => "INGRESOS", "tipo" => "titulo"],
        ["codigo" => "4.1", "nombre" => "Venta de Mercaderías/Servicios", "tipo" => "detalle"],
        ["codigo" => "5", "nombre" => "GASTOS", "tipo" => "titulo"],
        ["codigo" => "5.1", "nombre" => "Costo de Ventas", "tipo" => "detalle"],
        ["codigo" => "5.2", "nombre" => "Gastos de Administración", "tipo" => "detalle"],
    ];
}

/**
 * Registra una actividad en el log
 */
function log_activity($message) {
    $pdo = getDB();
    $username = $_SESSION['username'] ?? 'Sistema';
    
    try {
        $stmt = $pdo->prepare("INSERT INTO activity_log (username, action) VALUES (?, ?)");
        $stmt->execute([$username, $message]);
        
        // Limpiar entradas antiguas
        $stmt = $pdo->prepare("DELETE FROM activity_log WHERE id NOT IN (SELECT id FROM (SELECT id FROM activity_log ORDER BY created_at DESC LIMIT ?) t)");
        $stmt->execute([MAX_LOG_ENTRIES]);
    } catch (PDOException $e) {
        error_log("Error logging activity: " . $e->getMessage());
    }
}

/**
 * Obtiene la configuración del sistema
 */
function get_config() {
    $pdo = getDB();
    $stmt = $pdo->query("SELECT clave, valor FROM configuracion");
    $config = [];
    while ($row = $stmt->fetch()) {
        $config[$row['clave']] = $row['valor'];
    }
    return $config;
}

/**
 * Guarda la configuración del sistema
 */
function save_config($key, $value) {
    $pdo = getDB();
    $stmt = $pdo->prepare("INSERT INTO configuracion (clave, valor) VALUES (?, ?) ON DUPLICATE KEY UPDATE valor = ?");
    $stmt->execute([$key, $value, $value]);
}

/**
 * Valida un ID de empresa
 */
function validate_empresa_id($empresa_id) {
    if (empty($empresa_id) || strpos($empresa_id, '.') !== false || strpos($empresa_id, '/') !== false) {
        http_response_code(400);
        die(json_encode(['error' => 'ID de empresa no válido']));
    }
    return $empresa_id;
}

/**
 * Llama a la API de IA
 */
function call_ai_api($prompt) {
    $config = get_config();
    $api_key = $config['ai_api_key'] ?? '';

    if (empty($api_key)) {
        return "Error: La API Key de la IA no ha sido configurada.";
    }

    $url = '';
    $headers = [];
    $data = [];
    $ai_response_path = '';
    $system_prompt_content = 'Eres un experto analista financiero y contable. Analiza los datos proporcionados y ofrece un resumen claro, conciso y útil para un gerente de negocios. Identifica tendencias, puntos clave y posibles áreas de mejora. Responde en español y usa formato markdown para una mejor legibilidad.';

    if (str_starts_with($api_key, 'sk-')) { // OpenAI
        $url = 'https://api.openai.com/v1/chat/completions';
        $headers = [
            'Content-Type: application/json',
            'Authorization: Bearer ' . $api_key
        ];
        $data = [
            'model' => 'gpt-3.5-turbo',
            'messages' => [
                ['role' => 'system', 'content' => $system_prompt_content],
                ['role' => 'user', 'content' => $prompt]
            ],
            'temperature' => 0.5,
        ];
        $ai_response_path = 'openai';
    } elseif (str_starts_with($api_key, 'AIzaSy')) { // Google AI
        $model = 'gemini-1.5-flash-latest';
        $url = "https://generativelanguage.googleapis.com/v1beta/models/{$model}:generateContent?key=" . $api_key;
        $headers = ['Content-Type: application/json'];
        $data = [
            'contents' => [
                ['parts' => [['text' => $system_prompt_content . "\n\n" . $prompt]]]
            ],
            'generationConfig' => ['temperature' => 0.5]
        ];
        $ai_response_path = 'gemini';
    } else {
        return "Error: Formato de API Key no reconocido.";
    }

    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    if ($http_code !== 200) {
        return "Error al contactar la API de IA (Código: $http_code): " . $response;
    }
    
    $result = json_decode($response, true);

    if ($ai_response_path === 'openai') {
        return $result['choices'][0]['message']['content'] ?? 'No se recibió una respuesta válida de la IA.';
    } elseif ($ai_response_path === 'gemini') {
        return $result['candidates'][0]['content']['parts'][0]['text'] ?? 'Error en la respuesta de Gemini.';
    }
    
    return 'Error interno al procesar la respuesta de la IA.';
}

/**
 * Verifica autenticación
 */
function check_auth() {
    if (!isset($_SESSION['username'])) {
        http_response_code(401);
        echo json_encode(['error' => 'No autorizado. Por favor, inicie sesión.']);
        exit;
    }
}

/**
 * Verifica que sea administrador
 */
function check_admin_auth() {
    check_auth();
    if ($_SESSION['categoria'] !== 'Administrador') {
        http_response_code(403);
        echo json_encode(['error' => 'Acceso denegado. Se requiere rol de Administrador.']);
        exit;
    }
}

// --- MANEJADORES DE AUTENTICACIÓN ---

/**
 * Maneja el login
 */
function handle_login($input) {
    $pdo = getDB();
    $username = $input['username'] ?? '';
    $password = $input['password'] ?? '';
    $remember = $input['remember'] ?? false;

    $stmt = $pdo->prepare("SELECT * FROM users WHERE username = ?");
    $stmt->execute([$username]);
    $user = $stmt->fetch();

    if ($user && password_verify($password, $user['password'])) {
        $_SESSION['user_id'] = $user['id'];
        $_SESSION['username'] = $user['username'];
        $_SESSION['categoria'] = $user['categoria'];

        if ($remember) {
            $token = bin2hex(random_bytes(32));
            $token_hash = hash('sha256', $token);
            
            $stmt = $pdo->prepare("UPDATE users SET remember_token = ? WHERE id = ?");
            $stmt->execute([$token_hash, $user['id']]);

            $cookie_options = [
                'expires' => time() + (86400 * 30),
                'path' => '/',
                'httponly' => true,
                'samesite' => 'Strict'
            ];
            setcookie('remember_me', $token, $cookie_options);
        }

        echo json_encode(['success' => true]);
        return;
    }

    http_response_code(401);
    echo json_encode(['error' => 'Usuario o contraseña incorrectos.']);
}

/**
 * Maneja el registro
 */
function handle_register($input) {
    $pdo = getDB();
    
    $username = $input['username'] ?? '';
    $password = $input['password'] ?? '';
    $nombrecompleto = $input['nombrecompleto'] ?? '';
    $email = $input['email'] ?? '';

    if (empty($username) || empty($password) || empty($email)) {
        http_response_code(400);
        echo json_encode(['error' => 'Datos requeridos faltantes.']);
        return;
    }

    // Verificar si el usuario ya existe
    $stmt = $pdo->prepare("SELECT id FROM users WHERE username = ?");
    $stmt->execute([$username]);
    if ($stmt->fetch()) {
        http_response_code(409);
        echo json_encode(['error' => 'El nombre de usuario ya existe.']);
        return;
    }

    $user_id = uniqid();
    $stmt = $pdo->prepare("INSERT INTO users (id, username, password, nombrecompleto, documento, email, telefono, direccion, ciudad, pais, categoria) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
    $stmt->execute([
        $user_id,
        $username,
        password_hash($password, PASSWORD_DEFAULT),
        $nombrecompleto,
        $input['documento'] ?? '',
        $email,
        $input['telefono'] ?? '',
        $input['direccion'] ?? '',
        $input['ciudad'] ?? '',
        $input['pais'] ?? '',
        $input['categoria'] ?? 'Digitador'
    ]);

    echo json_encode(['success' => true]);
}

/**
 * Maneja el logout
 */
function handle_logout() {
    session_unset();
    session_destroy();

    if (isset($_COOKIE['remember_me'])) {
        unset($_COOKIE['remember_me']);
        setcookie('remember_me', '', time() - 3600, '/');
    }

    header('Location: app/login.php');
    exit;
}

/**
 * Verifica la sesión
 */
function handle_check_session() {
    if (isset($_SESSION['username'])) {
        echo json_encode(['loggedIn' => true, 'username' => $_SESSION['username'], 'categoria' => $_SESSION['categoria']]);
        return;
    }

    // Verificar cookie de recordar
    if (isset($_COOKIE['remember_me'])) {
        $token = $_COOKIE['remember_me'];
        $token_hash = hash('sha256', $token);

        $pdo = getDB();
        $stmt = $pdo->prepare("SELECT * FROM users WHERE remember_token = ?");
        $stmt->execute([$token_hash]);
        $user = $stmt->fetch();

        if ($user) {
            $_SESSION['user_id'] = $user['id'];
            $_SESSION['username'] = $user['username'];
            $_SESSION['categoria'] = $user['categoria'];
            echo json_encode(['loggedIn' => true, 'username' => $_SESSION['username'], 'categoria' => $_SESSION['categoria']]);
            return;
        }
    }

    echo json_encode(['loggedIn' => false]);
}

// --- MANEJADORES DE EMPRESAS ---

/**
 * Obtiene todas las empresas (filtradas por permisos)
 */
function handle_get_empresas() {
    $pdo = getDB();
    $user_id = $_SESSION['user_id'];
    $categoria = $_SESSION['categoria'];

    if ($categoria === 'Administrador' || $categoria === 'Contador') {
        $stmt = $pdo->query("SELECT * FROM empresas ORDER BY nombre");
        $empresas = $stmt->fetchAll();
    } else {
        // Digitador: solo empresas asignadas
        $stmt = $pdo->prepare("
            SELECT e.* FROM empresas e
            INNER JOIN empresa_digitadores ed ON e.id = ed.empresa_id
            WHERE ed.digitador_id = ?
            ORDER BY e.nombre
        ");
        $stmt->execute([$user_id]);
        $empresas = $stmt->fetchAll();
    }

    // Cargar digitadores y timbrados para cada empresa
    foreach ($empresas as &$empresa) {
        // Digitadores
        $stmt = $pdo->prepare("SELECT digitador_id FROM empresa_digitadores WHERE empresa_id = ?");
        $stmt->execute([$empresa['id']]);
        $empresa['digitadores'] = array_column($stmt->fetchAll(), 'digitador_id');

        // Timbrados
        $stmt = $pdo->prepare("SELECT numero, fecha_inicio as inicio, fecha_fin as fin FROM timbrados WHERE empresa_id = ?");
        $stmt->execute([$empresa['id']]);
        $empresa['timbrados'] = $stmt->fetchAll();

        // Presentaciones
        $stmt = $pdo->prepare("SELECT * FROM presentaciones_fiscales WHERE empresa_id = ?");
        $stmt->execute([$empresa['id']]);
        $presentaciones = $stmt->fetch();
        $empresa['presentaciones'] = $presentaciones ?: [];
    }

    echo json_encode($empresas);
}

/**
 * Crea una nueva empresa
 */
function handle_create_empresa() {
    if ($_SESSION['categoria'] === 'Digitador') {
        http_response_code(403);
        echo json_encode(['error' => 'Acceso denegado.']);
        return;
    }

    $pdo = getDB();
    $input = $_POST;

    if (empty($input['nombre']) || empty($input['ruc'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Nombre y RUC son obligatorios.']);
        return;
    }

    $empresa_id = uniqid();
    $logo_path = null;

    // Manejar subida de logo
    if (isset($_FILES['logo']) && $_FILES['logo']['error'] == 0) {
        if (!is_dir(UPLOAD_DIR)) {
            mkdir(UPLOAD_DIR, 0755, true);
        }
        $extension = pathinfo($_FILES['logo']['name'], PATHINFO_EXTENSION);
        $logo_filename = $empresa_id . '_logo.' . $extension;
        $logo_path = UPLOAD_DIR . $logo_filename;
        move_uploaded_file($_FILES['logo']['tmp_name'], $logo_path);
    }

    try {
        $pdo->beginTransaction();

        // Insertar empresa
        $stmt = $pdo->prepare("INSERT INTO empresas (id, nombre, ruc, tipo, responsable_contabilidad, telefono, direccion, ciudad, pais, actividad_economica, observaciones, contador_id, logo) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
        $stmt->execute([
            $empresa_id,
            $input['nombre'],
            $input['ruc'],
            $input['tipo'] ?? 'fisica',
            $input['responsable_contabilidad'] ?? '',
            $input['telefono'] ?? '',
            $input['direccion'] ?? '',
            $input['ciudad'] ?? '',
            $input['pais'] ?? '',
            $input['actividad_economica'] ?? '',
            $input['observaciones'] ?? '',
            $input['contador_id'] ?? null,
            $logo_path
        ]);

        // Insertar digitadores
        if (!empty($input['digitadores']) && is_array($input['digitadores'])) {
            $stmt = $pdo->prepare("INSERT INTO empresa_digitadores (empresa_id, digitador_id) VALUES (?, ?)");
            foreach ($input['digitadores'] as $digitador_id) {
                $stmt->execute([$empresa_id, $digitador_id]);
            }
        }

        // Insertar timbrados
        $timbrados = json_decode($input['timbrados_json'] ?? '[]', true);
        if (!empty($timbrados)) {
            $stmt = $pdo->prepare("INSERT INTO timbrados (empresa_id, numero, fecha_inicio, fecha_fin) VALUES (?, ?, ?, ?)");
            foreach ($timbrados as $timbrado) {
                $stmt->execute([$empresa_id, $timbrado['numero'], $timbrado['inicio'], $timbrado['fin']]);
            }
        }

        // Insertar presentaciones fiscales
        if (!empty($input['presentaciones'])) {
            $p = $input['presentaciones'];
            $stmt = $pdo->prepare("INSERT INTO presentaciones_fiscales (empresa_id, iva_dia, irp_dia, irp_mes, irp_periodo, ire_dia, ire_mes, ire_periodo, inf_dia, inf_mes) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
            $stmt->execute([
                $empresa_id,
                $p['iva_dia'] ?? null,
                $p['irp_dia'] ?? null,
                $p['irp_mes'] ?? null,
                $p['irp_periodo'] ?? null,
                $p['ire_dia'] ?? null,
                $p['ire_mes'] ?? null,
                $p['ire_periodo'] ?? null,
                $p['inf_dia'] ?? null,
                $p['inf_mes'] ?? null
            ]);
        }

        // Insertar plan de cuentas por defecto
        $plan_default = get_plan_de_cuentas_default();
        $stmt = $pdo->prepare("INSERT INTO plan_cuentas (empresa_id, codigo, nombre, tipo) VALUES (?, ?, ?, ?)");
        foreach ($plan_default as $cuenta) {
            $stmt->execute([$empresa_id, $cuenta['codigo'], $cuenta['nombre'], $cuenta['tipo']]);
        }

        $pdo->commit();

        log_activity("Creó la empresa '{$input['nombre']}' (ID: {$empresa_id})");

        // Devolver la empresa creada
        $stmt = $pdo->prepare("SELECT * FROM empresas WHERE id = ?");
        $stmt->execute([$empresa_id]);
        echo json_encode($stmt->fetch());

    } catch (PDOException $e) {
        $pdo->rollBack();
        http_response_code(500);
        echo json_encode(['error' => 'Error al crear empresa: ' . $e->getMessage()]);
    }
}

/**
 * Actualiza una empresa
 */
function handle_update_empresa() {
    if ($_SESSION['categoria'] === 'Digitador') {
        http_response_code(403);
        echo json_encode(['error' => 'Acceso denegado.']);
        return;
    }

    $pdo = getDB();
    $input = $_POST;
    $empresa_id = validate_empresa_id($input['id'] ?? '');

    try {
        $pdo->beginTransaction();

        $logo_path = null;
        
        // Si hay nuevo logo
        if (isset($_FILES['logo']) && $_FILES['logo']['error'] == 0) {
            // Obtener logo anterior para eliminarlo
            $stmt = $pdo->prepare("SELECT logo FROM empresas WHERE id = ?");
            $stmt->execute([$empresa_id]);
            $old_logo = $stmt->fetchColumn();
            if ($old_logo && file_exists($old_logo)) {
                unlink($old_logo);
            }

            $extension = pathinfo($_FILES['logo']['name'], PATHINFO_EXTENSION);
            $logo_filename = $empresa_id . '_logo.' . $extension;
            $logo_path = UPLOAD_DIR . $logo_filename;
            move_uploaded_file($_FILES['logo']['tmp_name'], $logo_path);
        }

        // Actualizar empresa
        $sql = "UPDATE empresas SET nombre = ?, ruc = ?, tipo = ?, responsable_contabilidad = ?, telefono = ?, direccion = ?, ciudad = ?, pais = ?, actividad_economica = ?, observaciones = ?, contador_id = ?";
        $params = [
            $input['nombre'],
            $input['ruc'],
            $input['tipo'],
            $input['responsable_contabilidad'] ?? '',
            $input['telefono'] ?? '',
            $input['direccion'] ?? '',
            $input['ciudad'] ?? '',
            $input['pais'] ?? '',
            $input['actividad_economica'] ?? '',
            $input['observaciones'] ?? '',
            $input['contador_id'] ?? null
        ];

        if ($logo_path) {
            $sql .= ", logo = ?";
            $params[] = $logo_path;
        }

        $sql .= " WHERE id = ?";
        $params[] = $empresa_id;

        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);

        // Actualizar digitadores
        $pdo->prepare("DELETE FROM empresa_digitadores WHERE empresa_id = ?")->execute([$empresa_id]);
        if (!empty($input['digitadores']) && is_array($input['digitadores'])) {
            $stmt = $pdo->prepare("INSERT INTO empresa_digitadores (empresa_id, digitador_id) VALUES (?, ?)");
            foreach ($input['digitadores'] as $digitador_id) {
                $stmt->execute([$empresa_id, $digitador_id]);
            }
        }

        // Actualizar timbrados
        $pdo->prepare("DELETE FROM timbrados WHERE empresa_id = ?")->execute([$empresa_id]);
        $timbrados = json_decode($input['timbrados_json'] ?? '[]', true);
        if (!empty($timbrados)) {
            $stmt = $pdo->prepare("INSERT INTO timbrados (empresa_id, numero, fecha_inicio, fecha_fin) VALUES (?, ?, ?, ?)");
            foreach ($timbrados as $timbrado) {
                $stmt->execute([$empresa_id, $timbrado['numero'], $timbrado['inicio'], $timbrado['fin']]);
            }
        }

        // Actualizar presentaciones
        $pdo->prepare("DELETE FROM presentaciones_fiscales WHERE empresa_id = ?")->execute([$empresa_id]);
        if (!empty($input['presentaciones'])) {
            $p = $input['presentaciones'];
            $stmt = $pdo->prepare("INSERT INTO presentaciones_fiscales (empresa_id, iva_dia, irp_dia, irp_mes, irp_periodo, ire_dia, ire_mes, ire_periodo, inf_dia, inf_mes) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
            $stmt->execute([
                $empresa_id,
                $p['iva_dia'] ?? null,
                $p['irp_dia'] ?? null,
                $p['irp_mes'] ?? null,
                $p['irp_periodo'] ?? null,
                $p['ire_dia'] ?? null,
                $p['ire_mes'] ?? null,
                $p['ire_periodo'] ?? null,
                $p['inf_dia'] ?? null,
                $p['inf_mes'] ?? null
            ]);
        }

        $pdo->commit();

        log_activity("Actualizó la empresa '{$input['nombre']}' (ID: {$empresa_id})");

        // Devolver empresa actualizada
        $stmt = $pdo->prepare("SELECT * FROM empresas WHERE id = ?");
        $stmt->execute([$empresa_id]);
        echo json_encode($stmt->fetch());

    } catch (PDOException $e) {
        $pdo->rollBack();
        http_response_code(500);
        echo json_encode(['error' => 'Error al actualizar empresa: ' . $e->getMessage()]);
    }
}

/**
 * Elimina una empresa
 */
function handle_delete_empresa($input) {
    if ($_SESSION['categoria'] === 'Digitador') {
        http_response_code(403);
        echo json_encode(['error' => 'Acceso denegado.']);
        return;
    }

    $pdo = getDB();
    $empresa_id = validate_empresa_id($input['id'] ?? '');

    try {
        // Obtener nombre y logo antes de eliminar
        $stmt = $pdo->prepare("SELECT nombre, logo FROM empresas WHERE id = ?");
        $stmt->execute([$empresa_id]);
        $empresa = $stmt->fetch();

        if (!$empresa) {
            http_response_code(404);
            echo json_encode(['error' => 'Empresa no encontrada']);
            return;
        }

        // Eliminar logo si existe
        if ($empresa['logo'] && file_exists($empresa['logo'])) {
            unlink($empresa['logo']);
        }

        // La eliminación en cascada se encarga del resto
        $stmt = $pdo->prepare("DELETE FROM empresas WHERE id = ?");
        $stmt->execute([$empresa_id]);

        log_activity("Eliminó la empresa '{$empresa['nombre']}' (ID: {$empresa_id})");

        echo json_encode(['success' => true]);

    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Error al eliminar empresa: ' . $e->getMessage()]);
    }
}

// Continúa en el siguiente mensaje debido a la longitud...

// --- MANEJADORES DE PLAN DE CUENTAS ---

/**
 * Obtiene el plan de cuentas de una empresa
 */
function handle_get_plan_de_cuentas() {
    $pdo = getDB();
    $empresa_id = validate_empresa_id($_GET['empresa_id'] ?? '');

    $stmt = $pdo->prepare("SELECT codigo, nombre, tipo FROM plan_cuentas WHERE empresa_id = ? ORDER BY codigo");
    $stmt->execute([$empresa_id]);
    echo json_encode($stmt->fetchAll());
}

/**
 * Actualiza el plan de cuentas
 */
function handle_update_plan_de_cuentas($input) {
    $pdo = getDB();
    $empresa_id = validate_empresa_id($input['empresa_id'] ?? '');
    $cuentas = $input['cuentas'] ?? [];

    try {
        $pdo->beginTransaction();

        // Eliminar cuentas anteriores
        $pdo->prepare("DELETE FROM plan_cuentas WHERE empresa_id = ?")->execute([$empresa_id]);

        // Insertar nuevas cuentas
        $stmt = $pdo->prepare("INSERT INTO plan_cuentas (empresa_id, codigo, nombre, tipo) VALUES (?, ?, ?, ?)");
        foreach ($cuentas as $cuenta) {
            $stmt->execute([$empresa_id, $cuenta['codigo'], $cuenta['nombre'], $cuenta['tipo']]);
        }

        $pdo->commit();
        echo json_encode(['success' => true]);

    } catch (PDOException $e) {
        $pdo->rollBack();
        http_response_code(500);
        echo json_encode(['error' => 'Error al actualizar plan de cuentas']);
    }
}

// --- MANEJADORES DE ASIENTOS ---

/**
 * Obtiene asientos de una empresa
 */
function handle_get_asientos() {
    $pdo = getDB();
    $empresa_id = validate_empresa_id($_GET['empresa_id'] ?? '');
    $fecha_desde = $_GET['fecha_desde'] ?? '';
    $fecha_hasta = $_GET['fecha_hasta'] ?? '';

    $sql = "SELECT * FROM asientos WHERE empresa_id = ?";
    $params = [$empresa_id];

    if ($fecha_desde) {
        $sql .= " AND fecha >= ?";
        $params[] = $fecha_desde;
    }
    if ($fecha_hasta) {
        $sql .= " AND fecha <= ?";
        $params[] = $fecha_hasta;
    }

    $sql .= " ORDER BY fecha DESC";

    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);
    $asientos = $stmt->fetchAll();

    // Cargar movimientos para cada asiento
    $stmt_mov = $pdo->prepare("SELECT cuenta_codigo, descripcion, debe, haber FROM movimientos WHERE asiento_id = ?");
    foreach ($asientos as &$asiento) {
        $stmt_mov->execute([$asiento['id']]);
        $asiento['movimientos'] = $stmt_mov->fetchAll();
    }

    echo json_encode($asientos);
}

/**
 * Crea un asiento (venta o compra)
 */
function handle_create_asiento($input) {
    $pdo = getDB();
    $empresa_id = validate_empresa_id($input['empresa_id'] ?? '');

    // Obtener plan de cuentas para nombres
    $stmt = $pdo->prepare("SELECT codigo, nombre FROM plan_cuentas WHERE empresa_id = ?");
    $stmt->execute([$empresa_id]);
    $plan = [];
    while ($row = $stmt->fetch()) {
        $plan[$row['codigo']] = $row['nombre'];
    }

    $get_cuenta_nombre = function($codigo) use ($plan) {
        return $plan[$codigo] ?? '';
    };

    $monto = floatval($input['monto']);
    $iva_rate = 1.1;
    if ($input['iva_tipo'] == '5') $iva_rate = 1.05;
    if ($input['iva_tipo'] == '0') $iva_rate = 1.0;

    $base = round($monto / $iva_rate, 2);
    $iva = $monto - $base;

    $cuenta_pago_codigo = $input['cuenta_pago_codigo'];
    $cuenta_pago_nombre = $get_cuenta_nombre($cuenta_pago_codigo);

    $movimientos = [];

    if ($input['tipo_factura'] == 'venta') {
        $cuenta_ingreso_codigo = $input['cuenta_ingreso_codigo'];
        $cuenta_ingreso_nombre = $get_cuenta_nombre($cuenta_ingreso_codigo);
        $movimientos = [
            ["cuenta_codigo" => $cuenta_pago_codigo, "descripcion" => $cuenta_pago_nombre, "debe" => $monto, "haber" => 0],
            ["cuenta_codigo" => $cuenta_ingreso_codigo, "descripcion" => $cuenta_ingreso_nombre, "debe" => 0, "haber" => $base],
            ["cuenta_codigo" => "2.1.3", "descripcion" => "IVA Débito Fiscal", "debe" => 0, "haber" => $iva],
        ];
    } else {
        $cuenta_gasto_codigo = $input['cuenta_gasto_codigo'];
        $cuenta_gasto_nombre = $get_cuenta_nombre($cuenta_gasto_codigo);
        $movimientos = [
            ["cuenta_codigo" => $cuenta_gasto_codigo, "descripcion" => $cuenta_gasto_nombre, "debe" => $base, "haber" => 0],
            ["cuenta_codigo" => "1.1.4", "descripcion" => "IVA Crédito Fiscal", "debe" => $iva, "haber" => 0],
            ["cuenta_codigo" => $cuenta_pago_codigo, "descripcion" => $cuenta_pago_nombre, "debe" => 0, "haber" => $monto],
        ];
    }

    try {
        $pdo->beginTransaction();

        $asiento_id = uniqid();
        $stmt = $pdo->prepare("INSERT INTO asientos (id, empresa_id, fecha, comprobante_nro, descripcion, ruc_cliente_proveedor, timbrado, tipo_comprobante, condicion_operacion, es_electronica, moneda, tipo_factura, iva_tipo) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
        $stmt->execute([
            $asiento_id,
            $empresa_id,
            $input['fecha'],
            $input['comprobante_nro'],
            $input['descripcion'],
            $input['ruc_cliente_proveedor'],
            $input['timbrado'],
            $input['tipo_comprobante'],
            $input['condicion_operacion'],
            $input['es_electronica'] ?? false,
            $input['moneda'],
            $input['tipo_factura'],
            $input['iva_tipo']
        ]);

        // Insertar movimientos
        $stmt = $pdo->prepare("INSERT INTO movimientos (asiento_id, cuenta_codigo, descripcion, debe, haber) VALUES (?, ?, ?, ?, ?)");
        foreach ($movimientos as $mov) {
            $stmt->execute([$asiento_id, $mov['cuenta_codigo'], $mov['descripcion'], $mov['debe'], $mov['haber']]);
        }

        $pdo->commit();

        // Devolver el asiento creado
        $stmt = $pdo->prepare("SELECT * FROM asientos WHERE id = ?");
        $stmt->execute([$asiento_id]);
        $asiento = $stmt->fetch();
        
        $stmt = $pdo->prepare("SELECT cuenta_codigo, descripcion, debe, haber FROM movimientos WHERE asiento_id = ?");
        $stmt->execute([$asiento_id]);
        $asiento['movimientos'] = $stmt->fetchAll();

        echo json_encode($asiento);

    } catch (PDOException $e) {
        $pdo->rollBack();
        http_response_code(500);
        echo json_encode(['error' => 'Error al crear asiento']);
    }
}

/**
 * Actualiza un asiento
 */
function handle_update_asiento($input) {
    $pdo = getDB();
    $empresa_id = validate_empresa_id($input['empresa_id'] ?? '');
    $asiento_id = $input['asiento_id'] ?? '';

    // Similar a create_asiento pero con UPDATE
    // Obtener plan de cuentas
    $stmt = $pdo->prepare("SELECT codigo, nombre FROM plan_cuentas WHERE empresa_id = ?");
    $stmt->execute([$empresa_id]);
    $plan = [];
    while ($row = $stmt->fetch()) {
        $plan[$row['codigo']] = $row['nombre'];
    }

    $get_cuenta_nombre = function($codigo) use ($plan) {
        return $plan[$codigo] ?? '';
    };

    $monto = floatval($input['monto']);
    $iva_rate = 1.1;
    if ($input['iva_tipo'] == '5') $iva_rate = 1.05;
    if ($input['iva_tipo'] == '0') $iva_rate = 1.0;

    $base = round($monto / $iva_rate, 2);
    $iva = $monto - $base;

    $cuenta_pago_codigo = $input['cuenta_pago_codigo'];
    $cuenta_pago_nombre = $get_cuenta_nombre($cuenta_pago_codigo);

    $movimientos = [];

    if ($input['tipo_factura'] == 'venta') {
        $cuenta_ingreso_codigo = $input['cuenta_ingreso_codigo'];
        $cuenta_ingreso_nombre = $get_cuenta_nombre($cuenta_ingreso_codigo);
        $movimientos = [
            ["cuenta_codigo" => $cuenta_pago_codigo, "descripcion" => $cuenta_pago_nombre, "debe" => $monto, "haber" => 0],
            ["cuenta_codigo" => $cuenta_ingreso_codigo, "descripcion" => $cuenta_ingreso_nombre, "debe" => 0, "haber" => $base],
            ["cuenta_codigo" => "2.1.3", "descripcion" => "IVA Débito Fiscal", "debe" => 0, "haber" => $iva],
        ];
    } else {
        $cuenta_gasto_codigo = $input['cuenta_gasto_codigo'];
        $cuenta_gasto_nombre = $get_cuenta_nombre($cuenta_gasto_codigo);
        $movimientos = [
            ["cuenta_codigo" => $cuenta_gasto_codigo, "descripcion" => $cuenta_gasto_nombre, "debe" => $base, "haber" => 0],
            ["cuenta_codigo" => "1.1.4", "descripcion" => "IVA Crédito Fiscal", "debe" => $iva, "haber" => 0],
            ["cuenta_codigo" => $cuenta_pago_codigo, "descripcion" => $cuenta_pago_nombre, "debe" => 0, "haber" => $monto],
        ];
    }

    try {
        $pdo->beginTransaction();

        $stmt = $pdo->prepare("UPDATE asientos SET fecha = ?, comprobante_nro = ?, descripcion = ?, ruc_cliente_proveedor = ?, timbrado = ?, tipo_comprobante = ?, condicion_operacion = ?, es_electronica = ?, moneda = ?, tipo_factura = ?, iva_tipo = ? WHERE id = ? AND empresa_id = ?");
        $stmt->execute([
            $input['fecha'],
            $input['comprobante_nro'],
            $input['descripcion'],
            $input['ruc_cliente_proveedor'],
            $input['timbrado'],
            $input['tipo_comprobante'],
            $input['condicion_operacion'],
            $input['es_electronica'] ?? false,
            $input['moneda'],
            $input['tipo_factura'],
            $input['iva_tipo'],
            $asiento_id,
            $empresa_id
        ]);

        // Eliminar y recrear movimientos
        $pdo->prepare("DELETE FROM movimientos WHERE asiento_id = ?")->execute([$asiento_id]);

        $stmt = $pdo->prepare("INSERT INTO movimientos (asiento_id, cuenta_codigo, descripcion, debe, haber) VALUES (?, ?, ?, ?, ?)");
        foreach ($movimientos as $mov) {
            $stmt->execute([$asiento_id, $mov['cuenta_codigo'], $mov['descripcion'], $mov['debe'], $mov['haber']]);
        }

        $pdo->commit();

        // Devolver asiento actualizado
        $stmt = $pdo->prepare("SELECT * FROM asientos WHERE id = ?");
        $stmt->execute([$asiento_id]);
        $asiento = $stmt->fetch();
        
        $stmt = $pdo->prepare("SELECT cuenta_codigo, descripcion, debe, haber FROM movimientos WHERE asiento_id = ?");
        $stmt->execute([$asiento_id]);
        $asiento['movimientos'] = $stmt->fetchAll();

        echo json_encode($asiento);

    } catch (PDOException $e) {
        $pdo->rollBack();
        http_response_code(500);
        echo json_encode(['error' => 'Error al actualizar asiento']);
    }
}

/**
 * Elimina un asiento
 */
function handle_delete_asiento($input) {
    $pdo = getDB();
    $empresa_id = validate_empresa_id($input['empresa_id'] ?? '');
    $asiento_id = $input['asiento_id'] ?? '';

    try {
        $stmt = $pdo->prepare("DELETE FROM asientos WHERE id = ? AND empresa_id = ?");
        $stmt->execute([$asiento_id, $empresa_id]);
        echo json_encode(['success' => true]);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Error al eliminar asiento']);
    }
}

/**
 * Guarda un asiento manual
 */
function handle_save_manual_asiento($input) {
    $pdo = getDB();
    $empresa_id = validate_empresa_id($input['empresa_id'] ?? '');
    $asiento = $input['asiento'];

    try {
        $pdo->beginTransaction();

        if (!empty($asiento['id'])) {
            // Actualizar asiento existente
            $stmt = $pdo->prepare("UPDATE asientos SET fecha = ?, comprobante_nro = ?, descripcion = ? WHERE id = ? AND empresa_id = ?");
            $stmt->execute([
                $asiento['fecha'],
                $asiento['comprobante_nro'],
                $asiento['descripcion'],
                $asiento['id'],
                $empresa_id
            ]);
            $asiento_id = $asiento['id'];

            // Eliminar movimientos anteriores
            $pdo->prepare("DELETE FROM movimientos WHERE asiento_id = ?")->execute([$asiento_id]);
        } else {
            // Crear nuevo asiento
            $asiento_id = uniqid();
            $stmt = $pdo->prepare("INSERT INTO asientos (id, empresa_id, fecha, comprobante_nro, descripcion, tipo_factura) VALUES (?, ?, ?, ?, ?, 'manual')");
            $stmt->execute([
                $asiento_id,
                $empresa_id,
                $asiento['fecha'],
                $asiento['comprobante_nro'],
                $asiento['descripcion']
            ]);
        }

        // Insertar movimientos
        $stmt = $pdo->prepare("INSERT INTO movimientos (asiento_id, cuenta_codigo, descripcion, debe, haber) VALUES (?, ?, ?, ?, ?)");
        foreach ($asiento['movimientos'] as $mov) {
            $stmt->execute([
                $asiento_id,
                $mov['cuenta_codigo'],
                $mov['descripcion'],
                $mov['debe'],
                $mov['haber']
            ]);
        }

        $pdo->commit();
        echo json_encode(['success' => true, 'asiento_id' => $asiento_id]);

    } catch (PDOException $e) {
        $pdo->rollBack();
        http_response_code(500);
        echo json_encode(['error' => 'Error al guardar asiento manual']);
    }
}

// --- EXPORTACIÓN ---

/**
 * Exporta asientos a CSV/Excel
 */
function handle_export_asientos() {
    $pdo = getDB();
    $empresa_id = validate_empresa_id($_GET['empresa_id'] ?? '');
    $tipo_factura = $_GET['action'] == 'export_ventas' ? 'venta' : 'compra';
    $format = $_GET['format'] ?? 'csv';

    $stmt = $pdo->prepare("
        SELECT a.*, GROUP_CONCAT(CONCAT(m.cuenta_codigo, ':', m.debe, ':', m.haber) SEPARATOR '|') as movimientos_concat
        FROM asientos a
        LEFT JOIN movimientos m ON a.id = m.asiento_id
        WHERE a.empresa_id = ? AND a.tipo_factura = ?
        GROUP BY a.id
        ORDER BY a.fecha
    ");
    $stmt->execute([$empresa_id, $tipo_factura]);
    $asientos = $stmt->fetchAll();

    $filename = $tipo_factura . 's_rg90.' . ($format == 'excel' ? 'xls' : 'csv');
    header('Content-Type: text/csv; charset=utf-8');
    header('Content-Disposition: attachment; filename="' . $filename . '"');

    $output = fopen('php://output', 'w');
    
    $cabecera = [
        'Fecha', 
        ($tipo_factura == 'venta' ? 'RUC Cliente' : 'RUC Proveedor'),
        ($tipo_factura == 'venta' ? 'Nombre Cliente' : 'Nombre Proveedor'),
        'Timbrado', 
        'Nro Factura', 
        'Gravado 10%', 
        'IVA 10%', 
        'Gravado 5%', 
        'IVA 5%', 
        'Exento', 
        'Total Factura', 
        'Condicion', 
        'Electronica'
    ];
    fputcsv($output, $cabecera);

    foreach ($asientos as $factura) {
        // Procesar movimientos concatenados
        $movs = explode('|', $factura['movimientos_concat']);
        $iva_10 = 0; $base_10 = 0; $iva_5 = 0; $base_5 = 0; $exento = 0;
        $monto = 0;

        foreach ($movs as $mov_str) {
            $parts = explode(':', $mov_str);
            if (count($parts) == 3) {
                $codigo = $parts[0];
                $debe = floatval($parts[1]);
                $haber = floatval($parts[2]);

                if ($tipo_factura == 'venta' && ($codigo == '1.1.1' || $codigo == '1.1.2')) {
                    $monto = $debe;
                } elseif ($tipo_factura == 'compra' && ($codigo == '1.1.1' || $codigo == '2.1.1')) {
                    $monto = $haber;
                }

                if ($codigo == '1.1.4') $iva_10 = $debe;
                if ($codigo == '2.1.3') $iva_10 = $haber;
                if ($codigo[0] == '5') $base_10 = $debe;
                if ($codigo[0] == '4') $base_10 = $haber;
            }
        }

        if ($monto == 0 && !empty($movs)) {
            $first_mov = explode(':', $movs[0]);
            $monto = max(floatval($first_mov[1]), floatval($first_mov[2]));
        }

        if ($iva_10 == 0) $exento = $monto;

        $linea = [
            $factura['fecha'],
            $factura['ruc_cliente_proveedor'],
            str_replace('Factura ' . $tipo_factura . ' - ', '', $factura['descripcion']),
            $factura['timbrado'],
            $factura['comprobante_nro'],
            number_format($base_10, 2, ',', ''),
            number_format($iva_10, 2, ',', ''),
            number_format($base_5, 2, ',', ''),
            number_format($iva_5, 2, ',', ''),
            number_format($exento, 2, ',', ''),
            number_format($monto, 2, ',', ''),
            $factura['condicion_operacion'],
            ($factura['es_electronica'] ?? false) ? 'Si' : 'No'
        ];
        fputcsv($output, $linea);
    }

    fclose($output);
    exit;
}


// --- MANEJADORES DE USUARIOS ---

/**
 * Obtiene todos los usuarios
 */
function handle_get_users() {
    $pdo = getDB();
    $stmt = $pdo->query("SELECT id, username, nombrecompleto, documento, email, telefono, direccion, ciudad, pais, categoria FROM users");
    echo json_encode($stmt->fetchAll());
}

/**
 * Crea un nuevo usuario
 */
function handle_create_user($input) {
    $pdo = getDB();
    
    try {
        $user_id = uniqid();
        $stmt = $pdo->prepare("INSERT INTO users (id, username, password, nombrecompleto, documento, email, telefono, direccion, ciudad, pais, categoria) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
        $stmt->execute([
            $user_id,
            $input['username'],
            password_hash($input['password'], PASSWORD_DEFAULT),
            $input['nombrecompleto'] ?? '',
            $input['documento'] ?? '',
            $input['email'],
            $input['telefono'] ?? '',
            $input['direccion'] ?? '',
            $input['ciudad'] ?? '',
            $input['pais'] ?? '',
            $input['categoria'] ?? 'Digitador'
        ]);
        echo json_encode(['success' => true, 'user_id' => $user_id]);
    } catch (PDOException $e) {
        http_response_code(400);
        echo json_encode(['error' => 'Error al crear usuario: ' . $e->getMessage()]);
    }
}

/**
 * Actualiza un usuario
 */
function handle_update_user($input) {
    $pdo = getDB();
    $user_id = $input['id'] ?? '';

    $sql = "UPDATE users SET username = ?, nombrecompleto = ?, documento = ?, email = ?, telefono = ?, direccion = ?, ciudad = ?, pais = ?, categoria = ?";
    $params = [
        $input['username'],
        $input['nombrecompleto'] ?? '',
        $input['documento'] ?? '',
        $input['email'],
        $input['telefono'] ?? '',
        $input['direccion'] ?? '',
        $input['ciudad'] ?? '',
        $input['pais'] ?? '',
        $input['categoria'] ?? 'Digitador'
    ];

    if (!empty($input['password'])) {
        $sql .= ", password = ?";
        $params[] = password_hash($input['password'], PASSWORD_DEFAULT);
    }

    $sql .= " WHERE id = ?";
    $params[] = $user_id;

    try {
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        echo json_encode(['success' => true]);
    } catch (PDOException $e) {
        http_response_code(400);
        echo json_encode(['error' => 'Error al actualizar usuario']);
    }
}

/**
 * Elimina un usuario
 */
function handle_delete_user($input) {
    $pdo = getDB();
    $user_id = $input['id'] ?? '';

    if ($user_id === $_SESSION['user_id']) {
        http_response_code(403);
        echo json_encode(['error' => 'No puedes eliminar tu propio usuario.']);
        return;
    }

    try {
        $stmt = $pdo->prepare("DELETE FROM users WHERE id = ?");
        $stmt->execute([$user_id]);
        log_activity("Eliminó al usuario con ID: {$user_id}");
        echo json_encode(['success' => true]);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Error al eliminar usuario']);
    }
}

// --- FUNCIONES AUXILIARES ---

/**
 * Obtiene digitadores
 */
function handle_get_digitadores() {
    $pdo = getDB();
    $stmt = $pdo->prepare("SELECT id, nombrecompleto FROM users WHERE categoria = 'Digitador'");
    $stmt->execute();
    echo json_encode($stmt->fetchAll());
}

/**
 * Obtiene contadores
 */
function handle_get_contadores() {
    $pdo = getDB();
    $stmt = $pdo->prepare("SELECT id, nombrecompleto FROM users WHERE categoria IN ('Administrador', 'Contador')");
    $stmt->execute();
    echo json_encode($stmt->fetchAll());
}

/**
 * Obtiene vencimientos para el calendario
 */
function handle_get_vencimientos() {
    $pdo = getDB();
    $eventos = [];
    $current_year = date('Y');
    
    $periodos_map = [
        'mes' => 1, 'bimestre' => 2, 'trimestre' => 3, 
        'cuatrimestre' => 4, 'semestre' => 6, 'año' => 12
    ];

    // Timbrados
    $stmt = $pdo->query("SELECT t.fecha_fin, e.id, e.nombre FROM timbrados t JOIN empresas e ON t.empresa_id = e.id");
    while ($row = $stmt->fetch()) {
        $eventos[] = [
            'title' => 'Vence Timbrado: ' . $row['nombre'],
            'start' => $row['fecha_fin'],
            'url' => 'index.html#empresa=' . $row['id'],
            'allDay' => true,
            'color' => '#dc3545'
        ];
    }

    // Presentaciones fiscales
    $stmt = $pdo->query("SELECT p.*, e.id, e.nombre FROM presentaciones_fiscales p JOIN empresas e ON p.empresa_id = e.id");
    while ($row = $stmt->fetch()) {
        // IVA mensual
        if ($row['iva_dia']) {
            for ($month = 1; $month <= 12; $month++) {
                $eventos[] = [
                    'title' => 'IVA: ' . $row['nombre'],
                    'start' => sprintf('%s-%02d-%02d', $current_year, $month, $row['iva_dia']),
                    'url' => 'index.html#empresa=' . $row['id'],
                    'allDay' => true,
                    'color' => '#0d6efd'
                ];
            }
        }

        // IRP periódico
        if ($row['irp_dia'] && $row['irp_mes'] && $row['irp_periodo']) {
            $paso = $periodos_map[$row['irp_periodo']];
            for ($month = intval($row['irp_mes']); $month <= 12; $month += $paso) {
                $eventos[] = [
                    'title' => 'IRP: ' . $row['nombre'],
                    'start' => sprintf('%s-%02d-%02d', $current_year, $month, $row['irp_dia']),
                    'url' => 'index.html#empresa=' . $row['id'],
                    'allDay' => true,
                    'color' => '#198754'
                ];
            }
        }

        // IRE periódico
        if ($row['ire_dia'] && $row['ire_mes'] && $row['ire_periodo']) {
            $paso = $periodos_map[$row['ire_periodo']];
            for ($month = intval($row['ire_mes']); $month <= 12; $month += $paso) {
                $eventos[] = [
                    'title' => 'IRE: ' . $row['nombre'],
                    'start' => sprintf('%s-%02d-%02d', $current_year, $month, $row['ire_dia']),
                    'url' => 'index.html#empresa=' . $row['id'],
                    'allDay' => true,
                    'color' => '#198754'
                ];
            }
        }

        // Informes anuales
        if ($row['inf_dia'] && $row['inf_mes']) {
            $eventos[] = [
                'title' => 'Inf. Financ.: ' . $row['nombre'],
                'start' => sprintf('%s-%02d-%02d', $current_year, $row['inf_mes'], $row['inf_dia']),
                'url' => 'index.html#empresa=' . $row['id'],
                'allDay' => true,
                'color' => '#ffc107',
                'textColor' => '#000'
            ];
        }
    }

    echo json_encode($eventos);
}

/**
 * Obtiene el log de actividad
 */
function handle_get_activity_log() {
    $pdo = getDB();
    $stmt = $pdo->prepare("SELECT username, action, DATE_FORMAT(created_at, '%Y-%m-%d %H:%i:%s') as timestamp FROM activity_log ORDER BY created_at DESC LIMIT ?");
    $stmt->execute([MAX_LOG_ENTRIES]);
    echo json_encode($stmt->fetchAll());
}

/**
 * Analiza con IA
 */
function handle_analizar_con_ia($input) {
    $tipo_analisis = $input['tipo'] ?? '';
    $datos_json = $input['datos'] ?? '';

    if (empty($tipo_analisis) || empty($datos_json)) {
        http_response_code(400);
        echo json_encode(['error' => 'Faltan parámetros para el análisis.']);
        return;
    }

    $prompt = "Analiza los siguientes datos de un reporte de '$tipo_analisis' y proporciona un resumen ejecutivo, puntos clave y recomendaciones. Responde en español y usa formato markdown para una mejor legibilidad:\n\n" . $datos_json;
    $analisis = call_ai_api($prompt);

    if (str_starts_with($analisis, 'Error:')) {
        http_response_code(500);
        echo json_encode(['analisis' => $analisis]);
    } else {
        echo json_encode(['analisis' => $analisis]);
    }
}

/**
 * Obtiene configuración
 */
function handle_get_settings() {
    $config = get_config();
    echo json_encode($config);
}

/**
 * Guarda configuración
 */
function handle_save_settings($input) {
    save_config('ai_api_key', $input['ai_api_key'] ?? '');
    echo json_encode(['success' => true]);
}

/**
 * Obtiene datos para reporte de IVA
 */
function handle_get_iva_report_data() {
    $pdo = getDB();
    $empresa_id = validate_empresa_id($_GET['empresa_id'] ?? '');
    $fecha_desde = $_GET['fecha_desde'] ?? '';
    $fecha_hasta = $_GET['fecha_hasta'] ?? '';

    $sql = "SELECT * FROM asientos WHERE empresa_id = ? AND tipo_factura IN ('venta', 'compra')";
    $params = [$empresa_id];

    if ($fecha_desde) {
        $sql .= " AND fecha >= ?";
        $params[] = $fecha_desde;
    }
    if ($fecha_hasta) {
        $sql .= " AND fecha <= ?";
        $params[] = $fecha_hasta;
    }

    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);
    $asientos = $stmt->fetchAll();

    $reporte = [
        'ventas' => [
            '10' => ['count' => 0, 'total' => 0, 'base' => 0, 'iva' => 0],
            '5'  => ['count' => 0, 'total' => 0, 'base' => 0, 'iva' => 0],
            '0'  => ['count' => 0, 'total' => 0, 'base' => 0, 'iva' => 0],
        ],
        'compras' => [
            '10' => ['count' => 0, 'total' => 0, 'base' => 0, 'iva' => 0],
            '5'  => ['count' => 0, 'total' => 0, 'base' => 0, 'iva' => 0],
            '0'  => ['count' => 0, 'total' => 0, 'base' => 0, 'iva' => 0],
        ],
    ];

    $stmt_mov = $pdo->prepare("SELECT cuenta_codigo, debe, haber FROM movimientos WHERE asiento_id = ?");

    foreach ($asientos as $asiento) {
        $stmt_mov->execute([$asiento['id']]);
        $movimientos = $stmt_mov->fetchAll();

        $tipo_factura = $asiento['tipo_factura'];
        $total_factura = 0;
        $base_imponible = 0;
        $monto_iva = 0;

        if ($tipo_factura === 'venta') {
            foreach ($movimientos as $m) {
                if ($m['cuenta_codigo'] == '1.1.1' || $m['cuenta_codigo'] == '1.1.2') {
                    $total_factura += $m['debe'];
                }
                if ($m['cuenta_codigo'][0] == '4') {
                    $base_imponible += $m['haber'];
                }
                if ($m['cuenta_codigo'] == '2.1.3') {
                    $monto_iva += $m['haber'];
                }
            }
        } else {
            foreach ($movimientos as $m) {
                if ($m['cuenta_codigo'] == '1.1.1' || $m['cuenta_codigo'] == '2.1.1') {
                    $total_factura += $m['haber'];
                }
                if ($m['cuenta_codigo'][0] == '5') {
                    $base_imponible += $m['debe'];
                }
                if ($m['cuenta_codigo'] == '1.1.4') {
                    $monto_iva += $m['debe'];
                }
            }
        }

        $iva_tipo_calculado = '0';
        if ($base_imponible > 0 && $monto_iva > 0) {
            $rate = round(($monto_iva / $base_imponible) * 100);
            if ($rate >= 9 && $rate <= 11) $iva_tipo_calculado = '10';
            elseif ($rate >= 4 && $rate <= 6) $iva_tipo_calculado = '5';
        }

        $key = $tipo_factura . 's';
        $reporte[$key][$iva_tipo_calculado]['count']++;
        $reporte[$key][$iva_tipo_calculado]['total'] += $total_factura;
        $reporte[$key][$iva_tipo_calculado]['base'] += $base_imponible;
        $reporte[$key][$iva_tipo_calculado]['iva'] += $monto_iva;
    }

    echo json_encode($reporte);
}


// --- GENERACIÓN DE REPORTES HTML ---

/**
 * Función auxiliar para procesar datos de mayor contable
 */
function get_processed_mayor_data($empresa_id, $fecha_desde, $fecha_hasta) {
    $pdo = getDB();
    
    // Obtener asientos
    $sql = "SELECT * FROM asientos WHERE empresa_id = ?";
    $params = [$empresa_id];
    if ($fecha_desde) {
        $sql .= " AND fecha >= ?";
        $params[] = $fecha_desde;
    }
    if ($fecha_hasta) {
        $sql .= " AND fecha <= ?";
        $params[] = $fecha_hasta;
    }
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);
    $asientos = $stmt->fetchAll();
    
    // Obtener plan de cuentas
    $stmt = $pdo->prepare("SELECT codigo, nombre, tipo FROM plan_cuentas WHERE empresa_id = ?");
    $stmt->execute([$empresa_id]);
    $plan = $stmt->fetchAll();
    
    // Obtener empresa
    $stmt = $pdo->prepare("SELECT * FROM empresas WHERE id = ?");
    $stmt->execute([$empresa_id]);
    $empresa = $stmt->fetch();
    
    // Procesar mayor
    $mayor = [];
    foreach ($plan as $cuenta) {
        $mayor[$cuenta['codigo']] = [
            'nombre' => $cuenta['nombre'],
            'tipo' => $cuenta['tipo'],
            'totalDebe' => 0,
            'totalHaber' => 0
        ];
    }
    
    // Cargar movimientos para cada asiento
    $stmt_mov = $pdo->prepare("SELECT cuenta_codigo, debe, haber FROM movimientos WHERE asiento_id = ?");
    foreach ($asientos as &$asiento) {
        $stmt_mov->execute([$asiento['id']]);
        $movimientos = $stmt_mov->fetchAll();
        $asiento['movimientos'] = $movimientos;
        
        foreach ($movimientos as $mov) {
            if (isset($mayor[$mov['cuenta_codigo']])) {
                $mayor[$mov['cuenta_codigo']]['totalDebe'] += floatval($mov['debe']);
                $mayor[$mov['cuenta_codigo']]['totalHaber'] += floatval($mov['haber']);
            }
        }
    }
    
    return [
        'asientos' => $asientos,
        'plan' => $plan,
        'empresa' => $empresa,
        'mayor' => $mayor
    ];
}

/**
 * Genera reporte de Balance General
 */
function handle_generar_reporte_balance_general() {
    header('Content-Type: text/html; charset=utf-8');
    
    $empresa_id = validate_empresa_id($_GET['empresa_id'] ?? '');
    $fecha_desde = $_GET['fecha_desde'] ?? '';
    $fecha_hasta = $_GET['fecha_hasta'] ?? '';
    
    $data = get_processed_mayor_data($empresa_id, $fecha_desde, $fecha_hasta);
    $mayor = $data['mayor'];
    $empresa = $data['empresa'];
    
    // Calcular resultado del ejercicio
    $total_ingresos = 0;
    $total_gastos = 0;
    foreach ($mayor as $codigo => $cuenta) {
        $saldo = $cuenta['totalDebe'] - $cuenta['totalHaber'];
        if ($codigo[0] == '4') $total_ingresos += -$saldo;
        if ($codigo[0] == '5') $total_gastos += $saldo;
    }
    $resultado_ejercicio = $total_ingresos - $total_gastos;
    
    // Clasificar cuentas
    $html_activo = '';
    $html_pasivo = '';
    $html_patrimonio = '';
    $total_activo = 0;
    $total_pasivo = 0;
    $total_patrimonio = 0;
    
    foreach ($mayor as $codigo => $cuenta) {
        $saldo = $cuenta['totalDebe'] - $cuenta['totalHaber'];
        if ($saldo == 0 && $cuenta['tipo'] == 'detalle') continue;
        
        $clase = $cuenta['tipo'] == 'titulo' ? 'class="fw-bold"' : 'style="padding-left: 20px;"';
        $monto_formateado = number_format($saldo, 2, ',', '.');
        
        if ($codigo[0] == '1') {
            if ($cuenta['tipo'] == 'detalle') $total_activo += $saldo;
            $html_activo .= "<tr><td {$clase}>($codigo) {$cuenta['nombre']}</td><td class=\"text-end\">{$monto_formateado}</td></tr>";
        } else if ($codigo[0] == '2') {
            $monto_pasivo = -$saldo;
            if ($cuenta['tipo'] == 'detalle') $total_pasivo += $monto_pasivo;
            $html_pasivo .= "<tr><td {$clase}>($codigo) {$cuenta['nombre']}</td><td class=\"text-end\">" . number_format($monto_pasivo, 2, ',', '.') . "</td></tr>";
        } else if ($codigo[0] == '3') {
            $monto_patrimonio = -$saldo;
            if ($cuenta['tipo'] == 'detalle') $total_patrimonio += $monto_patrimonio;
            $html_patrimonio .= "<tr><td {$clase}>($codigo) {$cuenta['nombre']}</td><td class=\"text-end\">" . number_format($monto_patrimonio, 2, ',', '.') . "</td></tr>";
        }
    }
    
    $total_patrimonio += $resultado_ejercicio;
    $html_patrimonio .= '<tr><td style="padding-left: 20px;">Resultado del Ejercicio</td><td class="text-end">' . number_format($resultado_ejercicio, 2, ',', '.') . '</td></tr>';
    $total_pasivo_patrimonio = $total_pasivo + $total_patrimonio;
    
    echo '<!DOCTYPE html><html lang="es"><head><meta charset="UTF-8"><title>Balance General</title><style>body{font-family:sans-serif;margin:20px}table{width:100%;border-collapse:collapse}th,td{padding:8px;text-align:left}.text-end{text-align:right}.header{text-align:center;margin-bottom:20px}h1,h2,h3{margin:0}.total-row{font-weight:bold;border-top:2px solid #333}.gran-total{font-size:1.1em;font-weight:bold;border-top:3px double #333;margin-top:10px}.container{display:flex;width:100%}.col{width:50%;padding:0 15px}@media print{body{margin:0} @page { size: A4; margin: 2.5cm; }}</style></head><body>';
    echo '<div class="header"><h1>Balance General</h1>';
    echo '<h2>' . htmlspecialchars($empresa['nombre']) . '</h2>';
    echo '<h3>RUC: ' . htmlspecialchars($empresa['ruc']) . '</h3>';
    echo '<p>Período del ' . htmlspecialchars($fecha_desde) . ' al ' . htmlspecialchars($fecha_hasta) . '</p></div>';
    echo '<div class="container">';
    echo '<div class="col"><h3>Activo</h3><table>' . $html_activo . '<tr class="total-row"><td><strong>Total Activo</strong></td><td class="text-end"><strong>' . number_format($total_activo, 2, ',', '.') . '</strong></td></tr></table></div>';
    echo '<div class="col"><h3>Pasivo</h3><table>' . $html_pasivo . '<tr class="total-row"><td><strong>Total Pasivo</strong></td><td class="text-end"><strong>' . number_format($total_pasivo, 2, ',', '.') . '</strong></td></tr></table><br><h3>Patrimonio Neto</h3><table>' . $html_patrimonio . '<tr class="total-row"><td><strong>Total Patrimonio Neto</strong></td><td class="text-end"><strong>' . number_format($total_patrimonio, 2, ',', '.') . '</strong></td></tr></table><br><table class="gran-total"><tr><td><strong>Total Pasivo y Patrimonio</strong></td><td class="text-end"><strong>' . number_format($total_pasivo_patrimonio, 2, ',', '.') . '</strong></td></tr></table></div>';
    echo '</div></body></html>';
    exit;
}

/**
 * Genera reporte de Estado de Resultados  
 */
function handle_generar_reporte_estado_resultados() {
    header('Content-Type: text/html; charset=utf-8');
    
    $empresa_id = validate_empresa_id($_GET['empresa_id'] ?? '');
    $fecha_desde = $_GET['fecha_desde'] ?? '';
    $fecha_hasta = $_GET['fecha_hasta'] ?? '';
    
    $data = get_processed_mayor_data($empresa_id, $fecha_desde, $fecha_hasta);
    $mayor = $data['mayor'];
    $empresa = $data['empresa'];
    
    $total_ingresos = 0;
    $total_gastos = 0;
    $html_ingresos = '';
    $html_gastos = '';
    
    foreach ($mayor as $codigo => $cuenta) {
        $saldo = $cuenta['totalDebe'] - $cuenta['totalHaber'];
        if ($codigo[0] == '4' && $saldo != 0) {
            $saldo_acreedor = -$saldo;
            $total_ingresos += $saldo_acreedor;
            $html_ingresos .= '<tr><td>(' . htmlspecialchars($codigo) . ') ' . htmlspecialchars($cuenta['nombre']) . '</td><td class="text-end">' . number_format($saldo_acreedor, 2, ',', '.') . '</td></tr>';
        }
        if ($codigo[0] == '5' && $saldo != 0) {
            $saldo_deudor = $saldo;
            $total_gastos += $saldo_deudor;
            $html_gastos .= '<tr><td>(' . htmlspecialchars($codigo) . ') ' . htmlspecialchars($cuenta['nombre']) . '</td><td class="text-end">' . number_format($saldo_deudor, 2, ',', '.') . '</td></tr>';
        }
    }
    
    $resultado = $total_ingresos - $total_gastos;
    
    echo '<!DOCTYPE html><html lang="es"><head><meta charset="UTF-8"><title>Estado de Resultados</title><style>body{font-family:sans-serif;margin:20px}table{width:100%;border-collapse:collapse}th,td{padding:8px;text-align:left}.text-end{text-align:right}.header{text-align:center;margin-bottom:20px}h1,h2,h3{margin:0}.total-row{font-weight:bold;border-top:2px solid #333}.gran-total{font-size:1.1em;font-weight:bold;border-top:3px double #333;margin-top:10px}@media print{body{margin:0} @page { size: A4; margin: 2.5cm; }}</style></head><body>';
    echo '<div class="header"><h1>Estado de Resultados</h1>';
    echo '<h2>' . htmlspecialchars($empresa['nombre']) . '</h2>';
    echo '<h3>RUC: ' . htmlspecialchars($empresa['ruc']) . '</h3>';
    echo '<p>Período del ' . htmlspecialchars($fecha_desde) . ' al ' . htmlspecialchars($fecha_hasta) . '</p></div>';
    echo '<h3>Ingresos</h3><table>' . $html_ingresos . '<tr class="total-row"><td><strong>Total Ingresos</strong></td><td class="text-end"><strong>' . number_format($total_ingresos, 2, ',', '.') . '</strong></td></tr></table><br>';
    echo '<h3>Gastos</h3><table>' . $html_gastos . '<tr class="total-row"><td><strong>Total Gastos</strong></td><td class="text-end"><strong>' . number_format($total_gastos, 2, ',', '.') . '</strong></td></tr></table><br>';
    echo '<table class="gran-total"><tr><td><strong>' . ($resultado >= 0 ? 'Utilidad del Ejercicio' : 'Pérdida del Ejercicio') . '</strong></td><td class="text-end"><strong>' . number_format($resultado, 2, ',', '.') . '</strong></td></tr></table>';
    echo '</body></html>';
    exit;
}

/**
 * Genera reporte de Libro Diario
 */
function handle_generar_reporte_libro_diario() {
    header('Content-Type: text/html; charset=utf-8');
    
    $empresa_id = validate_empresa_id($_GET['empresa_id'] ?? '');
    $fecha_desde = $_GET['fecha_desde'] ?? '';
    $fecha_hasta = $_GET['fecha_hasta'] ?? '';
    
    $data = get_processed_mayor_data($empresa_id, $fecha_desde, $fecha_hasta);
    $asientos = $data['asientos'];
    $empresa = $data['empresa'];
    
    usort($asientos, fn($a, $b) => strcmp($a['fecha'], $b['fecha']));
    
    echo '<!DOCTYPE html><html lang="es"><head><meta charset="UTF-8"><title>Libro Diario</title><style>body{font-family:sans-serif;margin:20px}table{width:100%;border-collapse:collapse}th,td{padding:8px;text-align:left;border:1px solid #ddd}.text-end{text-align:right}.header{text-align:center;margin-bottom:20px}h1,h2,h3{margin:0}tfoot{font-weight:bold;background-color:#f2f2f2}@media print{body{margin:0} @page { size: A4; margin: 2.5cm; }}</style></head><body>';
    echo '<div class="header"><h1>Libro Diario</h1>';
    echo '<h2>' . htmlspecialchars($empresa['nombre']) . '</h2>';
    echo '<h3>RUC: ' . htmlspecialchars($empresa['ruc']) . '</h3>';
    echo '<p>Período del ' . htmlspecialchars($fecha_desde) . ' al ' . htmlspecialchars($fecha_hasta) . '</p></div>';
    
    echo '<table><thead><tr><th>Fecha</th><th>Cuenta</th><th>Descripción</th><th class="text-end">Debe</th><th class="text-end">Haber</th></tr></thead><tbody>';
    
    $totalDebe = 0;
    $totalHaber = 0;
    
    foreach ($asientos as $asiento) {
        echo '<tr style="background-color:#f8f9fa; font-weight:bold;"><td colspan="2">Asiento #' . substr($asiento['id'], 0, 6) . '</td><td colspan="3">' . htmlspecialchars($asiento['descripcion']) . ' (' . htmlspecialchars($asiento['comprobante_nro']) . ')</td></tr>';
        foreach ($asiento['movimientos'] as $mov) {
            $debe = floatval($mov['debe']);
            $haber = floatval($mov['haber']);
            $totalDebe += $debe;
            $totalHaber += $haber;
            
            echo '<tr><td>' . htmlspecialchars($asiento['fecha']) . '</td><td>(' . htmlspecialchars($mov['cuenta_codigo']) . ') ' . htmlspecialchars($mov['descripcion']) . '</td><td></td><td class="text-end">' . ($debe > 0 ? number_format($debe, 2, ',', '.') : '') . '</td><td class="text-end">' . ($haber > 0 ? number_format($haber, 2, ',', '.') : '') . '</td></tr>';
        }
    }
    
    echo '</tbody><tfoot><tr><td colspan="3" style="text-align:right;"><strong>TOTALES</strong></td><td class="text-end"><strong>' . number_format($totalDebe, 2, ',', '.') . '</strong></td><td class="text-end"><strong>' . number_format($totalHaber, 2, ',', '.') . '</strong></td></tr></tfoot></table>';
    echo '</body></html>';
    exit;
}

/**
 * Genera reporte de Sumas y Saldos
 */
function handle_generar_reporte_sumas_y_saldos() {
    header('Content-Type: text/html; charset=utf-8');
    
    $empresa_id = validate_empresa_id($_GET['empresa_id'] ?? '');
    $fecha_desde = $_GET['fecha_desde'] ?? '';
    $fecha_hasta = $_GET['fecha_hasta'] ?? '';
    
    $data = get_processed_mayor_data($empresa_id, $fecha_desde, $fecha_hasta);
    $mayor = $data['mayor'];
    $plan = $data['plan'];
    $empresa = $data['empresa'];
    
    echo '<!DOCTYPE html><html lang="es"><head><meta charset="UTF-8"><title>Sumas y Saldos</title><style>body{font-family:sans-serif;margin:20px}table{width:100%;border-collapse:collapse}th,td{padding:8px;text-align:left;border:1px solid #ddd}.text-end{text-align:right}.header{text-align:center;margin-bottom:20px}h1,h2,h3{margin:0}tfoot{font-weight:bold;background-color:#f2f2f2}@media print{body{margin:0} @page { size: A4; margin: 2.5cm; }}</style></head><body>';
    echo '<div class="header"><h1>Balance de Comprobación de Sumas y Saldos</h1>';
    echo '<h2>' . htmlspecialchars($empresa['nombre']) . '</h2>';
    echo '<h3>RUC: ' . htmlspecialchars($empresa['ruc']) . '</h3>';
    echo '<p>Período del ' . htmlspecialchars($fecha_desde) . ' al ' . htmlspecialchars($fecha_hasta) . '</p></div>';
    
    echo '<table><thead><tr><th rowspan="2">Código</th><th rowspan="2">Cuenta</th><th colspan="2" style="text-align:center;">Sumas</th><th colspan="2" style="text-align:center;">Saldos</th></tr><tr><th class="text-end">Debe</th><th class="text-end">Haber</th><th class="text-end">Deudor</th><th class="text-end">Acreedor</th></tr></thead><tbody>';
    
    $granTotalDebe = 0;
    $granTotalHaber = 0;
    $granTotalDeudor = 0;
    $granTotalAcreedor = 0;
    
    foreach ($plan as $cuenta_plan) {
        $codigo = $cuenta_plan['codigo'];
        $cuenta = $mayor[$codigo];
        if ($cuenta['totalDebe'] > 0 || $cuenta['totalHaber'] > 0) {
            $saldo = $cuenta['totalDebe'] - $cuenta['totalHaber'];
            $granTotalDebe += $cuenta['totalDebe'];
            $granTotalHaber += $cuenta['totalHaber'];
            
            $saldoDeudor = $saldo > 0 ? $saldo : 0;
            $saldoAcreedor = $saldo < 0 ? -$saldo : 0;
            $granTotalDeudor += $saldoDeudor;
            $granTotalAcreedor += $saldoAcreedor;
            
            echo '<tr><td>' . htmlspecialchars($codigo) . '</td><td>' . htmlspecialchars($cuenta['nombre']) . '</td><td class="text-end">' . number_format($cuenta['totalDebe'], 2, ',', '.') . '</td><td class="text-end">' . number_format($cuenta['totalHaber'], 2, ',', '.') . '</td><td class="text-end">' . ($saldoDeudor > 0 ? number_format($saldoDeudor, 2, ',', '.') : '') . '</td><td class="text-end">' . ($saldoAcreedor > 0 ? number_format($saldoAcreedor, 2, ',', '.') : '') . '</td></tr>';
        }
    }
    
    echo '</tbody><tfoot><tr><td colspan="2" style="text-align:right;"><strong>TOTALES</strong></td><td class="text-end"><strong>' . number_format($granTotalDebe, 2, ',', '.') . '</strong></td><td class="text-end"><strong>' . number_format($granTotalHaber, 2, ',', '.') . '</strong></td><td class="text-end"><strong>' . number_format($granTotalDeudor, 2, ',', '.') . '</strong></td><td class="text-end"><strong>' . number_format($granTotalAcreedor, 2, ',', '.') . '</strong></td></tr></tfoot></table>';
    echo '</body></html>';
    exit;
}

/**
 * Genera reporte de Plan de Cuentas
 */
function handle_generar_reporte_plan_de_cuentas() {
    header('Content-Type: text/html; charset=utf-8');
    
    $pdo = getDB();
    $empresa_id = validate_empresa_id($_GET['empresa_id'] ?? '');
    
    $stmt = $pdo->prepare("SELECT * FROM empresas WHERE id = ?");
    $stmt->execute([$empresa_id]);
    $empresa = $stmt->fetch();
    
    $stmt = $pdo->prepare("SELECT codigo, nombre, tipo FROM plan_cuentas WHERE empresa_id = ? ORDER BY codigo");
    $stmt->execute([$empresa_id]);
    $plan = $stmt->fetchAll();
    
    echo '<!DOCTYPE html><html lang="es"><head><meta charset="UTF-8"><title>Plan de Cuentas</title><style>body{font-family:sans-serif;margin:20px}table{width:100%;border-collapse:collapse}th,td{padding:8px;text-align:left;border:1px solid #ddd}.header{text-align:center;margin-bottom:20px}h1,h2,h3{margin:0}.titulo{font-weight:bold;background-color:#f2f2f2}@media print{body{margin:0} @page { size: A4; margin: 2.5cm; }}</style></head><body>';
    echo '<div class="header"><h1>Plan de Cuentas</h1>';
    echo '<h2>' . htmlspecialchars($empresa['nombre']) . '</h2>';
    echo '<h3>RUC: ' . htmlspecialchars($empresa['ruc']) . '</h3></div>';
    
    echo '<table><thead><tr><th>Código</th><th>Nombre</th></tr></thead><tbody>';
    foreach ($plan as $cuenta) {
        $clase = $cuenta['tipo'] === 'titulo' ? 'class="titulo"' : '';
        echo "<tr {$clase}><td>" . htmlspecialchars($cuenta['codigo']) . "</td><td>" . htmlspecialchars($cuenta['nombre']) . "</td></tr>";
    }
    echo '</tbody></table></body></html>';
    exit;
}

/**
 * Genera reporte de IVA
 */
function handle_generar_reporte_iva() {
    header('Content-Type: text/html; charset=utf-8');
    
    $empresa_id = validate_empresa_id($_GET['empresa_id'] ?? '');
    
    // Simular llamada a get_iva_report_data
    ob_start();
    handle_get_iva_report_data();
    $reporte_json = ob_get_clean();
    $reporte = json_decode($reporte_json, true);
    
    $pdo = getDB();
    $stmt = $pdo->prepare("SELECT * FROM empresas WHERE id = ?");
    $stmt->execute([$empresa_id]);
    $empresa = $stmt->fetch();
    
    $fecha_desde = $_GET['fecha_desde'] ?? '';
    $fecha_hasta = $_GET['fecha_hasta'] ?? '';
    
    $total_iva_debito = $reporte['ventas']['10']['iva'] + $reporte['ventas']['5']['iva'];
    $total_iva_credito = $reporte['compras']['10']['iva'] + $reporte['compras']['5']['iva'];
    $resultado = $total_iva_debito - $total_iva_credito;
    $resultado_texto = $resultado >= 0 ? 'IVA A PAGAR' : 'SALDO A FAVOR DEL CONTRIBUYENTE';
    
    $render_table = function($titulo, $data) {
        $html = "<h4>{$titulo}</h4><table class='detail-table'><thead><tr><th>Tipo IVA</th><th>Cant. Facturas</th><th class='text-end'>Total Facturado</th><th class='text-end'>Base Imponible</th><th class='text-end'>Total IVA</th></tr></thead><tbody>";
        $total_count = 0; $total_total = 0; $total_base = 0; $total_iva = 0;
        foreach ($data as $tipo => $valores) {
            if ($valores['count'] == 0) continue;
            $html .= "<tr><td>IVA {$tipo}%</td><td>{$valores['count']}</td><td class='text-end'>" . number_format($valores['total'], 2, ',', '.') . "</td><td class='text-end'>" . number_format($valores['base'], 2, ',', '.') . "</td><td class='text-end'>" . number_format($valores['iva'], 2, ',', '.') . "</td></tr>";
            $total_count += $valores['count']; $total_total += $valores['total']; $total_base += $valores['base']; $total_iva += $valores['iva'];
        }
        $html .= "</tbody><tfoot><tr><td><strong>TOTAL</strong></td><td><strong>{$total_count}</strong></td><td class='text-end'><strong>" . number_format($total_total, 2, ',', '.') . "</strong></td><td class='text-end'><strong>" . number_format($total_base, 2, ',', '.') . "</strong></td><td class='text-end'><strong>" . number_format($total_iva, 2, ',', '.') . "</strong></td></tr></tfoot></table>";
        return $html;
    };
    
    echo '<!DOCTYPE html><html lang="es"><head><meta charset="UTF-8"><title>Liquidación de IVA</title><style>body{font-family:sans-serif;margin:20px} .detail-table{width:100%;border-collapse:collapse;font-size:1em;margin-bottom:25px} .summary-table{width:60%;margin:20px auto;border-collapse:collapse;font-size:1.2em} th,td{padding:8px;text-align:left;border-bottom:1px solid #ddd} .detail-table th, .detail-table td {border: 1px solid #ddd;} .detail-table thead {background-color: #f2f2f2;} .detail-table tfoot {font-weight: bold; background-color: #e9ecef;} .text-end{text-align:right} .header{text-align:center;margin-bottom:40px} h1,h2,h3,h4{margin:0; margin-bottom: 10px;} .total-row{font-weight:bold;border-top:2px solid #333;font-size:1.3em} @media print{body{margin:0} @page { size: A4; margin: 2.5cm; }}</style></head><body>';
    echo '<div class="header"><h1>Liquidación de Impuesto al Valor Agregado (IVA)</h1>';
    echo '<h2>' . htmlspecialchars($empresa['nombre']) . '</h2>';
    echo '<h3>RUC: ' . htmlspecialchars($empresa['ruc']) . '</h3>';
    echo '<p>Período del ' . htmlspecialchars($fecha_desde) . ' al ' . htmlspecialchars($fecha_hasta) . '</p></div>';
    echo $render_table('IVA Débito Fiscal (Ventas)', $reporte['ventas']);
    echo $render_table('IVA Crédito Fiscal (Compras)', $reporte['compras']);
    echo '<hr style="margin: 40px 0;"><h3 style="text-align:center;">Resumen de Liquidación</h3>';
    echo '<table class="summary-table">';
    echo '<tr><td>(+) Total IVA Débito Fiscal</td><td class="text-end">' . number_format($total_iva_debito, 2, ',', '.') . '</td></tr>';
    echo '<tr><td>(-) Total IVA Crédito Fiscal</td><td class="text-end">' . number_format($total_iva_credito, 2, ',', '.') . '</td></tr>';
    echo '<tr class="total-row"><td>' . $resultado_texto . '</td><td class="text-end">' . number_format(abs($resultado), 2, ',', '.') . '</td></tr>';
    echo '</table></body></html>';
    exit;
}

// --- ENRUTADOR PRINCIPAL ---

$action = $_REQUEST['action'] ?? '';
$method = $_SERVER['REQUEST_METHOD'];

if ($method === 'GET') {
    // Acciones que devuelven HTML
    if ($action == 'generar_reporte_balance_general') {
        check_auth();
        handle_generar_reporte_balance_general();
    }
    if ($action == 'generar_reporte_estado_resultados') {
        check_auth();
        handle_generar_reporte_estado_resultados();
    }
    if ($action == 'generar_reporte_libro_diario') {
        check_auth();
        handle_generar_reporte_libro_diario();
    }
    if ($action == 'generar_reporte_sumas_y_saldos') {
        check_auth();
        handle_generar_reporte_sumas_y_saldos();
    }
    if ($action == 'generar_reporte_plan_de_cuentas') {
        check_auth();
        handle_generar_reporte_plan_de_cuentas();
    }
    if ($action == 'generar_reporte_iva') {
        check_auth();
        handle_generar_reporte_iva();
    }
    if ($action == 'export_ventas' || $action == 'export_compras') {
        check_auth();
        handle_export_asientos();
    }
    if ($action == 'logout') {
        handle_logout();
    }

    // Acciones que devuelven JSON
    header('Content-Type: application/json');
    switch ($action) {
        case 'check_session':
            handle_check_session();
            break;
        case 'get_empresas':
            check_auth();
            handle_get_empresas();
            break;
        case 'get_digitadores':
            check_auth();
            handle_get_digitadores();
            break;
        case 'get_contadores':
            check_auth();
            handle_get_contadores();
            break;
        case 'get_vencimientos':
            check_auth();
            handle_get_vencimientos();
            break;
        case 'get_plan_de_cuentas':
            check_auth();
            handle_get_plan_de_cuentas();
            break;
        case 'get_asientos':
            check_auth();
            handle_get_asientos();
            break;
        case 'get_users':
            check_admin_auth();
            handle_get_users();
            break;
        case 'get_activity_log':
            check_auth();
            handle_get_activity_log();
            break;
        case 'get_settings':
            check_admin_auth();
            handle_get_settings();
            break;
        case 'get_iva_report_data':
            check_auth();
            handle_get_iva_report_data();
            break;
        default:
            http_response_code(404);
            echo json_encode(['error' => 'Acción GET no encontrada']);
            break;
    }
} elseif ($method === 'POST') {
    header('Content-Type: application/json');

    $is_json_request = isset($_SERVER['CONTENT_TYPE']) && strpos($_SERVER['CONTENT_TYPE'], 'application/json') !== false;
    if ($is_json_request) {
        $input = json_decode(file_get_contents('php://input'), true) ?? [];
    } else {
        $input = [];
    }

    if ($action === 'login') {
        handle_login($input);
        exit;
    }

    if ($action === 'register') {
        handle_register($input);
        exit;
    }

    check_auth();

    $post_actions = [
        'create_empresa' => fn() => handle_create_empresa(),
        'update_empresa' => fn() => handle_update_empresa(),
        'delete_empresa' => fn() => handle_delete_empresa($input),
        'update_plan_de_cuentas' => fn() => handle_update_plan_de_cuentas($input),
        'create_asiento' => fn() => handle_create_asiento($input),
        'update_asiento' => fn() => handle_update_asiento($input),
        'delete_asiento' => fn() => handle_delete_asiento($input),
        'save_manual_asiento' => fn() => handle_save_manual_asiento($input),
        'create_user' => fn() => handle_create_user($input),
        'update_user' => fn() => handle_update_user($input),
        'delete_user' => fn() => handle_delete_user($input),
        'save_settings' => fn() => handle_save_settings($input),
        'analizar_con_ia' => fn() => handle_analizar_con_ia($input),
    ];

    if (isset($post_actions[$action])) {
        $post_actions[$action]();
    } else {
        http_response_code(404);
        echo json_encode(['error' => 'Acción POST no encontrada']);
    }
}

exit;
?>
