/**
 * LexiPro - JavaScript Global
 * Version 1.0.0
 */

// Objeto global LexiPro
const LexiPro = {
    // Configuración
    config: {
        apiUrl: '/api.php',
        loadingClass: 'loading',
        errorClass: 'error'
    },

    // Inicialización
    init: function() {
        this.setupEventListeners();
        this.setupTooltips();
        this.setupModals();
        console.log('LexiPro inicializado');
    },

    // Event Listeners
    setupEventListeners: function() {
        // Confirmación de eliminación
        document.querySelectorAll('[data-confirm]').forEach(el => {
            el.addEventListener('click', function(e) {
                const message = this.getAttribute('data-confirm') || '¿Está seguro?';
                if (!confirm(message)) {
                    e.preventDefault();
                    return false;
                }
            });
        });

        // Formularios AJAX
        document.querySelectorAll('form[data-ajax]').forEach(form => {
            form.addEventListener('submit', function(e) {
                e.preventDefault();
                LexiPro.submitAjaxForm(this);
            });
        });
    },

    // Tooltips
    setupTooltips: function() {
        document.querySelectorAll('[data-tooltip]').forEach(el => {
            el.addEventListener('mouseenter', function() {
                const tooltip = this.getAttribute('data-tooltip');
                LexiPro.showTooltip(this, tooltip);
            });
        });
    },

    // Modales
    setupModals: function() {
        // Abrir modal
        document.querySelectorAll('[data-modal]').forEach(trigger => {
            trigger.addEventListener('click', function(e) {
                e.preventDefault();
                const modalId = this.getAttribute('data-modal');
                LexiPro.openModal(modalId);
            });
        });

        // Cerrar modal
        document.querySelectorAll('.modal-close').forEach(closeBtn => {
            closeBtn.addEventListener('click', function() {
                const modal = this.closest('.modal');
                LexiPro.closeModal(modal);
            });
        });

        // Cerrar al hacer clic fuera
        document.querySelectorAll('.modal').forEach(modal => {
            modal.addEventListener('click', function(e) {
                if (e.target === this) {
                    LexiPro.closeModal(this);
                }
            });
        });
    },

    // Abrir modal
    openModal: function(modalId) {
        const modal = document.getElementById(modalId);
        if (modal) {
            modal.classList.add('active');
            document.body.style.overflow = 'hidden';
        }
    },

    // Cerrar modal
    closeModal: function(modal) {
        if (typeof modal === 'string') {
            modal = document.getElementById(modal);
        }
        if (modal) {
            modal.classList.remove('active');
            document.body.style.overflow = '';
        }
    },

    // Enviar formulario AJAX
    submitAjaxForm: function(form) {
        const formData = new FormData(form);
        const url = form.getAttribute('action') || this.config.apiUrl;
        
        this.showLoader(form);

        fetch(url, {
            method: 'POST',
            body: formData
        })
        .then(response => response.json())
        .then(data => {
            this.hideLoader(form);
            if (data.success) {
                this.showNotification(data.message || 'Operación exitosa', 'success');
                if (data.redirect) {
                    setTimeout(() => window.location.href = data.redirect, 1000);
                }
            } else {
                this.showNotification(data.message || 'Error en la operación', 'error');
            }
        })
        .catch(error => {
            this.hideLoader(form);
            this.showNotification('Error de conexión', 'error');
            console.error('Error:', error);
        });
    },

    // Mostrar loader
    showLoader: function(element) {
        if (!element) return;
        element.classList.add(this.config.loadingClass);
        const loader = document.createElement('div');
        loader.className = 'loader';
        loader.setAttribute('data-loader', 'true');
        element.appendChild(loader);
    },

    // Ocultar loader
    hideLoader: function(element) {
        if (!element) return;
        element.classList.remove(this.config.loadingClass);
        const loader = element.querySelector('[data-loader]');
        if (loader) loader.remove();
    },

    // Mostrar notificación
    showNotification: function(message, type = 'info', duration = 4000) {
        const notification = document.createElement('div');
        notification.className = `notification notification-${type}`;
        notification.innerHTML = `
            <div class="notification-content">
                <span class="notification-icon">${this.getNotificationIcon(type)}</span>
                <span class="notification-message">${message}</span>
            </div>
        `;
        
        // Agregar estilos inline si no existen
        if (!document.getElementById('notification-styles')) {
            const style = document.createElement('style');
            style.id = 'notification-styles';
            style.textContent = `
                .notification {
                    position: fixed;
                    top: 20px;
                    right: 20px;
                    padding: 15px 20px;
                    border-radius: 8px;
                    box-shadow: 0 4px 12px rgba(0,0,0,0.15);
                    z-index: 9999;
                    animation: slideIn 0.3s ease;
                }
                .notification-success { background: #10b981; color: white; }
                .notification-error { background: #ef4444; color: white; }
                .notification-warning { background: #f59e0b; color: white; }
                .notification-info { background: #3b82f6; color: white; }
                .notification-content { display: flex; align-items: center; gap: 10px; }
                @keyframes slideIn {
                    from { transform: translateX(100%); opacity: 0; }
                    to { transform: translateX(0); opacity: 1; }
                }
            `;
            document.head.appendChild(style);
        }

        document.body.appendChild(notification);

        setTimeout(() => {
            notification.style.animation = 'slideIn 0.3s ease reverse';
            setTimeout(() => notification.remove(), 300);
        }, duration);
    },

    // Iconos de notificación
    getNotificationIcon: function(type) {
        const icons = {
            success: '✓',
            error: '✗',
            warning: '⚠',
            info: 'ℹ'
        };
        return icons[type] || icons.info;
    },

    // Mostrar tooltip
    showTooltip: function(element, text) {
        // Implementación simple de tooltip
        const tooltip = document.createElement('div');
        tooltip.className = 'tooltip';
        tooltip.textContent = text;
        document.body.appendChild(tooltip);

        const rect = element.getBoundingClientRect();
        tooltip.style.position = 'absolute';
        tooltip.style.top = (rect.top - tooltip.offsetHeight - 10) + 'px';
        tooltip.style.left = (rect.left + rect.width / 2 - tooltip.offsetWidth / 2) + 'px';

        element.addEventListener('mouseleave', function() {
            tooltip.remove();
        }, { once: true });
    },

    // Formatear fecha
    formatDate: function(date, format = 'dd/mm/yyyy') {
        const d = new Date(date);
        const day = String(d.getDate()).padStart(2, '0');
        const month = String(d.getMonth() + 1).padStart(2, '0');
        const year = d.getFullYear();

        return format
            .replace('dd', day)
            .replace('mm', month)
            .replace('yyyy', year);
    },

    // Formatear moneda
    formatMoney: function(amount, currency = 'Gs.') {
        const formatted = new Intl.NumberFormat('es-PY').format(amount);
        return `${currency} ${formatted}`;
    },

    // Validar email
    validateEmail: function(email) {
        const re = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        return re.test(email);
    },

    // Validar RUC paraguayo
    validateRUC: function(ruc) {
        const cleaned = ruc.replace(/[^0-9]/g, '');
        return cleaned.length >= 6 && cleaned.length <= 9;
    },

    // Debounce para búsquedas
    debounce: function(func, wait) {
        let timeout;
        return function executedFunction(...args) {
            const later = () => {
                clearTimeout(timeout);
                func(...args);
            };
            clearTimeout(timeout);
            timeout = setTimeout(later, wait);
        };
    },

    // Hacer petición GET
    get: function(url, callback) {
        fetch(url)
            .then(response => response.json())
            .then(data => callback(null, data))
            .catch(error => callback(error, null));
    },

    // Hacer petición POST
    post: function(url, data, callback) {
        fetch(url, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify(data)
        })
        .then(response => response.json())
        .then(data => callback(null, data))
        .catch(error => callback(error, null));
    },

    // Confirmar acción
    confirm: function(message, callback) {
        if (window.confirm(message)) {
            callback();
        }
    }
};

// Inicializar cuando el DOM esté listo
if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', () => LexiPro.init());
} else {
    LexiPro.init();
}

// Exportar para uso global
window.LexiPro = LexiPro;