document.addEventListener('DOMContentLoaded', () => {
    // Elementos del DOM
    const placeholderIcon = "data:image/svg+xml,%3Csvg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 24 24' fill='%23bdc3c7'%3E%3Cpath d='M12 12c2.21 0 4-1.79 4-4s-1.79-4-4-4-4 1.79-4 4 1.79 4 4 4zm0 2c-2.67 0-8 1.34-8 4v2h16v-2c0-2.66-5.33-4-8-4z'/%3E%3C/svg%3E";

    const API_URL = 'api.php?entity=profesionales';

    const CONFIG_API_URL = 'api.php?entity=configuracion';
    const profesionalForm = document.getElementById('profesional-form');
    const tableBody = document.getElementById('profesionales-table-body');
    const modal = document.getElementById('profesional-modal');
    const openModalBtn = document.getElementById('open-modal-btn');
    const closeBtn = document.querySelector('.close-btn');
    const searchInput = document.getElementById('search-input');
    const tableHead = document.querySelector('#profesionales-table thead');
    const fotoInput = document.getElementById('foto');
    const fotoPreview = document.getElementById('foto-preview');
    const clearFotoBtn = document.getElementById('clear-foto-btn');
    let allProfesionales = []; 

    // Inicializar el input de teléfono para profesionales
    const telefonoInput = document.querySelector("#telefono");
    const iti = window.intlTelInput(telefonoInput, {
        initialCountry: "py", // Paraguay por defecto
        separateDialCode: true,
        utilsScript: "https://cdnjs.cloudflare.com/ajax/libs/intl-tel-input/17.0.8/js/utils.js",
    });

    // --- Control del Modal ---
    const openModal = () => modal.style.display = "block";
    const closeModal = () => {
        modal.style.display = "none";
        profesionalForm.reset(); // Limpia el formulario
        fotoPreview.src = placeholderIcon; 
        document.getElementById('profesional-id').value = ''; 
    };

    openModalBtn.onclick = openModal;
    closeBtn.onclick = closeModal;
    window.onclick = (event) => {
        if (event.target == modal) {
            closeModal();
        }
    }

    // Previsualización de la foto
    fotoInput.addEventListener('input', () => {
        const url = fotoInput.value;
        fotoPreview.src = url && url.startsWith('http') ? url : placeholderIcon;
    });

    // Limpiar campo de foto y vista previa
    clearFotoBtn.addEventListener('click', () => {
        fotoInput.value = '';
        fotoPreview.src = placeholderIcon;
    });

    // --- Lógica de la Aplicación ---

    // Función que solo se encarga de RENDERIZAR la tabla con los datos que le pases
    const renderTable = (profesionales) => {
        tableBody.innerHTML = '';
        profesionales.forEach(prof => {
            let fotoSrc = (prof.foto && prof.foto.startsWith('http')) ? prof.foto : placeholderIcon;
            // Añadir el parámetro anti-caché solo si no es una data:URL
            if (fotoSrc.startsWith('http')) {
                fotoSrc += `?v=${new Date().getTime()}`;
            }
            // Preparar enlace de WhatsApp
            const cleanPhoneNumber = (prof.telefono || '').replace(/\D/g, '');
            const whatsappLink = cleanPhoneNumber ? `https://wa.me/${cleanPhoneNumber}` : '#';
            const whatsappIcon = cleanPhoneNumber ? `<a href="${whatsappLink}" target="_blank" class="whatsapp-icon" title="Enviar WhatsApp"><svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="#25D366"><path d="M19.7,4.3c-1.9-1.9-4.4-2.9-7.1-2.9c-5.6,0-10.1,4.5-10.1,10.1c0,1.8,0.5,3.5,1.3,5l-1.5,5.5l5.6-1.5 c1.4,0.8,3,1.3,4.7,1.3h0c5.6,0,10.1-4.5,10.1-10.1C22.6,8.7,21.6,6.2,19.7,4.3z M12.6,20.5h0c-1.6,0-3.2-0.5-4.5-1.4l-0.3-0.2 l-3.4,0.9l0.9-3.3l-0.2-0.3c-0.9-1.4-1.5-3-1.5-4.7c0-4.5,3.6-8.1,8.1-8.1c2.2,0,4.3,0.9,5.8,2.4c1.5,1.5,2.4,3.5,2.4,5.8 C20.7,16.9,17.1,20.5,12.6,20.5z M18.2,14.8c-0.3-0.1-1.5-0.7-1.7-0.8c-0.2-0.1-0.4-0.1-0.6,0.1c-0.2,0.2-0.6,0.8-0.8,0.9 c-0.1,0.1-0.3,0.2-0.5,0.1c-0.2-0.1-1-0.4-1.9-1.2c-0.7-0.6-1.2-1.4-1.3-1.6c-0.1-0.2,0-0.4,0.1-0.5c0.1-0.1,0.2-0.3,0.4-0.4 c0.1-0.1,0.2-0.2,0.2-0.4c0.1-0.1,0-0.3-0.1-0.4c-0.1-0.1-0.6-1.5-0.8-2c-0.2-0.5-0.4-0.5-0.6-0.5c-0.2,0-0.4,0-0.6,0 c-0.2,0-0.5,0.1-0.8,0.4c-0.3,0.3-1,1-1,2.5c0,1.5,1,2.9,1.2,3.1c0.2,0.2,2,3.2,4.9,4.3c0.7,0.3,1.2,0.4,1.6,0.5 c0.7,0.2,1.3,0.1,1.8-0.1c0.5-0.2,1.5-0.6,1.7-1.2c0.2-0.6,0.2-1.1,0.1-1.2C18.6,14.9,18.4,14.9,18.2,14.8z"></path></svg></a>` : '';

            const row = document.createElement('tr');
            row.innerHTML = `
                <td data-label="Foto"><img src="${fotoSrc}" alt="Foto de ${prof.nombre}" class="profile-pic"></td>
                <td data-label="Nombre">${prof.nombre || ''}</td>
                <td data-label="Apellido">${prof.apellido}</td>
                <td data-label="Especialidad">${prof.especialidad}</td>
                <td data-label="Email">${prof.email}</td>
                <td data-label="Teléfono">${prof.telefono || ''} ${whatsappIcon}</td>
                <td data-label="Estado"><span class="status status-${prof.status || 'inactivo'}">${prof.status || 'inactivo'}</span></td>
                <td data-label="Acciones">
                    <button class="edit-btn" data-id="${prof.id}">Editar</button>
                    <button class="delete-btn" data-id="${prof.id}">Eliminar</button>
                </td>
            `;
            tableBody.appendChild(row);
        });
    };

    // Función para OBTENER los datos de la API y luego renderizar la tabla
    const fetchAndRenderProfesionales = async () => {
        const response = await fetch(API_URL);
        allProfesionales = await response.json(); // Guardamos los datos en nuestra variable
        
        // Ordenamos inicialmente por nombre al cargar
        allProfesionales.sort((a, b) => a.nombre.localeCompare(b.nombre));

        renderTable(allProfesionales); // Renderizamos la tabla completa
        searchInput.value = ''; // Limpiamos el campo de búsqueda
    };

    profesionalForm.addEventListener('submit', async (e) => {
        e.preventDefault();

        const id = document.getElementById('profesional-id').value;        
        // Construimos el objeto de datos manualmente para asegurar que todo se envíe.
        const profesionalData = {
            nombre: document.getElementById('nombre').value,
            apellido: document.getElementById('apellido').value,
            matricula: document.getElementById('matricula').value,
            especialidad: document.getElementById('especialidad').value,
            email: document.getElementById('email').value,
            telefono: iti.getNumber(), // Obtener el número completo con código de país
            biografia: document.getElementById('biografia').value,
            foto: document.getElementById('foto').value,
            status: document.getElementById('status').value,
            password: document.getElementById('password').value,
            role: document.getElementById('role').value,
        };

        let response;
        let url = API_URL;

        if (id) {
            // --- Lógica de ACTUALIZACIÓN (PUT) ---
            url = `${API_URL}&id=${id}`;
        }

        response = await fetch(url, { // Siempre usamos POST, la API distingue por la presencia del ID en la URL
            method: 'POST',
            headers: {
                // Especificamos que estamos enviando JSON
                'Content-Type': 'application/json'
            },
            body: JSON.stringify(profesionalData)
        });

        const result = await response.json();
        console.log(result.message);

        closeModal(); // Cierra el modal
        fetchAndRenderProfesionales(); // Vuelve a cargar y mostrar todos los profesionales
    });

    // --- Event Listener para el campo de búsqueda ---
    searchInput.addEventListener('input', (e) => {
        const searchTerm = e.target.value.toLowerCase();

        const filteredProfesionales = allProfesionales.filter(prof => {
            const fullName = `${prof.nombre} ${prof.apellido}`.toLowerCase();
            return fullName.includes(searchTerm);
        });

        // Renderizamos la tabla solo con los profesionales filtrados
        renderTable(filteredProfesionales);
    });

    // --- Event Listener para ordenar la tabla ---
    let sortState = { key: 'nombre', order: 'asc' }; // Estado inicial de ordenación

    tableHead.addEventListener('click', (e) => {
        const target = e.target;
        // Comprobamos si hemos hecho clic en una cabecera con 'data-sort'
        if (target.tagName === 'TH' && target.dataset.sort) {
            const sortKey = target.dataset.sort;

            // Si es la misma columna, invertimos el orden. Si no, ordenamos ascendentemente.
            const order = (sortState.key === sortKey && sortState.order === 'asc') ? 'desc' : 'asc';

            allProfesionales.sort((a, b) => {
                const valA = a[sortKey].toLowerCase();
                const valB = b[sortKey].toLowerCase();
                return order === 'asc' ? valA.localeCompare(valB) : valB.localeCompare(valA);
            });

            sortState = { key: sortKey, order: order }; // Actualizamos el estado
            renderTable(allProfesionales); // Volvemos a dibujar la tabla con los datos ordenados
        }
    });

    // --- Event Listener para botones de la tabla (Editar/Eliminar) ---
    tableBody.addEventListener('click', async (e) => {
        const target = e.target;
        const id = target.dataset.id;

        // Botón Eliminar
        if (target.classList.contains('delete-btn')) {
            // Pedimos confirmación al usuario
            const confirmar = confirm('¿Estás seguro de que quieres eliminar a este profesional?');

            if (confirmar) {
                const response = await fetch(`${API_URL}&id=${id}`, {
                    method: 'DELETE'
                });

                const result = await response.json();
                console.log(result.message);

                fetchAndRenderProfesionales(); // Recargamos la tabla
            }
        }

        // Botón Editar
        if (target.classList.contains('edit-btn')) {
            // 1. Encontrar los datos del profesional en nuestro array local 'allProfesionales'
            const profesionalAEditar = allProfesionales.find(p => p.id === id);

            // 2. Rellenar el formulario del modal con esos datos
            document.getElementById('profesional-id').value = profesionalAEditar.id;
            document.getElementById('nombre').value = profesionalAEditar.nombre;
            document.getElementById('apellido').value = profesionalAEditar.apellido;
            document.getElementById('matricula').value = profesionalAEditar.matricula || '';
            document.getElementById('especialidad').value = profesionalAEditar.especialidad;
            document.getElementById('email').value = profesionalAEditar.email;
            iti.setNumber(profesionalAEditar.telefono || ''); // Establecer el número en el input
            document.getElementById('biografia').value = profesionalAEditar.biografia || '';
            document.getElementById('status').value = profesionalAEditar.status || 'activo';
            document.getElementById('foto').value = profesionalAEditar.foto || '';
            document.getElementById('role').value = profesionalAEditar.role || 'profesional';
            fotoPreview.src = (profesionalAEditar.foto && profesionalAEditar.foto.startsWith('http')) ? `${profesionalAEditar.foto}?v=${new Date().getTime()}` : placeholderIcon;
            
            // Limpiar el campo de contraseña por seguridad
            document.getElementById('password').value = '';
            openModal();
        }
    });

    // Carga inicial de profesionales
    fetchAndRenderProfesionales();
});
