<?php
// Función para leer la base de datos
function get_data() {
    $database_file = 'database.json';
    if (!file_exists($database_file)) return null;
    return json_decode(file_get_contents($database_file), true);
}

$paciente_id = $_GET['id'] ?? null;
if (!$paciente_id) die('Error: No se especificó un ID de paciente.');

$start_date = $_GET['start_date'] ?? null;
$end_date = $_GET['end_date'] ?? null;

$data = get_data();
$paciente = null;
$config = $data['configuracion'] ?? [];
$profesionales = $data['profesionales'] ?? [];

// Buscar paciente
foreach ($data['pacientes'] as $p) {
    if ($p['id'] === $paciente_id) {
        $paciente = $p;
        break;
    }
}

if (!$paciente) die('Error: Paciente no encontrado.');

// Filtrar sesiones y citas del paciente
$sesiones_paciente = array_filter($data['sesiones'] ?? [], function($s) use ($paciente_id, $start_date, $end_date) {
    if ($s['paciente_id'] !== $paciente_id) return false;
    if ($start_date && strtotime($s['fecha']) < strtotime($start_date)) return false;
    // Para la fecha de fin, incluimos el día completo
    if ($end_date && strtotime($s['fecha']) > strtotime($end_date . ' 23:59:59')) return false;
    return true;
});

$citas_paciente = array_filter($data['citas'] ?? [], function($c) use ($paciente_id, $start_date, $end_date) {
    if ($c['paciente_id'] !== $paciente_id) return false;
    $fecha_cita = $c['fecha'];
    if ($start_date && strtotime($fecha_cita) < strtotime($start_date)) return false;
    if ($end_date && strtotime($fecha_cita) > strtotime($end_date)) return false;
    return true;
});

// Ordenar por fecha (más recientes primero)
usort($sesiones_paciente, fn($a, $b) => strtotime($b['fecha']) - strtotime($a['fecha']));
usort($citas_paciente, fn($a, $b) => strtotime($b['fecha'].' '.$b['hora']) - strtotime($a['fecha'].' '.$a['hora']));

$nombre_completo = htmlspecialchars($paciente['nombre'] . ' ' . $paciente['apellido']);
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <title>Ficha de Paciente - <?php echo $nombre_completo; ?></title>
    <style>
        body { font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, "Helvetica Neue", Arial, sans-serif; line-height: 1.6; color: #333; background-color: #f4f4f4; margin: 0; padding: 0; }
        .page { width: 21cm; min-height: 29.7cm; padding: 2cm; margin: 1cm auto; border: 1px #D3D3D3 solid; border-radius: 5px; background: white; box-shadow: 0 0 5px rgba(0, 0, 0, 0.1); }
        .toolbar { text-align: center; padding: 10px; background-color: #333; }
        .toolbar button { padding: 10px 20px; font-size: 16px; cursor: pointer; background-color: #3498db; color: white; border: none; border-radius: 5px; }
        .report-header { display: flex; justify-content: space-between; align-items: flex-start; border-bottom: 2px solid #3498db; padding-bottom: 15px; margin-bottom: 20px; }
        .report-header .clinic-info img { max-width: 100px; max-height: 100px; }
        .report-header .clinic-info h2 { margin: 0; color: #3498db; }
        .report-header .report-title { text-align: right; }
        .report-header .report-title h1 { margin: 0; font-size: 2em; }
        .section { margin-bottom: 30px; }
        .section h3 { border-bottom: 1px solid #ccc; padding-bottom: 8px; margin-bottom: 15px; color: #3498db; }
        dl { display: grid; grid-template-columns: 200px 1fr; gap: 10px 20px; }
        dt { font-weight: bold; color: #555; }
        dd { margin-left: 0; }
        .nota-item { border: 1px solid #eee; border-left: 4px solid #3498db; padding: 15px; margin-bottom: 15px; border-radius: 5px; page-break-inside: avoid; }
        .nota-header h4 { margin: 0; }
        .nota-meta { font-size: 0.9em; color: #777; margin: 5px 0 10px; }
        .nota-body { font-size: 0.95em; }
        .nota-adjuntos { margin-top: 10px; font-size: 0.9em; }
        .cita-item { display: flex; justify-content: space-between; padding: 8px; border-bottom: 1px solid #eee; }
        @media print {
            body, .page { margin: 0; box-shadow: none; border: none; }
            .toolbar { display: none; }
        }
    </style>
</head>
<body>
    <div class="toolbar">
        <button onclick="window.print()">Imprimir o Guardar como PDF</button>
    </div>
    <div class="page">
        <div class="report-header">
            <div class="clinic-info">
                <?php if (!empty($config['clinica_logo'])): ?>
                    <img src="<?php echo htmlspecialchars($config['clinica_logo']); ?>" alt="Logo Clínica">
                <?php endif; ?>
                <h2><?php echo htmlspecialchars($config['clinica_nombre'] ?? 'Clínica'); ?></h2>
            </div>
            <div class="report-title">
                <h1>Ficha de Paciente</h1>
                <p>Fecha: <?php echo date('d/m/Y'); ?></p>
            </div>
        </div>

        <div class="section">
            <h3>Datos del Paciente</h3>
            <dl>
                <dt>Nombre Completo</dt><dd><?php echo $nombre_completo; ?></dd>
                <dt>Documento</dt><dd><?php echo htmlspecialchars($paciente['dni'] ?: 'No especificado'); ?></dd>
                <dt>Fecha de Nacimiento</dt><dd><?php echo htmlspecialchars($paciente['fecha_nacimiento'] ? date('d/m/Y', strtotime($paciente['fecha_nacimiento'])) : 'No especificada'); ?></dd>
                <dt>Teléfono</dt><dd>
                    <?php 
                        $clean_phone = preg_replace('/\D/', '', $paciente['telefono'] ?? '');
                        $whatsapp_link = $clean_phone ? 'https://wa.me/' . $clean_phone : '#';
                        echo htmlspecialchars($paciente['telefono'] ?: 'No especificado');
                        if ($clean_phone) echo ' <a href="' . $whatsapp_link . '" target="_blank" style="text-decoration: none; vertical-align: middle;" title="Enviar WhatsApp"><svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="#25D366"><path d="M19.7,4.3c-1.9-1.9-4.4-2.9-7.1-2.9c-5.6,0-10.1,4.5-10.1,10.1c0,1.8,0.5,3.5,1.3,5l-1.5,5.5l5.6-1.5 c1.4,0.8,3,1.3,4.7,1.3h0c5.6,0,10.1-4.5,10.1-10.1C22.6,8.7,21.6,6.2,19.7,4.3z M12.6,20.5h0c-1.6,0-3.2-0.5-4.5-1.4l-0.3-0.2 l-3.4,0.9l0.9-3.3l-0.2-0.3c-0.9-1.4-1.5-3-1.5-4.7c0-4.5,3.6-8.1,8.1-8.1c2.2,0,4.3,0.9,5.8,2.4c1.5,1.5,2.4,3.5,2.4,5.8 C20.7,16.9,17.1,20.5,12.6,20.5z M18.2,14.8c-0.3-0.1-1.5-0.7-1.7-0.8c-0.2-0.1-0.4-0.1-0.6,0.1c-0.2,0.2-0.6,0.8-0.8,0.9 c-0.1,0.1-0.3,0.2-0.5,0.1c-0.2-0.1-1-0.4-1.9-1.2c-0.7-0.6-1.2-1.4-1.3-1.6c-0.1-0.2,0-0.4,0.1-0.5c0.1-0.1,0.2-0.3,0.4-0.4 c0.1-0.1,0.2-0.2,0.2-0.4c0.1-0.1,0-0.3-0.1-0.4c-0.1-0.1-0.6-1.5-0.8-2c-0.2-0.5-0.4-0.5-0.6-0.5c-0.2,0-0.4,0-0.6,0 c-0.2,0-0.5,0.1-0.8,0.4c-0.3,0.3-1,1-1,2.5c0,1.5,1,2.9,1.2,3.1c0.2,0.2,2,3.2,4.9,4.3c0.7,0.3,1.2,0.4,1.6,0.5 c0.7,0.2,1.3,0.1,1.8-0.1c0.5-0.2,1.5-0.6,1.7-1.2c0.2-0.6,0.2-1.1,0.1-1.2C18.6,14.9,18.4,14.9,18.2,14.8z"></path></svg></a>';
                    ?>
                </dd>
                <dt>Email</dt><dd><?php echo htmlspecialchars($paciente['email'] ?: 'No especificado'); ?></dd>
                <?php
                    $prof_asignado = 'No asignado';
                    if (!empty($paciente['profesional_asignado_id'])) {
                        foreach ($profesionales as $prof) {
                            if ($prof['id'] === $paciente['profesional_asignado_id']) {
                                $prof_asignado = htmlspecialchars($prof['nombre'] . ' ' . $prof['apellido']);
                                break;
                            }
                        }
                    }
                ?>
                <dt>Profesional Asignado</dt><dd><?php echo $prof_asignado; ?></dd>
                <dt>Motivo de Consulta</dt><dd><?php echo nl2br(htmlspecialchars($paciente['motivo_consulta'] ?: 'No especificado')); ?></dd>
                <dt>Antecedentes</dt><dd><?php echo nl2br(htmlspecialchars($paciente['antecedentes'] ?: 'No especificado')); ?></dd>
            </dl>
        </div>

        <div class="section">
            <h3>Historial de Sesiones (Bitácora)</h3>
            <?php if (empty($sesiones_paciente)): ?>
                <p>No hay notas de sesión registradas para este paciente.</p>
            <?php else: ?>
                <?php foreach ($sesiones_paciente as $sesion): ?>
                    <?php
                        $prof_sesion = 'Desconocido';
                        foreach ($profesionales as $prof) {
                            if ($prof['id'] === $sesion['profesional_id']) {
                                $prof_sesion = htmlspecialchars($prof['nombre'] . ' ' . $prof['apellido']);
                                break;
                            }
                        }
                    ?>
                    <div class="nota-item">
                        <div class="nota-header">
                            <h4><?php echo htmlspecialchars($sesion['titulo']); ?></h4>
                        </div>
                        <p class="nota-meta">
                            Por <strong><?php echo $prof_sesion; ?></strong> el 
                            <?php echo date('d/m/Y', strtotime($sesion['fecha'])); ?>
                        </p>
                        <div class="nota-body">
                            <?php echo $sesion['contenido']; // El contenido ya es HTML ?>
                        </div>
                        <?php if (!empty($sesion['adjuntos'])): ?>
                            <div class="nota-adjuntos">
                                <strong>Adjuntos:</strong>
                                <?php foreach ($sesion['adjuntos'] as $adjunto): ?>
                                    <span><?php echo htmlspecialchars($adjunto['nombre']); ?></span>
                                <?php endforeach; ?>
                            </div>
                        <?php endif; ?>
                    </div>
                <?php endforeach; ?>
            <?php endif; ?>
        </div>

        <div class="section">
            <h3>Historial de Citas</h3>
            <?php if (empty($citas_paciente)): ?>
                <p>No hay citas registradas para este paciente.</p>
            <?php else: ?>
                <?php foreach ($citas_paciente as $cita): ?>
                     <?php
                        $prof_cita = 'Desconocido';
                        foreach ($profesionales as $prof) {
                            if ($prof['id'] === $cita['profesional_id']) {
                                $prof_cita = htmlspecialchars($prof['nombre'] . ' ' . $prof['apellido']);
                                break;
                            }
                        }
                    ?>
                    <div class="cita-item">
                        <span>
                            <strong><?php echo date('d/m/Y H:i', strtotime($cita['fecha'].' '.$cita['hora'])); ?></strong>
                            con <?php echo $prof_cita; ?>
                        </span>
                        <span>Estado: <strong><?php echo htmlspecialchars($cita['estado']); ?></strong></span>
                    </div>
                <?php endforeach; ?>
            <?php endif; ?>
        </div>

    </div>
</body>
</html>