<?php
// Función para leer la base de datos
function get_data() {
    $database_file = 'database.json';
    if (!file_exists($database_file)) return null;
    return json_decode(file_get_contents($database_file), true);
}

$start_date_str = $_GET['start_date'] ?? null;
$end_date_str = $_GET['end_date'] ?? null;

if (!$start_date_str || !$end_date_str) {
    die('Error: Debe especificar una fecha de inicio y una fecha de fin.');
}

$start_date = strtotime($start_date_str);
$end_date = strtotime($end_date_str . ' 23:59:59');

$data = get_data();
$config = $data['configuracion'] ?? [];
$profesionales = $data['profesionales'] ?? [];
$sesiones = $data['sesiones'] ?? [];

$report_data = [];

// Inicializar datos para cada profesional
foreach ($profesionales as $prof) {
    $report_data[$prof['id']] = [
        'nombre' => $prof['nombre'] . ' ' . $prof['apellido'],
        'pacientes_unicos' => [],
        'total_sesiones' => 0
    ];
}

// Procesar sesiones dentro del rango de fechas
foreach ($sesiones as $sesion) {
    $fecha_sesion = strtotime($sesion['fecha']);
    if ($fecha_sesion >= $start_date && $fecha_sesion <= $end_date) {
        $prof_id = $sesion['profesional_id'];
        if (isset($report_data[$prof_id]) && isset($sesion['paciente_id'])) {
            $report_data[$prof_id]['total_sesiones']++;
            // Usamos el ID del paciente como clave para contar pacientes únicos
            $report_data[$prof_id]['pacientes_unicos'][$sesion['paciente_id']] = true;
        }
    }
}

// Finalizar el conteo de pacientes únicos
foreach ($report_data as $prof_id => &$prof_data) {
    $prof_data['total_pacientes_unicos'] = count($prof_data['pacientes_unicos']);
}

// Ordenar por número de sesiones (de mayor a menor)
uasort($report_data, fn($a, $b) => $b['total_sesiones'] - $a['total_sesiones']);

?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <title>Reporte de Actividad por Profesional</title>
    <link rel="stylesheet" href="reporte_print.css">
    <!-- Incluimos Chart.js desde un CDN -->
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
</head>
<body>
    <div class="toolbar">
        <button onclick="window.print()">Imprimir o Guardar como PDF</button>
    </div>
    <div class="page">
        <div class="report-header">
            <h1>Reporte de Actividad por Profesional</h1>
            <p>
                Período del <strong><?php echo date('d/m/Y', $start_date); ?></strong> 
                al <strong><?php echo date('d/m/Y', $end_date); ?></strong>
            </p>
        </div>

        <div class="chart-container">
            <canvas id="activityChart"></canvas>
        </div>

        <h3>Datos Detallados</h3>
        <table>
            <thead>
                <tr>
                    <th>Profesional</th>
                    <th>N° de Pacientes Únicos Atendidos</th>
                    <th>N° Total de Sesiones</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($report_data as $data_row): ?>
                    <tr>
                        <td><?php echo htmlspecialchars($data_row['nombre']); ?></td>
                        <td><?php echo $data_row['total_pacientes_unicos']; ?></td>
                        <td><?php echo $data_row['total_sesiones']; ?></td>
                    </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    </div>

    <script>
        // Pasamos los datos de PHP a JavaScript
        const reportData = <?php echo json_encode(array_values($report_data)); ?>;

        // Extraemos las etiquetas y los datos para el gráfico
        const labels = reportData.map(item => item.nombre);
        const pacientesData = reportData.map(item => item.total_pacientes_unicos);
        const sesionesData = reportData.map(item => item.total_sesiones);

        const ctx = document.getElementById('activityChart').getContext('2d');
        const activityChart = new Chart(ctx, {
            type: 'bar',
            data: {
                labels: labels,
                datasets: [
                    {
                        label: 'Pacientes Únicos',
                        data: pacientesData,
                        backgroundColor: 'rgba(52, 152, 219, 0.5)',
                        borderColor: 'rgba(52, 152, 219, 1)',
                        borderWidth: 1
                    },
                    {
                        label: 'Total Sesiones',
                        data: sesionesData,
                        backgroundColor: 'rgba(26, 188, 156, 0.5)',
                        borderColor: 'rgba(26, 188, 156, 1)',
                        borderWidth: 1
                    }
                ]
            },
            options: {
                responsive: true,
                plugins: {
                    legend: {
                        position: 'top',
                    },
                    title: {
                        display: true,
                        text: 'Comparativa de Actividad por Profesional'
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        ticks: {
                            stepSize: 1 // Asegura que el eje Y vaya de 1 en 1 si los números son pequeños
                        }
                    }
                }
            }
        });
    </script>
</body>
</html>