document.addEventListener('DOMContentLoaded', () => {
    const CONFIG_API_URL = 'api.php?entity=configuracion';
    const PACIENTES_API_URL = 'api.php?entity=pacientes';
    const PROFESIONALES_API_URL = 'api.php?entity=profesionales';
    const SESIONES_API_URL = 'api.php?entity=sesiones';
    const CITAS_API_URL = 'api.php?entity=citas';

    // --- Lógica de Pestañas (Tabs) ---
    const tabLinks = document.querySelectorAll('.tab-link');
    const tabContents = document.querySelectorAll('.tab-content');

    tabLinks.forEach(link => {
        link.addEventListener('click', () => {
            const tabId = link.dataset.tab;
            tabLinks.forEach(l => l.classList.remove('active'));
            tabContents.forEach(c => c.classList.remove('active'));
            link.classList.add('active');
            document.getElementById(tabId).classList.add('active');
        });
    });

    // --- Lógica de Ayuda de Formato ---
    const helpToggle = document.getElementById('markdown-help-toggle');
    const helpText = document.getElementById('markdown-help-text');

    helpToggle.addEventListener('click', () => {
        const isVisible = helpText.style.display === 'block';
        helpText.style.display = isVisible ? 'none' : 'block';
    });

    // --- Lógica del Formulario Colapsable ---
    const toggleFormBtn = document.getElementById('toggle-form-btn');
    const notaFormWrapper = document.getElementById('nota-form-wrapper');

    if (toggleFormBtn && notaFormWrapper) {
        toggleFormBtn.addEventListener('click', () => {
            const isVisible = notaFormWrapper.style.display === 'block';
            notaFormWrapper.style.display = isVisible ? 'none' : 'block';
            toggleFormBtn.classList.toggle('active');
            if (!isVisible) {
                document.getElementById('nota-titulo').focus();
            }
        });
    }

    // --- Lógica de Notas de Sesión ---
    const notasList = document.getElementById('notas-list');
    const notaForm = document.getElementById('nota-form');
    let pacienteId; // Guardaremos el ID del paciente aquí

    let allProfesionales = []; // Guardaremos los profesionales aquí
    let allNotas = []; // Guardaremos las notas aquí

    const renderNotas = (notas) => {
        notasList.innerHTML = '';
        if (notas.length === 0) {
            notasList.innerHTML = '<p>No hay notas para este paciente.</p>';
            return;
        }

        // Ordenar notas por fecha, de más reciente a más antigua
        notas.sort((a, b) => new Date(b.fecha) - new Date(a.fecha));

        notas.forEach(nota => {
            const profesional = allProfesionales.find(p => p.id === nota.profesional_id);
            const nombreProfesional = profesional ? `${profesional.nombre} ${profesional.apellido}` : 'Desconocido';
            
            const notaElement = document.createElement('div');
            notaElement.className = 'nota-item';
            notaElement.innerHTML = `
                <div class="nota-header">
                    <h3>${nota.titulo}</h3>
                    <div class="nota-actions">
                        <button class="edit-nota-btn" data-id="${nota.id}">Editar</button>
                        <button class="delete-nota-btn" data-id="${nota.id}">Eliminar</button>
                    </div>
                </div>
                <p class="nota-meta">
                    Escrito por <strong>${nombreProfesional}</strong> el 
                    ${new Date(nota.fecha).toLocaleDateString('es-ES', { day: '2-digit', month: '2-digit', year: 'numeric' })}
                </p>
                <div class="nota-body">${nota.contenido}</div>
            `;

            // Renderizar adjuntos si existen
            if (nota.adjuntos && nota.adjuntos.length > 0) {
                const adjuntosContainer = document.createElement('div');
                adjuntosContainer.className = 'nota-adjuntos';
                let adjuntosHTML = '<h4>Archivos Adjuntos:</h4>';
                nota.adjuntos.forEach(adjunto => {
                    adjuntosHTML += `
                        <div class="adjunto-item">
                            <a href="${adjunto.url}" target="_blank" title="${adjunto.nombre}">${adjunto.nombre}</a>
                            <button class="delete-adjunto-btn" data-nota-id="${nota.id}" data-adjunto-url="${adjunto.url}" title="Eliminar adjunto">&times;</button>
                        </div>`;
                });
                adjuntosContainer.innerHTML = adjuntosHTML;
                notaElement.appendChild(adjuntosContainer);
            }

            notasList.appendChild(notaElement);
        });
    };

    const fetchAndRenderNotas = async () => {
        const response = await fetch(`${SESIONES_API_URL}&paciente_id=${pacienteId}`);
        allNotas = await response.json();
        renderNotas(allNotas);
    };

    // --- Lógica de Citas del Paciente ---
    const renderCitasPaciente = (citas) => {
        const programadasList = document.getElementById('citas-programadas-list');
        const pasadasList = document.getElementById('citas-pasadas-list');
        programadasList.innerHTML = '';
        pasadasList.innerHTML = '';

        if (citas.length === 0) {
            programadasList.innerHTML = '<p>No hay citas registradas para este paciente.</p>';
            return;
        }

        citas.sort((a, b) => new Date(b.fecha + 'T' + b.hora) - new Date(a.fecha + 'T' + a.hora));

        const hoy = new Date();
        hoy.setHours(0, 0, 0, 0);

        let hayProgramadas = false;
        let hayPasadas = false;

        citas.forEach(cita => {
            const profesional = allProfesionales.find(p => p.id === cita.profesional_id);
            const nombreProfesional = profesional ? `${profesional.nombre} ${profesional.apellido}` : 'Desconocido';
            const fechaCita = new Date(cita.fecha + 'T00:00:00');
            const esPasada = fechaCita < hoy || ['Realizada', 'Cancelada'].includes(cita.estado);

            let actionsHTML = '';
            // Si la cita está confirmada, mostrar el botón para marcarla como realizada.
            if (cita.estado === 'Confirmada') {
                actionsHTML = `<button class="action-btn mark-realizada-btn" data-id="${cita.id}">Marcar Realizada</button>`;
            }

            const citaElement = document.createElement('div');
            citaElement.className = 'cita-item-detalle';
            citaElement.innerHTML = `
                <div class="cita-info-detalle" title="${cita.notas || ''}">
                    <p class="fecha-hora">${new Date(cita.fecha + 'T' + cita.hora).toLocaleString('es-ES', { dateStyle: 'full', timeStyle: 'short' })}</p>
                    <p>Con: <strong>${nombreProfesional}</strong></p>
                    <p class="estado">Estado: <span class="status status-cita-${cita.estado.toLowerCase()}">${cita.estado}</span></p>
                </div>
                <div class="cita-actions-detalle">
                    ${actionsHTML}
                </div>
            `;

            if (esPasada) {
                pasadasList.appendChild(citaElement);
                hayPasadas = true;
            } else {
                programadasList.appendChild(citaElement);
                hayProgramadas = true;
            }
        });

        if (!hayProgramadas) programadasList.innerHTML = '<p>No hay próximas citas.</p>';
        if (!hayPasadas) pasadasList.innerHTML = '<p>No hay citas en el historial.</p>';
    };

    // --- Event Listener para marcar cita como realizada ---
    document.querySelector('.citas-list-container').addEventListener('click', async (e) => {
        if (e.target.classList.contains('mark-realizada-btn')) {
            const citaId = e.target.dataset.id;
            const confirmar = confirm('¿Estás seguro de que quieres marcar esta cita como "Realizada"?');

            if (confirmar) {
                try {
                    await fetch(`${CITAS_API_URL}&id=${citaId}`, {
                        method: 'POST',
                        headers: { 'Content-Type': 'application/json' },
                        body: JSON.stringify({ estado: 'Realizada' })
                    });

                    // Volver a cargar los datos de la página para reflejar el cambio
                    await loadPageData();

                } catch (error) {
                    console.error('Error al actualizar la cita:', error);
                    alert('Hubo un error al actualizar el estado de la cita.');
                }
            }
        }
    });

    // --- Cargar Datos ---
    const loadPageData = async () => {
        // 1. Obtener el ID del paciente de la URL
        const urlParams = new URLSearchParams(window.location.search);
        pacienteId = urlParams.get('id');

        if (!pacienteId) {
            document.querySelector('.container').innerHTML = '<h1>No se especificó un paciente.</h1>';
            return;
        }

        try {
            // 2. Cargar todos los datos necesarios en paralelo
            const [pacientesRes, profesionalesRes, citasRes] = await Promise.all([
                fetch(PACIENTES_API_URL),
                fetch(PROFESIONALES_API_URL),
                fetch(`${CITAS_API_URL}&paciente_id=${pacienteId}`)
            ]);

            const allPacientes = await pacientesRes.json();
            allProfesionales = await profesionalesRes.json(); // Guardar en la variable global
            const citasPaciente = await citasRes.json();

            // 3. Usar la configuración global ya cargada
            document.getElementById('clinic-name').textContent = GLOBAL_CONFIG.clinica_nombre || 'Clinic App';
            if (GLOBAL_CONFIG.clinica_logo) document.getElementById('clinic-logo').src = GLOBAL_CONFIG.clinica_logo;

            // 4. Encontrar el paciente y su profesional
            const paciente = allPacientes.find(p => p.id === pacienteId);
            if (!paciente) {
                document.querySelector('.container').innerHTML = '<h1>Paciente no encontrado.</h1>';
                return;
            }

            const profesional = allProfesionales.find(prof => prof.id === paciente.profesional_asignado_id);

            // Preparar enlace de WhatsApp para el paciente
            const cleanPhoneNumber = (paciente.telefono || '').replace(/\D/g, '');
            const whatsappLink = cleanPhoneNumber ? `https://wa.me/${cleanPhoneNumber}` : '#';
            const whatsappIcon = cleanPhoneNumber ? `<a href="${whatsappLink}" target="_blank" class="whatsapp-icon" title="Enviar WhatsApp"><svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="#25D366"><path d="M19.7,4.3c-1.9-1.9-4.4-2.9-7.1-2.9c-5.6,0-10.1,4.5-10.1,10.1c0,1.8,0.5,3.5,1.3,5l-1.5,5.5l5.6-1.5 c1.4,0.8,3,1.3,4.7,1.3h0c5.6,0,10.1-4.5,10.1-10.1C22.6,8.7,21.6,6.2,19.7,4.3z M12.6,20.5h0c-1.6,0-3.2-0.5-4.5-1.4l-0.3-0.2 l-3.4,0.9l0.9-3.3l-0.2-0.3c-0.9-1.4-1.5-3-1.5-4.7c0-4.5,3.6-8.1,8.1-8.1c2.2,0,4.3,0.9,5.8,2.4c1.5,1.5,2.4,3.5,2.4,5.8 C20.7,16.9,17.1,20.5,12.6,20.5z M18.2,14.8c-0.3-0.1-1.5-0.7-1.7-0.8c-0.2-0.1-0.4-0.1-0.6,0.1c-0.2,0.2-0.6,0.8-0.8,0.9 c-0.1,0.1-0.3,0.2-0.5,0.1c-0.2-0.1-1-0.4-1.9-1.2c-0.7-0.6-1.2-1.4-1.3-1.6c-0.1-0.2,0-0.4,0.1-0.5c0.1-0.1,0.2-0.3,0.4-0.4 c0.1-0.1,0.2-0.2,0.2-0.4c0.1-0.1,0-0.3-0.1-0.4c-0.1-0.1-0.6-1.5-0.8-2c-0.2-0.5-0.4-0.5-0.6-0.5c-0.2,0-0.4,0-0.6,0 c-0.2,0-0.5,0.1-0.8,0.4c-0.3,0.3-1,1-1,2.5c0,1.5,1,2.9,1.2,3.1c0.2,0.2,2,3.2,4.9,4.3c0.7,0.3,1.2,0.4,1.6,0.5 c0.7,0.2,1.3,0.1,1.8-0.1c0.5-0.2,1.5-0.6,1.7-1.2c0.2-0.6,0.2-1.1,0.1-1.2C18.6,14.9,18.4,14.9,18.2,14.8z"></path></svg></a>` : '';

            // 5. Rellenar la tarjeta de cabecera del paciente
            document.getElementById('patient-name').textContent = `${paciente.nombre} ${paciente.apellido}`;
            document.getElementById('patient-dni').textContent = paciente.dni || 'No especificado';
            document.getElementById('patient-phone').innerHTML = `${paciente.telefono || 'No especificado'} ${whatsappIcon}`;
            document.getElementById('patient-email').textContent = paciente.email || 'No especificado';
            document.getElementById('patient-profesional').textContent = profesional ? `${profesional.nombre} ${profesional.apellido}` : 'No asignado';
            
            const statusBadge = document.getElementById('patient-status-badge');
            statusBadge.textContent = paciente.estado || 'inactivo';
            statusBadge.className = `status status-${paciente.estado || 'inactivo'}`;

            // 6. Rellenar la pestaña de Datos Personales
            const detailsContent = document.getElementById('patient-details-content');
            detailsContent.innerHTML = `
                <dl class="details-list">
                    <dt>Nombre Completo</dt>
                    <dd>${paciente.nombre} ${paciente.apellido}</dd>
                    
                    <dt>Documento de Identidad</dt>
                    <dd>${paciente.dni || 'No especificado'}</dd>
                    
                    <dt>Fecha de Nacimiento</dt>
                    <dd>${paciente.fecha_nacimiento ? new Date(paciente.fecha_nacimiento).toLocaleDateString('es-ES', { day: '2-digit', month: '2-digit', year: 'numeric' }) : 'No especificada'}</dd>
                    
                    <dt>Motivo de Consulta</dt>
                    <dd>${paciente.motivo_consulta || 'No especificado'}</dd>
                    
                    <dt>Antecedentes Relevantes</dt>
                    <dd>${paciente.antecedentes || 'No especificado'}</dd>
                </dl>
            `;

            // 7. Cargar las notas de sesión
            await fetchAndRenderNotas();

            // 8. Renderizar las citas del paciente
            renderCitasPaciente(citasPaciente);

        } catch (error) {
            console.error("Error al cargar los datos de la página:", error);
            document.querySelector('.container').innerHTML = '<h1>Error al cargar los datos.</h1>';
        }
    };

    // Event listener para el botón de editar en las notas
    notasList.addEventListener('click', async (e) => {
        if (e.target.classList.contains('edit-nota-btn')) {
            const notaId = e.target.dataset.id;
            const notaAEditar = allNotas.find(n => n.id === notaId);

            if (notaAEditar) {
                // Rellenar el formulario con los datos de la nota
                document.getElementById('nota-id').value = notaAEditar.id;
                document.getElementById('nota-titulo').value = notaAEditar.titulo;
                
                // Para convertir el HTML guardado de nuevo a texto plano para el textarea
                const tempDiv = document.createElement('div');
                tempDiv.innerHTML = notaAEditar.contenido.replace(/<br>/g, '\n');
                document.getElementById('nota-contenido').value = tempDiv.textContent || tempDiv.innerText || '';

                // Cambiar el texto del botón y hacer scroll hacia el formulario
                notaForm.querySelector('button[type="submit"]').textContent = 'Actualizar Nota';
                notaForm.scrollIntoView({ behavior: 'smooth' });

                // Nota: La edición de adjuntos no se implementa aquí. 
                // El usuario tendría que borrar la nota y crearla de nuevo para cambiar un adjunto.
                // Esto simplifica la lógica por ahora.
            }
        }

        if (e.target.classList.contains('delete-nota-btn')) {
            const notaId = e.target.dataset.id;
            
            const confirmar = confirm('¿Estás seguro de que quieres eliminar esta nota de sesión? Esta acción no se puede deshacer.');

            if (confirmar) {
                await fetch(`${SESIONES_API_URL}&id=${notaId}`, {
                    method: 'DELETE'
                });

                // Limpiar el formulario si la nota eliminada era la que se estaba editando
                if (document.getElementById('nota-id').value === notaId) {
                    notaForm.reset();
                    document.getElementById('nota-id').value = '';
                    notaForm.querySelector('button[type="submit"]').textContent = 'Guardar Nota';
                }

                await fetchAndRenderNotas();
            }
        }

        // Eliminar un adjunto específico
        if (e.target.classList.contains('delete-adjunto-btn')) {
            const notaId = e.target.dataset.notaId;
            const adjuntoUrl = e.target.dataset.adjuntoUrl;

            const confirmar = confirm('¿Estás seguro de que quieres eliminar este archivo adjunto?');

            if (confirmar) {
                // Usamos un parámetro 'action' para indicarle a la API qué hacer
                await fetch(`${SESIONES_API_URL}&id=${notaId}&action=delete_attachment`, {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({ url: adjuntoUrl })
                });

                // Recargar las notas para reflejar el cambio
                await fetchAndRenderNotas();
            }
        }
    });

    // Event listener para guardar una nueva nota
    notaForm.addEventListener('submit', async (e) => {
        e.preventDefault();

        const submitButton = notaForm.querySelector('button[type="submit"]');
        submitButton.disabled = true;
        submitButton.textContent = 'Guardando...';

        const adjuntoInput = document.getElementById('nota-adjunto');
        let adjuntos = [];

        // 1. Si hay un archivo, subirlo primero
        if (adjuntoInput.files.length > 0) {
            const formData = new FormData();
            formData.append('file', adjuntoInput.files[0]);

            try {
                const uploadResponse = await fetch('upload.php', {
                    method: 'POST',
                    body: formData
                });
                const uploadResult = await uploadResponse.json();
                if (uploadResult.status === 'success') {
                    adjuntos.push({ nombre: uploadResult.original_name, url: uploadResult.url });
                } else {
                    throw new Error(uploadResult.message);
                }
            } catch (error) {
                alert('Error al subir el archivo adjunto: ' + error.message);
                submitButton.disabled = false;
                submitButton.textContent = 'Guardar Nota';
                return; // Detener el proceso si la subida falla
            }
        }

        let contenido = document.getElementById('nota-contenido').value;

        // --- Convertir Markdown simple a HTML ---
        // Títulos (ej: # Mi Título)
        contenido = contenido.replace(/^# (.*$)/gim, '<h3>$1</h3>');
        // Subtítulos (ej: ## Mi Subtítulo)
        contenido = contenido.replace(/^## (.*$)/gim, '<h4>$1</h4>');
        // Negrita (ej: **texto en negrita**)
        contenido = contenido.replace(/\*\*(.*?)\*\*/g, '<strong>$1</strong>');
        // Cursiva (ej: *texto en cursiva*)
        contenido = contenido.replace(/\*(.*?)\*/g, '<em>$1</em>');
        // Listas de viñetas (ej: - Un elemento)
        contenido = contenido.replace(/^- (.*$)/gim, '<li>$1</li>');
        contenido = contenido.replace(/<\/li>\n<li>/g, '</li><li>'); // Corrige espacios entre items de lista
        contenido = contenido.replace(/(<li>.*<\/li>)/g, '<ul>$1</ul>');
        // Saltos de línea
        contenido = contenido.replace(/\n/g, '<br>');

        const notaId = document.getElementById('nota-id').value;
        
        // Si estamos editando, necesitamos mantener los adjuntos existentes
        if (notaId) {
            const notaExistente = allNotas.find(n => n.id === notaId);
            if (notaExistente && notaExistente.adjuntos) {
                adjuntos = [...notaExistente.adjuntos, ...adjuntos];
            }
        }

        const notaData = {
            paciente_id: pacienteId,
            profesional_id: LOGGED_IN_USER_ID, // Usar el ID del usuario logueado
            titulo: document.getElementById('nota-titulo').value,
            contenido: contenido, // Guardamos el contenido ya convertido a HTML
            fecha: new Date().toISOString(),
            adjuntos: adjuntos // Array con los archivos adjuntos
        };
        
        let url = SESIONES_API_URL;
        if (notaId) {
            url = `${SESIONES_API_URL}&id=${notaId}`;
        }

        await fetch(url, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify(notaData)
        });

        // Limpiar y restaurar el formulario
        notaForm.reset();
        document.getElementById('nota-id').value = '';
        // Ocultar el formulario después de guardar
        if (notaFormWrapper.style.display === 'block') {
            notaFormWrapper.style.display = 'none';
            toggleFormBtn.classList.remove('active');
        }

        submitButton.disabled = false;
        submitButton.textContent = 'Guardar Nota';
        await fetchAndRenderNotas();
    });

    loadPageData();
});