document.addEventListener('DOMContentLoaded', () => {
    /**
     * Carga la configuración global (nombre y logo de la clínica) en el encabezado.
     * La variable GLOBAL_CONFIG es inyectada por _header.php.
     */
    const loadGlobalHeader = () => {
        document.getElementById('clinic-name').textContent = GLOBAL_CONFIG.clinica_nombre || 'Clinic App';
        if (GLOBAL_CONFIG.clinica_logo) document.getElementById('clinic-logo').src = GLOBAL_CONFIG.clinica_logo;
    };
    loadGlobalHeader();

    /**
     * Lógica para el menú de navegación móvil (hamburguesa).
     */
    const mobileMenuBtn = document.getElementById('mobile-menu-btn');
    const mainNav = document.getElementById('main-nav');

    if (mobileMenuBtn && mainNav) {
        mobileMenuBtn.addEventListener('click', () => {
            mainNav.classList.toggle('active');
            mobileMenuBtn.classList.toggle('active');
        });
    }

    /**
     * Lógica para el sistema de notificaciones de citas por confirmar.
     */
    const loadNotifications = async () => {
        const bellBtn = document.getElementById('notification-bell-btn');
        const badge = document.getElementById('notification-badge');
        const panel = document.getElementById('notification-panel');
        const list = document.getElementById('notification-list');

        if (!bellBtn) return; // Si no estamos en una página con header, no hacer nada.

        try {
            // Cargamos los datos necesarios en paralelo
            const [citasRes, pacientesRes, profesionalesRes] = await Promise.all([
                fetch('api.php?entity=citas'),
                fetch('api.php?entity=pacientes'),
                fetch('api.php?entity=profesionales')
            ]);

            const citas = await citasRes.json();
            const pacientes = await pacientesRes.json();
            const profesionales = await profesionalesRes.json();

            const horasConfirmacion = parseInt(GLOBAL_CONFIG.horas_confirmar_cita, 10) || 24;
            const ahora = new Date();
            const limiteConfirmacion = new Date(ahora.getTime() + horasConfirmacion * 60 * 60 * 1000);

            const citasPorConfirmar = citas.filter(c => {
                const fechaCita = new Date(c.fecha + 'T' + c.hora);
                return c.estado === 'Programada' && fechaCita > ahora && fechaCita <= limiteConfirmacion;
            }).sort((a, b) => new Date(a.fecha + 'T' + a.hora) - new Date(b.fecha + 'T' + b.hora));

            // Actualizar el contador y la visibilidad
            if (citasPorConfirmar.length > 0) {
                badge.textContent = citasPorConfirmar.length;
                badge.style.display = 'block';
            } else {
                badge.style.display = 'none';
            }

            // Renderizar la lista de notificaciones
            list.innerHTML = '';
            if (citasPorConfirmar.length > 0) {
                citasPorConfirmar.forEach(cita => {
                    const paciente = pacientes.find(p => p.id === cita.paciente_id);
                    const profesional = profesionales.find(p => p.id === cita.profesional_id);
                    if (!paciente || !profesional) return;

                    const item = document.createElement('div');
                    item.className = 'notification-item';
                    item.dataset.citaId = cita.id;
                    item.innerHTML = `
                        <p>Confirmar cita de <strong>${paciente.nombre} ${paciente.apellido}</strong></p>
                        <p style="color: var(--color-text-light);">${new Date(cita.fecha + 'T' + cita.hora).toLocaleString('es-ES', { dateStyle: 'medium', timeStyle: 'short' })}</p>
                        <div class="actions">
                            <button class="button confirm-btn" data-action="Confirmada">Confirmar</button>
                            <button class="button cancel-btn" data-action="Cancelada">Cancelar</button>
                        </div>
                    `;
                    list.appendChild(item);
                });
            } else {
                list.innerHTML = '<p style="padding: 15px; text-align: center; color: var(--color-text-light);">No hay notificaciones.</p>';
            }

        } catch (error) {
            console.error("Error al cargar notificaciones:", error);
            list.innerHTML = '<p style="padding: 15px; color: var(--color-danger);">Error al cargar.</p>';
        }

        // Control de visibilidad del panel
        bellBtn.addEventListener('click', (e) => {
            e.stopPropagation();
            panel.style.display = panel.style.display === 'block' ? 'none' : 'block';
        });

        // Manejar acciones dentro del panel
        list.addEventListener('click', async (e) => {
            if (e.target.matches('.confirm-btn, .cancel-btn')) {
                const item = e.target.closest('.notification-item');
                const citaId = item.dataset.citaId;
                const newState = e.target.dataset.action;

                await fetch(`api.php?entity=citas&id=${citaId}`, {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({ estado: newState })
                });

                // Recargar las notificaciones para actualizar la lista y el contador
                await loadNotifications();
            }
        });
    };

    // Cargar notificaciones al iniciar
    loadNotifications();
});