// Objeto global para manejar la configuración de formato y las funciones de utilidad.
const AppFormatter = {
    // La configuración se carga desde la variable GLOBAL_CONFIG inyectada en el header.
    config: {
        separador_miles: '.',
        separador_decimales: ',',
        cantidad_decimales: 0,
        simbolo_moneda: 'G.',
    },

    /**
     * Carga la configuración de formato desde la base de datos.
     * @param {object} configData - El objeto de configuración de la API.
     */
    loadConfig(configData) {
        this.config.separador_miles = configData.separador_miles ? configData.separador_miles : this.config.separador_miles;
        this.config.separador_decimales = configData.separador_decimales ? configData.separador_decimales : this.config.separador_decimales;
        // Aseguramos que la cantidad de decimales sea un número, con 0 por defecto.
        this.config.cantidad_decimales = !isNaN(parseInt(configData.cantidad_decimales, 10)) ? parseInt(configData.cantidad_decimales, 10) : this.config.cantidad_decimales;
        this.config.simbolo_moneda = configData.simbolo_moneda ? configData.simbolo_moneda : this.config.simbolo_moneda;
    },

    /**
     * Formatea un número según la configuración cargada.
     * @param {number|string} value - El número a formatear.
     * @param {boolean} [withCurrency=true] - Si se debe incluir el símbolo de la moneda.
     * @returns {string} - El número formateado como una cadena.
     */
    formatNumber(value, withCurrency = true) {
        const num = parseFloat(value);
        if (isNaN(num)) {
            return value; // Devuelve el valor original si no es un número.
        }

        // 1. Convertir el número a una cadena con la cantidad correcta de decimales fijos.
        const fixedNum = num.toFixed(this.config.cantidad_decimales);
        let [integerPart, decimalPart] = fixedNum.split('.');

        // 2. Formatear la parte entera con el separador de miles.
        // Usamos una expresión regular para insertar el separador cada 3 dígitos desde la derecha.
        integerPart = integerPart.replace(/\B(?=(\d{3})+(?!\d))/g, this.config.separador_miles);

        // 3. Unir la parte entera y decimal (si existe) con el separador de decimales configurado.
        const formattedNumber = decimalPart 
            ? integerPart + this.config.separador_decimales + decimalPart
            : integerPart;

        // 4. Añadimos el símbolo de la moneda si se solicita.
        return withCurrency ? `${this.config.simbolo_moneda} ${formattedNumber}` : formattedNumber;
    },
};

// Cargar la configuración global al iniciar el script.
AppFormatter.loadConfig(window.GLOBAL_CONFIG || {});