document.addEventListener('DOMContentLoaded', () => {
    // --- URLs de la API ---
    const INGRESOS_API_URL = 'api.php?entity=financiero'; // 'financiero' se usa para ingresos
    const EGRESOS_API_URL = 'api.php?entity=egresos';
    const PACIENTES_API_URL = 'api.php?entity=pacientes';
    const CONFIG_API_URL = 'api.php?entity=configuracion';

    // --- Elementos del DOM ---
    // Ingresos
    const ingresoModal = document.getElementById('ingreso-modal');
    const openIngresoModalBtn = document.getElementById('open-ingreso-modal-btn');
    const ingresoForm = document.getElementById('ingreso-form');
    const ingresosTableBody = document.getElementById('ingresos-table-body');
    const searchInputIngresos = document.getElementById('search-input-ingresos');
    // Egresos
    const egresoModal = document.getElementById('egreso-modal');
    const openEgresoModalBtn = document.getElementById('open-egreso-modal-btn');
    const egresoForm = document.getElementById('egreso-form');
    const egresosTableBody = document.getElementById('egresos-table-body');
    const searchInputEgresos = document.getElementById('search-input-egresos');
    // Resumen y Filtros
    const summaryIngresos = document.getElementById('summary-ingresos');
    const summaryEgresos = document.getElementById('summary-egresos');
    const summaryBalance = document.getElementById('summary-balance');
    const filterButtons = document.querySelectorAll('.filter-btn');
    const startDateInput = document.getElementById('start-date');
    const endDateInput = document.getElementById('end-date');

    // --- Almacenamiento de datos ---
    let allIngresos = [];
    let allEgresos = [];
    let allPacientes = [];
    let financialChart = null; // Variable para mantener la instancia del gráfico

    // --- Estado de la UI ---
    let currentFilter = {
        period: 'all', // 'today', 'month', 'year', 'range', 'all'
        startDate: null,
        endDate: null
    };

    // --- Control de Modales ---
    const setupModal = (modal, openBtn, form, idField) => {
        const closeBtn = modal.querySelector('.close-btn');
        const open = () => modal.style.display = "block";
        const close = () => {
            modal.style.display = "none";
            form.reset();
            form.querySelector('.items-list')?.remove(); // Limpiar ítems multilínea si existen
            if (idField) document.getElementById(idField).value = '';
        };
        openBtn.onclick = open;
        closeBtn.onclick = close;
        window.addEventListener('click', (event) => {
            if (event.target == modal) close();
        });
        return { open, close };
    };
    const { open: openIngreso, close: closeIngreso } = setupModal(ingresoModal, openIngresoModalBtn, ingresoForm, 'ingreso-id');
    const { open: openEgreso, close: closeEgreso } = setupModal(egresoModal, openEgresoModalBtn, egresoForm, 'egreso-id');

    // --- Lógica de Ítems Multilínea ---
    const setupMultilineItems = (listId, addButtonId, totalInputId) => {
        const list = document.getElementById(listId);
        const addButton = document.getElementById(addButtonId);
        
        const updateTotals = () => {
            let total = 0;
            list.querySelectorAll('.item-row').forEach(row => {
                const cantidad = parseFloat(row.querySelector('.item-cantidad').value) || 0;
                const precio = parseFloat(row.querySelector('.item-precio').value) || 0;
                total += cantidad * precio;
            });
            document.getElementById(totalInputId).value = total; // El valor se guarda sin formato
        };

        const addItem = () => {
            const itemRow = document.createElement('div');
            itemRow.className = 'item-row';
            itemRow.innerHTML = `
                <input type="text" class="item-descripcion" placeholder="Descripción" required>
                <input type="number" class="item-cantidad" value="1" min="1" step="1" placeholder="Cant.">
                <input type="number" class="item-precio" placeholder="Precio Unit." step="0.01" required>
                <button type="button" class="remove-item-btn">-</button>
            `;
            list.appendChild(itemRow);

            itemRow.querySelector('.remove-item-btn').addEventListener('click', () => {
                itemRow.remove();
                updateTotals();
            });

            itemRow.querySelectorAll('input').forEach(input => {
                input.addEventListener('input', updateTotals);
            });
            updateTotals();
        };

        addButton.addEventListener('click', addItem);

        // Función para limpiar y añadir el primer item al abrir el modal
        const reset = () => {
            list.innerHTML = '';
            addItem();
        };

        return { reset, updateTotals };
    };

    const ingresoItemsManager = setupMultilineItems('ingreso-items-list', 'add-ingreso-item-btn', 'monto_total');
    const egresoItemsManager = setupMultilineItems('egreso-items-list', 'add-egreso-item-btn', 'egreso-monto-total');

    openIngresoModalBtn.addEventListener('click', ingresoItemsManager.reset);
    openEgresoModalBtn.addEventListener('click', egresoItemsManager.reset);

    const getItemsFromForm = (listId) => {
        const items = [];
        document.querySelectorAll(`#${listId} .item-row`).forEach(row => {
            items.push({
                descripcion: row.querySelector('.item-descripcion').value,
                cantidad: row.querySelector('.item-cantidad').value,
                precio: row.querySelector('.item-precio').value,
            });
        });
        return items;
    };

    // --- Lógica de Resumen y Filtros ---
    const updateSummary = (ingresos, egresos) => {
        const totalIngresos = ingresos.reduce((sum, item) => sum + parseFloat(item.monto_total), 0);
        const totalEgresos = egresos.reduce((sum, item) => sum + parseFloat(item.monto_total), 0);
        const balance = totalIngresos - totalEgresos;

        summaryIngresos.textContent = AppFormatter.formatNumber(totalIngresos);
        summaryEgresos.textContent = AppFormatter.formatNumber(totalEgresos);
        summaryBalance.textContent = AppFormatter.formatNumber(balance);

        summaryBalance.style.color = balance >= 0 ? 'var(--color-success)' : 'var(--color-danger)';
        summaryBalance.parentElement.style.borderColor = balance >= 0 ? 'var(--color-success)' : 'var(--color-danger)';
    };

    // --- Lógica del Gráfico ---
    const renderFinancialChart = (ingresos, egresos) => {
        const chartCanvas = document.getElementById('financial-overview-chart');
        const dataByDate = {};

        // Agrupar ingresos por fecha
        ingresos.forEach(item => {
            const date = item.fecha;
            if (!dataByDate[date]) dataByDate[date] = { ingresos: 0, egresos: 0 };
            dataByDate[date].ingresos += parseFloat(item.monto_total);
        });

        // Agrupar egresos por fecha
        egresos.forEach(item => {
            const date = item.fecha;
            if (!dataByDate[date]) dataByDate[date] = { ingresos: 0, egresos: 0 };
            dataByDate[date].egresos += parseFloat(item.monto_total);
        });

        const sortedDates = Object.keys(dataByDate).sort((a, b) => new Date(a) - new Date(b));
        
        const labels = sortedDates.map(date => new Date(date).toLocaleDateString('es-ES', { day: '2-digit', month: '2-digit' }));
        const ingresosData = sortedDates.map(date => dataByDate[date].ingresos);
        const egresosData = sortedDates.map(date => dataByDate[date].egresos);

        const chartData = {
            labels: labels,
            datasets: [
                {
                    label: 'Ingresos',
                    data: ingresosData,
                    borderColor: 'var(--color-success)',
                    backgroundColor: 'rgba(39, 174, 96, 0.1)',
                    fill: true,
                    tension: 0.3
                },
                {
                    label: 'Egresos',
                    data: egresosData,
                    borderColor: 'var(--color-danger)',
                    backgroundColor: 'rgba(192, 57, 43, 0.1)',
                    fill: true,
                    tension: 0.3
                }
            ]
        };

        if (financialChart) {
            // Si el gráfico ya existe, actualizamos sus datos
            financialChart.data = chartData;
            financialChart.update();
        } else {
            // Si no, lo creamos
            financialChart = new Chart(chartCanvas, {
                type: 'line',
                data: chartData,
                options: {
                    responsive: true,
                    plugins: { 
                        legend: { position: 'top' },
                        tooltip: {
                            callbacks: {
                                label: function(context) {
                                    let label = context.dataset.label || '';
                                    if (label) label += ': ';
                                    return label + AppFormatter.formatNumber(context.parsed.y);
                                }
                            }
                        }
                    },
                    scales: { y: { beginAtZero: true } }
                }
            });
        }
    };

    // --- Lógica de la Aplicación ---

    const populatePacientesDropdown = () => {
        const pacienteSelect = document.getElementById('paciente_id');
        pacienteSelect.innerHTML = '<option value="">-- Seleccionar Paciente --</option>';
        allPacientes.forEach(p => pacienteSelect.innerHTML += `<option value="${p.id}">${p.nombre} ${p.apellido}</option>`);
    };

    const renderIngresosTable = (ingresos) => {
        ingresosTableBody.innerHTML = '';
        ingresos.forEach(item => {
            const paciente = allPacientes.find(p => p.id === item.paciente_id);
            const row = document.createElement('tr');
            row.innerHTML = `
                <td data-label="Referencia">${item.referencia || 'N/A'}</td>
                <td data-label="Fecha">${new Date(item.fecha).toLocaleDateString('es-ES')}</td>
                <td data-label="Paciente">${paciente ? `${paciente.nombre} ${paciente.apellido}` : 'N/A'}</td>
                <td data-label="Monto" class="monto-ingreso">${AppFormatter.formatNumber(item.monto_total)}</td>
                <td data-label="Estado"><span class="status status-pago-${item.estado.toLowerCase()}">${item.estado}</span></td>
                <td data-label="Acciones" class="actions-cell">
                    <button class="edit-btn" data-id="${item.id}">Editar</button>
                    <button class="delete-btn" data-id="${item.id}">Eliminar</button>
                </td>
            `;
            ingresosTableBody.appendChild(row);
        });
    };

    const renderEgresosTable = (egresos) => {
        egresosTableBody.innerHTML = '';
        egresos.forEach(item => {
            const row = document.createElement('tr');
            row.innerHTML = `
                <td data-label="Fecha">${new Date(item.fecha).toLocaleDateString('es-ES')}</td>
                <td data-label="Concepto">${item.concepto || 'N/A'}</td>
                <td data-label="Proveedor">${item.proveedor || 'N/A'}</td>
                <td data-label="Monto" class="monto-egreso">${AppFormatter.formatNumber(item.monto_total)}</td>
                <td data-label="Acciones" class="actions-cell">
                    <button class="edit-btn" data-id="${item.id}">Editar</button>
                    <button class="delete-btn" data-id="${item.id}">Eliminar</button>
                </td>
            `;
            egresosTableBody.appendChild(row);
        });
    };

    const fetchAllDataAndRender = async () => {
        try {
            const [ingresosRes, egresosRes, pacientesRes, configRes] = await Promise.all([
                fetch(INGRESOS_API_URL),
                fetch(EGRESOS_API_URL),
                fetch(PACIENTES_API_URL),
                fetch(CONFIG_API_URL)
            ]);

            allIngresos = await ingresosRes.json();
            allEgresos = await egresosRes.json();
            allPacientes = await pacientesRes.json();

            document.getElementById('clinic-name').textContent = GLOBAL_CONFIG.clinica_nombre || 'Clinic App';
            if (GLOBAL_CONFIG.clinica_logo) document.getElementById('clinic-logo').src = GLOBAL_CONFIG.clinica_logo;

            allIngresos.sort((a, b) => new Date(b.fecha) - new Date(a.fecha));
            allEgresos.sort((a, b) => new Date(b.fecha) - new Date(a.fecha));

            applyFiltersAndRender(); // Aplicar filtros por defecto al cargar
            populatePacientesDropdown();

        } catch (error) {
            console.error("Error al cargar los datos:", error);
        }
    };

    // --- Manejadores de Formularios ---
    ingresoForm.addEventListener('submit', async (e) => {
        e.preventDefault();
        const id = document.getElementById('ingreso-id').value;
        const data = {
            referencia: document.getElementById('referencia').value,
            fecha: document.getElementById('fecha').value,
            paciente_ruc: document.getElementById('paciente_ruc').value,
            paciente_id: document.getElementById('paciente_id').value,
            monto_total: document.getElementById('monto_total').value,
            metodo_pago: document.getElementById('metodo_pago').value,
            iva_tipo: document.getElementById('iva_tipo').value,
            estado: document.getElementById('estado').value,
            items: getItemsFromForm('ingreso-items-list')
        };
        const url = id ? `${INGRESOS_API_URL}&id=${id}` : INGRESOS_API_URL;
        const response = await fetch(url, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify(data)
        });
        closeIngreso();
        await fetchAllDataAndRender();
    });

    egresoForm.addEventListener('submit', async (e) => {
        e.preventDefault();
        const id = document.getElementById('egreso-id').value;
        const data = {
            fecha: document.getElementById('egreso-fecha').value,
            monto_total: document.getElementById('egreso-monto-total').value,
            proveedor: document.getElementById('egreso-proveedor').value,
            egreso_ruc: document.getElementById('egreso_ruc').value,
            items: getItemsFromForm('egreso-items-list')
        };
        const url = id ? `${EGRESOS_API_URL}&id=${id}` : EGRESOS_API_URL;

        const response = await fetch(url, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify(data)
        });
        closeEgreso();
        await fetchAllDataAndRender();
    });

    // --- Manejadores de Eventos para Tablas ---
    ingresosTableBody.addEventListener('click', async (e) => {
        const target = e.target;
        const id = target.dataset.id;

        if (target.classList.contains('delete-btn')) {
            if (confirm('¿Estás seguro de que quieres eliminar este ingreso?')) {
                await fetch(`${INGRESOS_API_URL}&id=${id}`, { method: 'DELETE' });
                await fetchAllDataAndRender();
            }
        }

        if (target.classList.contains('edit-btn')) {
            const itemAEditar = allIngresos.find(item => item.id === id);
            if (itemAEditar) {
                document.getElementById('ingreso-id').value = itemAEditar.id;
                document.getElementById('referencia').value = itemAEditar.referencia;
                document.getElementById('paciente_ruc').value = itemAEditar.paciente_ruc;
                document.getElementById('fecha').value = itemAEditar.fecha;
                document.getElementById('paciente_id').value = itemAEditar.paciente_id;
                document.getElementById('monto_total').value = itemAEditar.monto_total;
                document.getElementById('metodo_pago').value = itemAEditar.metodo_pago;
                document.getElementById('iva_tipo').value = itemAEditar.iva_tipo;
                document.getElementById('estado').value = itemAEditar.estado;

                // Cargar ítems
                const list = document.getElementById('ingreso-items-list');
                list.innerHTML = '';
                itemAEditar.items.forEach(item => {
                    const itemRow = document.createElement('div');
                    itemRow.className = 'item-row';
                    itemRow.innerHTML = `
                        <input type="text" class="item-descripcion" value="${item.descripcion}" required>
                        <input type="number" class="item-cantidad" value="${item.cantidad}" min="1" step="1">
                        <input type="number" class="item-precio" value="${item.precio}" step="0.01" required>
                        <button type="button" class="remove-item-btn">-</button>`;
                    list.appendChild(itemRow);
                });
                openIngreso();
            }
        }
    });

    egresosTableBody.addEventListener('click', async (e) => {
        const target = e.target;
        const id = target.dataset.id;

        if (target.classList.contains('delete-btn')) {
            if (confirm('¿Estás seguro de que quieres eliminar este egreso?')) {
                await fetch(`${EGRESOS_API_URL}&id=${id}`, { method: 'DELETE' });
                await fetchAllDataAndRender();
            }
        }

        if (target.classList.contains('edit-btn')) {
            const itemAEditar = allEgresos.find(item => item.id === id);
            if (itemAEditar) {
                document.getElementById('egreso-id').value = itemAEditar.id;
                document.getElementById('egreso-fecha').value = itemAEditar.fecha;
                document.getElementById('egreso-monto-total').value = itemAEditar.monto_total;
                document.getElementById('egreso-proveedor').value = itemAEditar.proveedor;
                document.getElementById('egreso_ruc').value = itemAEditar.egreso_ruc;

                const list = document.getElementById('egreso-items-list');
                list.innerHTML = '';
                itemAEditar.items.forEach(item => {
                    const itemRow = document.createElement('div');
                    itemRow.className = 'item-row';
                    itemRow.innerHTML = `
                        <input type="text" class="item-descripcion" value="${item.descripcion}" required>
                        <input type="number" class="item-cantidad" value="${item.cantidad}" min="1" step="1">
                        <input type="number" class="item-precio" value="${item.precio}" step="0.01" required>
                        <button type="button" class="remove-item-btn">-</button>`;
                    list.appendChild(itemRow);
                });
                openEgreso();
            }
        }
    });

    // --- Lógica de Filtros de Fecha ---
    const applyFiltersAndRender = () => {
        const now = new Date();
        let startDate, endDate;

        if (currentFilter.period === 'today') {
            startDate = new Date(now.setHours(0, 0, 0, 0));
            endDate = new Date(now.setHours(23, 59, 59, 999));
        } else if (currentFilter.period === 'month') {
            startDate = new Date(now.getFullYear(), now.getMonth(), 1);
            endDate = new Date(now.getFullYear(), now.getMonth() + 1, 0, 23, 59, 59, 999);
        } else if (currentFilter.period === 'year') {
            startDate = new Date(now.getFullYear(), 0, 1);
            endDate = new Date(now.getFullYear(), 11, 31, 23, 59, 59, 999);
        } else if (currentFilter.period === 'range' && currentFilter.startDate && currentFilter.endDate) {
            startDate = new Date(currentFilter.startDate);
            startDate.setHours(0, 0, 0, 0);
            endDate = new Date(currentFilter.endDate);
            endDate.setHours(23, 59, 59, 999);
        } else { // Por defecto (o si es 'all'), mostrar todo
            renderIngresosTable(allIngresos);
            renderEgresosTable(allEgresos);
            updateSummary(allIngresos, allEgresos);
            return;
        }

        const filteredIngresos = allIngresos.filter(item => new Date(item.fecha) >= startDate && new Date(item.fecha) <= endDate);
        const filteredEgresos = allEgresos.filter(item => new Date(item.fecha) >= startDate && new Date(item.fecha) <= endDate);

        renderIngresosTable(filteredIngresos);
        renderEgresosTable(filteredEgresos);
        updateSummary(filteredIngresos, filteredEgresos);
        renderFinancialChart(filteredIngresos, filteredEgresos);
    };

    filterButtons.forEach(button => {
        button.addEventListener('click', () => {
            filterButtons.forEach(btn => btn.classList.remove('active'));
            button.classList.add('active');
            currentFilter.period = button.dataset.period;
            applyFiltersAndRender();
        });
    });

    [startDateInput, endDateInput].forEach(input => input.addEventListener('change', () => {
        currentFilter.period = 'range';
        currentFilter.startDate = startDateInput.value;
        currentFilter.endDate = endDateInput.value;
        filterButtons.forEach(btn => btn.classList.remove('active'));
        if (currentFilter.startDate && currentFilter.endDate) {
            applyFiltersAndRender();
        }
    }));

    // --- Búsqueda ---
    searchInputIngresos.addEventListener('input', (e) => {
        const searchTerm = e.target.value.toLowerCase();
        const filtered = allIngresos.filter(item => {
            const paciente = allPacientes.find(p => p.id === item.paciente_id);
            const pacienteNombre = paciente ? `${paciente.nombre} ${paciente.apellido}`.toLowerCase() : '';
            return pacienteNombre.includes(searchTerm);
        });
        renderIngresosTable(filtered);
    });

    searchInputEgresos.addEventListener('input', (e) => {
        const searchTerm = e.target.value.toLowerCase();
        const filtered = allEgresos.filter(item => {
            const concepto = (item.concepto || '').toLowerCase();
            const proveedor = (item.proveedor || '').toLowerCase();
            return concepto.includes(searchTerm) || proveedor.includes(searchTerm);
        });
        renderEgresosTable(filtered);
    });

    // Carga inicial de todos los datos
    fetchAllDataAndRender();
});