document.addEventListener('DOMContentLoaded', () => {
    // URLs de la API
    const PACIENTES_API_URL = 'api.php?entity=pacientes';
    const CITAS_API_URL = 'api.php?entity=citas';
    const INGRESOS_API_URL = 'api.php?entity=financiero';
    const ACTIVITY_LOG_API_URL = 'api.php?entity=activity_log';
    const PROFESIONALES_API_URL = 'api.php?entity=profesionales';

    // Elementos del DOM
    const metricPacientesActivos = document.getElementById('metric-pacientes-activos');
    const metricCitasHoy = document.getElementById('metric-citas-hoy');
    const metricIngresosMes = document.getElementById('metric-ingresos-mes');
    const proximasCitasList = document.getElementById('proximas-citas-list');
    const actividadRecienteList = document.getElementById('actividad-reciente-list');
    const citasPorConfirmarList = document.getElementById('citas-por-confirmar-list');

    const loadDashboardData = async () => {
        try {
            // Cargar todos los datos en paralelo, incluyendo el log de actividad
            const [pacientesRes, citasRes, ingresosRes, activityLogRes, profesionalesRes] = await Promise.all([
                fetch(PACIENTES_API_URL),
                fetch(CITAS_API_URL),
                fetch(INGRESOS_API_URL),
                fetch(ACTIVITY_LOG_API_URL),
                fetch(PROFESIONALES_API_URL)
            ]);

            const pacientes = await pacientesRes.json();
            const citas = await citasRes.json();
            const ingresos = await ingresosRes.json();
            const activityLog = await activityLogRes.json();
            const profesionales = await profesionalesRes.json();

            // --- 1. Calcular y mostrar métricas ---
            const pacientesActivos = pacientes.filter(p => p.estado === 'activo').length;
            metricPacientesActivos.textContent = pacientesActivos;

            const hoy = new Date().toISOString().split('T')[0];
            const citasHoy = citas.filter(c => c.fecha === hoy && c.estado !== 'Cancelada').length;
            metricCitasHoy.textContent = citasHoy;

            const mesActual = new Date().getMonth();
            const anioActual = new Date().getFullYear();
            const ingresosMes = ingresos
                .filter(i => {
                    const fechaIngreso = new Date(i.fecha);
                    return fechaIngreso.getMonth() === mesActual && fechaIngreso.getFullYear() === anioActual && i.estado === 'Pagada';
                })
                .reduce((sum, i) => sum + parseFloat(i.monto_total), 0);
            metricIngresosMes.textContent = AppFormatter.formatNumber(ingresosMes);

            // --- 2. Renderizar próximas citas de hoy ---
            const proximasCitas = citas
                .filter(c => c.fecha === hoy && c.estado !== 'Cancelada' && c.estado !== 'Realizada')
                .sort((a, b) => a.hora.localeCompare(b.hora));

            proximasCitasList.innerHTML = '';
            if (proximasCitas.length > 0) {
                proximasCitas.forEach(cita => {
                    const paciente = pacientes.find(p => p.id === cita.paciente_id);
                    const profesional = profesionales.find(p => p.id === cita.profesional_id);
                    const item = document.createElement('div');
                    item.className = 'list-item';
                    item.innerHTML = `
                        <span class="item-time">${cita.hora}</span>
                        <div class="item-info">
                            <p class="item-title">${paciente ? paciente.nombre + ' ' + paciente.apellido : 'Paciente no encontrado'}</p>
                            <p class="item-subtitle">con ${profesional ? profesional.nombre.split(' ')[0] : 'N/A'}</p>
                        </div>
                        <span class="status status-cita-${cita.estado.toLowerCase()}">${cita.estado}</span>
                    `;
                    proximasCitasList.appendChild(item);
                });
            } else {
                proximasCitasList.innerHTML = '<p class="empty-state">No hay más citas programadas para hoy.</p>';
            }

            // --- 3. Renderizar actividad reciente ---
            // Tomamos las 7 actividades más recientes del log
            const actividadReciente = activityLog.slice(0, 7);

            actividadRecienteList.innerHTML = '';
            if (actividadReciente.length > 0) {
                actividadReciente.forEach(act => {
                    const item = document.createElement('div');
                    item.className = 'list-item';
                    const icon = act.action.toLowerCase().includes('sesión') ? '👤' : '📝';
                    const title = act.action;
                    const subtitle = `${act.user} - ${new Date(act.timestamp).toLocaleString('es-ES', { day: 'numeric', month: 'short', hour: '2-digit', minute: '2-digit' })}`;

                    item.innerHTML = `
                        <div class="item-icon">${icon}</div>
                        <div class="item-info">
                            <p class="item-title">${title}</p>
                            <p class="item-subtitle">${subtitle}</p>
                        </div>
                    `;
                    actividadRecienteList.appendChild(item);
                });
            } else {
                actividadRecienteList.innerHTML = '<p class="empty-state">No hay actividad reciente.</p>';
            }

            // --- 4. Renderizar citas por confirmar ---
            const horasConfirmacion = parseInt(GLOBAL_CONFIG.horas_confirmar_cita, 10) || 24;
            const ahora = new Date();
            const limiteConfirmacion = new Date(ahora.getTime() + horasConfirmacion * 60 * 60 * 1000);

            const citasPorConfirmar = citas.filter(c => {
                const fechaCita = new Date(c.fecha + 'T' + c.hora);
                return c.estado === 'Programada' && fechaCita > ahora && fechaCita <= limiteConfirmacion;
            }).sort((a, b) => new Date(a.fecha + 'T' + a.hora) - new Date(b.fecha + 'T' + b.hora));

            citasPorConfirmarList.innerHTML = '';
            if (citasPorConfirmar.length > 0) {
                citasPorConfirmar.forEach(cita => {
                    const paciente = pacientes.find(p => p.id === cita.paciente_id);
                    const profesional = profesionales.find(p => p.id === cita.profesional_id);
                    if (!paciente || !profesional) return;

                    const fechaHoraCita = new Date(cita.fecha + 'T' + cita.hora).toLocaleString('es-ES', { dateStyle: 'full', timeStyle: 'short' });
                    const nombrePaciente = `${paciente.nombre} ${paciente.apellido}`;
                    const nombreProfesional = `${profesional.nombre} ${profesional.apellido}`;
                    
                    // Construir mensaje de WhatsApp
                    const mensajeWhatsapp = encodeURIComponent(`Hola, ${nombrePaciente}, queremos confirmar tu asistencia el dia ${fechaHoraCita}, con el Dr. ${nombreProfesional}. Aguardo tu confirmacion.`);
                    const cleanPhoneNumber = (paciente.telefono || '').replace(/\D/g, '');
                    const whatsappLink = cleanPhoneNumber ? `https://wa.me/${cleanPhoneNumber}?text=${mensajeWhatsapp}` : '#';

                    const card = document.createElement('div');
                    card.className = 'confirmation-card';
                    card.dataset.citaId = cita.id;
                    card.innerHTML = `
                        <p class="patient-name">${nombrePaciente}</p>
                        <p class="appointment-details">
                            ${fechaHoraCita}<br>
                            con <strong>${nombreProfesional}</strong>
                        </p>
                        <div class="confirmation-actions">
                            <a href="${whatsappLink}" target="_blank" class="button whatsapp-btn" ${!cleanPhoneNumber ? 'disabled' : ''}>WhatsApp</a>
                            <button class="button confirm-btn" data-action="Confirmada">Confirmar</button>
                            <button class="button cancel-btn" data-action="Cancelada">Cancelar</button>
                        </div>
                    `;
                    citasPorConfirmarList.appendChild(card);
                });
            } else {
                citasPorConfirmarList.innerHTML = '<p class="empty-state">No hay citas que necesiten confirmación en este momento.</p>';
            }

        } catch (error) {
            console.error("Error al cargar los datos del dashboard:", error);
        }
    };

    // --- Event Listener para los botones de confirmación/cancelación ---
    citasPorConfirmarList.addEventListener('click', async (e) => {
        if (e.target.matches('.confirm-btn, .cancel-btn')) {
            const card = e.target.closest('.confirmation-card');
            const citaId = card.dataset.citaId;
            const newState = e.target.dataset.action;

            // Actualizar en el servidor
            await fetch(`${CITAS_API_URL}&id=${citaId}`, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ estado: newState })
            });

            // Eliminar la tarjeta de la lista con una animación
            card.style.transition = 'opacity 0.5s ease, transform 0.5s ease';
            card.style.opacity = '0';
            card.style.transform = 'scale(0.9)';
            setTimeout(() => {
                card.remove();
                if (citasPorConfirmarList.children.length === 0) {
                    citasPorConfirmarList.innerHTML = '<p class="empty-state">No hay citas que necesiten confirmación en este momento.</p>';
                }
            }, 500);
        }
    });

    // Carga inicial de datos
    loadDashboardData();
});