document.addEventListener('DOMContentLoaded', () => {
    const CONFIG_API_URL = 'api.php?entity=configuracion';
    const form = document.getElementById('config-form');
    const activityLogBody = document.getElementById('activity-log-body');
    const logoInput = document.getElementById('clinica_logo');
    const logoPreview = document.getElementById('logo-preview');
    const clearLogoBtn = document.getElementById('clear-logo-btn');

    // Elementos de los filtros
    const filterStartDate = document.getElementById('filter-start-date');
    const filterEndDate = document.getElementById('filter-end-date');
    const filterUser = document.getElementById('filter-user');
    const applyFiltersBtn = document.getElementById('apply-filters-btn');
    const clearFiltersBtn = document.getElementById('clear-filters-btn');

    // Inicializar el input de teléfono para la configuración
    const telefonoInput = document.querySelector("#clinica_telefono");
    const iti = window.intlTelInput(telefonoInput, {
        initialCountry: "py", // Paraguay por defecto
        separateDialCode: true,
        utilsScript: "https://cdnjs.cloudflare.com/ajax/libs/intl-tel-input/17.0.8/js/utils.js",
    });

    // --- Lógica de Pestañas (Tabs) ---
    const tabLinks = document.querySelectorAll('.tab-link');
    const tabContents = document.querySelectorAll('.tab-content');

    // Ocultar el botón de guardar si no estamos en una pestaña de formulario
    const formActions = document.querySelector('.form-actions');
    const activityTabId = 'tab-activity';

    tabLinks.forEach(link => {
        link.addEventListener('click', () => {
            const tabId = link.dataset.tab;

            // Desactivar todas las pestañas y contenidos
            tabLinks.forEach(l => l.classList.remove('active'));
            tabContents.forEach(c => c.classList.remove('active'));

            // Activar la pestaña y el contenido seleccionados
            link.classList.add('active');
            document.getElementById(tabId).classList.add('active');

            // Si es la pestaña de actividad, la cargamos.
            if (tabId === activityTabId) {
                loadFilterUsers(); // <-- AÑADIMOS ESTA LÍNEA
                loadActivityLog();
                // Ocultamos el formulario y sus acciones
                if (form) form.style.display = 'none';
            } else {
                // Mostramos el formulario para las otras pestañas
                if (form) form.style.display = 'block';
            }
        });
    });

    // --- Lógica para el Registro de Actividad ---
    const loadActivityLog = async (params = {}) => {
        activityLogBody.innerHTML = '<tr><td colspan="3">Cargando...</td></tr>';
        
        // Construir la URL con los parámetros de filtro
        const query = new URLSearchParams(params).toString();
        const url = `api.php?entity=activity_log&${query}`;

        try {
            const response = await fetch(url);
            if (!response.ok) {
                if (response.status === 403) {
                    throw new Error('No tienes permiso para ver el registro de actividad.');
                }
                throw new Error('No tienes permiso para ver el registro de actividad.');
            }
            const logs = await response.json();

            activityLogBody.innerHTML = '';
            if (logs.length > 0) {
                logs.forEach(log => {
                    const row = `<tr>
                        <td>${new Date(log.timestamp).toLocaleString('es-ES')}</td>
                        <td>${log.user}</td>
                        <td>${log.action}</td>
                    </tr>`;
                    activityLogBody.innerHTML += row;
                });
            } else {
                activityLogBody.innerHTML = '<tr><td colspan="3">No hay actividad registrada.</td></tr>';
            }
        } catch (error) {
            activityLogBody.innerHTML = `<tr><td colspan="3" style="color: red;">${error.message}</td></tr>`;
        }
    };

    const loadFilterUsers = async () => {
        try {
            const response = await fetch('api.php?entity=activity_log&action=get_users');
            if (!response.ok) return;
            const users = await response.json();
            
            filterUser.innerHTML = '<option value="">Todos</option>'; // Reset
            users.forEach(user => {
                const option = document.createElement('option');
                option.value = user;
                option.textContent = user;
                filterUser.appendChild(option);
            });
        } catch (error) {
            console.error('Error al cargar usuarios para el filtro:', error);
        }
    };

    // Event Listeners para los filtros
    applyFiltersBtn.addEventListener('click', () => {
        const params = {
            start_date: filterStartDate.value,
            end_date: filterEndDate.value,
            user: filterUser.value,
        };
        // Eliminar parámetros vacíos
        Object.keys(params).forEach(key => params[key] === '' && delete params[key]);
        loadActivityLog(params);
    });

    clearFiltersBtn.addEventListener('click', () => {
        filterStartDate.value = '';
        filterEndDate.value = '';
        filterUser.value = '';
        loadActivityLog();
    });

    // Cargar la configuración existente al cargar la página
    const loadConfig = async () => {
        try {
            // Usamos la configuración global ya cargada en el header
            const config = GLOBAL_CONFIG;

            // Rellenar el encabezado principal
            document.getElementById('clinic-name').textContent = config.clinica_nombre || 'Clinic App';
            if (config.clinica_logo) {
                document.getElementById('clinic-logo').src = config.clinica_logo;
            }

            // Rellenar el formulario con los datos cargados
            for (const key in config) {
                const element = document.getElementById(key);
                if (element) {
                    if (key === 'clinica_telefono') {
                        iti.setNumber(config[key] || ''); // Establecer número de teléfono
                    } else {
                        element.value = config[key];
                    }
                }
            }

            // Mostrar el logo existente
            if (config.clinica_logo) {
                logoPreview.src = `${config.clinica_logo}?v=${new Date().getTime()}`;
            }

        } catch (error) {
            console.error('Error:', error);
            // No mostramos alerta si GLOBAL_CONFIG no existe, simplemente el formulario estará vacío.
        }
    };

    // Previsualizar el nuevo logo seleccionado
    logoInput.addEventListener('input', () => {
        const url = logoInput.value;
        logoPreview.src = url.startsWith('http') ? url : 'placeholder.png';
    });

    // Limpiar campo de logo y vista previa
    clearLogoBtn.addEventListener('click', () => {
        logoInput.value = '';
        logoPreview.src = 'placeholder.png';
    });

    // Guardar la configuración
    form.addEventListener('submit', async (e) => {
        e.preventDefault();

        // Construimos el objeto de datos manualmente
        const configData = {
            clinica_nombre: document.getElementById('clinica_nombre').value,
            clinica_logo: document.getElementById('clinica_logo').value,
            clinica_direccion: document.getElementById('clinica_direccion').value,
            clinica_telefono: iti.getNumber(), // Obtener número completo
            clinica_email: document.getElementById('clinica_email').value,
            legal_nombre: document.getElementById('legal_nombre').value,
            legal_documento: document.getElementById('legal_documento').value,
            facturacion_ruc: document.getElementById('facturacion_ruc').value,
            separador_miles: document.getElementById('separador_miles').value,
            separador_decimales: document.getElementById('separador_decimales').value,
            cantidad_decimales: document.getElementById('cantidad_decimales').value,
            simbolo_moneda: document.getElementById('simbolo_moneda').value,
            horas_confirmar_cita: document.getElementById('horas_confirmar_cita').value,
        };

        try {
            const response = await fetch(CONFIG_API_URL, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify(configData)
            });
            const result = await response.json();
            alert(result.message);
            // Recargamos la página completa para que la nueva configuración (GLOBAL_CONFIG)
            // esté disponible en toda la aplicación.
            window.location.reload();
        } catch (error) {
            console.error('Error al guardar:', error);
            alert('Hubo un error al guardar la configuración.');
        }
    });

    // Carga inicial
    loadConfig();

    // Si la pestaña de actividad es la activa por defecto, cargar el log
    if (document.querySelector('.tab-link.active')?.dataset.tab === activityTabId) {
        if (form) form.style.display = 'none';
        loadFilterUsers();
        loadActivityLog();
    } else {
        // Asegurarse de que el formulario sea visible si la pestaña activa no es la de actividad
        if (form) form.style.display = 'block';
    }
});