<?php
session_start();
header('Content-Type: application/json');

// Asegurarse de que el archivo de configuración existe antes de incluirlo.
$config_file = 'config_db.php';
if (!file_exists($config_file)) {
    header("HTTP/1.1 500 Internal Server Error");
    echo json_encode(['status' => 'error', 'message' => 'El archivo de configuración de la base de datos no se encuentra. La instalación podría no estar completa.']);
    exit;
}
require_once $config_file;

$input = json_decode(file_get_contents('php://input'), true);
$email = $input['email'] ?? '';
$password = $input['password'] ?? '';

if (empty($email) || empty($password)) {
    header("HTTP/1.1 400 Bad Request");
    echo json_encode(['status' => 'error', 'message' => 'Email y contraseña son requeridos.']);
    exit;
}

// Conectar a la base de datos
$conn = @new mysqli(DB_HOST, DB_USER, DB_PASS, DB_NAME);
if ($conn->connect_error) {
    header("HTTP/1.1 500 Internal Server Error");
    // Es una buena práctica no exponer detalles del error de conexión en producción.
    // Se podría registrar el error en un archivo de logs.
    error_log("Error de conexión a MySQL: " . $conn->connect_error);
    echo json_encode(['status' => 'error', 'message' => 'Error de conexión con el servidor.']);
    exit;
}

// Buscar al usuario por email en la tabla 'usuarios'
$stmt = $conn->prepare("SELECT id, nombre, password, role FROM usuarios WHERE email = ? AND status = 'activo'");
if ($stmt === false) {
    header("HTTP/1.1 500 Internal Server Error");
    error_log("Error al preparar la consulta: " . $conn->error);
    echo json_encode(['status' => 'error', 'message' => 'Error interno del servidor.']);
    exit;
}

$stmt->bind_param("s", $email);
$stmt->execute();
$result = $stmt->get_result();

if ($result->num_rows === 1) {
    $user = $result->fetch_assoc();

    // Verificar la contraseña
    if (password_verify($password, $user['password'])) {
        // Éxito: Regenerar ID de sesión para prevenir fijación de sesión
        session_regenerate_id(true);
        
        // Iniciar sesión
        $_SESSION['user_id'] = $user['id'];
        $_SESSION['user_name'] = $user['nombre'];
        $_SESSION['user_role'] = $user['role'];

        // La función log_activity() de api.php original usa JSON.
        // Se necesitaría adaptar para que use la base de datos MySQL.
        // Por ahora, se omite para asegurar que el login funcione.
        // log_activity('Inició sesión en el sistema.', $user['nombre']);

        echo json_encode(['status' => 'success', 'role' => $user['role']]);
        
        $stmt->close();
        $conn->close();
        exit;
    }
}

// Falla: Credenciales incorrectas o usuario no encontrado/inactivo
// Se envía un mensaje genérico para no dar pistas a posibles atacantes.
header("HTTP/1.1 401 Unauthorized");
echo json_encode(['status' => 'error', 'message' => 'Email o contraseña incorrectos.']);

if (isset($stmt)) {
    $stmt->close();
}
$conn->close();

?>
