<?php
if (session_status() == PHP_SESSION_NONE) {
    session_start(); // Iniciar sesión para acceder a los datos del usuario
}

header('Content-Type: application/json');

$database_file = 'database.json';
$valid_entities = ['profesionales', 'pacientes', 'configuracion', 'sesiones', 'citas', 'financiero', 'egresos', 'activity_log'];

// Determinar con qué entidad estamos trabajando (profesionales o pacientes)
$entity = $_GET['entity'] ?? null;
if ($entity && !in_array($entity, $valid_entities)) {
    header("HTTP/1.1 400 Bad Request"); // Error si la entidad es inválida
    echo json_encode(['status' => 'error', 'message' => 'Entidad inválida.']);
    exit;
}

// Función para leer la base de datos
function get_data() {
    global $database_file;
    if (!file_exists($database_file)) {
        file_put_contents($database_file, json_encode(['profesionales' => []]));
    }
    return json_decode(file_get_contents($database_file), true);
}

// Función para guardar la base de datos
function save_data($data) {
    global $database_file;
    file_put_contents($database_file, json_encode($data, JSON_PRETTY_PRINT));
}

// Función para registrar actividad
function log_activity($action, $user_name_override = null) {
    // Si no hay una sesión activa, no podemos determinar el usuario.
    // Esto es para evitar errores si se llama desde un contexto sin sesión.
    if (session_status() == PHP_SESSION_NONE) {
        session_start();
    }

    $user_name = $user_name_override ?? $_SESSION['user_name'] ?? 'Sistema';
    $log_entry = [
        'id' => uniqid('log_', true),
        'timestamp' => date('c'), // ISO 8601 format
        'user' => $user_name,
        'action' => $action
    ];
    $data = get_data(); // Usamos la función global para obtener los datos
    if (!isset($data['activity_log'])) {
        $data['activity_log'] = [];
    }
    array_unshift($data['activity_log'], $log_entry); // Añadir al principio
    save_data($data); // Usamos la función global para guardar
}

// Función para obtener los datos de una entidad, con filtros opcionales
function get_entity_data($data, $entity, $user_id, $user_role) {

    // --- Protección de Endpoints ---
    $restricted_for_profesional = ['financiero', 'egresos', 'profesionales', 'reportes'];
    $restricted_for_secretario = ['profesionales', 'reportes', 'configuracion'];

    if ($user_role === 'profesional' && in_array($entity, $restricted_for_profesional)) {
        header("HTTP/1.1 403 Forbidden");
        echo json_encode(['status' => 'error', 'message' => 'Acceso denegado para el rol profesional.']);
        exit;
    } elseif ($user_role === 'secretario' && in_array($entity, $restricted_for_secretario)) {
        header("HTTP/1.1 403 Forbidden");
        echo json_encode(['status' => 'error', 'message' => 'Acceso denegado para el rol secretario.']);
        exit;
    }

    // El log de actividad solo es visible para el admin
    if ($entity === 'activity_log' && $user_role !== 'admin') {
        header("HTTP/1.1 403 Forbidden");
        echo json_encode(['status' => 'error', 'message' => 'Acceso denegado.']);
        exit;
    }
    // Excepción: La configuración es pública para lectura (GET) para la página de login.
    if ($entity === 'configuracion') {
        return $data['configuracion'] ?? [];
    }

    $all_items = $data[$entity] ?? [];

    // --- Filtrado por Rol ---
    if ($user_role === 'profesional') { // El rol 'profesional' tiene la vista más restringida
        if ($entity === 'pacientes') {
            // Un profesional solo ve sus pacientes asignados
            return array_values(array_filter($all_items, fn($item) => ($item['profesional_asignado_id'] ?? '') === $user_id));
        }
        if ($entity === 'profesionales') {
            // Un profesional solo puede ver su propio perfil
            return array_values(array_filter($all_items, fn($item) => $item['id'] === $user_id));
        }
        if ($entity === 'citas') {
            // Un profesional solo ve sus citas
            return array_values(array_filter($all_items, fn($item) => ($item['profesional_id'] ?? '') === $user_id));
        }
        if ($entity === 'sesiones') {
            // Un profesional solo ve las sesiones que ha creado
            return array_values(array_filter($all_items, fn($item) => ($item['profesional_id'] ?? '') === $user_id));
        }
    }

    // El rol 'secretario' y 'admin' pueden ver todos los pacientes, profesionales y citas para poder gestionar.
    // La API ya no filtra para ellos en este punto, por lo que tienen acceso completo a estas entidades.

    // --- Filtros de GET (ej: por paciente_id) ---
    if ($entity === 'citas' || $entity === 'sesiones') {
        $paciente_id_filter = $_GET['paciente_id'] ?? null;
        if ($paciente_id_filter) {
            return array_values(array_filter($all_items, fn($item) => $item['paciente_id'] === $paciente_id_filter));
        }
    }

    // Si es admin o no hay filtro de rol específico, devolver todo (respetando filtros GET)
    return $all_items; 
}

// Función para crear una nueva entidad
function create_entity(&$data, $entity, $new_data) {
    $new_data['id'] = uniqid(rand(), true); // ID más único

    // Hashear la contraseña si existe
    if (!empty($new_data['password'])) {
        $new_data['password'] = password_hash($new_data['password'], PASSWORD_DEFAULT);
    }

    if (!isset($data[$entity])) {
        $data[$entity] = [];
    }
    $data[$entity][] = $new_data;
    save_data($data);
    return $new_data;
}

// Función para actualizar una entidad existente
function update_entity(&$data, $entity, $id, $update_data) {
    $item_found = false;
    foreach ($data[$entity] as $key => &$item) { // Usar '&' para modificar el array original
        if ($item['id'] === $id) {
            // --- INICIO: Lógica de Auditoría para Citas ---
            if ($entity === 'citas') {
                $original_item = $item; // Guardamos el estado original
                $changes = [];

                // Comparar campos clave y registrar cambios
                if (isset($update_data['fecha']) && $update_data['fecha'] !== $original_item['fecha']) {
                    $changes[] = "Fecha cambió de '{$original_item['fecha']}' a '{$update_data['fecha']}'";
                }
                if (isset($update_data['hora']) && $update_data['hora'] !== $original_item['hora']) {
                    $changes[] = "Hora cambió de '{$original_item['hora']}' a '{$update_data['hora']}'";
                }
                if (isset($update_data['estado']) && $update_data['estado'] !== $original_item['estado']) {
                    $changes[] = "Estado cambió de '{$original_item['estado']}' a '{$update_data['estado']}'";
                }
                if (isset($update_data['profesional_id']) && $update_data['profesional_id'] !== $original_item['profesional_id']) {
                    $changes[] = "Profesional fue reasignado";
                }

                if (!empty($changes)) {
                    $log_entry = [
                        'timestamp' => date('c'),
                        'user_id' => $_SESSION['user_id'] ?? 'sistema',
                        'user_name' => $_SESSION['user_name'] ?? 'Sistema',
                        'changes' => $changes
                    ];
                    if (!isset($item['history'])) $item['history'] = [];
                    array_unshift($item['history'], $log_entry); // Añadir al principio
                }
            }
            // --- FIN: Lógica de Auditoría ---

            // No actualizar la foto si viene vacía
            if (isset($update_data['foto']) && empty($update_data['foto'])) {
                unset($update_data['foto']);
            }

            // Hashear la contraseña solo si se proporcionó una nueva
            if (!empty($update_data['password'])) {
                $update_data['password'] = password_hash($update_data['password'], PASSWORD_DEFAULT);
            } else {
                unset($update_data['password']); // No actualizar si está vacía
            }

            // Fusionar los datos
            $data[$entity][$key] = array_merge($item, $update_data);
            $item_found = true;
            break;
        }
    }

    if ($item_found) {
        save_data($data);
    }
    return $item_found;
}

// --- Función para verificar permisos de escritura (POST, DELETE) ---
function authorize_write_action($entity, $user_role) {
    // Denegar por defecto si el rol es 'guest' o no está definido
    if ($user_role === 'guest' || !$user_role) {
        return false;
    }

    // El admin puede hacer todo
    if ($user_role === 'admin') {
        return true;
    }

    // Permisos para el rol 'secretario'
    $secretario_allowed = ['pacientes', 'citas', 'sesiones', 'financiero', 'egresos'];
    if ($user_role === 'secretario' && in_array($entity, $secretario_allowed)) {
        return true;
    }

    // Permisos para el rol 'profesional'
    $profesional_allowed = ['pacientes', 'citas', 'sesiones']; // Puede crear/editar sus propias sesiones/citas/pacientes
    if ($user_role === 'profesional' && in_array($entity, $profesional_allowed)) {
        return true;
    }

    return false; // Denegar por defecto
}

// --- Lógica Principal según el Método HTTP ---

// Esta comprobación asegura que la lógica de la API solo se ejecute cuando se llama directamente a api.php
// y no cuando se incluye desde otro archivo (como logout.php).
if (basename(__FILE__) == basename($_SERVER["SCRIPT_FILENAME"])) {

    $method = $_SERVER['REQUEST_METHOD'];

    if ($method === 'GET') {
        if (!$entity) {
            header("HTTP/1.1 400 Bad Request");
            echo json_encode(['status' => 'error', 'message' => 'Entidad no especificada para GET.']);
            exit;
        }
        $data = get_data();

        // Obtener datos del usuario de la sesión para filtrar
        // Para la página de login, user_id y user_role serán null, lo cual es esperado.
        $user_id = $_SESSION['user_id'] ?? null;
        // Si no hay rol, lo tratamos como un 'invitado' con permisos mínimos.
        $user_role = $_SESSION['user_role'] ?? 'guest';

        // --- Caso especial: Obtener usuarios para el filtro de logs ---
        if ($entity === 'activity_log' && ($_GET['action'] ?? null) === 'get_users') {
            if ($user_role !== 'admin') {
                header("HTTP/1.1 403 Forbidden");
                echo json_encode(['status' => 'error', 'message' => 'Acceso denegado.']);
                exit;
            }
            $logs = $data['activity_log'] ?? [];
            $users = array_unique(array_column($logs, 'user'));
            sort($users);
            echo json_encode($users);
            exit;
        }

        // --- Filtrado para el log de actividad ---
        if ($entity === 'activity_log') {
            $logs = $data['activity_log'] ?? [];
            $start_date = $_GET['start_date'] ?? null;
            $end_date = $_GET['end_date'] ?? null;
            $filter_user = $_GET['user'] ?? null;

            if ($start_date) $logs = array_filter($logs, fn($log) => substr($log['timestamp'], 0, 10) >= $start_date);
            if ($end_date) $logs = array_filter($logs, fn($log) => substr($log['timestamp'], 0, 10) <= $end_date);
            if ($filter_user) $logs = array_filter($logs, fn($log) => $log['user'] === $filter_user);
            echo json_encode(array_values($logs));
            exit;
        }

        $items = get_entity_data($data, $entity, $user_id, $user_role);
        echo json_encode($items);
    }

    if ($method === 'POST') {
        if (!$entity) {
            header("HTTP/1.1 400 Bad Request");
            echo json_encode(['status' => 'error', 'message' => 'Entidad no especificada para POST.']);
            exit;
        }

        // --- Autorización para POST ---
        $user_role = $_SESSION['user_role'] ?? 'guest';
        if (!authorize_write_action($entity, $user_role)) {
            // Excepción: la configuración solo la puede modificar el admin.
            if ($entity !== 'configuracion' || $user_role !== 'admin') {
                header("HTTP/1.1 403 Forbidden");
                echo json_encode(['status' => 'error', 'message' => 'Acceso denegado para realizar esta acción.']);
                exit;
            }
        }

        $input_data = json_decode(file_get_contents('php://input'), true);
        $data = get_data();
        $id_to_update = $_GET['id'] ?? null;

        // --- Caso especial: Configuración ---
        if ($entity === 'configuracion') {
            $current_config = $data['configuracion'] ?? [];
            // Usamos array_merge para actualizar los valores existentes y añadir los nuevos.
            $data['configuracion'] = array_merge($current_config, $input_data);
            save_data($data);
            log_activity("Actualizó la configuración del sistema.");
            echo json_encode(['status' => 'success', 'message' => 'Configuración guardada correctamente.']);
            exit;
        }

        // --- Caso especial: Acciones ---
        $action = $_GET['action'] ?? null;
        if ($id_to_update && $action === 'delete_attachment' && $entity === 'sesiones') {
            foreach ($data['sesiones'] as &$note) {
                if ($note['id'] === $id_to_update) {
                    $note['adjuntos'] = array_values(array_filter($note['adjuntos'] ?? [], function($adjunto) use ($input_data) {
                        return $adjunto['url'] !== $input_data['url'];
                    }));
                    break;
                }
            }
            save_data($data);
            log_activity("Eliminó un adjunto de una nota de sesión.");
            echo json_encode(['status' => 'success', 'message' => 'Adjunto eliminado correctamente.']);
            exit;
        }

        // --- Lógica de Creación vs. Actualización ---
        if ($id_to_update) {
            if (update_entity($data, $entity, $id_to_update, $input_data)) {
                // Mensaje de log más descriptivo para actualizaciones
                $log_message = "Actualizó un registro en '{$entity}' (ID: {$id_to_update}).";
                if (($entity === 'pacientes' || $entity === 'profesionales') && (isset($input_data['nombre']) || isset($input_data['apellido']))) {
                    // Buscamos el item actualizado para obtener su nombre completo
                    $updated_item = null;
                    foreach($data[$entity] as $item) {
                        if ($item['id'] === $id_to_update) {
                            $updated_item = $item;
                            break;
                        }
                    }
                    if ($updated_item) {
                        $nombre_completo = trim(($updated_item['nombre'] ?? '') . ' ' . ($updated_item['apellido'] ?? ''));
                        $log_message = "Actualizó los datos de '{$nombre_completo}' ({$entity}).";
                    }
                } else if ($entity === 'citas' && isset($input_data['estado'])) {
                    $log_message = "Actualizó el estado de una cita a '{$input_data['estado']}'.";
                }
                log_activity($log_message);
                echo json_encode(['status' => 'success', 'message' => 'Entidad actualizada correctamente.']);
            } else {
                header("HTTP/1.1 404 Not Found");
                echo json_encode(['status' => 'error', 'message' => 'Entidad no encontrada para actualizar.']);
            }
        } else {
            // Mensaje de log más descriptivo
            $log_message = "Creó una nueva entidad '{$entity}'.";
            if ($entity === 'citas' && isset($input_data['paciente_id'])) {
                $pacientes = $data['pacientes'] ?? [];
                $paciente_nombre = 'Desconocido';
                foreach($pacientes as $p) {
                    if ($p['id'] === $input_data['paciente_id']) {
                        $paciente_nombre = $p['nombre'] . ' ' . $p['apellido'];
                        break;
                    }
                }
                $log_message = "Creó una nueva cita para el paciente '{$paciente_nombre}'.";
            }

            $created_item = create_entity($data, $entity, $input_data);
            log_activity($log_message);
            header("HTTP/1.1 201 Created");
            echo json_encode(['status' => 'success', 'message' => 'Entidad creada correctamente.', 'data' => $created_item]);
        }
    }

    if ($method === 'DELETE') {
        if (!$entity) {
            header("HTTP/1.1 400 Bad Request");
            echo json_encode(['status' => 'error', 'message' => 'Entidad no especificada para DELETE.']);
            exit;
        }

        // --- Autorización para DELETE ---
        $user_role = $_SESSION['user_role'] ?? 'guest';
        if (!authorize_write_action($entity, $user_role)) {
            header("HTTP/11 403 Forbidden");
            echo json_encode(['status' => 'error', 'message' => 'Acceso denegado para eliminar esta entidad.']);
            exit;
        }
        $id_to_delete = $_GET['id'] ?? null;
        if ($id_to_delete) {
            $data = get_data();
            // Encontrar el item ANTES de borrarlo para poder registrar su nombre
            $item_to_delete = null;
            foreach ($data[$entity] ?? [] as $item) {
                if ($item['id'] === $id_to_delete) {
                    $item_to_delete = $item;
                    break;
                }
            }

            $initial_count = count($data[$entity] ?? []);
            $data[$entity] = array_values(array_filter($data[$entity], function($item) use ($id_to_delete) {
                return $item['id'] !== $id_to_delete;
            }));
            if (count($data[$entity]) < $initial_count) {
                save_data($data);
                $nombre_item = $item_to_delete ? ($item_to_delete['nombre'] ?? $item_to_delete['titulo'] ?? $entity) : $entity;
                $log_message = "Eliminó el registro '{$nombre_item}' de la entidad '{$entity}'.";
                log_activity($log_message);
                echo json_encode(['status' => 'success', 'message' => 'Entidad eliminada correctamente.']);
            } else {
                header("HTTP/1.1 404 Not Found");
                echo json_encode(['status' => 'error', 'message' => 'Entidad no encontrada para eliminar.']);
            }
        }
    }
}