<?php
session_start();

// Intentar cargar la configuración de la base de datos
if (file_exists(__DIR__ . '/config/db.php')) {
    require_once __DIR__ . '/config/db.php';
}

// --- CONFIGURACIÓN INICIAL ---
$data_dir = 'data/';
$config_file = $data_dir . 'config.json';
$empresas_file = $data_dir . 'empresas.json';
define('MAX_LOG_ENTRIES', 500);


// --- FUNCIONES DE AYUDA ---
function get_plan_de_cuentas_default() {
    return [
        ["codigo" => "1", "nombre" => "ACTIVO", "tipo" => "titulo"],
        ["codigo" => "1.1", "nombre" => "ACTIVO CORRIENTE", "tipo" => "titulo"],
        ["codigo" => "1.1.1", "nombre" => "Caja y Bancos", "tipo" => "detalle"],
        ["codigo" => "1.1.2", "nombre" => "Cuentas por Cobrar", "tipo" => "detalle"],
        ["codigo" => "1.1.3", "nombre" => "Inventarios", "tipo" => "detalle"],
        ["codigo" => "1.1.4", "nombre" => "IVA Crédito Fiscal", "tipo" => "detalle"],
        ["codigo" => "2", "nombre" => "PASIVO", "tipo" => "titulo"],
        ["codigo" => "2.1", "nombre" => "PASIVO CORRIENTE", "tipo" => "titulo"],
        ["codigo" => "2.1.1", "nombre" => "Cuentas por Pagar", "tipo" => "detalle"],
        ["codigo" => "2.1.2", "nombre" => "Impuestos por Pagar", "tipo" => "detalle"],
        ["codigo" => "2.1.3", "nombre" => "IVA Débito Fiscal", "tipo" => "detalle"],
        ["codigo" => "3", "nombre" => "PATRIMONIO NETO", "tipo" => "titulo"],
        ["codigo" => "3.1", "nombre" => "Capital Social", "tipo" => "detalle"],
        ["codigo" => "4", "nombre" => "INGRESOS", "tipo" => "titulo"],
        ["codigo" => "4.1", "nombre" => "Venta de Mercaderías/Servicios", "tipo" => "detalle"],
        ["codigo" => "5", "nombre" => "GASTOS", "tipo" => "titulo"],
        ["codigo" => "5.1", "nombre" => "Costo de Ventas", "tipo" => "detalle"],
        ["codigo" => "5.2", "nombre" => "Gastos de Administración", "tipo" => "detalle"],
    ];
}

function delete_directory($dir) {
    if (!file_exists($dir)) {
        return true;
    }

    if (!is_dir($dir)) {
        return unlink($dir);
    }

    foreach (scandir($dir) as $item) {
        if ($item == '.' || $item == '..') {
            continue;
        }

        if (!delete_directory($dir . DIRECTORY_SEPARATOR . $item)) {
            return false;
        }
    }

    return rmdir($dir);
}

function log_activity($message) {
    $log_file = $GLOBALS['data_dir'] . 'activity_log.json';
    $logs = file_exists($log_file) ? json_decode(file_get_contents($log_file), true) : [];

    $log_entry = [
        'timestamp' => date('Y-m-d H:i:s'),
        'user' => $_SESSION['username'] ?? 'Sistema',
        'action' => $message
    ];

    // Añadir al principio para que los más nuevos estén arriba
    array_unshift($logs, $log_entry);

    // Limitar el log a las últimas 500 entradas para no hacerlo muy grande
    $logs = array_slice($logs, 0, MAX_LOG_ENTRIES);

    file_put_contents($log_file, json_encode($logs, JSON_PRETTY_PRINT));
}

function get_config() {
    global $config_file;
    if (!file_exists($config_file)) {
        return ['ai_api_key' => ''];
    }
    return json_decode(file_get_contents($config_file), true);
}

function save_config($config) {
    global $config_file;
    file_put_contents($config_file, json_encode($config, JSON_PRETTY_PRINT));
}

function call_ai_api($prompt) {
    $config = get_config();
    $api_key = $config['ai_api_key'] ?? '';

    if (empty($api_key) || $api_key === 'TU_API_KEY_DE_OPENAI_O_GEMINI_AQUI') {
        return "Error: La API Key de la IA no ha sido configurada en api.php.";
    }

    $url = '';
    $headers = [];
    $data = [];
    $ai_response_path = '';
    $system_prompt_content = 'Eres un experto analista financiero y contable. Analiza los datos proporcionados y ofrece un resumen claro, conciso y útil para un gerente de negocios. Identifica tendencias, puntos clave y posibles áreas de mejora. Responde en español y usa formato markdown para una mejor legibilidad.';

    // Determinar el tipo de API basado en el prefijo de la clave
    if (str_starts_with($api_key, 'sk-')) { // Clave de OpenAI
        $url = 'https://api.openai.com/v1/chat/completions';
        $headers = [
            'Content-Type: application/json',
            'Authorization: Bearer ' . $api_key
        ];
        $data = [
            'model' => 'gpt-3.5-turbo', // Puedes cambiar a 'gpt-4' si tienes acceso
            'messages' => [
                ['role' => 'system', 'content' => $system_prompt_content],
                ['role' => 'user', 'content' => $prompt]
            ],
            'temperature' => 0.5,
        ];
        $ai_response_path = 'openai';
    } elseif (str_starts_with($api_key, 'AIzaSy')) { // Clave de Google AI Studio (Gemini)
        // Usamos la API v1beta y el modelo 'gemini-1.5-flash-latest', el más actual para claves de AI Studio.
        $model = 'gemini-1.5-flash-latest';
        $url = "https://generativelanguage.googleapis.com/v1beta/models/{$model}:generateContent?key=" . $api_key;
        $headers = [
            'Content-Type: application/json'
        ];
        $data = [
            'contents' => [
                [
                    'parts' => [
                        ['text' => $system_prompt_content . "\n\n" . $prompt]
                    ]
                ]
            ],
            'generationConfig' => [
                'temperature' => 0.5
            ]
        ];
        $ai_response_path = 'gemini';
    } else {
        return "Error: Formato de API Key no reconocido. Asegúrate de usar una clave de OpenAI (empieza con 'sk-') o Google AI Studio (empieza con 'AIzaSy').";
    }

    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    if ($http_code !== 200) {
        return "Error al contactar la API de IA (Código: $http_code): " . $response;
    }
    $result = json_decode($response, true);

    if ($ai_response_path === 'openai') {
        return $result['choices'][0]['message']['content'] ?? 'No se recibió una respuesta válida de la IA (OpenAI).';
    } elseif ($ai_response_path === 'gemini') {
        if (isset($result['candidates'][0]['content']['parts'][0]['text'])) {
            return $result['candidates'][0]['content']['parts'][0]['text'];
        }
        // Si la respuesta no tiene el formato esperado, devolvemos un error detallado.
        return 'Error: La respuesta de la API de Gemini no tiene el formato esperado. Respuesta recibida: ' . json_encode($result);
    }
    
    return 'Error interno al procesar la respuesta de la IA.';
}

/**
 * Valida un ID de empresa para prevenir ataques de Path Traversal.
 * Un ID válido no debe contener '.' o '/'.
 */
function validate_empresa_id($empresa_id) {
    if (empty($empresa_id) || strpos($empresa_id, '.') !== false || strpos($empresa_id, '/') !== false) {
        http_response_code(400);
        die('<h1>Error: ID de empresa no válido.</h1>');
    }
    return $empresa_id;
}

/**
 * Carga y procesa los datos contables para una empresa y un rango de fechas.
 * Devuelve un array con los datos necesarios para los reportes.
 */
function get_processed_accounting_data($empresa_id, $fecha_desde, $fecha_hasta) {
    global $data_dir, $empresas_file;

    $empresa_id = validate_empresa_id($empresa_id);

    // Cargar datos
    $asientos_file = $data_dir . $empresa_id . '/asientos.json';
    $plan_file = $data_dir . $empresa_id . '/plan_de_cuentas.json';

    if (!file_exists($asientos_file) || !file_exists($plan_file) || !file_exists($empresas_file)) {
        http_response_code(404);
        die('<h1>Error: No se encontraron los archivos necesarios para la empresa.</h1>');
    }

    $asientos = json_decode(file_get_contents($asientos_file), true) ?: [];
    $plan_de_cuentas = json_decode(file_get_contents($plan_file), true) ?: [];
    $empresas = json_decode(file_get_contents($empresas_file), true) ?: [];

    $empresa_actual = array_values(array_filter($empresas, fn($e) => $e['id'] === $empresa_id))[0] ?? null;

    // Filtrar asientos por fecha
    if (!empty($fecha_desde)) {
        $asientos = array_filter($asientos, fn($asiento) => $asiento['fecha'] >= $fecha_desde);
    }
    if (!empty($fecha_hasta)) {
        $asientos = array_filter($asientos, fn($asiento) => $asiento['fecha'] <= $fecha_hasta);
    }

    return [
        'asientos' => $asientos,
        'plan_de_cuentas' => $plan_de_cuentas,
        'empresa_actual' => $empresa_actual
    ];
}

// --- MANEJADORES DE ACCIONES (GET) ---
function handle_generar_reporte_balance_general($data_dir) {
    header('Content-Type: text/html; charset=utf-8');

        $empresa_id = $_GET['empresa_id'] ?? '';
        $fecha_desde = $_GET['fecha_desde'] ?? '';
        $fecha_hasta = $_GET['fecha_hasta'] ?? '';

        $data = get_processed_accounting_data($empresa_id, $fecha_desde, $fecha_hasta);
        $asientos = $data['asientos'];
        $plan_de_cuentas = $data['plan_de_cuentas'];
        $empresa_actual = $data['empresa_actual'];
        
        // Procesar datos
        $mayor = [];
        foreach ($plan_de_cuentas as $c) {
            $mayor[$c['codigo']] = ['nombre' => $c['nombre'], 'tipo' => $c['tipo'], 'totalDebe' => 0, 'totalHaber' => 0];
        }

        foreach ($asientos as $asiento) {
            foreach ($asiento['movimientos'] as $mov) {
                if (isset($mayor[$mov['cuenta_codigo']])) {
                    $mayor[$mov['cuenta_codigo']]['totalDebe'] += floatval($mov['debe']);
                    $mayor[$mov['cuenta_codigo']]['totalHaber'] += floatval($mov['haber']);
                }
            }
        }

        // Calcular resultado del ejercicio
        $total_ingresos = 0;
        $total_gastos = 0;
        foreach ($mayor as $codigo => $cuenta) {
            $saldo = $cuenta['totalDebe'] - $cuenta['totalHaber'];
            if (substr($codigo, 0, 1) == '4') { $total_ingresos += -$saldo; }
            if (substr($codigo, 0, 1) == '5') { $total_gastos += $saldo; }
        }
        $resultado_ejercicio = $total_ingresos - $total_gastos;

        // Clasificar cuentas para el Balance
        $html_activo = '';
        $html_pasivo = '';
        $html_patrimonio = '';
        $total_activo = 0;
        $total_pasivo = 0;
        $total_patrimonio = 0;

        foreach ($mayor as $codigo => $cuenta) {
            $saldo = $cuenta['totalDebe'] - $cuenta['totalHaber'];
            if ($saldo == 0 && $cuenta['tipo'] == 'detalle') continue;

            $clase = $cuenta['tipo'] == 'titulo' ? 'class="fw-bold"' : 'style="padding-left: 20px;"';
            $monto_formateado = number_format($saldo, 2, ',', '.');

            if (substr($codigo, 0, 1) == '1') { // Activo
                if ($cuenta['tipo'] == 'detalle') $total_activo += $saldo;
                $html_activo .= "<tr><td {$clase}>(" . htmlspecialchars($codigo) . ") " . htmlspecialchars($cuenta['nombre']) . "</td><td class=\"text-end\">{$monto_formateado}</td></tr>";
            } else if (substr($codigo, 0, 1) == '2') { // Pasivo
                $monto_pasivo = -$saldo;
                if ($cuenta['tipo'] == 'detalle') $total_pasivo += $monto_pasivo;
                $html_pasivo .= "<tr><td {$clase}>(" . htmlspecialchars($codigo) . ") " . htmlspecialchars($cuenta['nombre']) . "</td><td class=\"text-end\">" . number_format($monto_pasivo, 2, ',', '.') . "</td></tr>";
            } else if (substr($codigo, 0, 1) == '3') { // Patrimonio
                $monto_patrimonio = -$saldo;
                if ($cuenta['tipo'] == 'detalle') $total_patrimonio += $monto_patrimonio;
                $html_patrimonio .= "<tr><td {$clase}>(" . htmlspecialchars($codigo) . ") " . htmlspecialchars($cuenta['nombre']) . "</td><td class=\"text-end\">" . number_format($monto_patrimonio, 2, ',', '.') . "</td></tr>";
            }
        }

        $total_patrimonio += $resultado_ejercicio;
        $html_patrimonio .= '<tr><td style="padding-left: 20px;">Resultado del Ejercicio</td><td class="text-end">' . number_format($resultado_ejercicio, 2, ',', '.') . '</td></tr>';
        $total_pasivo_patrimonio = $total_pasivo + $total_patrimonio;

        // Generar HTML
        echo '<!DOCTYPE html><html lang="es"><head><meta charset="UTF-8"><title>Balance General</title><style>body{font-family:sans-serif;margin:20px}table{width:100%;border-collapse:collapse}th,td{padding:8px;text-align:left}.text-end{text-align:right}.header{text-align:center;margin-bottom:20px}h1,h2,h3{margin:0}.total-row{font-weight:bold;border-top:2px solid #333}.gran-total{font-size:1.1em;font-weight:bold;border-top:3px double #333;margin-top:10px}.container{display:flex;width:100%}.col{width:50%;padding:0 15px}@media print{body{margin:0}.no-print{display:none} @page { size: A4; margin: 2.5cm; @bottom-center { font-size: 10px; color: #888; content: "Bridgi \"El Puente Contable\". Dunamis Group 2025."; } }}</style></head><body>';
        echo '<div class="header"><h1>Balance General</h1>';
        if ($empresa_actual) {
            echo '<h2>' . htmlspecialchars($empresa_actual['nombre']) . '</h2>';
            echo '<h3>RUC: ' . htmlspecialchars($empresa_actual['ruc']) . '</h3>';
        }
        echo '<p>Período del ' . htmlspecialchars($fecha_desde) . ' al ' . htmlspecialchars($fecha_hasta) . '</p></div>';
        
        echo '<div class="container">';
        echo '<div class="col"><h3>Activo</h3><table>' . $html_activo . '<tr class="total-row"><td><strong>Total Activo</strong></td><td class="text-end"><strong>' . number_format($total_activo, 2, ',', '.') . '</strong></td></tr></table></div>';
        echo '<div class="col"><h3>Pasivo</h3><table>' . $html_pasivo . '<tr class="total-row"><td><strong>Total Pasivo</strong></td><td class="text-end"><strong>' . number_format($total_pasivo, 2, ',', '.') . '</strong></td></tr></table><br><h3>Patrimonio Neto</h3><table>' . $html_patrimonio . '<tr class="total-row"><td><strong>Total Patrimonio Neto</strong></td><td class="text-end"><strong>' . number_format($total_patrimonio, 2, ',', '.') . '</strong></td></tr></table><br><table class="gran-total"><tr><td><strong>Total Pasivo y Patrimonio</strong></td><td class="text-end"><strong>' . number_format($total_pasivo_patrimonio, 2, ',', '.') . '</strong></td></tr></table></div>';
        echo '</div>';

        echo '</body></html>';
        exit;
}

function handle_generar_reporte_estado_resultados($data_dir) {
    header('Content-Type: text/html; charset=utf-8');

        $empresa_id = $_GET['empresa_id'] ?? null;
        $fecha_desde = $_GET['fecha_desde'] ?? null;
        $fecha_hasta = $_GET['fecha_hasta'] ?? null;

        $data = get_processed_accounting_data($empresa_id, $fecha_desde, $fecha_hasta);
        $asientos = $data['asientos'];
        $plan_de_cuentas = $data['plan_de_cuentas'];
        $empresa_actual = $data['empresa_actual'];

        // Procesar datos
        $mayor = [];
        foreach ($plan_de_cuentas as $c) {
            $mayor[$c['codigo']] = ['nombre' => $c['nombre'], 'tipo' => $c['tipo'], 'totalDebe' => 0, 'totalHaber' => 0];
        }

        foreach ($asientos as $asiento) {
            foreach ($asiento['movimientos'] as $mov) {
                if (isset($mayor[$mov['cuenta_codigo']])) {
                    $mayor[$mov['cuenta_codigo']]['totalDebe'] += floatval($mov['debe']);
                    $mayor[$mov['cuenta_codigo']]['totalHaber'] += floatval($mov['haber']);
                }
            }
        }

        // Calcular totales para el reporte
        $total_ingresos = 0;
        $total_gastos = 0;
        $html_ingresos = '';
        $html_gastos = '';

        foreach ($mayor as $codigo => $cuenta) {
            $saldo = $cuenta['totalDebe'] - $cuenta['totalHaber'];
            if (substr($codigo, 0, 1) == '4' && $saldo != 0) { // Ingresos
                $saldo_acreedor = -$saldo;
                $total_ingresos += $saldo_acreedor;
                $html_ingresos .= '<tr><td>(' . htmlspecialchars($codigo) . ') ' . htmlspecialchars($cuenta['nombre']) . '</td><td class="text-end">' . number_format($saldo_acreedor, 2, ',', '.') . '</td></tr>';
            }
            if (substr($codigo, 0, 1) == '5' && $saldo != 0) { // Gastos
                $saldo_deudor = $saldo;
                $total_gastos += $saldo_deudor;
                $html_gastos .= '<tr><td>(' . htmlspecialchars($codigo) . ') ' . htmlspecialchars($cuenta['nombre']) . '</td><td class="text-end">' . number_format($saldo_deudor, 2, ',', '.') . '</td></tr>';
            }
        }

        $resultado = $total_ingresos - $total_gastos;

        // Generar HTML
        echo '<!DOCTYPE html>';
        echo '<html lang="es">';
        echo '<head>';
        echo '<meta charset="UTF-8">';
        echo '<title>Estado de Resultados</title><style>';
        echo 'body { font-family: sans-serif; margin: 20px; } table { width: 100%; border-collapse: collapse; } th, td { padding: 8px; text-align: left; } .text-end { text-align: right; } .header { text-align: center; margin-bottom: 20px; } h1, h2, h3 { margin: 0; } .total-row { font-weight: bold; border-top: 2px solid #333; } .gran-total { font-size: 1.1em; font-weight: bold; border-top: 3px double #333; margin-top: 10px; } @media print { body { margin: 0; } .no-print { display: none; } @page { size: A4; margin: 2.5cm; @bottom-center { font-size: 10px; color: #888; content: "Bridgi \"El Puente Contable\". Dunamis Group 2025."; } } }';
        echo '</style>';
        echo '</head>';
        echo '<body>';
        echo '<div class="header">';
        echo '<h1>Estado de Resultados</h1>';
        if ($empresa_actual) {
            echo '<h2>' . htmlspecialchars($empresa_actual['nombre']) . '</h2>';
            echo '<h3>RUC: ' . htmlspecialchars($empresa_actual['ruc']) . '</h3>';
        }
        echo '<p>Período del ' . htmlspecialchars($fecha_desde) . ' al ' . htmlspecialchars($fecha_hasta) . '</p>';
        echo '</div>';

        echo '<h3>Ingresos</h3>';
        echo '<table>' . $html_ingresos . '<tr class="total-row"><td><strong>Total Ingresos</strong></td><td class="text-end"><strong>' . number_format($total_ingresos, 2, ',', '.') . '</strong></td></tr></table>';
        
        echo '<br>';

        echo '<h3>Gastos</h3>';
        echo '<table>' . $html_gastos . '<tr class="total-row"><td><strong>Total Gastos</strong></td><td class="text-end"><strong>' . number_format($total_gastos, 2, ',', '.') . '</strong></td></tr></table>';

        echo '<br>';

        echo '<table class="gran-total"><tr><td><strong>' . ($resultado >= 0 ? 'Utilidad del Ejercicio' : 'Pérdida del Ejercicio') . '</strong></td><td class="text-end"><strong>' . number_format($resultado, 2, ',', '.') . '</strong></td></tr></table>';
        
        echo '</body>';
        echo '</html>';

        exit; // Salir para no mezclar con otras salidas JSON
}

function handle_generar_reporte_libro_diario($data_dir) {
    header('Content-Type: text/html; charset=utf-8');

    $empresa_id = $_GET['empresa_id'] ?? null;
    $fecha_desde = $_GET['fecha_desde'] ?? null;
    $fecha_hasta = $_GET['fecha_hasta'] ?? null;

    // Usamos la nueva función para cargar y filtrar datos de forma segura
    $data = get_processed_accounting_data($empresa_id, $fecha_desde, $fecha_hasta);
    $asientos = $data['asientos'];
    $empresa_actual = $data['empresa_actual'];

    usort($asientos, fn($a, $b) => strcmp($a['fecha'], $b['fecha']));

    // Generar HTML
    echo '<!DOCTYPE html><html lang="es"><head><meta charset="UTF-8"><title>Libro Diario</title>';
    echo '<style>
        body{font-family:sans-serif;margin:20px}
        table{width:100%;border-collapse:collapse}
        th,td{padding:8px;text-align:left;border:1px solid #ddd}
        .text-end{text-align:right}
        .header{text-align:center;margin-bottom:20px; page-break-after: avoid;}
        h1,h2,h3{margin:0}
        tfoot{font-weight:bold;background-color:#f2f2f2}
        @media print{
            body{margin:0}
            .no-print{display:none}
            @page { size: A4; margin: 2.5cm; @bottom-center { content: "Página " counter(page) " de " counter(pages); } @bottom-right { font-size: 10px; color: #888; content: "Bridgi \"El Puente Contable\". Dunamis Group 2025."; } }
        }
    </style>';
    echo '</head><body>';
    echo '<div class="header"><h1>Libro Diario</h1>';
    if ($empresa_actual) {
        echo '<h2>' . htmlspecialchars($empresa_actual['nombre']) . '</h2>';
        echo '<h3>RUC: ' . htmlspecialchars($empresa_actual['ruc']) . '</h3>';
    }
    echo '<p>Período del ' . htmlspecialchars($fecha_desde) . ' al ' . htmlspecialchars($fecha_hasta) . '</p></div>';
    
    $tableHTML = '<table>';
    $tableHTML .= '<thead><tr><th>Fecha</th><th>Cuenta</th><th>Descripción</th><th class="text-end">Debe</th><th class="text-end">Haber</th></tr></thead>';
    $tableHTML .= '<tbody>';

    $totalDebe = 0;
    $totalHaber = 0;

    foreach ($asientos as $asiento) {
        $tableHTML .= '<tr style="background-color:#f8f9fa; font-weight:bold;"><td colspan="2">Asiento #' . substr($asiento['id'], 0, 6) . '</td><td colspan="3">' . htmlspecialchars($asiento['descripcion']) . ' (' . htmlspecialchars($asiento['comprobante_nro']) . ')</td></tr>';
        foreach ($asiento['movimientos'] as $mov) {
            $debe = floatval($mov['debe']);
            $haber = floatval($mov['haber']);
            $totalDebe += $debe;
            $totalHaber += $haber;

            $tableHTML .= '<tr>';
            $tableHTML .= '<td>' . htmlspecialchars($asiento['fecha']) . '</td>';
            $tableHTML .= '<td>(' . htmlspecialchars($mov['cuenta_codigo']) . ') ' . htmlspecialchars($mov['descripcion']) . '</td>';
            $tableHTML .= '<td></td>';
            $tableHTML .= '<td class="text-end">' . ($debe > 0 ? number_format($debe, 2, ',', '.') : '') . '</td>';
            $tableHTML .= '<td class="text-end">' . ($haber > 0 ? number_format($haber, 2, ',', '.') : '') . '</td>';
            $tableHTML .= '</tr>';
        }
    }

    $tableHTML .= '</tbody>';
    $tableHTML .= '<tfoot>';
    $tableHTML .= '<tr>';
    $tableHTML .= '<td colspan="3" style="text-align:right;"><strong>TOTALES</strong></td>';
    $tableHTML .= '<td class="text-end"><strong>' . number_format($totalDebe, 2, ',', '.') . '</strong></td>';
    $tableHTML .= '<td class="text-end"><strong>' . number_format($totalHaber, 2, ',', '.') . '</strong></td>';
    $tableHTML .= '</tr>';
    $tableHTML .= '</tfoot>';
    $tableHTML .= '</table>';

    echo $tableHTML;
    
    echo '</body></html>';
    exit;
}

function handle_generar_reporte_sumas_y_saldos($data_dir) {
    header('Content-Type: text/html; charset=utf-8');

    $empresa_id = $_GET['empresa_id'] ?? null;
    $fecha_desde = $_GET['fecha_desde'] ?? null;
    $fecha_hasta = $_GET['fecha_hasta'] ?? null;

    $data = get_processed_accounting_data($empresa_id, $fecha_desde, $fecha_hasta);
    $asientos = $data['asientos'];
    $plan_de_cuentas = $data['plan_de_cuentas'];
    $empresa_actual = $data['empresa_actual'];

    // Ordenar plan de cuentas por código para el reporte
    usort($plan_de_cuentas, fn($a, $b) => strnatcmp($a['codigo'], $b['codigo']));

    // Procesar datos
    $mayor = [];
    foreach ($plan_de_cuentas as $c) {
        $mayor[$c['codigo']] = ['nombre' => $c['nombre'], 'totalDebe' => 0, 'totalHaber' => 0];
    }
    foreach ($asientos as $asiento) {
        foreach ($asiento['movimientos'] as $mov) {
            if (isset($mayor[$mov['cuenta_codigo']])) {
                $mayor[$mov['cuenta_codigo']]['totalDebe'] += floatval($mov['debe']);
                $mayor[$mov['cuenta_codigo']]['totalHaber'] += floatval($mov['haber']);
            }
        }
    }

    // Generar HTML
    echo '<!DOCTYPE html><html lang="es"><head><meta charset="UTF-8"><title>Balance de Sumas y Saldos</title><style>body{font-family:sans-serif;margin:20px}table{width:100%;border-collapse:collapse}th,td{padding:8px;text-align:left;border:1px solid #ddd}.text-end{text-align:right}.header{text-align:center;margin-bottom:20px}h1,h2,h3{margin:0}tfoot{font-weight:bold;background-color:#f2f2f2}@media print{body{margin:0}.no-print{display:none} @page { size: A4; margin: 2.5cm; @bottom-center { font-size: 10px; color: #888; content: "Bridgi \"El Puente Contable\". Dunamis Group 2025."; } }}</style>';
    echo '</head><body>';
    echo '<div class="header"><h1>Balance de Comprobación de Sumas y Saldos</h1>';
    if ($empresa_actual) {
        echo '<h2>' . htmlspecialchars($empresa_actual['nombre']) . '</h2>';
        echo '<h3>RUC: ' . htmlspecialchars($empresa_actual['ruc']) . '</h3>';
    }
    echo '<p>Período del ' . htmlspecialchars($fecha_desde) . ' al ' . htmlspecialchars($fecha_hasta) . '</p></div>';
    
    $tableHTML = '<table>';
    $tableHTML .= '<thead><tr><th rowspan="2">Código</th><th rowspan="2">Cuenta</th><th colspan="2" style="text-align:center;">Sumas</th><th colspan="2" style="text-align:center;">Saldos</th></tr><tr><th class="text-end">Debe</th><th class="text-end">Haber</th><th class="text-end">Deudor</th><th class="text-end">Acreedor</th></tr></thead>';
    $tableHTML .= '<tbody>';

    $granTotalDebe = 0;
    $granTotalHaber = 0;
    $granTotalDeudor = 0;
    $granTotalAcreedor = 0;

    foreach ($plan_de_cuentas as $cuenta_plan) {
        $codigo = $cuenta_plan['codigo'];
        $cuenta = $mayor[$codigo];
        if ($cuenta['totalDebe'] > 0 || $cuenta['totalHaber'] > 0) {
            $saldo = $cuenta['totalDebe'] - $cuenta['totalHaber'];
            $granTotalDebe += $cuenta['totalDebe'];
            $granTotalHaber += $cuenta['totalHaber'];
            
            $saldoDeudor = $saldo > 0 ? $saldo : 0;
            $saldoAcreedor = $saldo < 0 ? -$saldo : 0;
            $granTotalDeudor += $saldoDeudor;
            $granTotalAcreedor += $saldoAcreedor;

            $tableHTML .= '<tr>';
            $tableHTML .= '<td>' . htmlspecialchars($codigo) . '</td>';
            $tableHTML .= '<td>' . htmlspecialchars($cuenta['nombre']) . '</td>';
            $tableHTML .= '<td class="text-end">' . number_format($cuenta['totalDebe'], 2, ',', '.') . '</td>';
            $tableHTML .= '<td class="text-end">' . number_format($cuenta['totalHaber'], 2, ',', '.') . '</td>';
            $tableHTML .= '<td class="text-end">' . ($saldoDeudor > 0 ? number_format($saldoDeudor, 2, ',', '.') : '') . '</td>';
            $tableHTML .= '<td class="text-end">' . ($saldoAcreedor > 0 ? number_format($saldoAcreedor, 2, ',', '.') : '') . '</td>';
            $tableHTML .= '</tr>';
        }
    }

    $tableHTML .= '</tbody>';
    $tableHTML .= '<tfoot>';
    $tableHTML .= '<tr>';
    $tableHTML .= '<td colspan="2" style="text-align:right;"><strong>TOTALES</strong></td>';
    $tableHTML .= '<td class="text-end"><strong>' . number_format($granTotalDebe, 2, ',', '.') . '</strong></td>';
    $tableHTML .= '<td class="text-end"><strong>' . number_format($granTotalHaber, 2, ',', '.') . '</strong></td>';
    $tableHTML .= '<td class="text-end"><strong>' . number_format($granTotalDeudor, 2, ',', '.') . '</strong></td>';
    $tableHTML .= '<td class="text-end"><strong>' . number_format($granTotalAcreedor, 2, ',', '.') . '</strong></td>';
    $tableHTML .= '</tr>';
    $tableHTML .= '</tfoot>';
    $tableHTML .= '</table>';

    echo $tableHTML;
    
    echo '</body></html>';
    exit;
}

function handle_generar_reporte_plan_de_cuentas($data_dir) {
    header('Content-Type: text/html; charset=utf-8');

    $empresa_id = $_GET['empresa_id'] ?? null;

    $data = get_processed_accounting_data($empresa_id, null, null);
    $plan_de_cuentas = $data['plan_de_cuentas'];
    $empresa_actual = $data['empresa_actual'];

    // Ordenar plan de cuentas por código para el reporte
    usort($plan_de_cuentas, fn($a, $b) => strnatcmp($a['codigo'], $b['codigo']));


    // Generar HTML
    echo '<!DOCTYPE html><html lang="es"><head><meta charset="UTF-8"><title>Plan de Cuentas</title><style>body{font-family:sans-serif;margin:20px}table{width:100%;border-collapse:collapse}th,td{padding:8px;text-align:left;border:1px solid #ddd}.header{text-align:center;margin-bottom:20px}h1,h2,h3{margin:0}.titulo{font-weight:bold;background-color:#f2f2f2}@media print{body{margin:0}.no-print{display:none} @page { size: A4; margin: 2.5cm; @bottom-center { font-size: 10px; color: #888; content: "Bridgi \"El Puente Contable\". Dunamis Group 2025."; } }}</style>';
    echo '</head><body>';
    echo '<div class="header"><h1>Plan de Cuentas</h1>';
    if ($empresa_actual) {
        echo '<h2>' . htmlspecialchars($empresa_actual['nombre']) . '</h2>';
        echo '<h3>RUC: ' . htmlspecialchars($empresa_actual['ruc']) . '</h3>';
    }
    echo '</div>';

    $tableHTML = '<table><thead><tr><th>Código</th><th>Nombre</th></tr></thead><tbody>';
    foreach ($plan_de_cuentas as $cuenta) {
        $clase = $cuenta['tipo'] === 'titulo' ? 'class="titulo"' : '';
        $tableHTML .= "<tr {$clase}>";
        $tableHTML .= '<td>' . htmlspecialchars($cuenta['codigo']) . '</td>';
        $tableHTML .= '<td>' . htmlspecialchars($cuenta['nombre']) . '</td>';
        $tableHTML .= '</tr>';
    }
    $tableHTML .= '</tbody></table>';

    echo $tableHTML;
    echo '</body></html>';
    exit;
}

function handle_get_iva_report_data($data_dir) {
     $empresa_id = $_GET['empresa_id'] ?? '';
     $fecha_desde = $_GET['fecha_desde'] ?? '';
     $fecha_hasta = $_GET['fecha_hasta'] ?? '';
 
     $data = get_processed_accounting_data($empresa_id, $fecha_desde, $fecha_hasta);
     $asientos = $data['asientos'];
 
     $reporte = [
         'ventas' => [
             '10' => ['count' => 0, 'total' => 0, 'base' => 0, 'iva' => 0],
             '5'  => ['count' => 0, 'total' => 0, 'base' => 0, 'iva' => 0],
             '0'  => ['count' => 0, 'total' => 0, 'base' => 0, 'iva' => 0],
         ],
         'compras' => [
             '10' => ['count' => 0, 'total' => 0, 'base' => 0, 'iva' => 0],
             '5'  => ['count' => 0, 'total' => 0, 'base' => 0, 'iva' => 0],
             '0'  => ['count' => 0, 'total' => 0, 'base' => 0, 'iva' => 0],
         ],
     ];
 
     foreach ($asientos as $asiento) {
         if (!isset($asiento['tipo_factura'])) continue;
 
         $tipo_factura = $asiento['tipo_factura']; // 'venta' o 'compra'
         $iva_tipo = $asiento['iva_tipo'] ?? '0';
 
         $total_factura = 0;
         $base_imponible = 0;
         $monto_iva = 0;
 
        if ($tipo_factura === 'venta') {
            $total_factura = array_sum(array_column(array_filter($asiento['movimientos'], fn($m) => $m['cuenta_codigo'] === '1.1.1' || $m['cuenta_codigo'] === '1.1.2'), 'debe'));
            $base_imponible = array_sum(array_column(array_filter($asiento['movimientos'], fn($m) => substr($m['cuenta_codigo'], 0, 1) === '4'), 'haber'));
            $monto_iva = array_sum(array_column(array_filter($asiento['movimientos'], fn($m) => $m['cuenta_codigo'] === '2.1.3'), 'haber'));
        } elseif ($tipo_factura === 'compra') {
            $total_factura = array_sum(array_column(array_filter($asiento['movimientos'], fn($m) => $m['cuenta_codigo'] === '1.1.1' || $m['cuenta_codigo'] === '2.1.1'), 'haber'));
            $base_imponible = array_sum(array_column(array_filter($asiento['movimientos'], fn($m) => substr($m['cuenta_codigo'], 0, 1) === '5'), 'debe'));
            $monto_iva = array_sum(array_column(array_filter($asiento['movimientos'], fn($m) => $m['cuenta_codigo'] === '1.1.4'), 'debe'));
        }

        // Determinar el tipo de IVA basado en el cálculo, no en el valor guardado que puede ser incorrecto.
        $iva_tipo_calculado = '0';
        if ($base_imponible > 0 && $monto_iva > 0) {
            $rate = round(($monto_iva / $base_imponible) * 100);
            if ($rate >= 9 && $rate <= 11) $iva_tipo_calculado = '10'; // Margen para redondeos
            elseif ($rate >= 4 && $rate <= 6) $iva_tipo_calculado = '5';
        }

        $reporte[$tipo_factura . 's'][$iva_tipo_calculado]['count']++;
        $reporte[$tipo_factura . 's'][$iva_tipo_calculado]['total'] += $total_factura;
        $reporte[$tipo_factura . 's'][$iva_tipo_calculado]['base'] += $base_imponible;
        $reporte[$tipo_factura . 's'][$iva_tipo_calculado]['iva'] += $monto_iva;
     }
 
     echo json_encode($reporte);
 }

function handle_generar_reporte_iva($data_dir) {
    header('Content-Type: text/html; charset=utf-8');

    $empresa_id = $_GET['empresa_id'] ?? '';
    $fecha_desde = $_GET['fecha_desde'] ?? '';
    $fecha_hasta = $_GET['fecha_hasta'] ?? '';

    // Simular la llamada a la función de datos para obtener el reporte procesado
    ob_start();
    handle_get_iva_report_data($data_dir);
    $reporte_json = ob_get_clean();
    $reporte = json_decode($reporte_json, true);
    $empresa_actual = get_processed_accounting_data($empresa_id, null, null)['empresa_actual'];

    $total_iva_debito = $reporte['ventas']['10']['iva'] + $reporte['ventas']['5']['iva'];
    $total_iva_credito = $reporte['compras']['10']['iva'] + $reporte['compras']['5']['iva'];
    $resultado = $total_iva_debito - $total_iva_credito;
    $resultado_texto = $resultado >= 0 ? 'IVA A PAGAR' : 'SALDO A FAVOR DEL CONTRIBUYENTE';

    $render_table = function($titulo, $data) {
        $html = "<h4>{$titulo}</h4><table class='detail-table'><thead><tr><th>Tipo IVA</th><th>Cant. Facturas</th><th class='text-end'>Total Facturado</th><th class='text-end'>Base Imponible</th><th class='text-end'>Total IVA</th></tr></thead><tbody>";
        $total_count = 0; $total_total = 0; $total_base = 0; $total_iva = 0;
        foreach ($data as $tipo => $valores) {
            if ($valores['count'] == 0) continue;
            $html .= "<tr><td>IVA {$tipo}%</td><td>{$valores['count']}</td><td class='text-end'>" . number_format($valores['total'], 2, ',', '.') . "</td><td class='text-end'>" . number_format($valores['base'], 2, ',', '.') . "</td><td class='text-end'>" . number_format($valores['iva'], 2, ',', '.') . "</td></tr>";
            $total_count += $valores['count']; $total_total += $valores['total']; $total_base += $valores['base']; $total_iva += $valores['iva'];
        }
        $html .= "</tbody><tfoot><tr><td><strong>TOTAL</strong></td><td><strong>{$total_count}</strong></td><td class='text-end'><strong>" . number_format($total_total, 2, ',', '.') . "</strong></td><td class='text-end'><strong>" . number_format($total_base, 2, ',', '.') . "</strong></td><td class='text-end'><strong>" . number_format($total_iva, 2, ',', '.') . "</strong></td></tr></tfoot></table>";
        return $html;
    };

    echo '<!DOCTYPE html><html lang="es"><head><meta charset="UTF-8"><title>Liquidación de IVA</title><style>body{font-family:sans-serif;margin:20px} .detail-table{width:100%;border-collapse:collapse;font-size:1em;margin-bottom:25px} .summary-table{width:60%;margin:20px auto;border-collapse:collapse;font-size:1.2em} th,td{padding:8px;text-align:left;border-bottom:1px solid #ddd} .detail-table th, .detail-table td {border: 1px solid #ddd;} .detail-table thead {background-color: #f2f2f2;} .detail-table tfoot {font-weight: bold; background-color: #e9ecef;} .text-end{text-align:right} .header{text-align:center;margin-bottom:40px} h1,h2,h3,h4{margin:0; margin-bottom: 10px;} .total-row{font-weight:bold;border-top:2px solid #333;font-size:1.3em} @media print{body{margin:0} .no-print{display:none} @page { size: A4; margin: 2.5cm; @bottom-center { font-size: 10px; color: #888; content: "Bridgi \"El Puente Contable\". Dunamis Group 2025."; } }}</style></head><body>';
    echo '<div class="header"><h1>Liquidación de Impuesto al Valor Agregado (IVA)</h1>';
    if ($empresa_actual) {
        echo '<h2>' . htmlspecialchars($empresa_actual['nombre']) . '</h2>';
        echo '<h3>RUC: ' . htmlspecialchars($empresa_actual['ruc']) . '</h3>';
    }
    echo '<p>Período del ' . htmlspecialchars($fecha_desde) . ' al ' . htmlspecialchars($fecha_hasta) . '</p></div>';
    
    echo $render_table('IVA Débito Fiscal (Ventas)', $reporte['ventas']);
    echo $render_table('IVA Crédito Fiscal (Compras)', $reporte['compras']);

    echo '<hr style="margin: 40px 0;">';
    echo '<h3 style="text-align:center;">Resumen de Liquidación</h3>';
    echo '<table class="summary-table">';
    echo '<tr><td>(+) Total IVA Débito Fiscal</td><td class="text-end">' . number_format($total_iva_debito, 2, ',', '.') . '</td></tr>';
    echo '<tr><td>(-) Total IVA Crédito Fiscal</td><td class="text-end">' . number_format($total_iva_credito, 2, ',', '.') . '</td></tr>';
    echo '<tr class="total-row"><td>' . $resultado_texto . '</td><td class="text-end">' . number_format(abs($resultado), 2, ',', '.') . '</td></tr>';
    echo '</table></body></html>';
    exit;
}

function handle_update_plan_de_cuentas($input, $data_dir) {
    if (function_exists('get_db_connection')) {
        $empresa_id = $input['empresa_id'] ?? '';
        $cuentas = $input['cuentas'] ?? [];

        $conn = get_db_connection();
        $conn->beginTransaction();
        try {
            $stmt_delete = $conn->prepare("DELETE FROM plan_de_cuentas WHERE empresa_id = :empresa_id");
            $stmt_delete->execute([':empresa_id' => $empresa_id]);

            $stmt_insert = $conn->prepare("INSERT INTO plan_de_cuentas (empresa_id, codigo, nombre, tipo) VALUES (:empresa_id, :codigo, :nombre, :tipo)");
            foreach ($cuentas as $cuenta) {
                $stmt_insert->execute([':empresa_id' => $empresa_id, ':codigo' => $cuenta['codigo'], ':nombre' => $cuenta['nombre'], ':tipo' => $cuenta['tipo']]);
            }
            $conn->commit();
            echo json_encode(['success' => true]);
        } catch (Exception $e) {
            $conn->rollBack();
            http_response_code(500);
            echo json_encode(['error' => 'Error al actualizar el plan de cuentas: ' . $e->getMessage()]);
        }
        return;
    }
        $empresa_id = $input['empresa_id'] ?? '';
        $empresa_id = validate_empresa_id($empresa_id);
        $cuentas = $input['cuentas'] ?? null;
        $plan_file = $data_dir . $empresa_id . '/plan_de_cuentas.json';

        if ($empresa_id && file_exists($plan_file)) {
            file_put_contents($plan_file, json_encode($cuentas, JSON_PRETTY_PRINT));
            echo json_encode(['success' => true]);
        } else {
            http_response_code(404);
            echo json_encode(['error' => 'Empresa no encontrada']);
        }
}

// --- MANEJADORES DE ACCIONES (POST) ---
function handle_save_manual_asiento($input, $data_dir) {
    if (function_exists('get_db_connection')) {
        $empresa_id = $input['empresa_id'] ?? '';
        $asiento_data = $input['asiento'] ?? [];
        $asiento_id = $asiento_data['id'] ?? null;

        $conn = get_db_connection();
        $conn->beginTransaction();
        try {
            if ($asiento_id) { // Actualización
                $stmt = $conn->prepare("UPDATE asientos SET fecha = :fecha, descripcion = :descripcion, comprobante_nro = :comprobante_nro WHERE id = :id AND empresa_id = :empresa_id");
                $stmt->execute([
                    ':fecha' => $asiento_data['fecha'],
                    ':descripcion' => $asiento_data['descripcion'],
                    ':comprobante_nro' => $asiento_data['comprobante_nro'],
                    ':id' => $asiento_id,
                    ':empresa_id' => $empresa_id
                ]);
                // Borrar movimientos antiguos
                $stmt_delete = $conn->prepare("DELETE FROM movimientos WHERE asiento_id = :asiento_id");
                $stmt_delete->execute([':asiento_id' => $asiento_id]);
            } else { // Creación
                $stmt = $conn->prepare("INSERT INTO asientos (empresa_id, fecha, descripcion, comprobante_nro, tipo_factura) VALUES (:empresa_id, :fecha, :descripcion, :comprobante_nro, 'manual')");
                $stmt->execute([
                    ':empresa_id' => $empresa_id,
                    ':fecha' => $asiento_data['fecha'],
                    ':descripcion' => $asiento_data['descripcion'],
                    ':comprobante_nro' => $asiento_data['comprobante_nro']
                ]);
                $asiento_id = $conn->lastInsertId();
            }

            // Insertar nuevos movimientos
            $stmt_mov = $conn->prepare("INSERT INTO movimientos (asiento_id, cuenta_codigo, descripcion, debe, haber) VALUES (:asiento_id, :cuenta_codigo, :descripcion, :debe, :haber)");
            foreach ($asiento_data['movimientos'] as $mov) {
                $stmt_mov->execute([
                    ':asiento_id' => $asiento_id,
                    ':cuenta_codigo' => $mov['cuenta_codigo'],
                    ':descripcion' => $mov['descripcion'],
                    ':debe' => $mov['debe'],
                    ':haber' => $mov['haber']
                ]);
            }

            $conn->commit();
            log_activity("Guardó/actualizó el asiento manual #{$asiento_id} para la empresa ID {$empresa_id}");
            echo json_encode(['success' => true, 'id' => $asiento_id]);
        } catch (Exception $e) {
            $conn->rollBack();
            http_response_code(500);
            echo json_encode(['error' => 'Error al guardar el asiento manual: ' . $e->getMessage()]);
        }
        return;
    }
}

function handle_create_asiento($input, $data_dir) {
    if (function_exists('get_db_connection')) {
        $conn = get_db_connection();
        $conn->beginTransaction();

        try {
            $empresa_id = $input['empresa_id'];
            $monto = floatval($input['monto']);
            $iva_tipo = intval($input['iva_tipo']);

            // 1. Calcular base e IVA
            $iva_rate_divisor = 1.0;
            if ($iva_tipo === 10) $iva_rate_divisor = 1.1;
            if ($iva_tipo === 5) $iva_rate_divisor = 1.05;
            $base = round($monto / $iva_rate_divisor, 2);
            $iva = $monto - $base;

            // 2. Insertar el asiento principal
            $sql_asiento = "INSERT INTO asientos (empresa_id, fecha, descripcion, comprobante_nro, timbrado, tipo_comprobante, condicion_operacion, es_electronica, ruc_cliente_proveedor, moneda, tipo_factura, iva_tipo) VALUES (:empresa_id, :fecha, :descripcion, :comprobante_nro, :timbrado, :tipo_comprobante, :condicion, :es_electronica, :ruc, :moneda, :tipo_factura, :iva_tipo)";
            $stmt_asiento = $conn->prepare($sql_asiento);
            $stmt_asiento->execute([
                ':empresa_id' => $empresa_id,
                ':fecha' => $input['fecha'],
                ':descripcion' => $input['descripcion'],
                ':comprobante_nro' => $input['comprobante_nro'],
                ':timbrado' => $input['timbrado'],
                ':tipo_comprobante' => $input['tipo_comprobante'],
                ':condicion' => $input['condicion_operacion'],
                ':es_electronica' => $input['es_electronica'] ? 1 : 0,
                ':ruc' => $input['ruc_cliente_proveedor'],
                ':moneda' => $input['moneda'],
                ':tipo_factura' => $input['tipo_factura'],
                ':iva_tipo' => $iva_tipo
            ]);
            $asiento_id = $conn->lastInsertId();

            // 3. Preparar los movimientos
            $plan_cuentas_stmt = $conn->prepare("SELECT nombre FROM plan_de_cuentas WHERE empresa_id = :empresa_id AND codigo = :codigo");
            $get_cuenta_nombre = function($codigo) use ($conn, $empresa_id, &$plan_cuentas_stmt) {
                $plan_cuentas_stmt->execute([':empresa_id' => $empresa_id, ':codigo' => $codigo]);
                return $plan_cuentas_stmt->fetchColumn() ?: 'Cuenta no encontrada';
            };

            $movimientos = [];
            $cuenta_pago_codigo = $input['cuenta_pago_codigo'];
            $cuenta_pago_nombre = $get_cuenta_nombre($cuenta_pago_codigo);

            if ($input['tipo_factura'] == 'venta') {
                $cuenta_ingreso_codigo = $input['cuenta_ingreso_codigo'];
                $cuenta_ingreso_nombre = $get_cuenta_nombre($cuenta_ingreso_codigo);
                $movimientos = [
                    ["cuenta_codigo" => $cuenta_pago_codigo, "descripcion" => $cuenta_pago_nombre, "debe" => $monto, "haber" => 0],
                    ["cuenta_codigo" => $cuenta_ingreso_codigo, "descripcion" => $cuenta_ingreso_nombre, "debe" => 0, "haber" => $base],
                ];
                if ($iva > 0) {
                    $movimientos[] = ["cuenta_codigo" => "2.1.3", "descripcion" => "IVA Débito Fiscal", "debe" => 0, "haber" => $iva];
                }
            } elseif ($input['tipo_factura'] == 'compra') {
                $cuenta_gasto_codigo = $input['cuenta_gasto_codigo'];
                $cuenta_gasto_nombre = $get_cuenta_nombre($cuenta_gasto_codigo);
                $movimientos = [
                    ["cuenta_codigo" => $cuenta_gasto_codigo, "descripcion" => $cuenta_gasto_nombre, "debe" => $base, "haber" => 0],
                    ["cuenta_codigo" => $cuenta_pago_codigo, "descripcion" => $cuenta_pago_nombre, "debe" => 0, "haber" => $monto],
                ];
                 if ($iva > 0) {
                    $movimientos[] = ["cuenta_codigo" => "1.1.4", "descripcion" => "IVA Crédito Fiscal", "debe" => $iva, "haber" => 0];
                }
            }

            // 4. Insertar los movimientos
            $sql_mov = "INSERT INTO movimientos (asiento_id, cuenta_codigo, descripcion, debe, haber) VALUES (:asiento_id, :cuenta_codigo, :descripcion, :debe, :haber)";
            $stmt_mov = $conn->prepare($sql_mov);
            foreach ($movimientos as $mov) {
                if (empty($mov['cuenta_codigo'])) continue; // No insertar movimientos sin cuenta
                $stmt_mov->execute([
                    ':asiento_id' => $asiento_id,
                    ':cuenta_codigo' => $mov['cuenta_codigo'],
                    ':descripcion' => $mov['descripcion'],
                    ':debe' => $mov['debe'],
                    ':haber' => $mov['haber']
                ]);
            }

            $conn->commit();
            log_activity("Creó el asiento #{$asiento_id} para la empresa ID {$empresa_id}");
            echo json_encode(['success' => true, 'id' => $asiento_id]);

        } catch (Exception $e) {
            $conn->rollBack();
            http_response_code(500);
            echo json_encode(['error' => 'Error al crear el asiento: ' . $e->getMessage()]);
        }
        return;
    }

        $empresa_id = $input['empresa_id'] ?? '';
        $empresa_id = validate_empresa_id($empresa_id);
        $asientos_file = $data_dir . $empresa_id . '/asientos.json';

        if (!$empresa_id || !is_dir($data_dir . $empresa_id)) {
            http_response_code(404);
            echo json_encode(['error' => 'Empresa no encontrada']);
            return;
        }

        $asientos = file_exists($asientos_file) ? json_decode(file_get_contents($asientos_file), true) : [];
        
        $plan_file = $data_dir . $empresa_id . '/plan_de_cuentas.json';
        $plan_de_cuentas = json_decode(file_get_contents($plan_file), true);

        $get_cuenta_nombre = function($codigo) use ($plan_de_cuentas) {
            foreach ($plan_de_cuentas as $cuenta) {
                if ($cuenta['codigo'] === $codigo) {
                    return $cuenta['nombre'];
                }
            }
            return '';
        };


        $monto = floatval($input['monto']);
        $movimientos = [];
        $iva_rate = 1.1; // Asumimos 10% por defecto

        // Lógica simple para IVA. Se puede expandir.
        if ($input['iva_tipo'] == '5') $iva_rate = 1.05;
        if ($input['iva_tipo'] == '0') $iva_rate = 1.0;

        $base = round($monto / $iva_rate, 2);
        $iva = $monto - $base;

        $cuenta_pago_codigo = $input['cuenta_pago_codigo'];
        $cuenta_pago_nombre = $get_cuenta_nombre($cuenta_pago_codigo);

        if ($input['tipo_factura'] == 'venta') {
            $cuenta_ingreso_codigo = $input['cuenta_ingreso_codigo'];
            $cuenta_ingreso_nombre = $get_cuenta_nombre($cuenta_ingreso_codigo);
            $movimientos = [
                ["cuenta_codigo" => $cuenta_pago_codigo, "descripcion" => $cuenta_pago_nombre, "debe" => $monto, "haber" => 0],
                ["cuenta_codigo" => $cuenta_ingreso_codigo, "descripcion" => $cuenta_ingreso_nombre, "debe" => 0, "haber" => $base],
                ["cuenta_codigo" => "2.1.3", "descripcion" => "IVA Débito Fiscal", "debe" => 0, "haber" => $iva],
            ];
        } elseif ($input['tipo_factura'] == 'compra') {
            $cuenta_gasto_codigo = $input['cuenta_gasto_codigo'];
            $cuenta_gasto_nombre = $get_cuenta_nombre($cuenta_gasto_codigo);
            $movimientos = [
                ["cuenta_codigo" => $cuenta_gasto_codigo, "descripcion" => $cuenta_gasto_nombre, "debe" => $base, "haber" => 0],
                ["cuenta_codigo" => "1.1.4", "descripcion" => "IVA Crédito Fiscal", "debe" => $iva, "haber" => 0],
                ["cuenta_codigo" => $cuenta_pago_codigo, "descripcion" => $cuenta_pago_nombre, "debe" => 0, "haber" => $monto],
            ];
        }

        $nuevo_asiento = [
            'id' => uniqid(),
            'fecha' => $input['fecha'],
            'comprobante_nro' => $input['comprobante_nro'],
            'descripcion' => $input['descripcion'],
            'movimientos' => $movimientos,
            'ruc_cliente_proveedor' => $input['ruc_cliente_proveedor'],
            'timbrado' => $input['timbrado'],
            'tipo_comprobante' => $input['tipo_comprobante'],
            'condicion_operacion' => $input['condicion_operacion'],
            'es_electronica' => $input['es_electronica'] ?? false,
            'moneda' => $input['moneda'],
            'tipo_factura' => $input['tipo_factura']
        ];

        $asientos[] = $nuevo_asiento;
        file_put_contents($asientos_file, json_encode($asientos, JSON_PRETTY_PRINT));
        echo json_encode($nuevo_asiento);
    }

function handle_delete_asiento($input, $data_dir) {
    if (function_exists('get_db_connection')) {
        $conn = get_db_connection();
        $empresa_id = $input['empresa_id'] ?? '';
        $asiento_id = $input['asiento_id'] ?? '';

        if (empty($empresa_id) || empty($asiento_id)) {
            http_response_code(400);
            echo json_encode(['error' => 'Faltan IDs de empresa o asiento.']);
            return;
        }

        try {
            $stmt = $conn->prepare("DELETE FROM asientos WHERE id = :asiento_id AND empresa_id = :empresa_id");
            $stmt->execute([':asiento_id' => $asiento_id, ':empresa_id' => $empresa_id]);
            log_activity("Eliminó el asiento #{$asiento_id} de la empresa ID {$empresa_id}");
            echo json_encode(['success' => $stmt->rowCount() > 0]);
        } catch (Exception $e) {
            http_response_code(500);
            echo json_encode(['error' => 'Error al eliminar el asiento: ' . $e->getMessage()]);
        }
        return;
    }
        $empresa_id = $input['empresa_id'] ?? '';
        $asiento_id = $input['asiento_id'] ?? '';
        $empresa_id = validate_empresa_id($empresa_id);
        $asientos_file = $data_dir . $empresa_id . '/asientos.json';

        if ($empresa_id && $asiento_id && file_exists($asientos_file)) {
            $asientos = json_decode(file_get_contents($asientos_file), true);
            $asientos_filtrados = array_filter($asientos, function($asiento) use ($asiento_id) {
                return $asiento['id'] !== $asiento_id;
            });

            file_put_contents($asientos_file, json_encode(array_values($asientos_filtrados), JSON_PRETTY_PRINT));
            echo json_encode(['success' => true]);
        } else {
            http_response_code(404);
            echo json_encode(['error' => 'Asiento o empresa no encontrada']);
        }
    }

function handle_update_asiento($input, $data_dir) {
    if (function_exists('get_db_connection')) {
        // Para la actualización, podemos simplemente borrar el asiento viejo y crear uno nuevo.
        // Es más simple y seguro que tratar de actualizar cada parte.
        handle_delete_asiento($input, $data_dir);
        handle_create_asiento($input, $data_dir);
        return;
    }
        $empresa_id = $input['empresa_id'] ?? '';
        $asiento_id = $input['asiento_id'] ?? '';
        $empresa_id = validate_empresa_id($empresa_id);
        $asientos_file = $data_dir . $empresa_id . '/asientos.json';

        if ($empresa_id && $asiento_id && file_exists($asientos_file)) {
            $asientos = json_decode(file_get_contents($asientos_file), true);
            
            $plan_file = $data_dir . $empresa_id . '/plan_de_cuentas.json';
            $plan_de_cuentas = json_decode(file_get_contents($plan_file), true);

            $get_cuenta_nombre = function($codigo) use ($plan_de_cuentas) {
                foreach ($plan_de_cuentas as $cuenta) {
                    if ($cuenta['codigo'] === $codigo) {
                        return $cuenta['nombre'];
                    }
                }
                return '';
            };

            $monto = floatval($input['monto']);
            $iva_rate = 1.1; // Asumimos 10% por defecto
            if ($input['iva_tipo'] == '5') $iva_rate = 1.05;
            if ($input['iva_tipo'] == '0') $iva_rate = 1.0;

            $base = round($monto / $iva_rate, 2);
            $iva = $monto - $base;

            $cuenta_pago_codigo = $input['cuenta_pago_codigo'];
            $cuenta_pago_nombre = $get_cuenta_nombre($cuenta_pago_codigo);

            if ($input['tipo_factura'] == 'venta') {
                $cuenta_ingreso_codigo = $input['cuenta_ingreso_codigo'];
                $cuenta_ingreso_nombre = $get_cuenta_nombre($cuenta_ingreso_codigo);
                $movimientos = [
                    ["cuenta_codigo" => $cuenta_pago_codigo, "descripcion" => $cuenta_pago_nombre, "debe" => $monto, "haber" => 0],
                    ["cuenta_codigo" => $cuenta_ingreso_codigo, "descripcion" => $cuenta_ingreso_nombre, "debe" => 0, "haber" => $base],
                    ["cuenta_codigo" => "2.1.3", "descripcion" => "IVA Débito Fiscal", "debe" => 0, "haber" => $iva],
                ];
            } elseif ($input['tipo_factura'] == 'compra') {
                $cuenta_gasto_codigo = $input['cuenta_gasto_codigo'];
                $cuenta_gasto_nombre = $get_cuenta_nombre($cuenta_gasto_codigo);
                $movimientos = [
                    ["cuenta_codigo" => $cuenta_gasto_codigo, "descripcion" => $cuenta_gasto_nombre, "debe" => $base, "haber" => 0],
                    ["cuenta_codigo" => "1.1.4", "descripcion" => "IVA Crédito Fiscal", "debe" => $iva, "haber" => 0],
                    ["cuenta_codigo" => $cuenta_pago_codigo, "descripcion" => $cuenta_pago_nombre, "debe" => 0, "haber" => $monto],
                ];
            }

            $updated_asiento = [
                'id' => $asiento_id,
                'fecha' => $input['fecha'],
                'comprobante_nro' => $input['comprobante_nro'],
                'descripcion' => $input['descripcion'],
                'movimientos' => $movimientos,
                'ruc_cliente_proveedor' => $input['ruc_cliente_proveedor'],
                'timbrado' => $input['timbrado'],
                'tipo_comprobante' => $input['tipo_comprobante'],
                'condicion_operacion' => $input['condicion_operacion'],
                'es_electronica' => $input['es_electronica'] ?? false,
                'moneda' => $input['moneda'],
                'tipo_factura' => $input['tipo_factura']
            ];

            $asiento_encontrado = false;
            foreach ($asientos as $key => $asiento) {
                if ($asiento['id'] === $asiento_id) {
                    $asientos[$key] = $updated_asiento;
                    $asiento_encontrado = true;
                    break;
                }
            }

            if ($asiento_encontrado) {
                file_put_contents($asientos_file, json_encode($asientos, JSON_PRETTY_PRINT));
                echo json_encode($updated_asiento);
            } else {
                http_response_code(404);
                echo json_encode(['error' => 'Asiento no encontrado para actualizar']);
            }
        } else {
            http_response_code(404);
            echo json_encode(['error' => 'Asiento o empresa no encontrada']);
        }
    }

function handle_get_plan_de_cuentas($data_dir) {
    if (function_exists('get_db_connection')) {
        $empresa_id = $_GET['empresa_id'] ?? '';
        if (empty($empresa_id)) {
            http_response_code(400);
            echo json_encode(['error' => 'Falta el ID de la empresa.']);
            return;
        }
        $conn = get_db_connection();
        $stmt = $conn->prepare("SELECT codigo, nombre, tipo FROM plan_de_cuentas WHERE empresa_id = :empresa_id ORDER BY codigo ASC");
        $stmt->execute([':empresa_id' => $empresa_id]);
        $cuentas = $stmt->fetchAll();
        echo json_encode($cuentas);
        return;
    }

        $empresa_id = $_GET['empresa_id'] ?? '';
        $empresa_id = validate_empresa_id($empresa_id);
        $plan_file = $data_dir . $empresa_id . '/plan_de_cuentas.json';
        if ($empresa_id && file_exists($plan_file)) {
            echo file_get_contents($plan_file);
        } else {
            http_response_code(404);
            echo json_encode(['error' => 'Plan de cuentas no encontrado']);
        }
}

function handle_get_asientos($data_dir) {
    if (function_exists('get_db_connection')) {
        $empresa_id = $_GET['empresa_id'] ?? '';
        $fecha_desde = $_GET['fecha_desde'] ?? '';
        $fecha_hasta = $_GET['fecha_hasta'] ?? '';

        $conn = get_db_connection();
        $sql = "SELECT * FROM asientos WHERE empresa_id = :empresa_id";
        $params = [':empresa_id' => $empresa_id];

        if (!empty($fecha_desde)) {
            $sql .= " AND fecha >= :fecha_desde";
            $params[':fecha_desde'] = $fecha_desde;
        }
        if (!empty($fecha_hasta)) {
            $sql .= " AND fecha <= :fecha_hasta";
            $params[':fecha_hasta'] = $fecha_hasta;
        }
        $sql .= " ORDER BY fecha DESC, id DESC";

        $stmt = $conn->prepare($sql);
        $stmt->execute($params);
        $asientos = $stmt->fetchAll();

        if (empty($asientos)) {
            echo json_encode([]);
            return;
        }

        // Obtener todos los movimientos de una vez (evita N+1)
        $asiento_ids = array_column($asientos, 'id');
        $placeholders = implode(',', array_fill(0, count($asiento_ids), '?'));
        $stmt_mov = $conn->prepare("SELECT * FROM movimientos WHERE asiento_id IN ($placeholders)");
        $stmt_mov->execute($asiento_ids);
        $movimientos_agrupados = $stmt_mov->fetchAll(PDO::FETCH_GROUP);

        // Inyectar movimientos en sus asientos
        foreach ($asientos as &$asiento) {
            $asiento['movimientos'] = $movimientos_agrupados[$asiento['id']] ?? [];
        }

        echo json_encode($asientos);
        return;
    }
        $empresa_id = $_GET['empresa_id'] ?? '';
        $empresa_id = validate_empresa_id($empresa_id);
        $fecha_desde = $_GET['fecha_desde'] ?? '';
        $fecha_hasta = $_GET['fecha_hasta'] ?? '';

        $asientos_file = $data_dir . $empresa_id . '/asientos.json';

        if ($empresa_id && file_exists($asientos_file)) {
            $asientos = json_decode(file_get_contents($asientos_file), true);

            if (!empty($fecha_desde)) {
                $asientos = array_filter($asientos, function($asiento) use ($fecha_desde) {
                    return $asiento['fecha'] >= $fecha_desde;
                });
            }

            if (!empty($fecha_hasta)) {
                $asientos = array_filter($asientos, function($asiento) use ($fecha_hasta) {
                    return $asiento['fecha'] <= $fecha_hasta;
                });
            }

            echo json_encode(array_values($asientos));
        } else {
            echo json_encode([]); // Devuelve array vacío si no hay asientos
        }
}

function handle_get_empresas($empresas_file) {
    if (function_exists('get_db_connection')) {
        $conn = get_db_connection();
        $user_id = $_SESSION['user_id'];
        $user_category = $_SESSION['categoria'];

        $sql = "";
        $params = [];

        if ($user_category === 'Administrador' || $user_category === 'Contador') {
            // Los contadores y admins ven todas las empresas
            $sql = "SELECT * FROM empresas ORDER BY nombre ASC";
        } elseif ($user_category === 'Digitador') {
            // Los digitadores solo ven las empresas a las que están asignados
            $sql = "SELECT e.* FROM empresas e 
                    JOIN empresa_digitadores ed ON e.id = ed.empresa_id 
                    WHERE ed.user_id = :user_id 
                    ORDER BY e.nombre ASC";
            $params[':user_id'] = $user_id;
        }

        if (empty($sql)) {
            echo json_encode([]);
            return;
        }

        $stmt = $conn->prepare($sql);
        $stmt->execute($params);
        $empresas = $stmt->fetchAll();

        // Si no hay empresas, no hay nada más que hacer.
        if (empty($empresas)) {
            echo json_encode([]);
            return;
        }

        // --- INICIO DE LA CORRECCIÓN ---
        // 1. Obtener los IDs de todas las empresas cargadas
        $empresa_ids = array_column($empresas, 'id');

        // 2. Hacer UNA sola consulta para traer todos los digitadores asignados a esas empresas
        $placeholders = implode(',', array_fill(0, count($empresa_ids), '?'));
        $stmt_digitadores = $conn->prepare("SELECT empresa_id, user_id FROM empresa_digitadores WHERE empresa_id IN ($placeholders)");
        $stmt_digitadores->execute($empresa_ids);
        $asignaciones = $stmt_digitadores->fetchAll(PDO::FETCH_GROUP | PDO::FETCH_COLUMN);

        // 2.1. Hacer UNA sola consulta para traer todos los timbrados de esas empresas
        $stmt_timbrados = $conn->prepare("SELECT empresa_id, numero, inicio, fin FROM timbrados WHERE empresa_id IN ($placeholders) ORDER BY inicio DESC");
        $stmt_timbrados->execute($empresa_ids);
        // Agrupar los timbrados por empresa_id
        $timbrados_por_empresa = [];
        while ($row = $stmt_timbrados->fetch()) {
            $timbrados_por_empresa[$row['empresa_id']][] = $row;
        }

        // 3. Inyectar los datos en el array de empresas
        foreach ($empresas as &$empresa) {
            // Decodificar el JSON de presentaciones
            $empresa['presentaciones'] = json_decode($empresa['presentaciones_json'], true) ?? [];
            unset($empresa['presentaciones_json']); // Limpiar para no enviar data redundante

            // Añadir el array de digitadores (ya existente)
            $empresa['digitadores'] = $asignaciones[$empresa['id']] ?? [];

            // Añadir el array de timbrados
            $empresa['timbrados'] = $timbrados_por_empresa[$empresa['id']] ?? [];
        }

        echo json_encode($empresas);
        return;
    }

    if (!file_exists($empresas_file)) {
        echo json_encode([]);
        return;
    }

    $empresas = json_decode(file_get_contents($empresas_file), true);
    $user_id = $_SESSION['user_id'];
    $user_category = $_SESSION['categoria'];

    if ($user_category === 'Administrador' || $user_category === 'Contador') {
        echo json_encode($empresas);
    } elseif ($user_category === 'Digitador') {
        $empresas_filtradas = array_filter($empresas, function($empresa) use ($user_id) {
            return isset($empresa['digitadores']) && in_array($user_id, $empresa['digitadores']);
        });
        echo json_encode(array_values($empresas_filtradas));
    } else {
        echo json_encode([]); // Por defecto, no mostrar nada si no tiene rol
    }
}

function handle_create_empresa($input, $data_dir, $empresas_file) {
    if (function_exists('get_db_connection')) {
        if ($_SESSION['categoria'] === 'Digitador') {
            http_response_code(403);
            echo json_encode(['error' => 'Acceso denegado.']);
            return;
        }

        $conn = get_db_connection();
        $input = $_POST; // Datos vienen de FormData

        if (empty($input['nombre']) || empty($input['ruc'])) {
            http_response_code(400);
            echo json_encode(['error' => 'El nombre y el RUC de la empresa son obligatorios.']);
            return;
        }

        $logo_path_for_db = null;
        $empresa_id = null;

        try {
            $conn->beginTransaction();

            // 1. Insertar la empresa
            $sql_empresa = "INSERT INTO empresas (nombre, ruc, tipo, responsable_contabilidad, telefono, direccion, ciudad, pais, actividad_economica, observaciones, contador_id, presentaciones_json) VALUES (:nombre, :ruc, :tipo, :responsable, :telefono, :direccion, :ciudad, :pais, :actividad, :obs, :contador_id, :presentaciones)";
            $stmt_empresa = $conn->prepare($sql_empresa);
            $stmt_empresa->execute([
                ':nombre' => $input['nombre'],
                ':ruc' => $input['ruc'],
                ':tipo' => $input['tipo'] ?? 'fisica',
                ':responsable' => $input['responsable_contabilidad'] ?? null,
                ':telefono' => $input['telefono'] ?? null,
                ':direccion' => $input['direccion'] ?? null,
                ':ciudad' => $input['ciudad'] ?? null,
                ':pais' => $input['pais'] ?? null,
                ':actividad' => $input['actividad_economica'] ?? null,
                ':obs' => $input['observaciones'] ?? null,
                ':contador_id' => empty($input['contador_id']) ? null : $input['contador_id'],
                ':presentaciones' => json_encode($input['presentaciones'] ?? [])
            ]);
            $empresa_id = $conn->lastInsertId();

            // 2. Manejar subida de logo
            if (isset($_FILES['logo']) && $_FILES['logo']['error'] == 0) {
                $empresa_dir_for_files = 'data/' . $empresa_id;
                if (!is_dir($empresa_dir_for_files)) {
                    mkdir($empresa_dir_for_files, 0777, true);
                }
                $logo_filename = 'logo.' . pathinfo($_FILES['logo']['name'], PATHINFO_EXTENSION);
                $logo_path_on_server = $empresa_dir_for_files . '/' . $logo_filename;
                if (move_uploaded_file($_FILES['logo']['tmp_name'], $logo_path_on_server)) {
                    $logo_path_for_db = $logo_path_on_server;
                    // Actualizar la ruta del logo en la BD
                    $stmt_logo = $conn->prepare("UPDATE empresas SET logo = :logo WHERE id = :id");
                    $stmt_logo->execute([':logo' => $logo_path_for_db, ':id' => $empresa_id]);
                }
            }

            // 3. Insertar timbrados
            $timbrados = json_decode($input['timbrados_json'] ?? '[]', true);
            if (!empty($timbrados)) {
                $sql_timbrado = "INSERT INTO timbrados (empresa_id, numero, inicio, fin) VALUES (:empresa_id, :numero, :inicio, :fin)";
                $stmt_timbrado = $conn->prepare($sql_timbrado);
                foreach ($timbrados as $timbrado) {
                    $stmt_timbrado->execute([
                        ':empresa_id' => $empresa_id,
                        ':numero' => $timbrado['numero'],
                        ':inicio' => $timbrado['inicio'],
                        ':fin' => $timbrado['fin']
                    ]);
                }
            }

            // 4. Asignar digitadores
            $digitadores = $input['digitadores'] ?? [];
            if (!empty($digitadores)) {
                $sql_digitador = "INSERT INTO empresa_digitadores (empresa_id, user_id) VALUES (:empresa_id, :user_id)";
                $stmt_digitador = $conn->prepare($sql_digitador);
                foreach ($digitadores as $digitador_id) {
                    $stmt_digitador->execute([':empresa_id' => $empresa_id, ':user_id' => $digitador_id]);
                }
            }
            
            // 5. Crear plan de cuentas por defecto para la nueva empresa
            $sql_plan = "INSERT INTO plan_de_cuentas (empresa_id, codigo, nombre, tipo) VALUES (:empresa_id, :codigo, :nombre, :tipo)";
            $stmt_plan = $conn->prepare($sql_plan);
            foreach(get_plan_de_cuentas_default() as $cuenta) {
                $stmt_plan->execute([
                    ':empresa_id' => $empresa_id,
                    ':codigo' => $cuenta['codigo'],
                    ':nombre' => $cuenta['nombre'],
                    ':tipo' => $cuenta['tipo']
                ]);
            }

            $conn->commit();
            log_activity("Creó la empresa '{$input['nombre']}' (ID: {$empresa_id})");
            echo json_encode(['success' => true, 'id' => $empresa_id, 'logo' => $logo_path_for_db]);

        } catch (Exception $e) {
            $conn->rollBack();
            http_response_code(500);
            echo json_encode(['error' => 'Error al crear la empresa: ' . $e->getMessage()]);
        }
        return;
    }

    if ($_SESSION['categoria'] === 'Digitador') {
        http_response_code(403);
        echo json_encode(['error' => 'Acceso denegado.']);
        return;
    }

    // Los datos vienen directamente de $_POST y $_FILES para FormData.
    $input = $_POST;

    if (empty($input['nombre']) || empty($input['ruc'])) {
        http_response_code(400);
        echo json_encode(['error' => 'El nombre y el RUC de la empresa son obligatorios.']);
        return;
    }
    $empresas = file_exists($empresas_file) ? json_decode(file_get_contents($empresas_file), true) : [];

    $empresa_id = uniqid();

    $new_empresa = [
        'id' => $empresa_id,
        'nombre' => $input['nombre'],
        'ruc' => $input['ruc'],
        'tipo' => $input['tipo'] ?? 'fisica',
        'responsable_contabilidad' => $input['responsable_contabilidad'] ?? '',
        'telefono' => $input['telefono'] ?? '',
        'direccion' => $input['direccion'] ?? '',
        'ciudad' => $input['ciudad'] ?? '',
        'pais' => $input['pais'] ?? '',
        'actividad_economica' => $input['actividad_economica'] ?? '',
        'observaciones' => $input['observaciones'] ?? '',
        'digitadores' => isset($input['digitadores']) && is_array($input['digitadores']) ? $input['digitadores'] : [],
        'contador_id' => $input['contador_id'] ?? null,
        'timbrados' => json_decode($input['timbrados_json'] ?? '[]', true),
        'presentaciones' => $input['presentaciones'] ?? [],
        'logo' => null
    ];

    // Manejar subida de logo
    if (isset($_FILES['logo']) && $_FILES['logo']['error'] == 0) {
        $empresa_dir = $data_dir . $empresa_id;
        if (!is_dir($empresa_dir)) {
            mkdir($empresa_dir, 0777, true);
        }
        $logo_filename = 'logo' . '.' . pathinfo($_FILES['logo']['name'], PATHINFO_EXTENSION);
        $logo_path = $empresa_dir . '/' . $logo_filename;
        if (move_uploaded_file($_FILES['logo']['tmp_name'], $logo_path)) {
            $new_empresa['logo'] = str_replace('\\', '/', substr($logo_path, strlen(dirname(dirname(__DIR__))) + 1));
        }
    }

    $empresas[] = $new_empresa;
    file_put_contents($empresas_file, json_encode($empresas, JSON_PRETTY_PRINT));

    // Crear directorio y plan de cuentas para la nueva empresa
    $empresa_dir = $data_dir . $new_empresa['id'];
    if (!is_dir($empresa_dir)) {
        mkdir($empresa_dir, 0777, true);
    }
    file_put_contents($empresa_dir . '/plan_de_cuentas.json', json_encode(get_plan_de_cuentas_default(), JSON_PRETTY_PRINT));

    log_activity("Creó la empresa '{$new_empresa['nombre']}' (ID: {$new_empresa['id']})");
    echo json_encode($new_empresa);
}

function handle_update_empresa($input, $data_dir, $empresas_file) {
    if (function_exists('get_db_connection')) {
        if ($_SESSION['categoria'] === 'Digitador') {
            http_response_code(403);
            echo json_encode(['error' => 'Acceso denegado.']);
            return;
        }

        $conn = get_db_connection();
        $input = $_POST; // Datos vienen de FormData
        $empresa_id = $input['id'] ?? '';

        if (empty($empresa_id)) {
            http_response_code(400);
            echo json_encode(['error' => 'Falta el ID de la empresa.']);
            return;
        }

        try {
            $conn->beginTransaction();

            // 1. Actualizar datos principales de la empresa
            $sql_empresa = "UPDATE empresas SET 
                nombre = :nombre, ruc = :ruc, tipo = :tipo, responsable_contabilidad = :responsable, 
                telefono = :telefono, direccion = :direccion, ciudad = :ciudad, pais = :pais, 
                actividad_economica = :actividad, observaciones = :obs, contador_id = :contador_id, 
                presentaciones_json = :presentaciones 
                WHERE id = :id";
            $stmt_empresa = $conn->prepare($sql_empresa);
            $stmt_empresa->execute([
                ':id' => $empresa_id,
                ':nombre' => $input['nombre'],
                ':ruc' => $input['ruc'],
                ':tipo' => $input['tipo'] ?? 'fisica',
                ':responsable' => $input['responsable_contabilidad'] ?? null,
                ':telefono' => $input['telefono'] ?? null,
                ':direccion' => $input['direccion'] ?? null,
                ':ciudad' => $input['ciudad'] ?? null,
                ':pais' => $input['pais'] ?? null,
                ':actividad' => $input['actividad_economica'] ?? null,
                ':obs' => $input['observaciones'] ?? null,
                ':contador_id' => empty($input['contador_id']) ? null : $input['contador_id'],
                ':presentaciones' => json_encode($input['presentaciones'] ?? [])
            ]);

            // 2. Manejar subida de nuevo logo
            if (isset($_FILES['logo']) && $_FILES['logo']['error'] == 0) {
                // Opcional: borrar logo antiguo
                $stmt_old_logo = $conn->prepare("SELECT logo FROM empresas WHERE id = :id");
                $stmt_old_logo->execute([':id' => $empresa_id]);
                $old_logo_path = $stmt_old_logo->fetchColumn();
                if ($old_logo_path && file_exists($old_logo_path)) {
                    unlink($old_logo_path);
                }

                $empresa_dir_for_files = 'data/' . $empresa_id;
                if (!is_dir($empresa_dir_for_files)) {
                    mkdir($empresa_dir_for_files, 0777, true);
                }
                $logo_filename = 'logo.' . pathinfo($_FILES['logo']['name'], PATHINFO_EXTENSION);
                $logo_path_on_server = $empresa_dir_for_files . '/' . $logo_filename;
                if (move_uploaded_file($_FILES['logo']['tmp_name'], $logo_path_on_server)) {
                    $stmt_logo = $conn->prepare("UPDATE empresas SET logo = :logo WHERE id = :id");
                    $stmt_logo->execute([':logo' => $logo_path_on_server, ':id' => $empresa_id]);
                }
            }

            // 3. Actualizar timbrados (borrar los viejos e insertar los nuevos)
            $stmt_delete_timbrados = $conn->prepare("DELETE FROM timbrados WHERE empresa_id = :empresa_id");
            $stmt_delete_timbrados->execute([':empresa_id' => $empresa_id]);

            $timbrados = json_decode($input['timbrados_json'] ?? '[]', true);
            if (!empty($timbrados)) {
                $sql_timbrado = "INSERT INTO timbrados (empresa_id, numero, inicio, fin) VALUES (:empresa_id, :numero, :inicio, :fin)";
                $stmt_timbrado = $conn->prepare($sql_timbrado);
                foreach ($timbrados as $timbrado) {
                    $stmt_timbrado->execute([
                        ':empresa_id' => $empresa_id,
                        ':numero' => $timbrado['numero'],
                        ':inicio' => $timbrado['inicio'],
                        ':fin' => $timbrado['fin']
                    ]);
                }
            }

            // 4. Actualizar digitadores (borrar los viejos e insertar los nuevos)
            $stmt_delete_digitadores = $conn->prepare("DELETE FROM empresa_digitadores WHERE empresa_id = :empresa_id");
            $stmt_delete_digitadores->execute([':empresa_id' => $empresa_id]);

            $digitadores = $input['digitadores'] ?? [];
            if (!empty($digitadores)) {
                $sql_digitador = "INSERT INTO empresa_digitadores (empresa_id, user_id) VALUES (:empresa_id, :user_id)";
                $stmt_digitador = $conn->prepare($sql_digitador);
                foreach ($digitadores as $digitador_id) {
                    $stmt_digitador->execute([':empresa_id' => $empresa_id, ':user_id' => $digitador_id]);
                }
            }

            $conn->commit();
            log_activity("Actualizó la empresa '{$input['nombre']}' (ID: {$empresa_id})");
            
            // Devolver la empresa actualizada para refrescar el dashboard
            $stmt_final = $conn->prepare("SELECT * FROM empresas WHERE id = :id");
            $stmt_final->execute([':id' => $empresa_id]);
            $empresa_actualizada = $stmt_final->fetch();
            echo json_encode($empresa_actualizada);

        } catch (Exception $e) {
            $conn->rollBack();
            http_response_code(500);
            echo json_encode(['error' => 'Error al actualizar la empresa: ' . $e->getMessage()]);
        }
        return;
    }

    if ($_SESSION['categoria'] === 'Digitador') {
        http_response_code(403);
        echo json_encode(['error' => 'Acceso denegado.']);
        return;
    }

    // Los datos vienen de FormData, no de JSON
    $input = $_POST;
    $empresa_id = $input['id'] ?? '';
    $empresa_id = validate_empresa_id($empresa_id);
    if ($empresa_id && file_exists($empresas_file)) {
        $empresas = json_decode(file_get_contents($empresas_file), true);
        $empresa_encontrada = false;
        foreach ($empresas as $key => &$empresa) {
            if ($empresa['id'] === $empresa_id) {
                $empresas[$key]['nombre'] = $input['nombre'];
                $empresas[$key]['ruc'] = $input['ruc'];
                $empresas[$key]['tipo'] = $input['tipo'];
                $empresas[$key]['responsable_contabilidad'] = $input['responsable_contabilidad'];
                $empresas[$key]['telefono'] = $input['telefono'];
                $empresas[$key]['direccion'] = $input['direccion'];
                $empresas[$key]['ciudad'] = $input['ciudad'];
                $empresas[$key]['pais'] = $input['pais'];
                $empresas[$key]['actividad_economica'] = $input['actividad_economica'];
                $empresas[$key]['observaciones'] = $input['observaciones'];
                $empresas[$key]['digitadores'] = $input['digitadores'] ?? [];
                $empresas[$key]['contador_id'] = $input['contador_id'] ?? null;
                $empresas[$key]['timbrados'] = json_decode($input['timbrados_json'] ?? '[]', true);
                $empresas[$key]['presentaciones'] = $input['presentaciones'] ?? [];
                
                // Manejar subida de nuevo logo
                if (isset($_FILES['logo']) && $_FILES['logo']['error'] == 0) {
                    $empresa_dir = $data_dir . $empresa_id;
                    if (!is_dir($empresa_dir)) {
                        mkdir($empresa_dir, 0777, true);
                    }
                    $logo_filename = 'logo' . '.' . pathinfo($_FILES['logo']['name'], PATHINFO_EXTENSION);
                    $logo_path = $empresa_dir . '/' . $logo_filename;
                    if (move_uploaded_file($_FILES['logo']['tmp_name'], $logo_path)) {                        
                        $empresas[$key]['logo'] = str_replace('\\', '/', substr($logo_path, strlen(dirname(dirname(__DIR__))) + 1));
                    }
                }

                $empresa_encontrada = true;
                break;
            }
        }

        if ($empresa_encontrada) {
            file_put_contents($empresas_file, json_encode($empresas, JSON_PRETTY_PRINT));
            log_activity("Actualizó la empresa '{$input['nombre']}' (ID: {$input['id']})");
            echo json_encode($empresas[$key]); // Devolver la empresa actualizada con la ruta del logo
        } else {
            http_response_code(404);
            echo json_encode(['error' => 'Empresa no encontrada para actualizar']);
        }
    } else {
        http_response_code(404);
        echo json_encode(['error' => 'Empresa no encontrada']);
    }
}

function handle_delete_empresa($input, $data_dir, $empresas_file) {
    if (function_exists('get_db_connection')) {
        if ($_SESSION['categoria'] === 'Digitador') {
            http_response_code(403);
            echo json_encode(['error' => 'Acceso denegado.']);
            return;
        }
        $empresa_id = $input['id'] ?? '';
        if (empty($empresa_id)) {
            http_response_code(400);
            echo json_encode(['error' => 'Falta el ID de la empresa.']);
            return;
        }

        $conn = get_db_connection();

        // 1. Obtener datos de la empresa para el log y la ruta del logo
        $stmt = $conn->prepare("SELECT nombre, logo FROM empresas WHERE id = :id");
        $stmt->execute([':id' => $empresa_id]);
        $empresa = $stmt->fetch();

        if ($empresa) {
            // 2. Eliminar el logo si existe
            if ($empresa['logo'] && file_exists($empresa['logo'])) {
                unlink($empresa['logo']);
            }
            // 3. Eliminar la empresa de la BD (ON DELETE CASCADE se encarga del resto)
            $stmt_delete = $conn->prepare("DELETE FROM empresas WHERE id = :id");
            $stmt_delete->execute([':id' => $empresa_id]);
            log_activity("Eliminó la empresa '{$empresa['nombre']}' (ID: {$empresa_id})");
        }
        echo json_encode(['success' => true]);
        return;
    }
    if ($_SESSION['categoria'] === 'Digitador') {
        http_response_code(403);
        echo json_encode(['error' => 'Acceso denegado.']);
        return;
    }
    $empresa_id = $input['id'] ?? '';
    $empresa_id = validate_empresa_id($empresa_id);

    if ($empresa_id && file_exists($empresas_file)) {
        $empresas = json_decode(file_get_contents($empresas_file), true);
        $empresa_a_borrar = null;
        $empresa_encontrada_key = null;

        // 1. Encontrar la empresa y su nombre para el log
        foreach($empresas as $key => $emp) {
            if ($emp['id'] === $empresa_id) {
                $empresa_a_borrar = $emp;
                $empresa_encontrada_key = $key;
                break;
            }
        }

        if ($empresa_a_borrar) {
            // 2. Eliminar la entrada de la empresa del array
            array_splice($empresas, $empresa_encontrada_key, 1);
            file_put_contents($empresas_file, json_encode($empresas, JSON_PRETTY_PRINT));
            
            // 3. Eliminar el directorio completo de la empresa (con asientos, plan de cuentas, logo, etc.)
            $empresa_dir = $data_dir . $empresa_id;
            delete_directory($empresa_dir);

            // 4. Registrar la actividad
            log_activity("Eliminó la empresa '{$empresa_a_borrar['nombre']}' (ID: {$empresa_id})");
        }
        echo json_encode(['success' => true]);
    } else {
        http_response_code(404);
        echo json_encode(['error' => 'Empresa no encontrada']);
    }
}

function handle_export_asientos($data_dir) {
        $empresa_id = $_GET['empresa_id'] ?? '';
        $empresa_id = validate_empresa_id($empresa_id);
        $tipo_factura_export = $_GET['action'] == 'export_ventas' ? 'venta' : 'compra';
        $format = $_GET['format'] ?? 'csv';

        $asientos_file = $data_dir . $empresa_id . '/asientos.json';
        if (!file_exists($asientos_file)) {
            http_response_code(404);
            die('No se encontraron asientos para la empresa.');
        }

        $asientos = json_decode(file_get_contents($asientos_file), true);
        $facturas_a_exportar = array_filter($asientos, function($asiento) use ($tipo_factura_export) {
            return $asiento['tipo_factura'] == $tipo_factura_export;
        });

        $filename = $tipo_factura_export . 's_rg90.' . ($format == 'excel' ? 'xls' : 'csv');

        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename="' . $filename . '"'); // Note: CSVs don't have footers

        $output = fopen('php://output', 'w');
        
        // Escribir la cabecera
        $cabecera = [
            'Fecha', 
            ($tipo_factura_export == 'venta' ? 'RUC Cliente' : 'RUC Proveedor'),
            ($tipo_factura_export == 'venta' ? 'Nombre Cliente' : 'Nombre Proveedor'),
            'Timbrado', 
            'Nro Factura', 
            'Gravado 10%', 
            'IVA 10%', 
            'Gravado 5%', 
            'IVA 5%', 
            'Exento', 
            'Total Factura', 
            'Condicion', 
            'Electronica'
        ];
        fputcsv($output, $cabecera);

        // Escribir los datos
        foreach ($facturas_a_exportar as $factura) {
            $monto = 0;
            foreach($factura['movimientos'] as $mov) {
                if ($mov['haber'] > 0 && ($mov['cuenta_codigo'] == '1.1.1' || $mov['cuenta_codigo'] == '2.1.1')) {
                     $monto = $mov['haber'];
                     break;
                } 
                if ($mov['debe'] > 0 && ($mov['cuenta_codigo'] == '1.1.1' || $mov['cuenta_codigo'] == '2.1.1')) {
                     $monto = $mov['debe'];
                     break;
                }
            }
            if ($monto == 0) { // Fallback por si la lógica anterior no encuentra el monto
                 $monto = array_sum(array_column($factura['movimientos'], 'debe'));
            }

            $iva_10 = 0; $base_10 = 0; $iva_5 = 0; $base_5 = 0; $exento = 0;

            $movimiento_iva_credito = array_values(array_filter($factura['movimientos'], function($m) { return $m['cuenta_codigo'] == '1.1.4'; }));
            $movimiento_iva_debito = array_values(array_filter($factura['movimientos'], function($m) { return $m['cuenta_codigo'] == '2.1.3'; }));
            $movimiento_base_gasto = array_values(array_filter($factura['movimientos'], function($m) { return strpos($m['cuenta_codigo'], '5') === 0; }));
            $movimiento_base_ingreso = array_values(array_filter($factura['movimientos'], function($m) { return strpos($m['cuenta_codigo'], '4') === 0; }));

            $iva_monto = 0;
            $base_monto = 0;
            $iva_rate_percent = 0;

            if($tipo_factura_export == 'compra' && !empty($movimiento_iva_credito)) {
                $iva_monto = $movimiento_iva_credito[0]['debe'];
                $base_monto = $movimiento_base_gasto[0]['debe'];
            } else if ($tipo_factura_export == 'venta' && !empty($movimiento_iva_debito)) {
                $iva_monto = $movimiento_iva_debito[0]['haber'];
                $base_monto = $movimiento_base_ingreso[0]['haber'];
            }

            if ($base_monto > 0) {
                $iva_rate_percent = round(($iva_monto / $base_monto) * 100);
            }

            if ($iva_rate_percent == 10) {
                $iva_10 = $iva_monto;
                $base_10 = $base_monto;
            } elseif ($iva_rate_percent == 5) {
                $iva_5 = $iva_monto;
                $base_5 = $base_monto;
            } else {
                $exento = $monto;
            }

            $linea = [
                $factura['fecha'],
                $factura['ruc_cliente_proveedor'],
                str_replace('Factura ' . $tipo_factura_export . ' - ', '', $factura['descripcion']),
                $factura['timbrado'],
                $factura['comprobante_nro'],
                number_format($base_10, 2, ',', ''),
                number_format($iva_10, 2, ',', ''),
                number_format($base_5, 2, ',', ''),
                number_format($iva_5, 2, ',', ''),
                number_format($exento, 2, ',', ''),
                number_format($monto, 2, ',', ''),
                $factura['condicion_operacion'],
                ($factura['es_electronica'] ?? false) ? 'Si' : 'No'
            ];
            fputcsv($output, $linea);
        }

        fclose($output);
        exit;
}

function handle_get_digitadores($data_dir) {
    if (function_exists('get_db_connection')) {
        $conn = get_db_connection();
        $stmt = $conn->query("SELECT id, nombrecompleto FROM users WHERE categoria = 'Digitador' ORDER BY nombrecompleto ASC");
        $digitadores = $stmt->fetchAll();
        echo json_encode($digitadores);
        return;
    }

    $users_file = $data_dir . 'users.json';
    $users = file_exists($users_file) ? json_decode(file_get_contents($users_file), true) : [];

    $digitadores = array_filter($users, function($user) {
        return isset($user['categoria']) && $user['categoria'] === 'Digitador';
    });

    // Devolver solo id y nombre para el select
    $safe_digitadores = array_map(function($user) { return ['id' => $user['id'], 'nombrecompleto' => $user['nombrecompleto']]; }, $digitadores);

    echo json_encode(array_values($safe_digitadores));
}

function handle_get_contadores($data_dir) {
    if (function_exists('get_db_connection')) {
        $conn = get_db_connection();
        $stmt = $conn->query("SELECT id, nombrecompleto FROM users WHERE categoria IN ('Administrador', 'Contador') ORDER BY nombrecompleto ASC");
        $contadores = $stmt->fetchAll();
        echo json_encode($contadores);
        return;
    }

    $users_file = $data_dir . 'users.json';
    $users = file_exists($users_file) ? json_decode(file_get_contents($users_file), true) : [];

    $contadores = array_filter($users, function($user) {
        return isset($user['categoria']) && in_array($user['categoria'], ['Administrador', 'Contador']);
    });

    $safe_contadores = array_map(function($user) { return ['id' => $user['id'], 'nombrecompleto' => $user['nombrecompleto']]; }, $contadores);

    echo json_encode(array_values($safe_contadores));
}

function handle_get_vencimientos($empresas_file) {
    if (function_exists('get_db_connection')) {
        $conn = get_db_connection();
        $eventos = [];
        $current_year = date('Y');
        $user_id = $_SESSION['user_id'];
        $user_category = $_SESSION['categoria'];

        $sql_empresas = "";
        $params = [];

        if ($user_category === 'Administrador' || $user_category === 'Contador') {
            $sql_empresas = "SELECT id, nombre, presentaciones_json FROM empresas WHERE presentaciones_json IS NOT NULL AND presentaciones_json != '[]'";
        } elseif ($user_category === 'Digitador') {
            $sql_empresas = "SELECT e.id, e.nombre, e.presentaciones_json FROM empresas e 
                             JOIN empresa_digitadores ed ON e.id = ed.empresa_id 
                             WHERE ed.user_id = :user_id AND e.presentaciones_json IS NOT NULL AND e.presentaciones_json != '[]'";
            $params[':user_id'] = $user_id;
        } else {
            echo json_encode([]); // No mostrar nada si no tiene un rol válido
            return;
        }

        // 1. Vencimientos de Timbrados
        $stmt_timbrados = $conn->prepare("SELECT t.fin, e.nombre as empresa_nombre, e.id as empresa_id FROM timbrados t JOIN ($sql_empresas) e ON t.empresa_id = e.id WHERE t.fin IS NOT NULL");
        $stmt_timbrados->execute($params);
        while ($row = $stmt_timbrados->fetch()) {
            $eventos[] = [
                'title' => 'Vence Timbrado: ' . $row['empresa_nombre'],
                'start' => $row['fin'],
                'url'   => 'index.html#empresa=' . $row['empresa_id'],
                'allDay' => true,
                'color' => '#dc3545', // Rojo
                'extendedProps' => [ 'tipo' => 'timbrado' ]
            ];
        }

        // 2. Vencimientos de Presentaciones Fiscales
        $stmt_empresas = $conn->query("SELECT id, nombre, presentaciones_json FROM empresas WHERE presentaciones_json IS NOT NULL AND presentaciones_json != '[]'");
        $periodos_map = [
            'mes' => 1, 'bimestre' => 2, 'trimestre' => 3, 
            'cuatrimestre' => 4, 'semestre' => 6, 'año' => 12
        ];

        while ($empresa = $stmt_empresas->fetch()) {
            $presentaciones = json_decode($empresa['presentaciones_json'], true);
            if (empty($presentaciones)) continue;

            // IVA (mensual)
            if (!empty($presentaciones['iva_dia'])) {
                for ($month = 1; $month <= 12; $month++) {
                    $eventos[] = ['title' => 'IVA: ' . $empresa['nombre'], 'start' => sprintf('%s-%02d-%02d', $current_year, $month, $presentaciones['iva_dia']), 'url' => 'index.html#empresa=' . $empresa['id'], 'allDay' => true, 'color' => '#0d6efd', 'extendedProps' => [ 'tipo' => 'iva' ]];
                }
            }

            // IRP e IRE (periódico)
            foreach (['irp', 'ire'] as $impuesto) {
                if (!empty($presentaciones[$impuesto.'_dia']) && !empty($presentaciones[$impuesto.'_mes']) && !empty($presentaciones[$impuesto.'_periodo']) && isset($periodos_map[$presentaciones[$impuesto.'_periodo']])) {
                    $paso = $periodos_map[$presentaciones[$impuesto.'_periodo']];
                    for ($month = intval($presentaciones[$impuesto.'_mes']); $month <= 12; $month += $paso) {
                        $eventos[] = ['title' => strtoupper($impuesto) . ': ' . $empresa['nombre'], 'start' => sprintf('%s-%02d-%02d', $current_year, $month, $presentaciones[$impuesto.'_dia']), 'url' => 'index.html#empresa=' . $empresa['id'], 'allDay' => true, 'color' => '#198754', 'extendedProps' => [ 'tipo' => $impuesto ]];
                    }
                }
            }

            // Informes Financieros (anual)
            if (!empty($presentaciones['inf_dia']) && !empty($presentaciones['inf_mes'])) {
                 $eventos[] = [
                    'title' => 'Inf. Financ.: ' . $empresa['nombre'],
                    'start' => sprintf('%s-%02d-%02d', $current_year, $presentaciones['inf_mes'], $presentaciones['inf_dia']),
                    'url'   => 'index.html#empresa=' . $empresa['id'],
                    'allDay' => true,
                    'color' => '#ffc107',
                    'textColor' => '#000',
                    'extendedProps' => [ 'tipo' => 'informe' ]
                ];
            }
        }

        echo json_encode($eventos);
        return;
    }

    if (!file_exists($empresas_file)) {
        echo json_encode([]);
        return;
    }

    $empresas = json_decode(file_get_contents($empresas_file), true);
    $user_id = $_SESSION['user_id'];
    $user_category = $_SESSION['categoria'];
    $eventos = [];

    $empresas_visibles = [];

    $current_year = date('Y');
    $periodos_map = [
        'mes' => 1, 'bimestre' => 2, 'trimestre' => 3, 
        'cuatrimestre' => 4, 'semestre' => 6, 'año' => 12
    ];

    if ($user_category === 'Administrador' || $user_category === 'Contador') {
        $empresas_visibles = $empresas;
    } elseif ($user_category === 'Digitador') {
        $empresas_visibles = array_filter($empresas, function($empresa) use ($user_id) {
            return isset($empresa['digitadores']) && in_array($user_id, $empresa['digitadores']);
        });
    }

    foreach ($empresas_visibles as $empresa) {
        $empresa_id = $empresa['id'];
        $empresa_nombre = $empresa['nombre'];

        // 1. Vencimientos de Timbrados (ya existente)
        if (isset($empresa['timbrados']) && is_array($empresa['timbrados'])) {
            foreach ($empresa['timbrados'] as $timbrado) {
                if (!empty($timbrado['fin'])) {
                    $eventos[] = [
                        'title' => 'Vence Timbrado: ' . $empresa_nombre,
                        'start' => $timbrado['fin'],
                        'url'   => 'index.html#empresa=' . $empresa_id,
                        'allDay' => true,
                        'color' => '#dc3545' // Rojo para timbrados
                    ];
                }
            }
        }

        // 2. Vencimientos de Presentaciones Fiscales
        if (isset($empresa['presentaciones']) && is_array($empresa['presentaciones'])) {
            $p = $empresa['presentaciones'];

            // IVA (mensual)
            if (!empty($p['iva_dia'])) {
                for ($month = 1; $month <= 12; $month++) {
                    $eventos[] = [
                        'title' => 'IVA: ' . $empresa_nombre,
                        'start' => sprintf('%s-%02d-%02d', $current_year, $month, $p['iva_dia']),
                        'url'   => 'index.html#empresa=' . $empresa_id,
                        'allDay' => true,
                        'color' => '#0d6efd' // Azul para IVA
                    ];
                }
            }

            // IRP e IRE (periódico)
            foreach (['irp', 'ire'] as $impuesto) {
                if (!empty($p[$impuesto.'_dia']) && !empty($p[$impuesto.'_mes']) && !empty($p[$impuesto.'_periodo'])) {
                    $paso = $periodos_map[$p[$impuesto.'_periodo']];
                    for ($month = intval($p[$impuesto.'_mes']); $month <= 12; $month += $paso) {
                        $eventos[] = [
                            'title' => strtoupper($impuesto) . ': ' . $empresa_nombre,
                            'start' => sprintf('%s-%02d-%02d', $current_year, $month, $p[$impuesto.'_dia']),
                            'url'   => 'index.html#empresa=' . $empresa_id,
                            'allDay' => true,
                            'color' => '#198754' // Verde para IRP/IRE
                        ];
                    }
                }
            }

            // Informes Financieros (anual)
            if (!empty($p['inf_dia']) && !empty($p['inf_mes'])) {
                 $eventos[] = [
                    'title' => 'Inf. Financ.: ' . $empresa_nombre,
                    'start' => sprintf('%s-%02d-%02d', $current_year, $p['inf_mes'], $p['inf_dia']),
                    'url'   => 'index.html#empresa=' . $empresa_id,
                    'allDay' => true,
                    'color' => '#ffc107', // Amarillo para informes
                    'textColor' => '#000' // Texto negro para mejor legibilidad en fondo amarillo
                ];
            }
        }
    }
    echo json_encode($eventos);
}

function handle_get_activity_log($data_dir) {
    if ($_SESSION['categoria'] === 'Digitador') {
        http_response_code(403);
        echo json_encode(['error' => 'Acceso denegado.']);
        return;
    }
    $log_file = $data_dir . 'activity_log.json';
    if (file_exists($log_file)) {
        echo file_get_contents($log_file);
    } else {
        echo json_encode([]);
    }
}

function handle_get_users($data_dir) {
    // Si la conexión a la BD existe, usamos la nueva lógica con MySQL
    if (function_exists('get_db_connection')) {
        $conn = get_db_connection();
        // Seleccionamos todos los campos excepto la contraseña
        $stmt = $conn->query("SELECT id, username, nombrecompleto, email, categoria, documento, telefono, direccion, ciudad, pais FROM users ORDER BY nombrecompleto ASC");
        $users = $stmt->fetchAll();
        echo json_encode($users);
        return;
    }

    // --- Lógica antigua con JSON (fallback) ---
    $users_file = $data_dir . 'users.json';
    $users = file_exists($users_file) ? json_decode(file_get_contents($users_file), true) : [];

    // Nunca devolver las contraseñas al frontend
    $safe_users = array_map(function($user) {
        unset($user['password']);
        return $user;
    }, $users);

    echo json_encode($safe_users);
}

function handle_update_user($input, $data_dir) {
    // Si la conexión a la BD existe, usamos la nueva lógica con MySQL
    if (function_exists('get_db_connection')) {
        check_admin_auth();
        $conn = get_db_connection();

        $sql_parts = [];
        $params = [':id' => $input['id']];

        // Lista de campos que se pueden actualizar
        $fields = ['username', 'nombrecompleto', 'email', 'categoria', 'documento', 'telefono', 'direccion', 'ciudad', 'pais'];
        foreach ($fields as $field) {
            if (isset($input[$field])) {
                $sql_parts[] = "`$field` = :$field";
                $params[":$field"] = $input[$field];
            }
        }

        // Actualizar contraseña solo si se proporciona una nueva
        if (!empty($input['password'])) {
            $sql_parts[] = "`password` = :password";
            $params[':password'] = password_hash($input['password'], PASSWORD_DEFAULT);
        }

        if (empty($sql_parts)) {
            echo json_encode(['success' => true, 'message' => 'No hay nada que actualizar.']);
            return;
        }

        $sql = "UPDATE users SET " . implode(', ', $sql_parts) . " WHERE id = :id";
        $stmt = $conn->prepare($sql);
        $stmt->execute($params);
        echo json_encode(['success' => $stmt->rowCount() > 0]);
        return;
    }
    $users_file = $data_dir . 'users.json';
    $users = file_exists($users_file) ? json_decode(file_get_contents($users_file), true) : [];

    $user_id = $input['id'] ?? '';
    $username = $input['username'] ?? '';
    $password = $input['password'] ?? '';
    $nombrecompleto = $input['nombrecompleto'] ?? '';
    $documento = $input['documento'] ?? '';
    $email = $input['email'] ?? '';
    $telefono = $input['telefono'] ?? '';
    $direccion = $input['direccion'] ?? '';
    $ciudad = $input['ciudad'] ?? '';
    $pais = $input['pais'] ?? '';
    $categoria = $input['categoria'] ?? 'Digitador';

    if (empty($user_id) || empty($username)) {
        http_response_code(400);
        echo json_encode(['error' => 'ID de usuario y nombre de usuario son requeridos.']);
        return;
    }

    $user_found = false;
    foreach ($users as &$user) {
        if ($user['id'] === $user_id) {
            $user['username'] = $username;
            $user['nombrecompleto'] = $nombrecompleto;
            $user['documento'] = $documento;
            $user['email'] = $email;
            $user['telefono'] = $telefono;
            $user['direccion'] = $direccion;
            $user['ciudad'] = $ciudad;
            $user['pais'] = $pais;
            $user['categoria'] = $categoria;
            if (!empty($password)) {
                $user['password'] = password_hash($password, PASSWORD_DEFAULT);
            }
            $user_found = true;
            break;
        }
    }

    if ($user_found) {
        file_put_contents($users_file, json_encode($users, JSON_PRETTY_PRINT));
        echo json_encode(['success' => true]);
    } else {
        http_response_code(404);
        echo json_encode(['error' => 'Usuario no encontrado.']);
    }
}

function handle_delete_user($input, $data_dir) {
    // Si la conexión a la BD existe, usamos la nueva lógica con MySQL
    if (function_exists('get_db_connection')) {
        check_admin_auth();
        $user_id_to_delete = $input['id'] ?? '';

        if (empty($user_id_to_delete)) {
            http_response_code(400);
            echo json_encode(['error' => 'Falta el ID del usuario a eliminar.']);
            return;
        }
        if ($user_id_to_delete == $_SESSION['user_id']) {
            http_response_code(403);
            echo json_encode(['error' => 'No puedes eliminar tu propio usuario.']);
            return;
        }
        $conn = get_db_connection();
        $stmt = $conn->prepare("DELETE FROM users WHERE id = :id");
        $stmt->execute([':id' => $user_id_to_delete]);
        echo json_encode(['success' => $stmt->rowCount() > 0]);
        return;
    }
    $user_id_to_delete = $input['id'] ?? '';

    if (empty($user_id_to_delete)) {
        http_response_code(400);
        echo json_encode(['error' => 'Falta el ID del usuario a eliminar.']);
        return;
    }

    if ($user_id_to_delete === $_SESSION['user_id']) {
        http_response_code(403);
        echo json_encode(['error' => 'No puedes eliminar tu propio usuario.']);
        return;
    }

    $users_file = $data_dir . 'users.json';
    if (file_exists($users_file)) {
        $users = json_decode(file_get_contents($users_file), true);
        $users_filtrados = array_filter($users, fn($user) => $user['id'] !== $user_id_to_delete);

        file_put_contents($users_file, json_encode(array_values($users_filtrados), JSON_PRETTY_PRINT));
        log_activity("Eliminó al usuario con ID: {$user_id_to_delete}");
        echo json_encode(['success' => true]);
    }
}

function handle_register($input, $data_dir) {
    // Si la conexión a la BD existe, usamos la nueva lógica con MySQL
    if (function_exists('get_db_connection')) {
        $conn = get_db_connection();

        // Validaciones
        if (empty($input['username']) || empty($input['password']) || empty($input['nombrecompleto']) || empty($input['email'])) {
            http_response_code(400);
            echo json_encode(['error' => 'Todos los campos marcados con * son obligatorios.']);
            return;
        }

        // Verificar si el usuario o email ya existen
        $stmt = $conn->prepare("SELECT id FROM users WHERE username = :username OR email = :email");
        $stmt->execute([':username' => $input['username'], ':email' => $input['email']]);
        if ($stmt->fetch()) {
            http_response_code(409);
            echo json_encode(['error' => 'El nombre de usuario o el correo electrónico ya están en uso.']);
            return;
        }

        // Insertar nuevo usuario
        $sql = "INSERT INTO users (username, password, nombrecompleto, email, categoria, documento, telefono, direccion, ciudad, pais) VALUES (:username, :password, :nombrecompleto, :email, :categoria, :documento, :telefono, :direccion, :ciudad, :pais)";
        $stmt = $conn->prepare($sql);

        $hashed_password = password_hash($input['password'], PASSWORD_DEFAULT);

        $stmt->execute([
            ':username' => $input['username'], ':password' => $hashed_password, ':nombrecompleto' => $input['nombrecompleto'],
            ':email' => $input['email'], ':categoria' => $input['categoria'] ?? 'Digitador', ':documento' => $input['documento'] ?? null,
            ':telefono' => $input['telefono'] ?? null, ':direccion' => $input['direccion'] ?? null, ':ciudad' => $input['ciudad'] ?? null, ':pais' => $input['pais'] ?? null
        ]);

        echo json_encode(['success' => true]);
        return;
    }

    // --- Lógica antigua con JSON (fallback) ---
    $users_file = $data_dir . 'users.json';
    $users = file_exists($users_file) ? json_decode(file_get_contents($users_file), true) : [];

    $username = $input['username'] ?? '';
    $password = $input['password'] ?? '';
    $nombrecompleto = $input['nombrecompleto'] ?? '';
    $documento = $input['documento'] ?? '';
    $email = $input['email'] ?? '';
    $telefono = $input['telefono'] ?? '';
    $direccion = $input['direccion'] ?? '';
    $ciudad = $input['ciudad'] ?? '';
    $pais = $input['pais'] ?? '';
    $categoria = $input['categoria'] ?? 'Digitador';


    if (empty($username) || empty($password) || empty($nombrecompleto) || empty($email)) {
        http_response_code(400);
        echo json_encode(['error' => 'Usuario y contraseña son requeridos.']);
        return;
    }

    foreach ($users as $user) {
        if ($user['username'] === $username) {
            http_response_code(409);
            echo json_encode(['error' => 'El nombre de usuario ya existe.']);
            return;
        }
    }

    $new_user = [
        'id' => uniqid(),
        'username' => $username,
        'password' => password_hash($password, PASSWORD_DEFAULT),
        'nombrecompleto' => $nombrecompleto,
        'documento' => $documento,
        'email' => $email,
        'telefono' => $telefono,
        'direccion' => $direccion,
        'ciudad' => $ciudad,
        'pais' => $pais,
        'categoria' => $categoria,
        'remember_token' => null
    ];

    $users[] = $new_user;
    file_put_contents($users_file, json_encode($users, JSON_PRETTY_PRINT));
    echo json_encode(['success' => true]);
}

function handle_login($input, $data_dir) {
    // Si la conexión a la BD existe, usamos la nueva lógica con MySQL
    if (function_exists('get_db_connection')) {
        $conn = get_db_connection();
        $username = $input['username'] ?? '';
        $password = $input['password'] ?? '';
        $remember = $input['remember'] ?? false;

        $stmt = $conn->prepare("SELECT * FROM users WHERE username = :username");
        $stmt->execute([':username' => $username]);
        $user = $stmt->fetch();

        if ($user && password_verify($password, $user['password'])) {
            $_SESSION['user_id'] = $user['id'];
            $_SESSION['username'] = $user['username'];
            $_SESSION['categoria'] = $user['categoria'];

            if ($remember) {
                $token = bin2hex(random_bytes(32));
                $token_hash = hash('sha256', $token);

                // Guardar el token hasheado en la base de datos
                $update_stmt = $conn->prepare("UPDATE users SET remember_token = :token WHERE id = :id");
                $update_stmt->execute([':token' => $token_hash, ':id' => $user['id']]);

                // Enviar el token original en la cookie
                $cookie_options = [
                    'expires' => time() + (86400 * 30), // 30 días
                    'path' => '/',
                    'httponly' => true,
                    'samesite' => 'Strict'
                    // 'secure' => true, // Descomentar en producción con HTTPS
                ];
                setcookie('remember_me', $token, $cookie_options);
            }

            echo json_encode(['success' => true]);
            return;
        }

        http_response_code(401);
        echo json_encode(['error' => 'Usuario o contraseña incorrectos.']);
        return;
    }


    // --- Lógica antigua con JSON (fallback) ---
    $users_file = $data_dir . 'users.json';
    if (!file_exists($users_file)) {
        http_response_code(401);
        echo json_encode(['error' => 'Usuario o contraseña incorrectos.']);
        return;
    }

    $users = json_decode(file_get_contents($users_file), true);
    $username = $input['username'] ?? '';
    $password = $input['password'] ?? '';
    $remember = $input['remember'] ?? false;

    foreach ($users as &$user) {
        if ($user['username'] === $username && password_verify($password, $user['password'])) {
            $_SESSION['user_id'] = $user['id'];
            $_SESSION['username'] = $user['username'];
            $_SESSION['categoria'] = $user['categoria'] ?? 'Digitador';

            if ($remember) {
                $token = bin2hex(random_bytes(32));
                $user['remember_token'] = hash('sha256', $token);
                
                // Guardar el token hasheado en el archivo de usuarios
                file_put_contents($users_file, json_encode($users, JSON_PRETTY_PRINT));

                // Enviar el token original en la cookie
                $cookie_options = [
                    'expires' => time() + (86400 * 30), // 30 días
                    'path' => '/',
                    'httponly' => true,
                    'samesite' => 'Strict'
                    // 'secure' => true, // Descomentar en producción con HTTPS
                ];
                setcookie('remember_me', $token, $cookie_options);
            }

            echo json_encode(['success' => true]);
            return;
        }
    }

    http_response_code(401);
    echo json_encode(['error' => 'Usuario o contraseña incorrectos.']);
}

function handle_logout() {
    // Si hay conexión a la BD, limpiar el token de "recuérdame"
    if (function_exists('get_db_connection') && isset($_SESSION['user_id'])) {
        try {
            $conn = get_db_connection();
            $stmt = $conn->prepare("UPDATE users SET remember_token = NULL WHERE id = :id");
            $stmt->execute([':id' => $_SESSION['user_id']]);
        } catch (PDOException $e) {
            // No detener el logout si falla la BD, solo registrar el error si es necesario.
            // error_log("Error al limpiar token de sesión: " . $e->getMessage());
        }
    }

    session_unset();
    session_destroy();

    // Limpiar la cookie de "recordar sesión"
    if (isset($_COOKIE['remember_me'])) {
        unset($_COOKIE['remember_me']);
        $cookie_options = [ 'expires' => time() - 3600, 'path' => '/', 'httponly' => true, 'samesite' => 'Strict' ];
        setcookie('remember_me', '', $cookie_options);
    }

    header('Location: app/login.php');
    exit;
}

function handle_check_session() {
    if (isset($_SESSION['username'])) {
        echo json_encode(['loggedIn' => true, 'username' => $_SESSION['username'], 'categoria' => $_SESSION['categoria']]);
        return;
    }

    // Si hay conexión a la BD, usamos la nueva lógica con MySQL para "recuérdame"
    if (function_exists('get_db_connection') && isset($_COOKIE['remember_me'])) {
        $token = $_COOKIE['remember_me'];
        $token_hash = hash('sha256', $token);

        $conn = get_db_connection();
        $stmt = $conn->prepare("SELECT * FROM users WHERE remember_token = :token");
        $stmt->execute([':token' => $token_hash]);
        $user = $stmt->fetch();

        if ($user) {
            // Iniciar sesión al usuario
            $_SESSION['user_id'] = $user['id'];
            $_SESSION['username'] = $user['username'];
            $_SESSION['categoria'] = $user['categoria'];
            echo json_encode(['loggedIn' => true, 'username' => $_SESSION['username'], 'categoria' => $_SESSION['categoria']]);
            return;
        }
    }

    if (isset($_COOKIE['remember_me'])) {
        $token = $_COOKIE['remember_me'];
        $token_hash = hash('sha256', $token);

        $users_file = $GLOBALS['data_dir'] . 'users.json';
        if (file_exists($users_file)) {
            $users = json_decode(file_get_contents($users_file), true);
            foreach ($users as $user) {
                if (isset($user['remember_token']) && hash_equals($user['remember_token'], $token_hash)) {
                    // Iniciar sesión al usuario
                    $_SESSION['user_id'] = $user['id'];
                    $_SESSION['username'] = $user['username'];
                    $_SESSION['categoria'] = $user['categoria'] ?? 'Digitador';
                    echo json_encode(['loggedIn' => true, 'username' => $_SESSION['username'], 'categoria' => $_SESSION['categoria']]);
                    return;
                }
            }
        }
    }

    // Si no hay sesión ni cookie válida
    echo json_encode(['loggedIn' => false]);
}

function check_auth() {
    if (!isset($_SESSION['username'])) {
        http_response_code(401);
        echo json_encode(['error' => 'No autorizado. Por favor, inicie sesión.']);
        exit;
    }
}

function check_admin_auth() {
    check_auth();
    if ($_SESSION['categoria'] !== 'Administrador') {
        http_response_code(403);
        echo json_encode(['error' => 'Acceso denegado. Se requiere rol de Administrador.']);
        exit;
    }
}

function handle_analizar_con_ia($input) {
    $tipo_analisis = $input['tipo'] ?? '';
    $datos_json = $input['datos'] ?? '';

    if (empty($tipo_analisis) || empty($datos_json)) {
        http_response_code(400);
        echo json_encode(['error' => 'Faltan parámetros para el análisis.']);
        return;
    }

    $prompt = "Analiza los siguientes datos de un reporte de '$tipo_analisis' y proporciona un resumen ejecutivo, puntos clave y recomendaciones. Responde en español y usa formato markdown para una mejor legibilidad: \n\n" . $datos_json;
    $analisis = call_ai_api($prompt);

    // Verificar si la respuesta de la IA es un error
    if (str_starts_with($analisis, 'Error:')) {
        http_response_code(500); // Internal Server Error o Bad Gateway
        echo json_encode(['analisis' => $analisis]); // Devolver el mensaje de error de la IA como parte del JSON
    } else {
        echo json_encode(['analisis' => $analisis]);
    }
}

function handle_get_settings() {
    // Reutiliza la función get_config que ya existe
    $config = get_config();
    echo json_encode($config);
}

function handle_save_settings($input) {
    // Reutiliza la función save_config que ya existe
    $current_config = get_config();
    // Actualiza solo la clave de la IA, manteniendo otras posibles configuraciones futuras
    $current_config['ai_api_key'] = $input['ai_api_key'] ?? '';
    save_config($current_config);
    echo json_encode(['success' => true]);
}

// --- ENRUTADOR PRINCIPAL ---
$action = $_REQUEST['action'] ?? '';
$method = $_SERVER['REQUEST_METHOD'];

if ($method === 'GET') {
    // Las acciones que no devuelven JSON deben manejarse primero para establecer las cabeceras correctas.
    if ($action == 'generar_reporte_balance_general') {
        check_auth();
        handle_generar_reporte_balance_general($data_dir);
        exit;
    }
    if ($action == 'generar_reporte_estado_resultados') {
        handle_generar_reporte_estado_resultados($data_dir);
        exit;
    }
    if ($action == 'generar_reporte_libro_diario') {
        check_auth();
        handle_generar_reporte_libro_diario($data_dir);
        exit;
    }
    if ($action == 'generar_reporte_sumas_y_saldos') {
        check_auth();
        handle_generar_reporte_sumas_y_saldos($data_dir);
        exit;
    }
    if ($action == 'generar_reporte_plan_de_cuentas') {
        check_auth();
        handle_generar_reporte_plan_de_cuentas($data_dir);
        exit;
    }
    if ($action == 'generar_reporte_iva') {
        check_auth();
        handle_generar_reporte_iva($data_dir);
        exit;
    }
    if ($action == 'export_ventas' || $action == 'export_compras') {
        check_auth();
        handle_export_asientos($data_dir);
        exit;
    }
    if ($action == 'exportar_rg90') {
        check_auth();
        handle_export_rg90($data_dir);
        exit;
    }
    if ($action == 'logout') {
        handle_logout();
        exit;
    }

    // Acciones que devuelven JSON
    header('Content-Type: application/json');
    switch ($action) {
        case 'check_session':
            handle_check_session();
            break;
        case 'get_empresas':
            check_auth();
            handle_get_empresas($empresas_file);
            break;
        case 'get_digitadores':
            check_auth();
            handle_get_digitadores($data_dir);
            break;
        case 'get_contadores':
            check_auth();
            handle_get_contadores($data_dir);
            break;
        case 'get_vencimientos':
            check_auth();
            handle_get_vencimientos($empresas_file);
            break;
        case 'get_plan_de_cuentas':
            check_auth();
            handle_get_plan_de_cuentas($data_dir);
            break;
        case 'get_asientos':
            check_auth();
            handle_get_asientos($data_dir);
            break;
        case 'get_users':
            check_admin_auth();
            handle_get_users($data_dir);
            break;
        case 'get_activity_log':
            check_auth();
            handle_get_activity_log($data_dir);
            break;
        case 'get_settings':
            check_admin_auth();
            handle_get_settings();
            break;
        case 'get_iva_report_data':
            check_auth();
            handle_get_iva_report_data($data_dir);
            break;
        default:
            http_response_code(404);
            echo json_encode(['error' => 'Acción GET no encontrada']);
            break;
    }
} elseif ($method === 'POST') {
    header('Content-Type: application/json');

    // Determinar si la petición es JSON o FormData
    $is_json_request = isset($_SERVER['CONTENT_TYPE']) && strpos($_SERVER['CONTENT_TYPE'], 'application/json') !== false;
    if ($is_json_request) {
        $input = json_decode(file_get_contents('php://input'), true) ?? [];
    } else {
        // Para FormData, los datos están en $_POST y $_FILES.
        // Las funciones que lo necesiten los leerán directamente. $input se deja vacío.
        $input = [];
    }

    if ($action === 'login') {
        handle_login($input, $data_dir);
        exit;
    } elseif ($action === 'create_user' || $action === 'register') { // Permitir registro sin sesión
        handle_register($input, $data_dir);
        exit;
    }

    check_auth(); // Proteger todas las demás acciones POST

    $post_actions = [
        'create_empresa' => fn() => handle_create_empresa(null, $data_dir, $empresas_file), // Admin/Contador - Usa $_POST
        'update_empresa' => fn() => handle_update_empresa(null, $data_dir, $empresas_file), // Admin/Contador - Usa $_POST
        'delete_empresa' => fn() => handle_delete_empresa($input, $data_dir, $empresas_file), // Admin/Contador - Usa JSON
        'update_plan_de_cuentas' => fn() => handle_update_plan_de_cuentas($input, $data_dir),
        'create_asiento' => fn() => handle_create_asiento($input, $data_dir),
        'update_asiento' => fn() => handle_update_asiento($input, $data_dir),
        'delete_asiento' => fn() => handle_delete_asiento($input, $data_dir),
        'save_manual_asiento' => fn() => handle_save_manual_asiento($input, $data_dir),
        'update_user' => fn() => handle_update_user($input, $data_dir), // Admin
        'delete_user' => fn() => handle_delete_user($input, $data_dir), // Admin
        'save_settings' => fn() => handle_save_settings($input), // Admin
    ];

    if (isset($post_actions[$action])) {
        $post_actions[$action]();
    } else {
        http_response_code(404);
        echo json_encode(['error' => 'Acción POST no encontrada']);
    }
}


function handle_export_rg90($data_dir) {
    // 1. Validar y obtener parámetros
    $empresa_id = $_GET['empresa_id'] ?? '';
    $periodo = $_GET['periodo'] ?? ''; // Formato esperado: YYYY-MM

    if (empty($empresa_id) || !preg_match('/^\d{4}-\d{2}$/', $periodo)) {
        http_response_code(400);
        die('<h1>Error: Se requiere un ID de empresa y un período válido (YYYY-MM).</h1>');
    }

    list($year, $month) = explode('-', $periodo);
    $fecha_desde = "$periodo-01";
    $fecha_hasta = date("Y-m-t", strtotime($fecha_desde));

    // 2. Cargar datos
    $data = get_processed_accounting_data($empresa_id, $fecha_desde, $fecha_hasta);
    $asientos = $data['asientos'];
    $empresa_actual = $data['empresa_actual'];

    if (!$empresa_actual) {
        http_response_code(404);
        die('<h1>Error: Empresa no encontrada.</h1>');
    }

    // 3. Preparar datos para el CSV
    $csv_rows = [];
    foreach ($asientos as $asiento) {
        // Omitir asientos que no son facturas de venta o compra
        if (!in_array($asiento['tipo_factura'], ['venta', 'compra'])) {
            continue;
        }

        // --- Mapeo y Cálculos ---
        $tipo_registro = ($asiento['tipo_factura'] == 'venta') ? 1 : 2;
        
        // Calcular montos
        $montos = calcular_montos_rg90($asiento);

        $row = [];
        if ($tipo_registro == 1) { // Venta
            $row = [
                1, // CÓDIGO TIPO DE REGISTRO (Venta)
                11, // CÓDIGO TIPO DE IDENTIFICACIÓN DEL COMPRADOR (Asumimos RUC)
                preg_replace('/[^0-9]/', '', $asiento['ruc_cliente_proveedor']), // NÚMERO DE IDENTIFICACIÓN
                '', // NOMBRE O RAZÓN SOCIAL (No requerido para RUC)
                mapear_codigo_comprobante($asiento['tipo_comprobante']),
                (new DateTime($asiento['fecha']))->format('d/m/Y'),
                preg_replace('/[^0-9]/', '', $asiento['timbrado']),
                $asiento['comprobante_nro'],
                $montos['gravado10'],
                $montos['gravado5'],
                $montos['exento'],
                $montos['total'],
                mapear_condicion_operacion($asiento['condicion_operacion']),
                ($asiento['moneda'] !== 'PYG' ? 'S' : 'N'),
                'S', // IMPUTA AL IVA (Asumimos S)
                'S', // IMPUTA AL IRE (Asumimos S)
                'N', // IMPUTA AL IRP-RSP (Asumimos N)
                '', // NÚMERO DEL COMPROBANTE DE VENTA ASOCIADO
                ''  // TIMBRADO DEL COMPROBANTE DE VENTA ASOCIADO
            ];
        } else { // Compra
            $row = [
                2, // CÓDIGO TIPO DE REGISTRO (Compra)
                11, // CÓDIGO TIPO DE IDENTIFICACIÓN DEL PROVEEDOR (Asumimos RUC)
                preg_replace('/[^0-9]/', '', $asiento['ruc_cliente_proveedor']),
                '', // NOMBRE O RAZÓN SOCIAL (No requerido para RUC)
                mapear_codigo_comprobante($asiento['tipo_comprobante']),
                (new DateTime($asiento['fecha']))->format('d/m/Y'),
                preg_replace('/[^0-9]/', '', $asiento['timbrado']),
                $asiento['comprobante_nro'],
                $montos['gravado10'],
                $montos['gravado5'],
                $montos['exento'],
                $montos['total'],
                mapear_condicion_operacion($asiento['condicion_operacion']),
                ($asiento['moneda'] !== 'PYG' ? 'S' : 'N'),
                'S', // IMPUTA AL IVA
                'S', // IMPUTA AL IRE
                'N', // IMPUTA AL IRP-RSP
                'N', // NO IMPUTA
                '',  // NÚMERO DEL COMPROBANTE DE VENTA ASOCIADO
                ''   // TIMBRADO DEL COMPROBANTE DE VENTA ASOCIADO
            ];
        }
        $csv_rows[] = $row;
    }

    // 4. Generar y descargar el archivo ZIP
    $ruc_sin_dv = explode('-', $empresa_actual['ruc'])[0];
    $periodo_str = $year . $month;
    $identificador_unico = substr(str_shuffle("0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ"), 0, 5);

    $csv_filename = "{$ruc_sin_dv}_REG_{$month}{$year}_{$identificador_unico}.csv";
    $zip_filename = "{$ruc_sin_dv}_REG_{$month}{$year}_{$identificador_unico}.zip";

    $temp_csv_path = tempnam(sys_get_temp_dir(), 'rg90_');
    $file_handle = fopen($temp_csv_path, 'w');
    
    foreach ($csv_rows as $row) {
        fputcsv($file_handle, $row, ','); // Usar coma como delimitador
    }
    fclose($file_handle);

    $zip = new ZipArchive();
    $zip_path = sys_get_temp_dir() . '/' . $zip_filename;

    if ($zip->open($zip_path, ZipArchive::CREATE | ZipArchive::OVERWRITE) === TRUE) {
        $zip->addFile($temp_csv_path, $csv_filename);
        $zip->close();

        // Limpiar el CSV temporal
        unlink($temp_csv_path);

        // Enviar el ZIP para descarga
        header('Content-Type: application/zip');
        header('Content-Disposition: attachment; filename="' . basename($zip_filename) . '"');
        header('Content-Length: ' . filesize($zip_path));
        flush();
        readfile($zip_path);

        // Limpiar el ZIP temporal
        unlink($zip_path);
        exit;
    } else {
        http_response_code(500);
        die('<h1>Error: No se pudo crear el archivo ZIP.</h1>');
    }
}

// --- Funciones de Ayuda para RG90 ---

function calcular_montos_rg90($asiento) {
    $total = 0;
    $iva10 = 0;
    $iva5 = 0;
    $base10 = 0;
    $base5 = 0;
    $exento = 0;

    // El total del comprobante es el 'debe' a Caja/Bancos (en ventas) o el 'haber' (en compras)
    foreach ($asiento['movimientos'] as $mov) {
        if ($asiento['tipo_factura'] == 'venta' && ($mov['cuenta_codigo'] == '1.1.1' || $mov['cuenta_codigo'] == '1.1.2')) {
            $total += $mov['debe'];
        } elseif ($asiento['tipo_factura'] == 'compra' && ($mov['cuenta_codigo'] == '1.1.1' || $mov['cuenta_codigo'] == '2.1.1')) {
            $total += $mov['haber'];
        }
    }
    if ($total == 0) { // Fallback si no se encuentra por cuenta de pago
        $total = array_sum(array_column($asiento['movimientos'], 'debe'));
    }

    $iva_debito = 0;
    $iva_credito = 0;
    $base_ingreso = 0;
    $base_gasto = 0;

    foreach ($asiento['movimientos'] as $mov) {
        if ($mov['cuenta_codigo'] == '2.1.3') $iva_debito += $mov['haber']; // IVA Débito Fiscal
        if ($mov['cuenta_codigo'] == '1.1.4') $iva_credito += $mov['debe']; // IVA Crédito Fiscal
        if (substr($mov['cuenta_codigo'], 0, 2) == '4.') $base_ingreso += $mov['haber'];
        if (substr($mov['cuenta_codigo'], 0, 2) == '5.') $base_gasto += $mov['debe'];
    }

    $iva_monto = ($asiento['tipo_factura'] == 'venta') ? $iva_debito : $iva_credito;
    $base_monto = ($asiento['tipo_factura'] == 'venta') ? $base_ingreso : $base_gasto;
    
    if ($iva_monto > 0 && $base_monto > 0) {
        $tasa_aprox = round(($iva_monto / $base_monto) * 100);
        if ($tasa_aprox >= 9 && $tasa_aprox <= 11) { // Rango flexible para 10%
            $iva10 = $iva_monto;
            $base10 = $base_monto;
        } elseif ($tasa_aprox >= 4 && $tasa_aprox <= 6) { // Rango flexible para 5%
            $iva5 = $iva_monto;
            $base5 = $base_monto;
        }
    } else {
        $exento = $total;
    }

    return [
        'gravado10' => round($base10 + $iva10),
        'gravado5' => round($base5 + $iva5),
        'exento' => round($exento),
        'total' => round($total)
    ];
}

function mapear_codigo_comprobante($tipo_comprobante_str) {
    $mapa = [
        'factura' => 109,
        'nota de credito' => 110,
        'nota de debito' => 111,
        'autofactura' => 101,
        'boleta de transporte publico de pasajeros' => 102,
        'boleta de venta' => 103,
        'boleta resimple' => 104,
        'boletos de loterias, juegos de azar' => 105,
        'boleto o ticket de transporte aereo' => 106,
        'despacho de importacion' => 107,
        'entrada a espectaculos publicos' => 108,
        'ticket maquina registradora' => 112,
    ];
    return $mapa[strtolower($tipo_comprobante_str)] ?? 109; // Default a Factura
}

function mapear_condicion_operacion($condicion_str) {
    $mapa = [
        'contado' => 1,
        'credito' => 2,
    ];
    return $mapa[strtolower($condicion_str)] ?? 1; // Default a Contado
}

exit; // Asegurarse de que el script termine aquí

?>