<?php
header('Content-Type: application/json');

$jsonDbFile = 'articulos.json';
$uploadDir = 'uploads/';

// --- Verificación de Permisos ---
// Comprueba si el directorio de subidas existe y si se puede escribir en él.
if (!is_dir($uploadDir) || !is_writable($uploadDir)) {
    http_response_code(500); // Internal Server Error
    echo json_encode(['status' => 'error', 'message' => 'Error de configuración del servidor: El directorio de subidas no existe o no tiene permisos de escritura.']);
    exit;
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Determinar si es una actualización o una creación
    $isUpdate = !empty($_POST['id']);
    $articleId = $isUpdate ? $_POST['id'] : uniqid('article_');

    // Validar campos de texto
    if (empty($_POST['title']) || empty($_POST['content'])) {
        http_response_code(400);
        echo json_encode(['status' => 'error', 'message' => 'El título y el contenido son obligatorios.']);
        exit;
    }

    // Leer la base de datos existente
    $articles = file_exists($jsonDbFile) ? json_decode(file_get_contents($jsonDbFile), true) : [];

    // Si es una actualización, encontrar el artículo existente para no perder la imagen si no se sube una nueva.
    $existingArticle = null;
    if ($isUpdate) {
        $articleIndex = -1;
        foreach ($articles as $index => $article) {
            if ($article['id'] === $articleId) {
                $existingArticle = $article;
                $articleIndex = $index;
                break;
            }
        }
    }
    // Manejo de la imagen del artículo
    $imagePath = null;
    if (isset($_FILES['image']) && $_FILES['image']['error'] === UPLOAD_ERR_OK) {
        $imageTmpName = $_FILES['image']['tmp_name'];
        $imageName = uniqid('img_') . '-' . basename($_FILES['image']['name']);
        $imagePath = $uploadDir . $imageName;
        if (!move_uploaded_file($imageTmpName, $imagePath)) {
            http_response_code(500);
            echo json_encode(['status' => 'error', 'message' => 'Error al subir la imagen.']);
            exit;
        }
        // Si es una actualización y se sube una nueva imagen, eliminar la antigua si existe.
        if ($isUpdate && isset($existingArticle['image']) && file_exists($existingArticle['image'])) {
            @unlink($existingArticle['image']);
        }
    } elseif ($isUpdate && isset($existingArticle['image'])) {
        // Si es una actualización y no se sube una nueva imagen, mantener la existente.
        $imagePath = $existingArticle['image'];
    }

    // Manejo del archivo descargable
    $filePath = null;
    $fileName = null;
    if (isset($_FILES['downloadable_file']) && $_FILES['downloadable_file']['error'] === UPLOAD_ERR_OK) {
        $fileTmpName = $_FILES['downloadable_file']['tmp_name'];
        $fileName = basename($_FILES['downloadable_file']['name']);
        $fileUniqueName = uniqid('file_') . '-' . $fileName;
        $filePath = $uploadDir . $fileUniqueName;
        if (!move_uploaded_file($fileTmpName, $filePath)) {
            http_response_code(500);
            echo json_encode(['status' => 'error', 'message' => 'Error al subir el archivo descargable.']);
            exit;
        }
        // Si es una actualización y se sube un nuevo archivo, eliminar el antiguo si existe.
        if ($isUpdate && isset($existingArticle['downloadable_file_path']) && file_exists($existingArticle['downloadable_file_path'])) {
            @unlink($existingArticle['downloadable_file_path']);
        }
    } elseif ($isUpdate && isset($existingArticle['downloadable_file_path'])) {
        // Mantener el archivo descargable existente si no se sube uno nuevo.
        $filePath = $existingArticle['downloadable_file_path'];
        $fileName = $existingArticle['downloadable_file_name'];
    }

    // Crear o actualizar el artículo
    $articleData = [
        'id' => $articleId,
        'title' => htmlspecialchars($_POST['title']),
        'subtitle' => htmlspecialchars($_POST['subtitle']),
        'category' => htmlspecialchars($_POST['category']),
        'content' => $_POST['content'], // Permitimos HTML, se debe tener cuidado al renderizar
        'image' => $imagePath,
        'downloadable_file_path' => $filePath,
        'downloadable_file_name' => $fileName,
        'timestamp' => $isUpdate && isset($existingArticle['timestamp']) ? $existingArticle['timestamp'] : date('Y-m-d H:i:s'),
        'last_updated' => $isUpdate ? date('Y-m-d H:i:s') : null
    ];

    if ($isUpdate) {
        if (isset($articleIndex) && $articleIndex > -1) {
            $articles[$articleIndex] = $articleData;
            $message = 'Artículo actualizado con éxito.';
        } else {
            http_response_code(404);
            echo json_encode(['status' => 'error', 'message' => 'El artículo a actualizar no fue encontrado.']);
            exit;
        }
    } else {
        array_unshift($articles, $articleData);
        $message = 'Artículo creado con éxito.';
    }

    // Guardar en el archivo JSON
    file_put_contents($jsonDbFile, json_encode($articles, JSON_PRETTY_PRINT));

    echo json_encode(['status' => 'success', 'message' => $message]);

} elseif ($_SERVER['REQUEST_METHOD'] === 'DELETE') {
    $input = file_get_contents('php://input');
    $data = json_decode($input, true);
    $articleIdToDelete = $data['id'] ?? null;

    if (!$articleIdToDelete) {
        http_response_code(400);
        echo json_encode(['status' => 'error', 'message' => 'No se proporcionó ID del artículo.']);
        exit;
    }

    $articles = file_exists($jsonDbFile) ? json_decode(file_get_contents($jsonDbFile), true) : [];
    
    $updatedArticles = array_values(array_filter($articles, function($article) use ($articleIdToDelete) {
        return $article['id'] !== $articleIdToDelete;
    }));

    file_put_contents($jsonDbFile, json_encode($updatedArticles, JSON_PRETTY_PRINT));

    echo json_encode(['status' => 'success', 'message' => 'Artículo eliminado.']);
} else {
    http_response_code(405);
    echo json_encode(['status' => 'error', 'message' => 'Método no permitido.']);
}
?>