<?php
// api.php

// Establecer la cabecera para devolver JSON
header('Content-Type: application/json');

// Solo permitir peticiones POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405); // Method Not Allowed
    echo json_encode(['success' => false, 'message' => 'Método no permitido.']);
    exit;
}

// Iniciar sesión para manejar el estado del usuario
session_start();

// Obtener los datos enviados en el cuerpo de la petición
$input = json_decode(file_get_contents('php://input'), true);

// Determinar la acción a realizar
// Si la petición es JSON, la acción está en $input. Si es FormData (con archivos), está en $_POST.
$action = $input['action'] ?? $_POST['action'] ?? 'login';

switch ($action) {
    case 'login':
        handleLogin($input);
        break;
    case 'createUser':
        handleCreateUser($input);
        break;
    case 'deleteUser':
        handleDeleteUser($input);
        break;
    case 'deleteContact':
        handleDeleteContact($input);
        break;
    case 'createBlogPost':
        handleCreateBlogPost();
        break;
    case 'deleteBlogPost':
        handleDeleteBlogPost($input);
        break;
    case 'getPostDetails':
        handleGetPostDetails($input);
        break;
    case 'updateBlogPost':
        handleUpdateBlogPost(); // No necesita $input porque usa $_POST y $_FILES
        break;
    default:
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'Acción no válida.']);
        break;
}

function handleLogin($data) {
    $username = $data['username'] ?? '';
    $password = $data['password'] ?? '';

    if (empty($username) || empty($password)) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'Usuario y contraseña son requeridos.']);
        exit;
    }

    $db = getDb();
    foreach ($db['users'] as $user) {
        if ($user['username'] === $username && password_verify($password, $user['password'])) {
            $_SESSION['user_logged_in'] = true;
            $_SESSION['username'] = $user['username'];
            echo json_encode(['success' => true, 'message' => 'Login exitoso.']);
            exit;
        }
    }

    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'Usuario o contraseña incorrectos.']);
}

function handleCreateUser($data) {
    checkAuth();
    $name = $data['name'] ?? '';
    $email = $data['email'] ?? '';
    $username = $data['username'] ?? '';
    $password = $data['password'] ?? '';

    if (empty($name) || empty($email) || empty($username) || empty($password)) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'Todos los campos son requeridos.']);
        exit;
    }

    $db = getDb();
    foreach ($db['users'] as $user) {
        if (isset($user['username']) && $user['username'] === $username) {
            http_response_code(409); // Conflict
            echo json_encode(['success' => false, 'message' => 'El nombre de usuario ya existe.']);
            exit;
        }
        if (isset($user['email']) && $user['email'] === $email) {
            http_response_code(409); // Conflict
            echo json_encode(['success' => false, 'message' => 'La dirección de correo electrónico ya está en uso.']);
            exit;
        }
    }

    $db['users'][] = [
        'name' => $name,
        'email' => $email,
        'username' => $username,
        'password' => password_hash($password, PASSWORD_DEFAULT)
    ];

    saveDb($db);
    echo json_encode(['success' => true, 'message' => 'Usuario creado exitosamente.']);
}

function handleDeleteUser($data) {
    checkAuth();
    $username = $data['username'] ?? '';

    if (empty($username)) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'Nombre de usuario requerido.']);
        exit;
    }

    if ($username === $_SESSION['username']) {
        http_response_code(403); // Forbidden
        echo json_encode(['success' => false, 'message' => 'No puedes eliminar a tu propio usuario.']);
        exit;
    }

    $db = getDb();
    $initial_count = count($db['users']);
    $db['users'] = array_filter($db['users'], fn($user) => $user['username'] !== $username);

    if (count($db['users']) < $initial_count) {
        saveDb($db);
        echo json_encode(['success' => true, 'message' => 'Usuario eliminado exitosamente.']);
    } else {
        http_response_code(404); // Not Found
        echo json_encode(['success' => false, 'message' => 'Usuario no encontrado.']);
    }
}

function handleDeleteContact($data) {
    checkAuth();
    $id = $data['id'] ?? '';

    if (empty($id)) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'ID del mensaje es requerido.']);
        exit;
    }

    $db = getDb();
    $initial_count = count($db['contacts']);
    $db['contacts'] = array_filter($db['contacts'], fn($contact) => $contact['id'] !== $id);

    if (count($db['contacts']) < $initial_count) {
        saveDb($db);
        echo json_encode(['success' => true, 'message' => 'Mensaje eliminado exitosamente.']);
    } else {
        http_response_code(404); // Not Found
        echo json_encode(['success' => false, 'message' => 'Mensaje no encontrado.']);
    }
}

function handleCreateBlogPost() {
    checkAuth();
    
    // Los datos ahora vienen de $_POST y $_FILES, no de $data (JSON)
    $title = trim($_POST['title'] ?? '');
    $subtitle = trim($_POST['subtitle'] ?? '');
    $category = trim($_POST['category'] ?? '');
    $content = trim($_POST['content'] ?? '');

    if (empty($title) || empty($subtitle) || empty($category) || empty($content)) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'Título, subtítulo, categoría y contenido son requeridos.']);
        exit;
    }

    // --- Manejo de la imagen principal ---
    if (!isset($_FILES['image']) || $_FILES['image']['error'] !== UPLOAD_ERR_OK) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'La imagen principal es obligatoria.']);
        exit;
    }
    $imageDir = 'uploads/blog/images/';
    if (!is_dir($imageDir)) mkdir($imageDir, 0777, true);
    $imageExtension = pathinfo($_FILES['image']['name'], PATHINFO_EXTENSION);
    $imageName = 'post_img_' . uniqid() . '.' . $imageExtension;
    $imagePath = $imageDir . $imageName;
    if (!move_uploaded_file($_FILES['image']['tmp_name'], $imagePath)) {
        http_response_code(500);
        echo json_encode(['success' => false, 'message' => 'Error al guardar la imagen.']);
        exit;
    }

    // --- Manejo de archivos adjuntos (opcional) ---
    $attachments = [];
    if (isset($_FILES['attachments'])) {
        $attachmentDir = 'uploads/blog/attachments/';
        if (!is_dir($attachmentDir)) mkdir($attachmentDir, 0777, true);

        $fileCount = count($_FILES['attachments']['name']);
        for ($i = 0; $i < $fileCount; $i++) {
            if ($_FILES['attachments']['error'][$i] === UPLOAD_ERR_OK) {
                $originalName = $_FILES['attachments']['name'][$i];
                $fileExtension = pathinfo($originalName, PATHINFO_EXTENSION);
                $fileName = 'attachment_' . uniqid() . '.' . $fileExtension;
                $filePath = $attachmentDir . $fileName;

                if (move_uploaded_file($_FILES['attachments']['tmp_name'][$i], $filePath)) {
                    $attachments[] = [
                        'filename' => $originalName,
                        'path' => $filePath
                    ];
                }
            }
        }
    }

    $db = getDb();
    $newPost = [
        'id' => 'post_' . uniqid(),
        'date' => date('Y-m-d H:i:s'),
        'author' => $_SESSION['username'],
        'title' => $title,
        'subtitle' => $subtitle,
        'image' => $imagePath,
        'category' => $category,
        'content' => $content,
        'attachments' => $attachments
    ];

    array_unshift($db['blogPosts'], $newPost);
    saveDb($db);

    echo json_encode(['success' => true, 'message' => 'Artículo publicado exitosamente.']);
}

function handleDeleteBlogPost($data) {
    checkAuth();
    // Esta función ahora recibe JSON, así que $data está bien
    $id = $data['id'] ?? '';

    if (empty($id)) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'ID del artículo es requerido.']);
        exit;
    }

    $db = getDb();
    $initial_count = count($db['blogPosts']);

    // Encontrar y eliminar archivos asociados
    foreach ($db['blogPosts'] as $post) {
        if ($post['id'] === $id) {
            if (isset($post['image']) && file_exists($post['image'])) {
                @unlink($post['image']);
            }
            if (isset($post['attachments'])) {
                foreach ($post['attachments'] as $attachment) {
                    if (file_exists($attachment['path'])) {
                        @unlink($attachment['path']);
                    }
                }
            }
            break;
        }
    }

    $db['blogPosts'] = array_filter($db['blogPosts'], fn($post) => $post['id'] !== $id);

    if (count($db['blogPosts']) < $initial_count) {
        saveDb($db);
        echo json_encode(['success' => true, 'message' => 'Artículo eliminado exitosamente.']);
    } else {
        http_response_code(404);
        echo json_encode(['success' => false, 'message' => 'Artículo no encontrado.']);
    }
}

function handleGetPostDetails($data) {
    checkAuth();
    $id = $data['id'] ?? '';

    if (empty($id)) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'ID del artículo es requerido.']);
        exit;
    }

    $db = getDb();
    foreach ($db['blogPosts'] as $post) {
        if ($post['id'] === $id) {
            echo json_encode(['success' => true, 'post' => $post]);
            exit;
        }
    }

    http_response_code(404);
    echo json_encode(['success' => false, 'message' => 'Artículo no encontrado.']);
}

function handleUpdateBlogPost() {
    checkAuth();

    $id = $_POST['id'] ?? '';
    if (empty($id)) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'ID del artículo es requerido.']);
        exit;
    }

    $db = getDb();
    $postIndex = -1;
    foreach ($db['blogPosts'] as $index => $p) {
        if ($p['id'] === $id) {
            $postIndex = $index;
            break;
        }
    }

    if ($postIndex === -1) {
        http_response_code(404);
        echo json_encode(['success' => false, 'message' => 'Artículo no encontrado.']);
        exit;
    }

    // Actualizar campos de texto
    $db['blogPosts'][$postIndex]['title'] = trim($_POST['title'] ?? $db['blogPosts'][$postIndex]['title']);
    $db['blogPosts'][$postIndex]['subtitle'] = trim($_POST['subtitle'] ?? $db['blogPosts'][$postIndex]['subtitle']);
    $db['blogPosts'][$postIndex]['category'] = trim($_POST['category'] ?? $db['blogPosts'][$postIndex]['category']);
    $db['blogPosts'][$postIndex]['content'] = trim($_POST['content'] ?? $db['blogPosts'][$postIndex]['content']);

    // Reemplazar imagen principal si se sube una nueva
    if (isset($_FILES['image']) && $_FILES['image']['error'] === UPLOAD_ERR_OK) {
        // Eliminar la imagen antigua
        if (isset($db['blogPosts'][$postIndex]['image']) && file_exists($db['blogPosts'][$postIndex]['image'])) {
            @unlink($db['blogPosts'][$postIndex]['image']);
        }
        // Guardar la nueva imagen
        $imageDir = 'uploads/blog/images/';
        $imageExtension = pathinfo($_FILES['image']['name'], PATHINFO_EXTENSION);
        $imageName = 'post_img_' . uniqid() . '.' . $imageExtension;
        $imagePath = $imageDir . $imageName;
        if (move_uploaded_file($_FILES['image']['tmp_name'], $imagePath)) {
            $db['blogPosts'][$postIndex]['image'] = $imagePath;
        }
    }

    // Eliminar adjuntos seleccionados
    $attachmentsToDelete = $_POST['delete_attachments'] ?? [];
    if (!empty($attachmentsToDelete)) {
        // Eliminar archivos del servidor
        foreach ($attachmentsToDelete as $pathToDelete) {
            if (file_exists($pathToDelete)) {
                @unlink($pathToDelete);
            }
        }
        // Filtrar los adjuntos del array en la BD
        $db['blogPosts'][$postIndex]['attachments'] = array_filter(
            $db['blogPosts'][$postIndex]['attachments'],
            fn($att) => !in_array($att['path'], $attachmentsToDelete)
        );
    }

    // Añadir nuevos adjuntos
    if (isset($_FILES['attachments'])) {
        $attachmentDir = 'uploads/blog/attachments/';
        $fileCount = count($_FILES['attachments']['name']);
        for ($i = 0; $i < $fileCount; $i++) {
            if ($_FILES['attachments']['error'][$i] === UPLOAD_ERR_OK) {
                $originalName = $_FILES['attachments']['name'][$i];
                $fileExtension = pathinfo($originalName, PATHINFO_EXTENSION);
                $fileName = 'attachment_' . uniqid() . '.' . $fileExtension;
                $filePath = $attachmentDir . $fileName;

                if (move_uploaded_file($_FILES['attachments']['tmp_name'][$i], $filePath)) {
                    $db['blogPosts'][$postIndex]['attachments'][] = [
                        'filename' => $originalName,
                        'path' => $filePath
                    ];
                }
            }
        }
    }

    saveDb($db);
    echo json_encode(['success' => true, 'message' => 'Artículo actualizado exitosamente.']);
}

function getDb() {
    $db_file = 'db.json';
    if (!file_exists($db_file)) die(json_encode(['success' => false, 'message' => 'DB error.']));
    return json_decode(file_get_contents($db_file), true);
}

function saveDb($data) {
    file_put_contents('db.json', json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE));
}

function checkAuth() {
    if (!isset($_SESSION['user_logged_in']) || $_SESSION['user_logged_in'] !== true) {
        http_response_code(401);
        echo json_encode(['success' => false, 'message' => 'No autorizado.']);
        exit;
    }
}