document.addEventListener('DOMContentLoaded', function () {

    // --- ELEMENTOS DEL DOM ---
    const projectNameTitle = document.getElementById('project-name-title');
    const mediaPlanListContainer = document.getElementById('media-plan-list-container');
    
    const mediaPlanModalEl = document.getElementById('mediaPlanModal');
    const mediaPlanModal = new bootstrap.Modal(mediaPlanModalEl);
    const saveMediaPlanBtn = document.getElementById('save-media-plan-btn');
    const planProfessionalsCheckboxes = document.getElementById('plan-professionals-checkboxes');

    // --- URLs Y ESTADO ---
    const urlParams = new URLSearchParams(window.location.search);
    const projectId = urlParams.get('project_id');

    const PROJECTS_API_URL = 'api/projects.php';
    const MEDIA_PLANS_API_URL = 'api/media_plans.php';
    const PROFESSIONALS_API_URL = 'api/professionals.php';

    const appState = {
        project: null,
        professionals: [],
        professionalsMap: new Map(),
        mediaPlans: [],
        isEditMode: false,
        currentPlanId: null
    };

    // --- FUNCIONES DE CARGA INICIAL ---

    async function initialLoad() {
        if (!projectId) {
            projectNameTitle.textContent = "Proyecto no especificado";
            return;
        }

        try {
            const [projectRes, plansRes, professionalsRes] = await Promise.all([
                fetch(`${PROJECTS_API_URL}?id=${projectId}`),
                fetch(`${MEDIA_PLANS_API_URL}?project_id=${projectId}`),
                fetch(PROFESSIONALS_API_URL)
            ]);

            if (!projectRes.ok) throw new Error('No se pudo cargar el proyecto.');
            if (!plansRes.ok) throw new Error('No se pudieron cargar los planes de medios.');
            if (!professionalsRes.ok) throw new Error('No se pudieron cargar los profesionales.');
            
            appState.project = await projectRes.json();
            appState.mediaPlans = await plansRes.json();
            appState.professionals = await professionalsRes.json();
            appState.professionalsMap = new Map(appState.professionals.map(p => [p.id, p.nombre_completo]));

            projectNameTitle.textContent = `Planes de Medios para: ${appState.project.nombre_proyecto}`;
            document.title = `Planes de Medios: ${appState.project.nombre_proyecto} - Platflow`;

            populateProfessionalsForNewPlan();
            renderMediaPlanList();

        } catch (error) {
            console.error("Error en la carga inicial:", error);
            projectNameTitle.textContent = "Error";
            mediaPlanListContainer.innerHTML = `<div class="alert alert-danger">${error.message}</div>`;
        }
    }
    
    function populateProfessionalsForNewPlan() {
        planProfessionalsCheckboxes.innerHTML = (appState.professionals.filter(p => p.estado === 'Activo') || []).map(prof => `
            <div class="form-check">
                <input class="form-check-input" type="checkbox" id="prof-plan-${prof.id}" value="${prof.id}">
                <label class="form-check-label" for="prof-plan-${prof.id}">${prof.nombre_completo} <small class="text-muted">(${prof.puesto})</small></label>
            </div>
        `).join('');
    }

    async function fetchMediaPlans() {
        try {
            const response = await fetch(`${MEDIA_PLANS_API_URL}?project_id=${projectId}`);
            if (!response.ok) throw new Error('Error al recargar los planes de medios.');
            appState.mediaPlans = await response.json();
            renderMediaPlanList();
        } catch (error) {
            console.error("Error:", error);
            mediaPlanListContainer.innerHTML = `<div class="alert alert-danger">${error.message}</div>`;
        }
    }

    async function saveMediaPlan() {
        const selectedProfessionals = Array.from(planProfessionalsCheckboxes.querySelectorAll('input:checked')).map(cb => parseInt(cb.value));

        const planData = {
            project_id: projectId,
            name: document.getElementById('plan-name').value,
            objective: document.getElementById('plan-objective').value,
            audience: document.getElementById('plan-audience').value,
            budget: document.getElementById('plan-budget').value,
            start_date: document.getElementById('plan-start-date').value,
            end_date: document.getElementById('plan-end-date').value,
            responsables: selectedProfessionals
        };

        if (!planData.name || !planData.start_date || !planData.end_date || !planData.budget) {
            alert('Nombre, fechas y presupuesto son obligatorios.');
            return;
        }

        const url = appState.isEditMode ? `${MEDIA_PLANS_API_URL}?id=${appState.currentPlanId}` : MEDIA_PLANS_API_URL;
        const method = appState.isEditMode ? 'PUT' : 'POST';

        try {
            const response = await fetch(url, { method, headers: { 'Content-Type': 'application/json' }, body: JSON.stringify(planData) });
            if (!response.ok) throw new Error(`Error al ${appState.isEditMode ? 'actualizar' : 'guardar'} el plan de medios.`);
            
            mediaPlanModal.hide();
            document.getElementById('media-plan-form').reset();
            fetchMediaPlans();
        } catch (error) {
            console.error(error);
            alert(error.message);
        }
    }

    async function deleteMediaPlan(planId) {
        if (!confirm('¿Estás seguro de que quieres eliminar este plan de medios y todas sus líneas? Esta acción no se puede deshacer.')) return;

        try {
            const response = await fetch(`${MEDIA_PLANS_API_URL}?id=${planId}`, { method: 'DELETE' });
            if (!response.ok) throw new Error('No se pudo eliminar el plan de medios.');
            fetchMediaPlans();
        } catch (error) {
            console.error(error);
            alert(error.message);
        }
    }

    function handlePlanEditClick(planId) {
        const plan = appState.mediaPlans.find(p => p.id === planId);
        if (!plan) return;

        appState.isEditMode = true;
        appState.currentPlanId = planId;

        document.getElementById('mediaPlanModalLabel').textContent = 'Editar Plan de Medios';
        document.getElementById('plan-name').value = plan.name;
        document.getElementById('plan-objective').value = plan.objective;
        document.getElementById('plan-audience').value = plan.audience;
        document.getElementById('plan-budget').value = plan.budget;
        document.getElementById('plan-start-date').value = plan.start_date;
        document.getElementById('plan-end-date').value = plan.end_date;

        // Marcar checkboxes de responsables
        planProfessionalsCheckboxes.querySelectorAll('input').forEach(cb => {
            cb.checked = (plan.responsables || []).includes(parseInt(cb.value));
        });

        mediaPlanModal.show();
    }

    function renderMediaPlanList() {
        mediaPlanListContainer.innerHTML = '';
        if (appState.mediaPlans.length === 0) {
            mediaPlanListContainer.innerHTML = '<p class="text-muted text-center col-12">No hay planes de medios creados para este proyecto. ¡Crea el primero!</p>';
            return;
        }

        const formatDate = (dateString) => {
            if (!dateString) return 'N/A';
            const [year, month, day] = dateString.split('-');
            if (!year || !month || !day) return dateString;
            return `${day}/${month}/${year}`;
        };

        const formatCurrency = (value) => new Intl.NumberFormat('es-PY', { style: 'currency', currency: 'PYG', maximumFractionDigits: 0 }).format(value);
        
        const getAssignedNames = (ids) => {
            if (!ids || ids.length === 0) return '<small class="text-muted">Sin asignar</small>';
            return ids.map(id => `<span class="badge bg-light text-dark me-1">${appState.professionalsMap.get(id) || 'N/A'}</span>`).join('');
        };

        appState.mediaPlans.forEach(plan => {
            const card = document.createElement('div');
            card.className = 'col-md-4 mb-4';
            card.innerHTML = `
                <div class="card h-100">
                    <div class="card-body d-flex flex-column">
                        <h5 class="card-title">${plan.name}</h5>
                        <h6 class="card-subtitle mb-2 text-primary">${formatCurrency(plan.budget)}</h6>
                        <p class="card-text text-muted small">
                            <i class="bi bi-calendar-range"></i> ${formatDate(plan.start_date)} al ${formatDate(plan.end_date)}
                        </p>
                        <div class="mb-2">
                            <small class="text-muted d-block">Responsables:</small>
                            ${getAssignedNames(plan.responsables)}
                        </div>
                        <div class="mt-auto d-flex justify-content-between">
                            <a href="media-plan-editor.html?plan_id=${plan.id}" class="btn btn-sm btn-primary">Abrir Editor</a>
                            <button class="btn btn-sm btn-outline-secondary edit-plan-btn" data-id="${plan.id}"><i class="bi bi-pencil"></i></button>
                            <button class="btn btn-sm btn-outline-danger delete-plan-btn" data-id="${plan.id}"><i class="bi bi-trash"></i></button>
                        </div>
                    </div>
                </div>
            `;
            mediaPlanListContainer.appendChild(card);
        });
    }

    // --- EVENT LISTENERS ---
    saveMediaPlanBtn.addEventListener('click', saveMediaPlan);

    document.getElementById('add-media-plan-btn').addEventListener('click', () => {
        appState.isEditMode = false;
        appState.currentPlanId = null;
        document.getElementById('mediaPlanModalLabel').textContent = 'Crear Nuevo Plan de Medios';
        document.getElementById('media-plan-form').reset();
    });

    mediaPlanListContainer.addEventListener('click', (e) => {
        const deleteBtn = e.target.closest('.delete-plan-btn');
        if (deleteBtn) {
            deleteMediaPlan(parseInt(deleteBtn.dataset.id));
        }
        const editBtn = e.target.closest('.edit-plan-btn');
        if (editBtn) {
            handlePlanEditClick(parseInt(editBtn.dataset.id));
        }
    });

    // --- CARGA INICIAL ---
    initialLoad();
});